/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: utils.c
 *
 * Description: Implementation of General utility APIs.
 */

#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <dirent.h>
#include <limits.h>
 
#include "aps.h"
#include "apsinternal.h"
#include "utils.h"
#include "cmdpipe.h"
#include "resultcode.h"

DEBUG_CHANNEL_DEFAULT(utils)

/* For GetCommandOutput() */
#define INITIAL_PATH_LENGTH 64
#define INITIAL_LINE_LENGTH 80

/* Special option for FindFile...() to use path as hint */
#define FINDFILEWITHHINTS_PATH (char *)-1

static Aps_Result FindFileWithHints(const char *fileName,
    const char *pathHint, char **filePath, mode_t access);
static Aps_Result InternalGetCommandOutput(
    const char *cliCommand, const char *cliArg,
    const char *cliCWD, int maxWait,
    char ***cmdOutput, int *outNumLines, int *cmdReturn);

/* --------------------------------------------------------------------
 * GetCommandOutput()
 *
 * Build an array from shell command output using TrackArray.
 *
 * Parameters: cliCommand - path of tool to be run [may be blank if in shell
 *                          search path]
 *             cliArgs  - command line arguments -- format string
 *                        NULL = none
 *             cliCWD   - path to use as the tool's current working directory
 *                        NULL = don't care / use current
 *             maxWait  - maximum time to wait for input before
 *                        forcefully killing the process (in seconds) or 0
 *                        to wait indefinitely
 *
 * Returns: (in param) cmdOutput   : command output array (NULL on fail)
 *                                   free with Aps_BufferRelease
 *          (in param) outNumLines : number of lines returned (0 on fail)
 *          (in param) cmdReturn   : command's return code (-1 if not run)
 *          (as result)            : APS extended error
 *
 * Notice, output contains terminating newlines after each line.
 * Strip these and other formatting chars using strstrip() / strtrim().
 */
Aps_Result GetCommandOutput(
    const char *cliCommand, /* shell command */
    const char *cliArg,     /* command arguments -- format string */
    const char *cliCWD,     /* current working dir for shell command */
    int   maxWait,          /* max time to wait for input */

    char ***cmdOutput,      /* returned, buffer */
    int    *outNumLines,    /* returned, number of lines */
    int    *cmdReturn,      /* returned, command's return code */

    ...)                    /* substitution arguments */
{
    va_list va;
    const char *argString;
    char argStorage[768]; /* pretty big! */
    Aps_Result result;

    /* prepare varargs list */
    va_start(va, cmdReturn);

    /* formatting needed? */
    if ((cliArg) && (strchr(cliArg, '%'))) {
        vsnprintf(argStorage, 768, cliArg, va);
        argString = argStorage;
    /* nope? */
    } else argString = cliArg;

    /* go run command */
    /* if we're not interested in output, we'll discard the values */
    if ((! cmdOutput)||(! outNumLines)||(! cmdReturn)) {
        char **cOut;
        int  cNum;
        int  cReturn;
        result = InternalGetCommandOutput(cliCommand, argString, cliCWD,
            maxWait, & cOut, & cNum, & cReturn);
        if (cOut) Aps_ReleaseBuffer(cOut);
    } else {
        result = InternalGetCommandOutput(cliCommand, argString, cliCWD,
            maxWait, cmdOutput, outNumLines, cmdReturn);
    }

    /* dispose of argument list */
    va_end(va);
    return result;
}

/* Internal function called by GetCommandOutput with preformatted
 * arguments, etc...
 */
static Aps_Result InternalGetCommandOutput(
    const char *cliCommand, /* shell command */
    const char *cliArg,     /* command arguments */
    const char *cliCWD,     /* current working dir for shell command */
    int   maxWait,          /* max time to wait for input */

    char ***cmdOutput,      /* returned, buffer */
    int    *outNumLines,    /* returned, number of lines */
    int    *cmdReturn)      /* returned, command's return code */
{
    FILE *pipeOutput = NULL;
    void *pipeHandle = NULL;
    Aps_Result errResult = APS_SUCCESS;
    char **outArray = NULL;
    char *oldPath;
    int   maxWaitQuit;

    /*** Setup return args in case of failure ***/
    *cmdOutput = NULL;
    *outNumLines = 0;
    *cmdReturn = -1;

    /*** Do some sanity checking ***/
    if (!(cliCommand && cmdOutput && (maxWait >= 0)))
        return APS_INVALID_PARAM;

    /* Set maxWaitQuit */
    /* ideally, the app should provide this value... */
    maxWaitQuit = 2; /* give the app 2 seconds to respond to SIGINT */

    /*** Create buffer ***/
    outArray = TrackArrayNew_PtrChar(NULL, 0);
    if (! outArray) return APS_OUT_OF_MEMORY;

    /*** Switch directories if need be ***/
    oldPath = NULL;
    if (cliCWD) {
        int pathSize = INITIAL_PATH_LENGTH;
        do {
            errno = 0;
            /* Try to allocate memory for path */
            if ( (oldPath = realloc(oldPath, pathSize)) ) {
                /* Try to get current directory
                 * if not long enough sets errno = ERANGE */
                if (getcwd(oldPath, pathSize)) {
                    pathSize *= 2; /* double size */
                }
            } else errno = ENOMEM; /* not set by some implementations */
        } while (errno == ERANGE);
        if (! errno) chdir(cliCWD); /* sets errno on failure */
        if (errno) {
            errResult = GetResultFromErrno();
            if (oldPath) free(oldPath);
            goto cleanup;
        }
    }

    /*** Start command ***/
    /* If we have an argument, must create new commandline */
    if (cliArg) {
        char *cmdLine;
        cmdLine = malloc(strlen(cliCommand) + strlen(cliArg) + 2);
        if (cmdLine) {
            sprintf(cmdLine, "%s %s", cliCommand, cliArg);
            pipeHandle = cmdpipeopen(cmdLine, NULL, &pipeOutput, &pipeOutput);
            if (! (pipeHandle && pipeOutput)) errResult = APS_SUBPROGRAM_FAILED;
	    free(cmdLine);
        } else errResult = APS_OUT_OF_MEMORY;
    } else {
    /* Otherwise, just use the one we got */
        pipeHandle = cmdpipeopen(cliCommand, NULL, &pipeOutput, &pipeOutput);
        if (! (pipeHandle && pipeOutput)) errResult = APS_SUBPROGRAM_FAILED;
    }

    /*** Switch directories back if need be ***/
    if (oldPath) {
        if (chdir(oldPath)) { /* failed ?!? what should we do? */ }
        free(oldPath);
    }
    if (pipeHandle && pipeOutput) {
    /*** Read command output until end / timeout, then close pipe ***/
        int     lineLen;
        char   *line, *linePos;
        fd_set  watch;
        struct  timeval tv;
        int     pipeFD = fileno(pipeOutput);

        /* Setup our watchdog timer (no hang) */
    	FD_ZERO(&watch);
        FD_SET(pipeFD, &watch);

        /* Read data in until finished, our buffer will
         * increase in size if not big enough to hold the entire
         * line. Does NOT assume lines are small.
         */
        lineLen = INITIAL_LINE_LENGTH;
        line = malloc(lineLen);
        if (line) {
            linePos = line;
            while (1) {
                char c;
                line[lineLen - 2] = 0; /* mark the end */
                if (maxWait) {
                    tv.tv_sec = maxWait;
                    tv.tv_usec = 0;
                    if (! select(pipeFD + 1, &watch, NULL, NULL, &tv)) {
                        /* no data available, so assume hung! bail out! */
                        errResult = APS_OPERATION_TIMEOUT;
                        break;
                    }
                }
                if (! fgets(linePos, lineLen - (linePos - line),
                    pipeOutput)) break;
                /* check last char of buffer */
                c = line[lineLen - 2];
                if ((c) && (c != '\n')) {
                    /* line was too long */
                    int   nLineLen = lineLen * 2;
                    char *nLine = realloc(line, nLineLen);
                    /* if we can't allocate, assume line ended here */
                    if (nLine) {
                        linePos = nLine + lineLen - 1;
                        lineLen = nLineLen;
                        line = nLine;
                        continue;
                    }
                }
                /* line fits, or couldn't be extended, add to buffer */
                {
                    char *cloneLine = (char *)TrackMemDupString(outArray,
                        line, 0);
                    if (! TrackArrayAddLast_PtrChar(& outArray, cloneLine)) {
                        errResult = APS_OUT_OF_MEMORY;
                        break;
                    }
                }
                (*outNumLines)++;
                linePos = line;
            }
            free(line);
        } else errResult = APS_OUT_OF_MEMORY;

        FD_ZERO(&watch); /* free resources */
    }

    /*** Close pipe and set return value ***/
    if (pipeHandle) {
        if (errResult != APS_OPERATION_TIMEOUT) {
            *cmdReturn = cmdpipeclose(pipeHandle, maxWait);
        } else *cmdReturn = -2;
        if (*cmdReturn == -2) { /* abort process */
            pid_t pid;
            pid = cmdpipegetpid(pipeHandle);
            kill(pid, SIGINT);
            *cmdReturn = cmdpipeclose(pipeHandle, maxWaitQuit);
            if (*cmdReturn == -2) {
                kill(pid, SIGKILL);
                *cmdReturn = cmdpipeclose(pipeHandle, 0);
            }
        }
    }

    /*** Perform other cleanup tasks ***/
cleanup:
    if (errResult != APS_SUCCESS) {
        if (outArray) TrackArrayDelete_PtrChar(outArray); /* kill our buffer */
        *outNumLines = 0; /* ensure consistent results */
    } else {
        *cmdOutput = outArray; /* no problems! return buffer */
    }
    return errResult;
}

/* --------------------------------------------------------------------
 * strstrip()
 *
 * Strip characters from a string
 *
 * Parameters: dest  - target string buffer (may equal src)
 *             src   - source string buffer
 *             chars - string containing chars to be stripped
 *
 * Dest must be at least as big as src.
 *
 * Returns: number of characters removed
 */
int strstrip(char *dest, const char *src, const char *chars) {
    int  numRemoved = 0;
    char c;

    while ( (c = *(src++)) ) {
        const char *test = chars;
        char  t;
        while ((t = *(test++)) && (t != c)); /* find char */
        if (t) { /* char was found */
            numRemoved++;
            continue;
        }
        *(dest++) = c;
    }
    *(dest++) = 0; /* terminate string */
    return numRemoved;
}

/* ---------------------------------------------------------------------
 * strtrim()
 *
 * Trim trailing characters from a string
 * Removes all but the last non-whitespace character in the string.
 * Also removes any returns or newlines.
 *
 * Parameters: dest  - target string buffer (may equal src)
 *             src   - source string buffer
 *
 * Dest must be at least as big as src.
 *
 * Returns: number of characters removed
 */
int strtrim(char *dest, const char *src) {
    int   numRemoved = 0;
    const char *temp = src;
    char  c;

    /* Works by locating next non-whitespace character and copying
       to that position; ends when none are found or end of string
       is reached */

    while ( (c = *(temp++)) ) {
        if ((c != ' ') && (c != '\t') &&  /* search for non-whitespace */
            (c != '\n') && (c != '\r')) { /* then copy to here */
            numRemoved = 0;  /* reset counter */
            do { *(dest++) = *(src++); } while (src != temp);
        } else numRemoved++; /* was skipped */
    }
    *(dest++) = 0;
    return numRemoved;
}


/* --------------------------------------------------------------------
 * strdelimit()
 *
 * Consolidate multiple token characters and replace with a single
 *   delimiting character.
 * Repeating token matches need not be of the same token (will match
 *   sequence of tokens and substitute a single delimiter).
 *
 * Parameters: dest  - target string buffer (may equal src)
 *             src   - source string buffer
 *             chars - string containing chars to be stripped
 *             dchar - replacement character (delimiter)
 *
 * Dest must be at least as big as src.
 *
 * Returns: number of delimiters inserted
 */
int strdelimit(char *dest, const char *src, const char *chars, char dchar) {
    int  numDelimit = 0;
    int  matchFlag = 0; /* set when the last character read matched */
    char c;

    while ( (c = *(src++)) ) {
        const char *test = chars;
        char  t;
        while ((t = *(test++)) && (t != c)); /* find char */
        if (t) { /* char was found */
            if (!matchFlag) {
                numDelimit++;
                *(dest++) = dchar;
                matchFlag = 1;
            }
            continue;
        }
        *(dest++) = c;
        matchFlag = 0;
    }
    *(dest++) = 0; /* terminate string */
    return numDelimit;
}

/* --------------------------------------------------------------------
 * strupdate()
 *
 * Update a target string by creating a copy of the source.  Frees the
 * old one unless it was NULL or "".
 *
 * A buffer containing only "" is NEVER FREED OR CREATED.
 *
 * Parameters: dest  - pointer to target string buffer
 *             src   - source string buffer
 *
 * Returns: new value of dest, or NULL on failure (old string freed
 *          anyways, but set to "" for safety)
 *
 * Remark: To free strings call this function with parameter NULL.
 */
char *strupdate(char **dest, const char *src)
{
    /* check for null pointer */
    ASSERT(dest);

    /* fall-through if same location */
    if (*dest != src) {
        /* if new non-empty string... */
        if ((src) && (*src)) {
            /* if old non-empty string... */
            if ((*dest) && (**dest)) {
                /* compare old strings */
                const char *dp = *dest, *sp = src;
                int len = 1;
                while ((*dp) && (*sp) && (*dp == *sp)) {
                    dp++; sp++; len++;
                }

                if (*dp != *sp) { /* different contents? */
                    if (*sp) { /* src is longer */
                        while (*(sp++)) len++;
                        if (! (*dest = realloc(*dest, len))) {
                            *dest = ""; /* couldn't allocate, set to "" */
                            return NULL;
                        }
                    }
                    /* copy new string */
                    strcpy(*dest, src);
                }
            } else {
                if (! (*dest = strdup(src))) {
                    *dest = ""; /* if can't copy set to "" */
                    return NULL;
                }
            }
        } else {
        /* if old non-empty string... */
        if ((*dest) && (**dest)) free(*dest);
        *dest = (src) ? "" : NULL;
        }
    }
    return *dest;
}

/* --------------------------------------------------------------------
 * strtotallen()
 *
 * Compute the total length of a string, including the 0 terminator.
 * Returns 0 if the passed string pointer was NULL.
 *
 * Parameters: str   - string buffer
 *
 * Returns: length of string + 1, unless str == NULL, then 0.
 */
size_t strtotallen(const char *str)
{
    size_t count = 0;
    if (str) {
        do { count++; } while ( *(str++) );
    }
    return count;
}

/* --------------------------------------------------------------------
 * strcpyupdate()
 *
 * Copy a string and increment the destination pointer past the 0
 * terminator.
 * For building compact lists of 0-terminated strings.
 *
 * Parameters: dest  - pointer to pointer to target string buffer
 *             src   - source string buffer, if NULL, returns NULL
 *
 * Returns: old dest pointer (before update), or NULL if src was NULL.
 */
char *strcpyupdate(char **dest, const char *src)
{
    char *oldDest;
    ASSERT(*dest);
    oldDest = *dest;
    if (src) {
        do { *((*dest)++) = *src; } while ( *(src++) );
    } else return NULL;
    return oldDest;
}

/* --------------------------------------------------------------------
 * strisdef()
 *
 * Determines if a string is defined by checking for "" or NULL and
 * returns 0 if so; non-zero if if not.
 *
 * Parameters: str  - string pointer to check
 *
 * Returns: 0 if the string pointer is NULL or the string has zero-length;
 *          non-zero if not.
 */
int strisdef(const char *str)
{
    return (str && *str);
}

/* --------------------------------------------------------------------
 * strupdatehostname()
 *
 * Copies the current local host name to the string following the same
 * conventions as strupdate().
 *
 * Parameters: str - pointer to pointer to string buffer
 */
char *strupdatehostname(char **str)
{
    int length = 32;
    /* Try to get hostname -- double length if insufficient */
    for (;;) {
        if (strisdef(*str)) free(*str);
        if (! (*str = malloc(length)) ) break;
        errno = 0;
        gethostname(*str, length);
        if (! errno) return *str;
        if (errno != EINVAL) break;
        length *= 2;
    }

    /* Trouble! */
    free(*str);
    return *str = "";
}

/* --------------------------------------------------------------------
 * strupdatejoin()
 *
 * Joins two new strings to an existing one.  The middle string is
 * intended to be used for inserting delimiters.
 *
 * Parameters: dest    - pointer to pointer to string buffer
 *             src1    - middle string -- omitted if dest is blank
 *             src2    - end string
 */
char *strupdatejoin(char **dest, const char *src1, const char *src2)
{
    /* if existing string, we must join */
    ASSERT(dest && src1 && src2);
    if (strisdef(*dest)) {
        char *temp = alloca(strlen(*dest) + strlen(src1) + strlen(src2));
        if (temp) sprintf(temp, "%s%s%s", *dest, src1, src2);
        return strupdate(dest, temp);
    }
    return strupdate(dest, src2);
}

/* --------------------------------------------------------------------
 * strcmpwild()
 *
 * Compare a string to a target using wildcards.
 *  '?' - matches single char
 *  '*' - matches anything (or nothing)
 *
 * Parameters: match  - string with wildcards
 *             target - target string
 *             icase  - ignore case flag (TRUE / FALSE)
 *
 * Returns   : 0 if matched, else non-zero.
 *
 * Not grep-like!!  Perhaps later.
 */
int strcmpwild(const char *match, const char *target, int icase)
{
    char c;
    /* loop through match string */
    while ( (c = *(match++)) ) {
        /** handle meta-characters **/
        if (c == '?') {
            /* match any single character */
            if (! *(target++)) return 1; /* must be some character */
        } else if (c == '*') {
            /* check all possible substrings [lazy] */
            /* only if chars left here */
            if (*match) {
                do {
                    if (strcmpwild(match, target, icase) == 0) return 0;
                } while (*(++target));
                return 1; /* failed to match any substring */
            /* else matches entire tail end of string */
            } else return 0;
        /** compare strings **/
        /* critical-path optimized */
        } else {
            /* try case-sensitive comparison first */
            char d = *(target++);
            if (c == d) continue;
            /* try case-insensitive comparison */
            if (! icase) return 1;
            c &= 0xdf;
            if (c != (d & 0xdf)) return 1;
            if ((c < 'A') || (c > 'Z')) return 1;
        }
    }
    return (*target != '\0'); /* must be done with string */
}

/* --------------------------------------------------------------------
 * strunquot()
 *
 * Unquote a string contained in any matching pair of characters.
 * Stray quoting chars are also removed unless quoted.
 *
 * Parameters: target - target string
 *             source - source string (may be same as target)
 *             quote  - quoting character pairs, each sequence of 2 chars
 *                      makes up a quoting pair, eg. "''\"\"<>()[]"
 *
 * Returns   : number of quoting pairs removed
 *
 * N.B. Only removes the outermost pair from a string. eg.
 *      [(abcd(efg)hijkl)] w/ "()[]" --> (abcd(efg)hijkl)
 *
 *      Most useful for unquoting strings in "" or ''.
 *
 * Future: concatenate quoted segments...
 */
int strunquot(char *target, const char *source, const char *quote)
{
    int l = strlen(source);
    if (l > 1) {
        char first = *source, last = source[l-1];
        char q;
        while ( (q = *(quote++)) ) {
            if (first == q) {
                if ( !(q = *(quote++)) ) break; /* check well-formedness */
                if (last == q) {
                    memmove(target, source + 1, l - 2);
                    target[l - 2] = '\0';
                    return 1;
                }
            } else quote++;
        }
    }
    if (target != source) memmove(target, source, l + 1);
    return 0;
}

/* ---------------------------------------------------------------------------
 * StrDupWithNull()
 *
 * Copies the contents of one string pointer to another, releasing any
 * existing string in the destination pointer, and treating NULL as a valid
 * value to be copied.
 *
 * Parameters: dest - The destination string pointer.
 *
 *             src  - The source string pointer.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result StrDupWithNull(char **dest, const char *src)
{
    /* Get rid of original string in destination, if it is non-NULL. */
    if (*dest != NULL) {
        free(*dest);
        *dest = NULL;
    }
    
    /* Copy new string to destination, if needed. */
    if (src != NULL) {
        *dest = malloc(strlen(src) + 1);
        if (*dest == NULL)
            return APS_OUT_OF_MEMORY;
        
        strcpy(*dest, src);
    }
    
    return APS_SUCCESS;
}

/* --------------------------------------------------------------------
 * FindProgram()
 *
 * Try to locate a program in command path and in special hint locations.
 *
 * Parameters: cmdName  - (usual) name of command that is sought
 *             pathHint - additional hints to storage locations (may be NULL)
 *                        (colon-separated)
 *
 * Returns: (in param) prgPath : newly allocated string containing
 *                                program path and filename, or NULL
 *                               on failure.
 *          (as result)        : APS extended error
 *
 * Default locations searched specified by APSCFG_PATH_DEFAULT_PGM_SEARCHDIR
 *
 * Path must be freed with free() when finished.
 * Eventually user may be able to specify aliases.
 * Returns APS_NOT_FOUND if not found.
 *         APS_ACCESS_DENIED if found but cannot be executed.
 */
Aps_Result FindProgram(const char *cmdName, const char *pathHint,
    char **prgPath)
{
    Aps_Result result = APS_NOT_FOUND;

    if (pathHint) result = FindFileWithHints(cmdName, pathHint, prgPath,
        S_IRUSR | S_IXUSR);
#ifdef APSCFG_PATH_DEFAULT_PGM_SEARCHDIR
    if (result != APS_SUCCESS)
        result = FindFileWithHints(cmdName,
            APSCFG_PATH_DEFAULT_PGM_SEARCHDIR, prgPath, S_IRUSR | S_IXUSR);
#endif
    if (result != APS_SUCCESS)
        result = FindFileWithHints(cmdName, FINDFILEWITHHINTS_PATH, prgPath,
        S_IRUSR | S_IXUSR);

    return result;
}

/* --------------------------------------------------------------------
 * FindFile()
 *
 * Try to locate a file in command path and in special hint locations.
 *
 * Parameters: fileName - (usual) name of file that is sought
 *             pathHint - additional hints to storage locations (may be NULL)
 *                        (colon-separated)
 *
 * Returns: (in param) prgPath : newly allocated string containing
 *                                program path and filename, or NULL
 *                               on failure.
 *          (as result)        : APS extended error
 *
 * Default locations searched specified by APSCFG_PATH_DEFAULT_FILE_SEARCHDIR
 *
 * Path must be freed with free() when finished.
 * Eventually user may be able to specify aliases.
 * Returns APS_NOT_FOUND if not found.
 */
Aps_Result FindFile(const char *fileName, const char *pathHint,
    char **filePath)
{
    Aps_Result result = APS_NOT_FOUND;

    if (pathHint) result = FindFileWithHints(fileName, pathHint, filePath,
        0);
#ifdef APSCFG_PATH_DEFAULT_FILE_SEARCHDIR
    if (result != APS_SUCCESS)
        result = FindFileWithHints(fileName,
            APSCFG_PATH_DEFAULT_FILE_SEARCHDIR, filePath, 0);
#endif
    if (result != APS_SUCCESS)
        result = FindFileWithHints(fileName, FINDFILEWITHHINTS_PATH, filePath,
        0);
    return result;
}

/* Find a file using hints */
static Aps_Result FindFileWithHints(const char *fileName,
    const char *pathHint, char **filePath, mode_t access)
{
    Aps_Result result   = APS_NOT_FOUND;
    int checkCurrentDir = 0;
    int filenameSize    = strlen(fileName);

    /** Search inside path and current dir? **/
    if (pathHint == FINDFILEWITHHINTS_PATH) {
        pathHint = getenv("PATH"); /* may return NULL */
        checkCurrentDir = 1; /* flag: search current dir */
    }

    /** Try all provided hints **/
    while ((checkCurrentDir) || (pathHint)) {
        struct stat info;
        const char *pathSource ;
        int         pathSize = 0;
        char       *path = NULL;

        if (checkCurrentDir) {
            checkCurrentDir = 0; /* do only once */
            pathSource = ".";
            pathSize = 1;
        } else if (pathHint[0]) {
            const char *colon;
            /* skip any extraneous colons */
            while (pathHint[0] == ':') pathHint++;
            pathSource = pathHint;

            /* find next colon */
            if ( (colon = strchr(pathHint, ':')) ) {
                pathSize = (colon - pathHint);
                pathHint = colon + 1;
            } else {
                /* no colon, use remainder */
                pathSize = strlen(pathHint);
                pathHint = NULL;
            }
        } else break; /* end! */

        if (pathSize == 0) break; /* nothing to do! */

        /* prepare buffer and build full path */
        path = alloca(pathSize + filenameSize + 2); /* place on stack */
        strncpy(path, pathSource, pathSize);   /* extract path */
        path[pathSize] = '/';
        strcpy(path + pathSize + 1, fileName); /* append filename */

        /* try to stat() file, if it exists */
        if (! stat(path, &info)) { /* else couldn't stat */
            if ((~ info.st_mode) & (access)) result = APS_ACCESS_DENIED;
            else {
                /* user has access!!
                 *
                 * Future obtain absolute path relative to root.
                 * This helps if we're using '~' to indicate the
                 * current user and setuid() is used.
                 * [if chroot is called for this process, we're still screwed!]
                 *
                 * For now, return what we've got.
                 */
                *filePath = malloc(strlen(path) + 1);
                if (*filePath) {
                    strcpy(*filePath, path);
                    result = APS_SUCCESS;
                } else result = APS_OUT_OF_MEMORY;
            }
            break; /* got it! */
        }
    }
    return result;
}


/* (Jeff, Jan 21)
 * I don't think this belongs in here... Perhaps utils-gs.c
 *   would be more appropriate for grouping Ghostscript-related
 *   common code.
 * I have updated it to use the new GetCommandOutput like the
 *   funcs in lprtransport.c.
 * Should be renamed to GetGSDeviceResolution as GetScriptOutput
 *   is not specific enough. (confusing)
 */

/*------------------------------------------------------------------------
 * GetScriptOutput()
 * Executes the print_config script and return gsdevice name and supported
 * resolution.
 *
 * Params :
 *          printerName  : name of the printer.
 *          ifFilterPath : if filter path from printcap entry.
 *          gsDevice     : buffer to store the gs device name.
 *          resolution   : buffer to store the supported resolution.
 */
int GetScriptOutput(const char* printerName, const char* ifFilterPath,
                    char** gsDevice, char** resolution )
{
    int commandResult = 0;
    Aps_Result result;
    char  **buffer;
    int     bufferNumLines;
    int i = 0;

    ASSERT(printerName && ifFilterPath);

    /* Preinitialize in case we can't read these values */
    *gsDevice = NULL;
    *resolution = NULL;

    /* Calling Script
     */
    result = GetCommandOutput( APSCFG_CONFIG_PATH_MODEL_PROBE_FILE,
        " %s  %s", NULL, APSCFG_CONFIG_TIMEOUT_MODEL_PROBE,
        &buffer, &bufferNumLines, &commandResult, ifFilterPath, printerName);
    if (result != APS_SUCCESS) {
        ERROR("Was unable to execute script %s, result code: %s",
            APSCFG_CONFIG_PATH_MODEL_PROBE_FILE, RESULTSTR(result));
    }

    TRACE("GetCommandOutput returned %d\n", commandResult);
    TRACE("Number of returned lines %d\n", bufferNumLines);

    /* Parsing the output produced by the script. */
    for(i = 0; i < bufferNumLines; i++)
    {
        /* Trim newlines out first */
        strstrip(buffer[i], buffer[i], "\n\r");
        TRACE("Line read: %s\n", buffer[i]);
        if (buffer[i][0]) { /* not a blank line */
            if( strncmp( buffer[i], "GSDEVICE", 8) == 0)
            {
                *gsDevice = realloc(*gsDevice, strlen(buffer[i])-8);
                if (*gsDevice != NULL)
                    strcpy(*gsDevice, &((buffer[i])[9]));
                continue;
            }
            if( strncmp( buffer[i], "RESOLUTION", 10) == 0)
            {
                *resolution = realloc(*resolution, strlen(buffer[i])-10);
                if (*resolution != NULL)
                    strcpy(*resolution, &((buffer[i])[11]));
                continue;
            }
        }
    }
    return 0;
}

/*------------------------------------------------------------------------
 * path_alloc()
 * 
 * Returns the buffer for path name.
 *
 * Params :
 *          size  : size of the buffer allocated for the path name
 *
 * Returns : 
 *          pointer to the path name buffer.
 */
char *path_alloc(int *size)
             /* returns allocated size, if nonnull */
{
    if (size) *size = APSCFG_MAX_PATH_SIZE;
    return malloc(APSCFG_MAX_PATH_SIZE);
}
 
/*------------------------------------------------------------------------
 * RemoveDirectory()
 * 
 * Checks the directory for the directories inside irecursively and remove 
 * it. 
 *
 * Params :
 *          directory  : 
 *          newDir
 *
 * Returns : 
 *          pointer to the path name buffer.
 */
int RemoveDirectory(char *directory, char *newDir)
{
    struct stat statbuf;
    struct dirent *dirp;
    DIR *dp;
    int size;
    char *newDirName;

    if (directory == NULL)
       return 0;
    /* Open the directory first */
    if ((dp = opendir(directory)) == NULL) {
        return 0;
    }
    /* Change the directory first
       if (chdir (directory) ){
       return 0;
       }
     */
    while ((dirp = readdir(dp)) != NULL) {
        if ((strcmp(dirp->d_name, ".") == 0) ||
            (strcmp(dirp->d_name, "..") == 0))
            continue;
 
        if ((newDirName = path_alloc(&size)) == NULL)
            return 0;
        /* Make the new Directory name */
        sprintf(newDirName, "%s/%s", directory, dirp->d_name);
        /* Check if the name passed is a directory */
        if (lstat(newDirName, &statbuf) < 0) {
            return 0;
        }
        if (S_ISDIR(statbuf.st_mode))
            RemoveDirectory(newDirName, directory);
        else {
            if (remove(newDirName)) {
                free(newDirName);
                return 0;
            }
            free(newDirName);
        }
    }
    /* Now the directory is empty so this directory now */
    if (remove(directory)) {
        free(directory);
        return 0;
    }
    closedir(dp);
    if (newDir == NULL)
        return 1;
 
    return 1;
}

/* --------------------------------------------------------------------
 * CopyFileSpecial()
 *
 * Efficiently copy a file and set its attributes and ownership.
 *
 * Parameters:  srcDir    - source dir or NULL (if spec'd '/' is appended)
 *              srcFile   - source file appended to dir
 *              destDir   - dest dir or NULL (if spec'd '/' is appended)
 *              destFile  - dest file appended to dir
 *              newProtection - dest file's new protection bits (mode_t)
 *              newOwner      - dest file's new owner (uid_t)
 *              newGroup      - dest file's new group (gid_t)
 */
Aps_Result CopyFileSpecial(
    const char *srcDir, const char *srcFile,
    const char *destDir, const char *destFile,
    mode_t newProtection, uid_t newOwner, gid_t newGroup)
{
    Aps_Result result = APS_SUCCESS;
    char *srcPath;
    char *destPath;
    int   srcFD, destFD;

    ASSERT(srcFile && destFile);

    /* Create source path */
    if (srcDir) {
        int size;
        srcPath = path_alloc(& size);
        if (! srcPath) return APS_OUT_OF_MEMORY;
        snprintf(srcPath, size, "%s/%s", srcDir, srcFile);
    } else srcPath = (char *)srcFile; /* cast's away const */

    /* Create dest path */
    if (destDir) {
        int size;
        destPath = path_alloc(& size);
        if (! destPath) {
            if (srcDir) free(srcPath);
            return APS_OUT_OF_MEMORY;
        }
        snprintf(destPath, size, "%s/%s", destDir, destFile);
    } else destPath = (char *)destFile; /* cast's away const */

    /* Open files & set protection / owner */
    errno = 0;
    if ( (srcFD = open(srcPath, O_RDONLY)) ) {
        errno = 0;
        if ( (destFD = open(destPath, O_CREAT | O_TRUNC | O_WRONLY,
            newProtection)) ) {
            char buf[APSCFG_COPY_BUFFER_SIZE];
            int  numRead;
            fchown(destFD, newOwner, newGroup);
            /* Copy files */
            while ( (numRead = read(srcFD, buf, sizeof(buf))) > 0) {
                write(destFD, buf, numRead);
            }
            close(destFD);
        } else result = GetResultFromErrno();
        close(srcFD);
    } else result = GetResultFromErrno();

    /* Done! */
    if (srcDir) free(srcPath);
    if (destDir) free(destPath);
    return result;
}

/* --------------------------------------------------------------------
 * CopyFileToEnd()
 *
 * Copies data from one file to another. Data is read from the current
 * position in the source file up to the end, and is written starting
 * at the current position in the destination file.
 *
 * Parameters: dest - A FILE * identifying the file to write to.
 *
 *             src  - A FILE * identifying the file to read from.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY on memory
 *             allocation failure.
 */
Aps_Result CopyFileToEnd(FILE *dest, FILE *src)
{
    char buffer[APSCFG_COPY_BUFFER_SIZE];
    int charsRead;

    /* Loop until we reach the end of the source file, copying chunks of data
     * from the source file to the destination file.
     */
    while(!feof(src)) {
        charsRead = fread(buffer, sizeof(char), sizeof(buffer) / sizeof(char),
                          src);
        if (charsRead == 0) {
            return GetResultFromErrno();
        }
        fwrite(buffer, sizeof(char), charsRead, dest);
    }

    return APS_SUCCESS;
}

/*********************************************
 *** REPLACE MISSING LIBRARY FUNCTIONALITY ***
 *********************************************/

#ifdef APSCFG_NEED_STRICMP
/* --------------------------------------------------------------------
 * stricmp()
 *
 * Compares two strings, like strcmp(), but ignoring case.
 *
 * Parameters: str1 - The first string to compare.
 *             str2 - The second string to compare.
 *
 * Returns: -1 if str1 comes before str2, 1 if str1 comes after str2,
 *          and 0 if the two strings are identical (ignoring case).
 */
int stricmp(const char *str1, const char *str2)
{
    while (*str1 && *str2) {
        /* Convert next character of each string to lower case. */
        char ch1 = tolower(*str1);
        char ch2 = tolower(*str2);

        if (ch1 < ch2) {
            return -1; /* String 1 comes before string 2. */
        } else if (ch1 > ch2) {
            return 1;  /* String 1 comes after string 2. */
        }

        /* Advance string pointers to next position in each string. */
        ++str1;
        ++str2;
    }
    if (!*str1 && !*str2) {
        return  0; /* Both strings are of equal length */
    } else if (!*str2) {
        return  1; /* String 1 is longer */
    } else {
        return -1; /* String 2 is longer */
    }
}
#endif /* APSCFG_NEED_STRICMP */

#ifdef APSCFG_NEED_STRNICMP
/* --------------------------------------------------------------------
 * strnicmp()
 *
 * Compares two strings, like strncmp(), but ignoring case.
 *
 * Parameters: str1 - The first string to compare.
 *             str2 - The second string to compare.
 *             num  - Max number of characters to compare.
 *
 * Returns: -1 if str1 comes before str2, 1 if str1 comes after str2,
 *          and 0 if the two strings are identical (ignoring case) after
 *          at least 'num' chars.
 */
int strnicmp(const char *str1, const char *str2, size_t num)
{
    while (*str1 && *str2 && (num--)) {
        /* Convert next character of each string to lower case. */
        char ch1 = tolower(*str1);
        char ch2 = tolower(*str2);

        if (ch1 < ch2) {
            return -1; /* String 1 comes before string 2. */
        } else if (ch1 > ch2) {
            return 1;  /* String 1 comes after string 2. */
        }

        /* Advance string pointers to next position in each string. */
        ++str1;
        ++str2;
    }
    if (!*str1 && !*str2) {
        return  0; /* Both strings are of equal length */
    } else if (!*str2) {
        return  1; /* String 1 is longer */
    } else {
        return -1; /* String 2 is longer */
    }
}
#endif /* APSCFG_NEED_STRNICMP */

#ifdef APSCFG_NEED_STRDUP
/* --------------------------------------------------------------------
 * strdup()
 *
 * Duplicate source string using malloc().  Result is an exactly copy
 * of the contents of the source string.
 *
 * Parameters: str  - String to duplicate.
 *
 * Returns:    Pointer to a new string containing copy of source, or
 *             NULL on failure.
 */
char *strdup(const char *str)
{
    size_t len = strlen(str) + 1;
    char *dest = malloc(len);
    if (dest) memcpy(dest, str, len);
    return dest;
}
#endif /* APSCFG_NEED_STRDUP */

