/*
 * Copyright 1998 Juniper Networks, Inc.
 * Copyright (c) 2000-2001, Stockholms Universitet
 * (Stockholm University, Stockholm Sweden)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the university nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: kpam.c,v 1.20 2002/09/19 14:33:37 lha Exp $ */

#include "config.h"

#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <krb5.h>

#define PAM_SM_AUTH

#include <security/pam_appl.h>
#include <security/pam_modules.h>

#include "kpam_opt.h"
#include "kpam_log.h"
#include "krb_stuff.h"

#define BUFSIZE 80

#define USER_PROMPT "user:"
#define PASSWD_PROMPT_TEMPL "%s's password:"

static int
parse_pname(char *buf, int bufsiz, const char *user,
    const char *instance, const char *realm)
{
	int n;
	if (instance != NULL && realm != NULL)
		n = snprintf(buf, bufsiz, "%s/%s@%s", user, instance, realm);
	else if (instance == NULL && realm != NULL)
		n = snprintf(buf, bufsiz, "%s@%s", user, realm);
	else if (instance != NULL && realm == NULL)
		n = snprintf(buf, bufsiz, "%s/%s", user, instance);
	else
		n = snprintf(buf, bufsiz, "%s", user);

	return n < bufsiz ? PAM_SUCCESS : PAM_BUF_ERR;
}

static int
pam_conv_pass(kpam_opt *kopt, pam_handle_t *pamh, MAYBE_CONST char *prompt)
{
	int retval;
	MAYBE_CONST void *item;
	const struct pam_conv *conv;
	struct pam_message msg;
	MAYBE_CONST struct pam_message *msgs[1];
	struct pam_response *resp;

	if ((retval = pam_get_item(pamh, PAM_CONV, &item)) !=
	    PAM_SUCCESS)
		return retval;
	conv = (const struct pam_conv *)item;
	msg.msg_style = PAM_PROMPT_ECHO_OFF;
	msg.msg = prompt;
	msgs[0] = &msg;
	if ((retval = conv->conv(1, msgs, &resp, conv->appdata_ptr)) !=
	    PAM_SUCCESS)
		return retval;
	if ((retval = pam_set_item(pamh, PAM_AUTHTOK, resp[0].resp)) !=
	    PAM_SUCCESS)
		return retval;
	memset(resp[0].resp, 0, strlen(resp[0].resp));
	free(resp[0].resp);
	free(resp);
	return PAM_SUCCESS;
}

int
pam_sm_authenticate(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	MAYBE_CONST char *user, *pass;
	char pname[BUFSIZE];
	char prompt[BUFSIZE];
	struct passwd *pwd;
	uid_t uid;
	kpam_opt kopt;
	krb_stuff kstuff;
	int n, ret;

	memset(&kstuff, 0, sizeof(kstuff));
	memset(&kopt, 0, sizeof(kopt));

	kopt.realm = NULL;

	ret = init_krb_stuff(&kstuff, &kopt);
	if (ret) {
	    kpam_syslog(&kopt, LOG_ALERT, "error inting krb stuff");
	    ret = PAM_SYSTEM_ERR;
	    goto out;
	}

	parse_options(&kopt, argc, argv);

	if (kopt.realm == NULL) {
		kpam_syslog(&kopt, LOG_ALERT, "aborting: realm is not set");
		ret = PAM_SYSTEM_ERR;
		goto out;
	}

	/*
	 * Begin with grabbing the username,
	 * control that it's a unix user
	 * and build the canonical kerberos principal.
	 */
	uid = getuid();
	
	D((&kopt, "service uid: %d", uid));

	ret = pam_get_user(pamh, &user, USER_PROMPT);
	if (ret != PAM_SUCCESS) {
		kpam_syslog(&kopt, LOG_CRIT, "%s", pam_strerror(pamh, ret));
		goto out;
	}
	if ((pwd = getpwnam(user)) == NULL) {
		D((&kopt, "unknown user: %s", user));
		ret = PAM_USER_UNKNOWN;
		goto out;
	}

	/*
	 * Different cases:
	 * nonroot -> root => "nonroot/root@REALM"
	 * root -> root => "root/root@REALM" or not allowed
	 * root -> user "user@REALM"
	 * user1 -> user2 "user2@REALM"
	 */
	if (strcmp("root", user) == 0) {
		struct passwd *pw_from;
		if ((uid == 0) && kopt.no_root) {
			D((&kopt, "no_root rule matched"));
			ret = PAM_AUTH_ERR;
			goto out;
		} else if ((pw_from = getpwuid(uid)) != NULL) {
			ret = parse_pname(pname, BUFSIZE,
			    pw_from->pw_name, kopt.root_inst, kopt.realm);
		} else {
			kpam_syslog(&kopt, LOG_CRIT,
			    "service uid[%d] is not a system account", uid);
			ret = PAM_SYSTEM_ERR;
			goto out;
		}
	} else
		ret = parse_pname(pname, BUFSIZE, user, NULL, kopt.realm);

	if (ret != PAM_SUCCESS) {
		kpam_syslog(&kopt, LOG_ERR, "buffer overflow");
		goto out;
	}

	/*
	 * Get kerberos password and try to authenticate the user
	 */
	if (kopt.try_first_pass || kopt.use_first_pass) {
		D((&kopt, "try_first_pass"));
		pam_get_item(pamh, PAM_AUTHTOK, (MAYBE_CONST void **)&pass);
		if (pass != NULL) {
			ret = krb_kauth(&kstuff, &kopt, pwd, pname, pass);
			if (ret != PAM_AUTH_ERR)
				goto out;
			else if ((ret == PAM_AUTH_ERR) && kopt.use_first_pass)
				goto out;
		} else
			D((&kopt, "no earlier pass set"));
	}

	n = snprintf(prompt, BUFSIZE, PASSWD_PROMPT_TEMPL, pname);
	if (n >= BUFSIZE) {
		ret = PAM_BUF_ERR;
		goto out;
	}

	D((&kopt, "read pass"));
	if ((ret = pam_conv_pass(&kopt, pamh, prompt)) != PAM_SUCCESS
	    || ((ret = pam_get_item(pamh, PAM_AUTHTOK, (MAYBE_CONST void **)&pass)) != PAM_SUCCESS))
		goto out;
	else
		ret = krb_kauth(&kstuff, &kopt, pwd, pname, pass);
	
	/*
	 * We always end up here for thorough cleanup ;)
	 */
out:
	if (ret == PAM_SUCCESS)
		kpam_syslog(&kopt, LOG_INFO, "kpam: success for user %s",
		   pname);
	else
    		kpam_syslog(&kopt, LOG_INFO, 
		    "kpam: failed with %d (krb error %d) for user %s",
		     ret, kstuff.auth_failed, pname);
	free_used_krb_stuff(&kstuff, &kopt);
	free_options_contents(&kopt);
	return ret;
}

int
pam_sm_setcred(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	return PAM_SUCCESS;
}
