/* mode: c; c-basic-offset: 8; */
/*
 * Copyright (c) 2000 - 2003, Stockholms Universitet
 * (Stockholm University, Stockholm Sweden)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the university nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: krb_stuff.c,v 1.21 2004/01/11 18:51:46 lha Exp $ */

#include "config.h"

#include <krb5.h>

#ifdef USE_KRB4
#include <krb.h>
#endif

#ifdef USE_AFS
#include <kafs.h>
#endif

#include <sys/types.h>
#include <pwd.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>

#include <security/pam_appl.h>
#include <security/pam_modules.h>

#include "kpam_opt.h"
#include "kpam_log.h"
#include "krb_stuff.h"

static int create_krb5_ccache(krb_stuff *kstuff, const kpam_opt *kopt,
    const char *pname);

static int create_krb5_creds(krb_stuff *kstuff, const kpam_opt *kopt,
    const char *passwd);

#ifdef USE_KRB4
static int create_krb4_tickets(krb_stuff *kstuff, const kpam_opt *kopt);
#endif


int
init_krb_stuff(krb_stuff *kstuff, kpam_opt *kopt)
{
	static char realm_string[400]; /* XXX */
	krb5_realm realm;
	krb5_error_code ret;

	kstuff->used = 0;
	kstuff->auth_failed = 0;

	if (krb5_init_context(&kstuff->context) == 0)
		kstuff->used |= KPAM_USED_CONTEXT;
	else
		return PAM_SYSTEM_ERR;

	/* try to get realm, if it failes, just ignore it, we'll catch
	 * it later */
	if (kopt->realm == NULL) {
		ret = krb5_get_default_realm(kstuff->context, &realm);
		if (ret || realm == NULL)
			return PAM_SUCCESS;

		strlcpy(realm_string, realm, sizeof(realm_string));
		free(realm);
		kopt->realm = realm_string;
	}

	return PAM_SUCCESS;
}

void
free_used_krb_stuff(krb_stuff *kstuff, const kpam_opt *kopt)
{
	if (kstuff->used & KPAM_USED_CREDS)
		krb5_free_creds_contents(kstuff->context, &kstuff->cred);
	if (kstuff->used & KPAM_USED_CCACHE)
		krb5_cc_close(kstuff->context, kstuff->ccache);
	if (kstuff->used & KPAM_USED_PRINCIPAL)
		krb5_free_principal(kstuff->context, kstuff->princ);
	if (kstuff->used & KPAM_USED_CONTEXT)
		krb5_free_context(kstuff->context);
#ifdef USE_KRB4
	if (kstuff->used & KPAM_USED_K4TMP)
		unlink(kstuff->k4tmp_file);
#endif
	if (kstuff->used & KPAM_USED_K5TMP)
		unlink(kstuff->k5tmp_file);
	
	kstuff->used = 0;
}

static int
create_krb5_ccache(krb_stuff *kstuff, const kpam_opt *kopt, const char *pname)
{
	if (krb5_parse_name(kstuff->context, pname, &kstuff->princ) == 0)
		kstuff->used |= KPAM_USED_PRINCIPAL;
	else
		return PAM_USER_UNKNOWN;

	if (krb5_cc_resolve(kstuff->context,
		kstuff->k5tmp_file, &kstuff->ccache) == 0) {
		kstuff->used |= KPAM_USED_CCACHE | KPAM_USED_K5TMP;
	} else
		return PAM_SYSTEM_ERR;

	return PAM_SUCCESS;
}

static int
create_krb5_creds(krb_stuff *kstuff, const kpam_opt *kopt, const char *pass)
{
	krb5_get_init_creds_opt opt;
	krb5_error_code ret;

	krb5_get_init_creds_opt_init(&opt);
	krb5_get_init_creds_opt_set_default_flags(kstuff->context, "kpam",
						  kstuff->princ->realm, &opt);

	if (kopt->forwardable)
		krb5_get_init_creds_opt_set_forwardable(&opt, 1);
	
	ret = krb5_get_init_creds_password(kstuff->context, &kstuff->cred,
	    kstuff->princ, pass, krb5_prompter_posix, NULL, 0, NULL,
	    &opt);
	if (ret != 0) {
		kstuff->auth_failed = ret;
		return PAM_AUTH_ERR;
	} else {
		kstuff->used |= KPAM_USED_CREDS;		

		ret = krb5_cc_initialize(kstuff->context, kstuff->ccache,
		    kstuff->cred.client);
	}

	if (ret == 0)
		ret = krb5_cc_store_cred(kstuff->context, kstuff->ccache,
		    &kstuff->cred);
	if (ret != 0)
		kstuff->auth_failed = ret;

	return ret == 0 ? PAM_SUCCESS : PAM_SYSTEM_ERR;
}

#ifdef USE_KRB4
static int
create_krb4_tickets(krb_stuff *kstuff, const kpam_opt *kopt)
{
	krb5_error_code ret;
	CREDENTIALS c;

	ret = krb524_convert_creds_kdc_ccache(kstuff->context, kstuff->ccache,
	    &(kstuff->cred), &c);
	if (ret == 0) {
		krb_set_tkt_string(kstuff->k4tmp_file);
		kstuff->used |= KPAM_USED_K4TMP;
		tf_setup(&c, c.pname, c.pinst);
		memset(&c, 0, sizeof (c));
		return PAM_SUCCESS;
	} else {
		kstuff->auth_failed = ret;
		return PAM_SYSTEM_ERR;
	}
}
#endif

#define TMP_TEMPL "/tmp/k%d_tmp_%u"
#define K5_TEMPL  "/tmp/krb5cc_%u"
#define K4_TEMPL  "/tmp/tkt%u"

static char *
get_default_cc_filename(const kpam_opt *kopt, const struct passwd *pwd)
{
    char *fn = NULL;

    asprintf(&fn, TMP_TEMPL, 5, (unsigned int)pwd->pw_uid);
    return fn;
}

#ifdef USE_AFS
void
kpam_create_afs_tokens(krb5_context context, 
		       const kpam_opt *kopt, 
		       const struct passwd *pwd)
{
	krb5_error_code ret;
	krb5_ccache id;
	char *fn;

	fn = getenv("KRB5CCNAME");
	if (fn)
		fn = strdup(fn);
	if (fn == NULL)
		fn = get_default_cc_filename(kopt, pwd);
	if (fn == NULL)
	    return;
	
	ret = krb5_cc_resolve(context,  fn, &id);
	if (ret) {
		free(fn);
		kpam_syslog(kopt, LOG_CRIT,
			    "afs_token: cc_resolve failed with %d", ret);
		return;
	}

	ret = krb5_afslog_uid_home(context, id, NULL, NULL, pwd->pw_uid, 
				   pwd->pw_dir);
	if (ret)
		kpam_syslog(kopt, LOG_CRIT, "afslog: failed with %d", ret);
	free(fn);
	krb5_cc_close(context, id);
}
#endif /* USE_AFS */

static int
file_trix(const char *name, uid_t uid, gid_t gid)
{
	int ret;

	if ((ret = chown(name, uid, gid)) != 0)
		unlink(name);

	return ret == 0 ? PAM_SUCCESS : PAM_SYSTEM_ERR;
}

void
kpam_copy_cc_to_new_name(pam_handle_t *pamh, const kpam_opt *kopt,
			 krb5_context context, struct passwd *pwd)
{
	krb5_ccache oldcc, newcc;
	krb5_error_code ret;
	char *fn;
	const char *const_fn;

	fn = get_default_cc_filename(kopt, pwd);
	if (fn == NULL) {
		kpam_syslog(kopt, LOG_CRIT,
			    "copy_cc: can't get default fn");
		return;
	}

	ret = krb5_cc_resolve(context, fn, &oldcc);
	if (ret) {
		free(fn);
		kpam_syslog(kopt, LOG_CRIT,
			    "copy_cc: error resolving new old cc name");
		return;
	}

	ret = krb5_cc_gen_new(context, &krb5_fcc_ops, &newcc);
	if (ret) {
		kpam_syslog(kopt, LOG_CRIT,
			    "copy_cc: error generating new cc");
		krb5_cc_close(context, oldcc);
		free(fn);
		return;
	}

	ret = krb5_cc_copy_cache(context, oldcc, newcc);
	if (ret) {
		kpam_syslog(kopt, LOG_CRIT,
			    "copy_cc: error generating new cc");
		krb5_cc_close(context, oldcc);
		krb5_cc_destroy(context, newcc);
		free(fn);
		return;
	}

	free(fn);

	const_fn = krb5_cc_get_name(context, newcc);
	if (const_fn) {
		char *penv;
		ret = file_trix(const_fn, pwd->pw_uid, pwd->pw_gid);
		asprintf(&penv, "KRB5CCNAME=FILE:%s", const_fn);
		pam_putenv(pamh, penv);
	}

	krb5_cc_close(context, newcc);
	krb5_cc_destroy(context, oldcc);
}


static int
install_krb_files(krb_stuff *kstuff, const kpam_opt *kopt,
    const struct passwd *pwd)
{
	int ret;

	if (kstuff->used & KPAM_USED_K5TMP) {
		kstuff->used &= ~KPAM_USED_K5TMP;
		ret = file_trix(kstuff->k5tmp_file, pwd->pw_uid, pwd->pw_gid);
		if (ret != PAM_SUCCESS)
			return ret;
	}

#ifdef USE_KRB4
	if (!kopt->no_krb4 && (kstuff->used & KPAM_USED_K4TMP)) {
		kstuff->used &= ~KPAM_USED_K4TMP;
		ret = file_trix(kstuff->k4tmp_file, pwd->pw_uid, pwd->pw_gid);
		if (ret != PAM_SUCCESS)
			return ret;
	}
#endif
	return PAM_SUCCESS;
}

int
krb_kauth(krb_stuff *kstuff, const kpam_opt *kopt,
    const struct passwd *pwd, const char *pname, const char *pass)
{
	int ret;
	pid_t pid = getpid();

	if (!kstuff->auth_failed != 0) {
		snprintf(kstuff->k5tmp_file, sizeof(kstuff->k5tmp_file),
			 TMP_TEMPL, 5, (unsigned int)pwd->pw_uid);
		ret = create_krb5_ccache(kstuff, kopt, pname);
		if (ret != PAM_SUCCESS)
			return ret;
	}

	ret = create_krb5_creds(kstuff, kopt, pass);
	if (ret != PAM_SUCCESS)
		return ret;

#ifdef USE_KRB4
	if (!kopt->no_krb4) {
		snprintf(kstuff->k4tmp_file, BUFSIZE, TMP_TEMPL, 4, 
			 (unsigned int)pid);
		if (create_krb4_tickets(kstuff, kopt) != PAM_SUCCESS)
			kpam_syslog(kopt, LOG_CRIT,
			    "Error getting KRB4 tickets for user: %s.",
			    pwd->pw_name);
	}
#endif

#ifdef USE_AFS
#if 0
	if (!kopt->no_afs) {
		create_afs_tokens(kstuff, kopt, pwd);
	}
#endif
#endif
	if ((ret = install_krb_files(kstuff, kopt, pwd)) != PAM_SUCCESS)
		return ret;

	return PAM_SUCCESS;
}
