#include <stdio.h>
#include <sys/time.h>
#include <stdlib.h>
#include <sipuri.h>
#include <sipuser.h>
#include <sipclient.h>
#include <sipmessage.h>
#include <siptransaction.h>
#include <sipcall.h>
#include <kconfig.h>
#include <kglobal.h>
#include <kphone/ksipregistrations.h>

SipCallMember::SipCallMember( SipCall *parent, const SipUri &uri )
{
	call = parent;
	memberuri = uri;
	contacturi = uri;
	local = 0;
	remote = 0;
	localExpiresTime = -1;
	status = Disconnected;
	parent->addMember( this );
	timer = new QTimer( this );
	connect( timer, SIGNAL( timeout() ), this, SLOT( call_timeout() ) );
}

SipCallMember::~SipCallMember( void )
{
	delete timer;
}

void SipCallMember::setContactUri( const SipUri &newcontact )
{
	contacturi = newcontact;
}

void SipCallMember::setUri( const SipUri &newuri )
{
	memberuri = newuri;
}

bool SipCallMember::requestSubscribe( int expiresTime, const QString &body, const MimeContentType &bodytype )
{
	if( expiresTime >= 0 ) {
		localExpiresTime = expiresTime;
	}
	if( body != QString::null ) {
		localsessiondesc = body;
		localsessiontype = bodytype;
	}
	if( status == AuthenticationRequired ||
			status == AuthenticationRequiredWithNewPassword ||
			status == AuthenticationRequired_Connected ) {
		QString u = call->getProxyUsername();
		QString p = call->getPassword();
		if( p.isEmpty() || status == AuthenticationRequiredWithNewPassword ) {
			status = SubscribeRequestedWithPassword;
			QString proxy = call->getSipProxy();
			SipUri localuri = call->localAddress();
			KSipAuthenticationRequest authreq( proxy, localuri.uri(), QString::null  );
			authreq.setUsername( u );
			authreq.setPassword( p );
			if( authreq.exec() ) {
				u = authreq.getUsername();
				p = authreq.getPassword();
				if( u.isEmpty() || p.isEmpty() ) {
					status = Disconnected;
					return false;
				}
				call->setPassword( p );
			} else {
				status = Disconnected;
				return false;
			}
		} else {  // if not AuthenticationRequiredWithNewPassword
			if (status == AuthenticationRequired ) {
				status = SubscribeRequestedWithPassword;
			} else {
				status = Connected;
			}
		}
		printf( "SipInvite: Authentication required\n" );
		if( authtype == ProxyDigestAuthenticationRequired ) {
			proxyauthresponse = Sip::getDigestResponse( u, p, "SUBSCRIBE", getContactUri().uri(), proxyauthstr );
		} else if( authtype == ProxyBasicAuthenticationRequired ) {
			proxyauthresponse = Sip::getBasicResponse( u, p );
		}
		printf( "SipInvite: Proxy Auth is '%s'\n", proxyauthresponse.latin1() );
		local = call->newRequest( this, Sip::SUBSCRIBE, localsessiondesc, localsessiontype,
			SipUri::null, proxyauthresponse, localExpiresTime );
	} else {
		status = SubscribeRequested;
		local = call->newRequest( this, Sip::SUBSCRIBE, body, bodytype, SipUri::null, QString::null, localExpiresTime );
	}
	if( localExpiresTime == 0 ) {
		contactUpdate( false );
		statusUpdated();
	} else if( localExpiresTime > 0 ) {
		timer->start( localExpiresTime * 900, TRUE );
	}
	if( local ) {
		connect( local, SIGNAL( statusUpdated() ), this, SLOT( localStatusUpdated() ) );
		statusUpdated();
		return true;
	} else {
		return false;
	}
}

bool SipCallMember::requestNotify( int expiresTime, const QString &body, const MimeContentType &bodytype )
{
	if( expiresTime >= 0 ) {
		localExpiresTime = expiresTime;
		if( localExpiresTime == 0 ) {
			localsessiondesc = QString::null;
			localsessiontype = QString::null;
		}
	} else if( expiresTime == -2 ) {
		localExpiresTime = expiresTime;
	}
	if( body != QString::null ) {
		localsessiondesc = body;
		localsessiontype = bodytype;
	}
	if( status == AuthenticationRequired ||
			status == AuthenticationRequiredWithNewPassword ||
			status == AuthenticationRequired_Connected ) {
		QString u = call->getProxyUsername();
		QString p = call->getPassword();
		if( p.isEmpty() || status == AuthenticationRequiredWithNewPassword ) {
			status = NotifyRequestedWithPassword;
			QString proxy = call->getSipProxy();
			SipUri localuri = call->localAddress();
			KSipAuthenticationRequest authreq( proxy, localuri.uri(), QString::null  );
			authreq.setUsername( u );
			authreq.setPassword( p );
			if( authreq.exec() ) {
				u = authreq.getUsername();
				p = authreq.getPassword();
				if( u.isEmpty() || p.isEmpty() ) {
					status = Disconnected;
					return false;
				}
				call->setPassword( p );
			} else {
				status = Disconnected;
				return false;
			}
		} else {  // if not AuthenticationRequiredWithNewPassword
			if (status == AuthenticationRequired ) {
				status = NotifyRequestedWithPassword;
			} else {
				status = Connected;
			}
		}
		printf( "SipInvite: Authentication required\n" );
		if( authtype == ProxyDigestAuthenticationRequired ) {
			proxyauthresponse = Sip::getDigestResponse( u, p, "NOTIFY", getContactUri().uri(), proxyauthstr );
		} else if( authtype == ProxyBasicAuthenticationRequired ) {
			proxyauthresponse = Sip::getBasicResponse( u, p );
		}
		printf( "SipInvite: Proxy Auth is '%s'\n", proxyauthresponse.latin1() );
		local = call->newRequest( this, Sip::NOTIFY, localsessiondesc, localsessiontype,
		SipUri::null, proxyauthresponse, localExpiresTime );
	} else {
		status = NotifyRequested;
		statusdesc = "Send Notify to remote end";
		local = call->newRequest( this, Sip::NOTIFY, body, bodytype, SipUri::null, QString::null, localExpiresTime );
	}
	if( local ) {
		connect( local, SIGNAL( statusUpdated() ), this, SLOT( localStatusUpdated() ) );
		statusUpdated();
		return true;
	} else {
		return false;
	}
}


bool SipCallMember::requestMessage( const QString &body, const MimeContentType &bodytype )
{
	if( body != QString::null ) {
		localsessiondesc = body;
		localsessiontype = bodytype;
	}
	if( status == AuthenticationRequired ||
			status == AuthenticationRequiredWithNewPassword ||
			status == AuthenticationRequired_Connected ) {
		QString u = call->getProxyUsername();
		QString p = call->getPassword();
		if( p.isEmpty() || status == AuthenticationRequiredWithNewPassword ) {
			status = MessageRequestedWithPassword;
			QString proxy = call->getSipProxy();
			SipUri localuri = call->localAddress();
			KSipAuthenticationRequest authreq( proxy, localuri.uri(), QString::null  );
			authreq.setUsername( u );
			authreq.setPassword( p );
			if( authreq.exec() ) {
				u = authreq.getUsername();
				p = authreq.getPassword();
				if( u.isEmpty() || p.isEmpty() ) {
					status = Disconnected;
					return false;
				}
				call->setPassword( p );
			} else {
				status = Disconnected;
				return false;
			}
		} else {  // if not AuthenticationRequiredWithNewPassword
			if (status == AuthenticationRequired ) {
				status = MessageRequestedWithPassword;
			} else {
				status = Connected;
			}
		}
		printf( "SipInvite: Authentication required\n" );
		if( authtype == ProxyDigestAuthenticationRequired ) {
			proxyauthresponse = Sip::getDigestResponse( u, p, "MESSAGE", getContactUri().uri(), proxyauthstr );
		} else if( authtype == ProxyBasicAuthenticationRequired ) {
			proxyauthresponse = Sip::getBasicResponse( u, p );
		}
		printf( "SipInvite: Proxy Auth is '%s'\n", proxyauthresponse.latin1() );
		local = call->newRequest( this, Sip::MESSAGE, localsessiondesc, localsessiontype,
		SipUri::null, proxyauthresponse );
	} else {
		if( status == Connected ) {
			statusdesc = "Requesting session update";
		} else {
			status = MessageRequested;
			statusdesc = "Requesting remote end to join session";
		}
		local = call->newRequest( this, Sip::MESSAGE, body, bodytype );
	}
	if( local ) {
		connect( local, SIGNAL( statusUpdated() ), this, SLOT( localStatusUpdated() ) );
		statusUpdated();
		return true;
	} else {
		return false;
	}
}

bool SipCallMember::requestInvite( const QString &body, const MimeContentType &bodytype )
{
	if( body != QString::null ) {
		localsessiondesc = body;
		localsessiontype = bodytype;
	}
	if( status == AuthenticationRequired ||
			status == AuthenticationRequiredWithNewPassword ||
			status == AuthenticationRequired_Connected ) {
		QString u = call->getProxyUsername();
		QString p = call->getPassword();
		if( p.isEmpty() || status == AuthenticationRequiredWithNewPassword ) {
			status = InviteRequestedWithPassword;
			QString proxy = call->getSipProxy();
			SipUri localuri = call->localAddress();
			KSipAuthenticationRequest authreq( proxy, localuri.uri(), QString::null  );
			authreq.setUsername( u );
			authreq.setPassword( p );
			if( authreq.exec() ) {
				u = authreq.getUsername();
				p = authreq.getPassword();
				if( u.isEmpty() || p.isEmpty() ) {
					status = Disconnected;
					return false;
				}
				call->setPassword( p );
			} else {
				status = Disconnected;
				return false;
			}
		} else {  // if not AuthenticationRequiredWithNewPassword
			if (status == AuthenticationRequired ) {
				status = InviteRequestedWithPassword;
			} else {
				status = Connected;
			}
		}
		printf( "SipInvite: Authentication required\n" );
		if( authtype == ProxyDigestAuthenticationRequired ) {
			proxyauthresponse = Sip::getDigestResponse( u, p, "INVITE", getContactUri().uri(), proxyauthstr );
		} else if( authtype == ProxyBasicAuthenticationRequired ) {
			proxyauthresponse = Sip::getBasicResponse( u, p );
		}
		printf( "SipInvite: Proxy Auth is '%s'\n", proxyauthresponse.latin1() );
		local = call->newRequest( this, Sip::INVITE, localsessiondesc, localsessiontype,
		SipUri::null, proxyauthresponse );
	} else {
		if( status == Connected ) {
			statusdesc = "Requesting session update";
		} else {
			status = InviteRequested;
			statusdesc = "Requesting remote end to join session";
		}
		local = call->newRequest( this, Sip::INVITE, body, bodytype );
	}
	if( local ) {
		connect( local, SIGNAL( statusUpdated() ), this, SLOT( localStatusUpdated() ) );
		statusUpdated();
		return true;
	} else {
		statusdesc = "Error, Check Request Uri";
		return false;
	}
}

void SipCallMember::requestDisconnect( const QString &body, const MimeContentType &bodytype )
{
	if( status == Disconnected ) return;
	if( body != QString::null ) {
		localsessiondesc = body;
		localsessiontype = bodytype;
	}
	if( status == AuthenticationRequired ) {
		status = Disconnecting;
		QString u = call->getProxyUsername();
		QString p = call->getPassword();
		if( p == QString::null ) {
			QString proxy = call->getSipProxy();
			SipUri localuri = call->localAddress();
			KSipAuthenticationRequest authreq( proxy, localuri.uri(), QString::null  );
			authreq.setUsername( u );
			authreq.setPassword( p );
			if( authreq.exec() ) {
				u = authreq.getUsername();
				p = authreq.getPassword();
				if( u.isEmpty() || p.isEmpty() ) {
					return;
				}
				call->setPassword(p);
			}
		}
		printf( "SipInvite: Authentication required\n" );
		if( authtype == ProxyDigestAuthenticationRequired ) {
			proxyauthresponse = Sip::getDigestResponse( u, p, "BYE", call->getLocaluri().uri(), proxyauthstr );
		} else if( authtype == ProxyBasicAuthenticationRequired ) {
			proxyauthresponse = Sip::getBasicResponse( u, p );
		}
		printf( "SipInvite: Proxy Auth is '%s'\n", proxyauthresponse.latin1() );
		local = call->newRequest( this, Sip::BYE, localsessiondesc, localsessiontype,
			SipUri::null, proxyauthresponse );
	} else {
		if( local ) {
			local->cancelRequest();
			disconnect( local, 0, this, 0 );
		}
		status = Disconnecting;
		statusdesc = "Disconnecting";
		local = call->newRequest( this, Sip::BYE, body, bodytype );
	}
	connect( local, SIGNAL( statusUpdated() ), this, SLOT( localStatusUpdated() ) );
	statusUpdated();
}

void SipCallMember::requestTransfer( const SipUri &transferto, const QString &body, const MimeContentType &bodytype )
{
	if( status == Disconnected ) return;
	if( local ) {
		local->cancelRequest();
		disconnect( local, 0, this, 0 );
	}
	status = Disconnecting;
	statusdesc = "Transfering";
	local = call->newRequest( this, Sip::BYE, body, bodytype, transferto );
	connect( local, SIGNAL( statusUpdated() ), this, SLOT( localStatusUpdated() ) );
	statusUpdated();
}

void SipCallMember::requestOptions( const QString &body, const MimeContentType &bodytype )
{
	if( local ) {
		local->cancelRequest();
		disconnect( local, 0, this, 0 );
	}
	statusdesc = "Querying options";
	local = call->newRequest( this, Sip::OPTIONS, body, bodytype );
	connect( local, SIGNAL( statusUpdated() ), this, SLOT( localStatusUpdated() ) );
	statusUpdated();
}

void SipCallMember::acceptInvite( const QString &body, const MimeContentType &bodytype )
{
	if( !remote ) return;
	if( body != QString::null ) {
		localsessiondesc = body;
		localsessiontype = bodytype;
	}
	remote->sendResponse( SipStatus( 200 ), body, bodytype );
	status = Connected;
	statusdesc = "Connected";
	statusUpdated();
}

void SipCallMember::notAcceptableHere( void )
{
	if( !remote ) return;
	remote->sendResponse( SipStatus( 488 ) );
	status = Disconnected;
	statusdesc = "Not Acceptable Here";
	statusUpdated();
}

void SipCallMember::declineInvite( const QString &body, const MimeContentType &bodytype )
{
	if( !remote ) return;
	remote->sendResponse( SipStatus( 603 ), body, bodytype );
	status = Disconnected;
	statusdesc = "Rejecting call invitation";
	statusUpdated();
}

void SipCallMember::localStatusUpdated( void )
{
	QString authstrtemp;
	MimeContentType mtype;
	SipStatus sipstatus;
	if( !local ) {
		printf( "SipCallMember: Received what was likely a retransmission, badly ignoring...\n" );
		return;
	}
	printf( "SipCallMember: localStatusUpdated: %d\n", local->getStatus().getCode() );
	mtype = local->getFinalContentType();
	if( mtype == MimeContentType( "application/sdp" ) ) {
		sessiondesc = local->getFinalMessageBody();
		localsessiontype = mtype;
	} else {
		recentbody = local->getFinalMessageBody();
		recentbodytype = mtype;
	}
	if( status == InviteRequested ||
			status == InviteRequestedWithPassword ||
			status == ReInviteRequested ||
			status == InviteEarlyDialog ) {
		if( local->wasCancelled() ) {
			status = Disconnected;
			statusdesc = "Request cancelled";
			local = 0;
		} else if( local->getStatus().getCode() < 200 ) {
			statusdesc = local->getStatus().getReasonPhrase();
			if( local->getStatus().getCode() != 100 ) {
				status = InviteEarlyDialog;
			}
		} else if( local->getStatus().getCode() < 300 ) {
			call->setCallStatus( SipCall::callInProgress );
			status = Connected;
			statusdesc = "Connected: " + local->getStatus().getReasonPhrase();
			local = 0;
		} else if( local->getStatus().getCode() < 400 ) {
			status = Redirected;
			statusdesc = "Redirected: " + local->getStatus().getReasonPhrase();
			redirectlist = local->getFinalContactList();
			local = 0;
		} else {
			if(local->getStatus().getCode() == 407){
				proxyauthstr = local->getFinalProxyAuthString();
				authstrtemp = proxyauthstr.lower();
				if( authstrtemp.contains( "digest" ) ) {
					authtype = ProxyDigestAuthenticationRequired;
				} else {
					authtype = ProxyBasicAuthenticationRequired;
				}
				if( status == InviteRequestedWithPassword ) {
					status = AuthenticationRequiredWithNewPassword;
				} else {
					status = AuthenticationRequired;
				}
				local = 0;
				requestInvite();
			} else {
				status = Disconnected;
				statusdesc = "Call Failed: " + local->getStatus().getReasonPhrase();
				local = 0;
			}
		}
	} else if( status == MessageRequested || status == MessageRequestedWithPassword ) {
		if( local->wasCancelled() ) {
			local = 0;
		} else if( local->getStatus().getCode() < 200 ) {
		} else if( local->getStatus().getCode() < 300 ) {
			local = 0;
		} else {
			if(local->getStatus().getCode() == 407){
				proxyauthstr = local->getFinalProxyAuthString();
				authstrtemp = proxyauthstr.lower();
				if( authstrtemp.contains( "digest" ) ) {
					authtype = ProxyDigestAuthenticationRequired;
				} else {
					authtype = ProxyBasicAuthenticationRequired;
				}
				if( status == MessageRequestedWithPassword ) {
					status = AuthenticationRequiredWithNewPassword;
				} else {
					status = AuthenticationRequired;
				}
				local = 0;
				requestMessage();
			} else {
				status = Disconnected;
				local = 0;
			}
		}
	} else if( status == SubscribeRequested || status == SubscribeRequestedWithPassword ) {
		if( local->wasCancelled() ) {
			status = Disconnected;
			local = 0;
		} else if( local->getStatus().getCode() < 200 ) {
		} else if( local->getStatus().getCode() < 300 ) {
			call->setCallStatus( SipCall::callInProgress );
			local = 0;
		} else if( local->getStatus().getCode() == 401) {
			proxyauthstr = local->getFinalWWWAuthString();
			authstrtemp = proxyauthstr.lower();
			if( authstrtemp.contains( "digest" ) ) {
				authtype = ProxyDigestAuthenticationRequired;
			} else {
				authtype = ProxyBasicAuthenticationRequired;
			}
			if( status == SubscribeRequestedWithPassword ) {
				status = AuthenticationRequiredWithNewPassword;
			} else {
				status = AuthenticationRequired;
			}
			local = 0;
			requestSubscribe();
		} else if( local->getStatus().getCode() == 407) {
			proxyauthstr = local->getFinalProxyAuthString();
			authstrtemp = proxyauthstr.lower();
			if( authstrtemp.contains( "digest" ) ) {
				authtype = ProxyDigestAuthenticationRequired;
			} else {
				authtype = ProxyBasicAuthenticationRequired;
			}
			if( status == NotifyRequestedWithPassword ) {
				status = AuthenticationRequiredWithNewPassword;
			} else {
				status = AuthenticationRequired;
			}
			local = 0;
			requestSubscribe();
		} else if( local->getStatus().getCode() < 500 ) {
			status = Disconnected;
			call->updateContact();
			local = 0;
		}
	} else if( status == NotifyRequested || status == NotifyRequestedWithPassword ) {
		if( local->wasCancelled() ) {
			status = Disconnected;
			local = 0;
		} else if( local->getStatus().getCode() < 200 ) {
		} else if( local->getStatus().getCode() < 300 ) {
			call->setCallStatus( SipCall::callInProgress );
			local = 0;
		} else {
			if( local->getStatus().getCode() == 407) {
				proxyauthstr = local->getFinalProxyAuthString();
				authstrtemp = proxyauthstr.lower();
				if( authstrtemp.contains( "digest" ) ) {
					authtype = ProxyDigestAuthenticationRequired;
				} else {
					authtype = ProxyBasicAuthenticationRequired;
				}
				if( status == NotifyRequestedWithPassword ) {
					status = AuthenticationRequiredWithNewPassword;
				} else {
					status = AuthenticationRequired;
				}
				local = 0;
				requestNotify();
			} else if( local->getStatus().getCode() == 401) {
				proxyauthstr = local->getFinalWWWAuthString();
				authstrtemp = proxyauthstr.lower();
				if( authstrtemp.contains( "digest" ) ) {
					authtype = ProxyDigestAuthenticationRequired;
				} else {
					authtype = ProxyBasicAuthenticationRequired;
				}
				if( status == SubscribeRequestedWithPassword ) {
					status = AuthenticationRequiredWithNewPassword;
				} else {
					status = AuthenticationRequired;
				}
				local = 0;
				requestNotify();
			} else if( local->getStatus().getCode() < 500 ) {
				status = Disconnected;
				call->setCallStatus( SipCall::callDead );
				local = 0;
			}
		}
	} else if( status == Connected ) {
		if( local->getStatus().getCode() > 200 ) {
			if( local->getStatus().getCode() == 407 ) {
				proxyauthstr = local->getFinalProxyAuthString();
				authstrtemp = proxyauthstr.lower();
				if( authstrtemp.contains( "digest" ) ) {
					authtype = ProxyDigestAuthenticationRequired;
				} else {
					authtype = ProxyBasicAuthenticationRequired;
				}
				status = AuthenticationRequired_Connected;
				local = 0;
 				requestInvite();
			} else {
				status = Disconnected;
				statusdesc = "Response: " + local->getStatus().getReasonPhrase();
				local = 0;
			}
		} else if( local->getStatus().getCode() == 200 ) {
			statusdesc = "Response: " + local->getStatus().getReasonPhrase();
			local = 0;
		}
	} else if( status == Disconnecting ) {
		if( local->getStatus().getCode() >= 200 ) {
			if( local->getStatus().getCode() == 407 ) {
				proxyauthstr = local->getFinalProxyAuthString();
				authstrtemp = proxyauthstr.lower();
				if( authstrtemp.contains( "digest" ) ) {
					authtype = ProxyDigestAuthenticationRequired;
				} else {
					authtype = ProxyBasicAuthenticationRequired;
				}
				status = AuthenticationRequired;
 				requestDisconnect();
			} else {
				status = Disconnected;
				statusdesc = "Response: " + local->getStatus().getReasonPhrase();
				local = 0;
			}
		}
	} else if( status == CancelPending ) {
		if( local->getStatus().getCode() == 200 ) {
			local = 0;
			requestDisconnect();
			status = Disconnected;
		} else if( local->getStatus().getCode() > 100 ) {
			local->cancelRequest();
			local = 0;
			status = Disconnected;
		}
	} else {
		if( local->getStatus().getCode() >= 200 ) {
			statusdesc = "Response: " + local->getStatus().getReasonPhrase();
			local = 0;
		} else {
			statusdesc = "Response: " + local->getStatus().getReasonPhrase();
		}
	}
	statusUpdated();
}

void SipCallMember::incomingTransaction( SipTransaction *newtrans )
{
	MimeContentType mtype;
	remote = newtrans;
	if( remote->getRequest()->getMethod() == Sip::INVITE ) {
		connect( remote, SIGNAL( statusUpdated() ), this, SLOT( remoteStatusUpdated() ) );
		if( status == Disconnected ) {
			status = RequestingInvite;
			statusdesc = "Invitation received";
			remote->sendResponse( SipStatus( 180 ) );
		} else {
			status = RequestingReInvite;
			statusdesc = "Session update requested";
		}
		mtype = remote->getRequestMessageContentType();
		if( mtype == MimeContentType( "application/sdp" ) ) {
			sessiontype = mtype;
			sessiondesc = remote->getRequestMessageBody();
		} else {
			recentbodytype = mtype;
			recentbody = remote->getRequestMessageBody();
		}
	} else if( remote->getRequest()->getMethod() == Sip::BYE ) {
		status = Disconnected;
		statusdesc = "Remote end disconnected";
		call->hideCallWidget();
	}
	statusUpdated();
}

void SipCallMember::remoteStatusUpdated( void )
{
	if( status == RequestingInvite ) {
		if( remote->wasCancelled() ) {
			status = Disconnected;
			statusdesc = "Request cancelled";
			statusUpdated();
		}
	}
}

void SipCallMember::cancelTransaction( void )
{
	if( local ) {
		if( status == InviteEarlyDialog ) {
			local->cancelRequest();
			status = Disconnected;
		} else {
			status = CancelPending;
		}
	}
}

QString SipCallMember::getSubject( void )
{
	return call->getSubject();
}

void SipCallMember::call_timeout()
{
	if( call->getCallType() == SipCall::outSubscribeCall ) {
		requestSubscribe( -1 );
	} else if( call->getCallType() == SipCall::inSubscribeCall ) {
		contactUpdate( false );
	};
}

void SipCallMember::contactUpdate( bool active, QString presence )
{
	call->setPresenceStatus( presence );
	if( active ) {
		call->setCallStatus( SipCall::callInProgress );
	} else {
		if( call->getCallStatus() != SipCall::callDead ) { 
			call->setCallStatus( SipCall::callUnconnected );
		}
		memberuri.setTag( QString::null );
	}
	statusUpdated();
}

void SipCallMember::timerStart( int time )
{
	timer->start( time );
}

//------------------------------
//           SipCall
//------------------------------

SipCall::SipCall( SipUser *local, const QString &id, SipCall::CallType ctype )
{
	callstatus = callUnconnected;
	tcpSocket = 0;
	if ( id == QString::null ) {
		callid = SipMessage::createCallId();
	} else {
		callid = id;
	}

	// Remember to nuke all members and transactions
	members.setAutoDelete( true );
	transactions.setAutoDelete( true );
	struct timeval tv;
	gettimeofday( &tv, NULL );
	srand( tv.tv_usec );
	lastseq = rand() % 8000;
	parent = local->parent();
	calltype = ctype;
	hasroute = false;
	localuri = local->getUri();
	localuri.generateTag();
	parent->addCall( this );
	presenceStatus = "";
	bodyMask = QString::null;
	contactstr = "";
}

SipCall::~SipCall( void )
{
	parent->deleteCall( this );
	if( (int)tcpSocket != 0 ) {
		delete tcpSocket;
	}
	transactions.clear();
	members.clear();
}

void SipCall::addMember( SipCallMember *newmember )
{
	members.append( newmember );
}

SipTransaction *SipCall::newRequest( SipCallMember *member, Sip::Method meth,
	const QString &body, const MimeContentType &bodytype, const SipUri &transferto,
	const QString &proxyauthentication, int expiresTime )
{
	SipTransaction *trans = new SipTransaction( lastseq++, member, this );
	if( trans->sendRequest( meth, body, bodytype, transferto, proxyauthentication, expiresTime ) ) {
		transactions.append( trans );

		// Audit the call
		auditCall();

		// Return the transaction object for tracking
		return trans;
	} else {
		delete trans;
		return 0;
	}
}

SipTransaction *SipCall::newRegister( const SipUri &registerserver, int expiresTime, const QString &authentication,
	const QString &proxyauthentication, const QString &qvalue, const QString &body, const MimeContentType &bodytype  )
{
	localuri.setTag( QString::null );
	SipTransaction *trans = new SipTransaction( lastseq++, new SipCallMember( this, localuri ), this );
	transactions.append( trans );
	trans->sendRegister( registerserver, expiresTime, authentication, proxyauthentication, body, bodytype, qvalue );
	return trans;
}

bool SipCall::sendRequest( SipMessage *reqmsg, bool contact, const SipUri &proxy )
{
	reqmsg->insertHeader( SipHeader::From, localuri.nameAddr() );
	reqmsg->insertHeader( SipHeader::Call_ID, callid );
	if( ( reqmsg->getMethod() == Sip::INVITE ) || ( reqmsg->getMethod() == Sip::MSG ) ) {
		reqmsg->insertHeader( SipHeader::Subject, getSubject() );
	}
	if( hasroute ) {
		if( route.getHead().uri().contains( ";lr" ) ) {
			reqmsg->insertHeader( SipHeader::Route, route.getUriList() );
		} else {
			reqmsg->setRequestUri( route.getHead() );
			SipUriList routewithouthead = route;
			routewithouthead.removeHead();
			reqmsg->insertHeader( SipHeader::Route, routewithouthead.getUriList() );
		}
	}
	if( parent->getExplicitProxyMode() ) {
		if( reqmsg->getMethod() == Sip::REGISTER || reqmsg->getMethod() == Sip::INVITE ||
				reqmsg->getMethod() == Sip::SUBSCRIBE || reqmsg->getMethod() == Sip::ACK ) {
			if( parent->isLooseRoute() ) {
				if( reqmsg->getMethod() == Sip::REGISTER ) {
					reqmsg->setRequestUri( SipUri( reqmsg->getHeaderData( SipHeader::To ) ).getRegisterUri() );
				}
				reqmsg->insertHeader( SipHeader::Route, parent->getExplicitProxyAddress() );
			} else if( parent->isStrictRoute() ) {
				if( reqmsg->getMethod() == Sip::REGISTER ) {
					reqmsg->insertHeader( SipHeader::Route, SipUri( reqmsg->getHeaderData( SipHeader::To ) ).getRouteUri() );
				} else {
					reqmsg->insertHeader( SipHeader::Route, "<" + reqmsg->getRequestUri().reqUri() + ">" );
				}
				reqmsg->setRequestUri( parent->getExplicitProxyAddress() );
			} else {
				if( reqmsg->getMethod() == Sip::REGISTER ) {
					reqmsg->setRequestUri( SipUri( reqmsg->getHeaderData( SipHeader::To ) ).getRegisterUri() );
				}
			}
		}
	}
	if( parent->isTcpSocket() ) {
		return parent->sendRequest( reqmsg, contact, proxy, &tcpSocket );
	} else {
		return parent->sendRequest( reqmsg, contact, proxy );
	}
}

void SipCall::sendResponse( SipMessage *responsemsg, bool contact )
{
	responsemsg->insertHeader( SipHeader::Call_ID, callid );
	responsemsg->insertHeader( SipHeader::To, localuri.nameAddr() );
	if( hasrecordroute ) {
		responsemsg->setRecordRoute( recordroute );
	}
	parent->sendResponse( responsemsg, contact );
}

void SipCall::sendRaw( SipMessage *msg )
{
	parent->sendRaw( msg );
}

SipCallMember *SipCall::incomingMessage( SipMessage *message )
{
	if( message->getType() == SipMessage::Request ) {
		return incomingRequest( message );
	} else if( message->getType() == SipMessage::Response ) {
		incomingResponse( message );
	} else {
		printf( "SipCall: Incoming message dropped (bad message type)\n" );
	}
	return 0;
}


void SipCall::incomingResponse( SipMessage *message )
{
	SipUri incominguri( message->getHeaderData( SipHeader::To ) );
	QString cseq = message->getHeaderData( SipHeader::CSeq );
	unsigned int seqnum = cseq.left( cseq.find( " " ) ).toUInt();
	printf( "SipCall: Incoming response\n" );
	SipTransaction *curtrans;
	for ( curtrans = transactions.first(); curtrans != 0; curtrans = transactions.next() ) {
		if ( ( incominguri == curtrans->getCallMember()->getUri() )
				&& ( seqnum == curtrans->getSeqNum() )
				&& ( curtrans->getDirection() == SipTransaction::LocalRequest ) ) {
			SipCallMember *member = getMember( incominguri );
			if( member == NULL ) {
				printf( "SipCall: Billy, you really messed something up\n" );
			} else if( message->getStatus().getCode() == 200 || message->getStatus().getCode() == 202 ) {
				printf( "SipCall: Checking for Contact and Record-Route\n" );

				// Update the Contact for this member
				if( message->getContactList().getListLength() > 0 ) {
					printf( "SipCall: Setting Contact for this Call Member\n" );
					member->setContactUri( message->getContactList().getHead() );
				}

				// Update the route
				if( message->getRecordRoute().getListLength() > 0 ) {
					hasroute = true;
					route = message->getRecordRoute();
					route.reverseList();
					if( !route.getHead().uri().contains( ";lr" ) ) {
						route.addToEnd( member->getContactUri() );
					}
				}
			}
			curtrans->incomingResponse( message );
			return;
		}
	}
	printf( "SipCall: Response dropped: No transaction found\n" );
}


SipCallMember *SipCall::incomingRequest( SipMessage *message )
{
	SipUri incominguri( message->getHeaderData( SipHeader::From ) );
	QString cseq = message->getHeaderData( SipHeader::CSeq );
	unsigned int seqnum = cseq.left( cseq.find( " " ) ).toUInt();
	SipTransaction *curtrans;
	printf( "SipCall: Incoming request\n" );

	// Update our identity if necessary
	SipUri touri( message->getHeaderData( SipHeader::To ) );
	if( touri != localuri ) {
		localuri = touri;
		if( !localuri.hasTag() ) {
			localuri.generateTag();
		}
	}

	if( ( message->getMethod() == Sip::ACK ) || ( message->getMethod() == Sip::CANCEL ) ) {
		for( curtrans = transactions.first(); curtrans != 0; curtrans = transactions.next() ) {
			if( ( incominguri == curtrans->getCallMember()->getUri() ) && ( seqnum == curtrans->getSeqNum() ) &&
					( curtrans->getDirection() == SipTransaction::RemoteRequest ) ) {
				curtrans->incomingRequest( message );
				return 0;
			}
		}
		printf( "SipCall: ACK/CANCEL recieved, but no matching transaction found\n" );
		return 0;
	}
	for( curtrans = transactions.first(); curtrans != 0; curtrans = transactions.next() ) {
		if( ( incominguri == curtrans->getCallMember()->getUri() ) && ( seqnum == curtrans->getSeqNum() ) &&
				( curtrans->getDirection() == SipTransaction::RemoteRequest ) ) {
			curtrans->incomingRequestRetransmission( message );
			return 0;
		}
	}

	// Find or create a member for this request
	SipCallMember *member = getMember( incominguri );
	if ( member == NULL ) {
		member = new SipCallMember( this, incominguri );
	}

	// Update the Contact for this member
	if( message->getContactList().getListLength() > 0 ) {
		member->setContactUri( message->getContactList().getHead() );
	}

	// Update the route
	if( message->getRecordRoute().getListLength() > 0 ) {
		hasrecordroute = true;
		recordroute = message->getRecordRoute();
		hasroute = true;
		route = recordroute;
		if( !route.getHead().uri().contains( ";lr" ) ) {
			route.addToEnd( member->getContactUri() );
		}
	}

	// Create a new transaction and process it
	printf( "SipCall: New transaction created\n" );
	SipTransaction *transaction = new SipTransaction( seqnum, member, this );
	transactions.append( transaction );
	transaction->incomingRequest( message );

	// Update member status based on this transaction
	member->incomingTransaction( transaction );
	return member;
}

void SipCall::setSubject( const QString& newsubject )
{
	subject = newsubject;
	subjectChanged();
}

SipCallMember *SipCall::getMember( const SipUri &uri )
{
	SipCallMember *m;
	for( m = members.first(); m != 0; m = members.next() ) {
		if ( uri == m->getUri() ) {
			return m;
		}
	}

	return NULL;
}

void SipCall::setCallType( CallType newtype )
{
	calltype = newtype;
	parent->callTypeUpdated();
}

void SipCall::auditCall( void )
{
	bool foundmemb = false;

	// If there are no active call members, set the call status to 'Dead'
	for( SipCallMember *memb = members.first(); memb != 0; memb = members.next() ) {
		if( ( memb->getStatus() != SipCallMember::Disconnecting ) &&
				( memb->getStatus() != SipCallMember::Disconnected ) ) {
			foundmemb = true;
		}
	}
	if( !foundmemb ) {
		callstatus = callDead;
		callStatusUpdated();
	}
}

QString SipCall::getProxyUsername( void )
{
	SipUser *u = parent->findUser( localuri );
	if( u != NULL ) {
		SipUri *uri = u->getMyUri();
		if( uri->hasProxyUsername() ) {
			return uri->getProxyUsername();
		} else {
			return QString::null;
		}
	} else {
		return QString::null;
	}
}

QString SipCall::getHostname( void )
{
	SipUser *u = parent->findUser( localuri );
	if( u != NULL ) {
		SipUri *uri = u->getMyUri();
		return uri->getHostname();
	} else {
		return QString::null;
	}
}

QString SipCall::getSipProxy( void )
{
	SipUser *u = parent->findUser( localuri );
	if( u != NULL ) {
		SipClient *c = u->parent();
		QString proxy = c->getSipProxy();
		if( !proxy.isEmpty() ) {
			return proxy;
		} else {
			if( c->getProxyMode() ) {
				return c->getSipProxySrv( u->getMyUri()->getHostname() );
			}
		}
	} 
	return QString::null;
}

void SipCall::setProxyUsername( QString newUsername )
{
	SipUser *u = parent->findUser( localuri );
	if( u != NULL ) {
		SipUri *uri = u->getMyUri();
		uri->setProxyUsername( newUsername );
	}
}

QString SipCall::getPassword( void )
{
	SipUser *u = parent->findUser( localuri );
	if( u != NULL ) {
		SipUri *uri = u->getMyUri();
		if( uri->hasPassword() ) {
			return uri->getPassword();
		} else {
			return QString::null;
		}
	} else {
		return QString::null;
	}
}

void SipCall::setPassword( QString newPassword )
{
	SipUser *u = parent->findUser( localuri );
	if( u != NULL ) {
		SipUri *uri = u->getMyUri();
		uri->setPassword( newPassword );
	}
}

void SipCall::hideCallWidget( void )
{
	parent->hideCallWidget( this );
}

void SipCall::updateContact( void )
{
	if( callstatus == callInProgress ) {
		callstatus = callDead;
		parent->updateSubscribes();
	}
}



#include "sipcall.moc"
