/*
 * This file is part of libdissipate, an open source SIP library.  Please see
 * http://www.div8.net/dissipate for more information.
 *
 * Copyright (c) 2000 Billy Biggs <bbiggs@div8.net>
 *
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 */

#ifndef SIPCLIENT_H_INCLUDED
#define SIPCLIENT_H_INCLUDED

#include <qobject.h>
#include <qstring.h>
#include <qlist.h>
#include <dissipate2/sipuri.h>
#include <dissipate2/udpmessagesocket.h>
#include <dissipate2/tcpmessagesocket.h>
#include <dissipate2/sipcall.h>
#include <dissipate2/sipuser.h>

class SipStatus;
class SipMessage;
class SipCallMember;
class MimeContentType;


// Stun

// define some basic types
typedef unsigned char  UInt8;
typedef unsigned short UInt16;
typedef unsigned int   UInt32;
typedef struct { unsigned char octet[16]; }  UInt128;

// define types for a stun message
const UInt16 BindRequestMsg          = 0x0001;
const UInt16 BindResponseMsg         = 0x0101;
const UInt16 BindErrorMsg            = 0x0111;
const UInt16 SharedSecretRequestMsg  = 0x0002;
const UInt16 SharedSecretResponseMsg = 0x0102;

/// define a structure to hold a stun address
const UInt8  IPv4Family = 0x01;
const UInt8  IPv6Family = 0x02;

// define  stun attribute
const UInt16 MappedAddress = 0x0001;

typedef struct
{
	UInt8 pad;
	UInt8 family;
	UInt16 port;
} StunAddrHdr;

typedef struct
{
	UInt16 type;
	UInt16 length;
} StunAtrHdr;

typedef struct
{
	UInt16 msgType;
	UInt16 msgLength;
	UInt128 id;
} StunMsgHdr;

typedef struct
{
	StunAddrHdr addrHdr;
	union {
		UInt32  v4addr;
		UInt128 v6addr;
	} addr;
} StunAddress;

typedef struct
{
	UInt16 type;
	UInt16 length;
	StunAddress address;
} StunAtrAddress;

typedef struct
{
	UInt16 type;
	UInt16 length;
	StunAddrHdr addrHdr;
	UInt32  v4addr;
} StunAtrAddress4;

// define the a simple request
typedef struct
{
	StunMsgHdr msgHdr;
} StunRequestSimple;



/**
 * @short	A Client for the Session Initiation Protocol (SIP).
 * @author	Billy Biggs <bbiggs@div8.net>
 *
 * SipClient is master class responsible for sending and receiving SIP
 * messages.
 * 
 * SipClient hosts two main lists: a list of active SIP calls
 * (SipCall), and a list of users of this class (SipUser).  It is also
 * responsible for the Contact URI in all messages.
 *
 */
class SipClient : public QObject
{ 
	Q_OBJECT

	friend class SipUser;
	friend class SipCall;
public:
	/**
	 * Creates a new SIP client with the specified explicit proxy.  This
	 * starts up the client listening on a UDP port.  By default, it tries
	 * to listen on port 5060, or whatever is specified in the DISSIPATE_PORT
	 * environment variable but if newListenport is not 0 then it specified listenport.
	 */
	SipClient( QObject *parent = 0, const char *name = 0, unsigned int newListenport = 0,
			bool newLooseRoute = true, bool newStrictRoute = true );

	/**
	 * Destructor for a SIP client.  This will delete all calls and all
	 * users of this client, and close the socket.
	 */
	~SipClient( void );

	/**
	 * This call does the select on all the client's listening sockets and
	 * performs any pending actions.  It should be called in some main loop
	 * somewhere.
	 *
	 * @param block	If set to true, this call will block until a message is
	 * received.
	 */
	void doSelect( bool block );

	void sendStunRequest( const QString uristr = QString::null );

	/**
	 * Returns the full contact URI for this client.  This is placed in all
	 * outgoing messages (except for when clearing registrations).
	 */
	const SipUri &getContactUri( void ) { return contacturi; }

	/**
	 * Turn on default user mode.  If incoming calls arrive at this client
	 * and there is no associated user matching the To line in the request,
	 * the default user will be used.
	 *
	 * Dissipate always uses a common contact URI for all registrations.
	 * This means that all incoming requests should have a request URI
	 * matching that of the announced Contact URI.  So, in order to check
	 * for a matching user, we match on the To header.
	 *
	 * When default user mode is off and a message is received when there
	 * is no user associted, the call is rejected with a 404 Not Found.
	 */
	void setDefaultUserMode( bool onoff );

	/**
	 * Returns true if we accept calls with a default user.
	 *
	 * @see setDefaultUserMode()
	 */
	bool getDefaultUserMode( void ) const { return defusermode; }

	/**
	 * Returns the current default user.
	 *
	 * @see setDefaultUserMode()
	 */
	SipUser *getDefaultUser( void ) const { return defuser; }

	/**
	 * Sets the current default user.
	 */
	void setDefaultUser( SipUser *user );

	/**
	 * Sets the URI to forward calls to if call forwarding is enabled.
	 */
	void setCallForwardUri( const SipUri &u );

	/**
	 * Returns the URI we are currently forwarding to.
	 */
	const SipUri &getCallForwardUri( void ) { return forwarduri; }

	/**
	 * Turns on or off call forwarding.
	 */
	void setCallForward( bool onoff );

	/**
	 * Returns the current state of call forwarding.
	 */
	bool getCallForward( void ) const { return fwmode; }

	/**
	 * Allows the user to set a message to be displayed when a client is
	 * forwarded.
	 */
	void setCallForwardMessage( const QString &newmessage );

	/**
	 * Returns our current forward message.
	 */
	QString getCallForwardMessage( void ) const { return fwbody; }

	/**
	 * Turns on or off busy mode.
	 */
	void setBusy( bool onoff );

	/**
	 * Returns the current state of busy mode.
	 */
	bool getBusy( void ) const { return busymode; }

	/**
	 * Allows the user to set a message to be displayed when the client
	 * discovers we are busy.
	 */
	void setBusyMessage( const QString &newmessage );

	/**
	 * Returns our current busy message.
	 */
	QString getBusyMessage( void ) const { return busybody; }

	/**
	 * Returns true if we are currently sending outgoing requests to the
	 * explicit proxy.
	 */
	bool getProxyMode( void ) { return useProxyDial; }

	/**
	 * Turn on explicit proxying.  When SipClient is acting in explicit
	 * proxy mode, all outgoing SIP messages will be sent to the proxy
	 * specified, regardless of what the Via dictates.
	 */
	void setExplicitProxyMode( bool eproxy );

	/**
	 * Returns true if we are currently sending outgoing requests to the
	 * explicit proxy.
	 */
	bool getExplicitProxyMode( void ) { return useExplicitProxy; }

	/**
	 * Specify the explicit proxy address for this client.  Address is in
	 * the format address[:port], defaulting to 5060 if the port is not
	 * specified.
	 */
	void setExplicitProxyAddress( const QString &newproxy );

	/**
	 * Returns the explicit proxy for this client.
	 */
	const QString &getExplicitProxyUri( void ) { return proxy; }

	/**
	 * Returns the explicit proxy address to route header.
	 */
	QString getExplicitProxyAddress( void );


	/**
	 * Sets our current max forwards.
	 */
	void setMaxForwards( int newmax );

	/**
	 * Returns our current max forwards.
	 */
	int getMaxForwards( void ) const { return maxforwards; }

	enum HideViaMode {
		DontHideVia,
		HideHop,
		HideRoute };

	/**
	 * Set Via hiding mode.
	 */
	void setHideViaMode( HideViaMode newmode );

	/**
	 * Returns the current Via hiding mode.
	 */
	HideViaMode getHideViaMode( void ) { return hidemode; }

	/**
	 * Returns an iterator to the list of active SIP calls.
	 */
	SipCallIterator getCallList( void ) const { return SipCallIterator( calls ); }

	/**
	 * Returns an iterator to the list of users of this client.
	 */
	SipUserIterator getUserList( void ) const { return SipUserIterator( users ); }

	/**
	 * Returns our User-Agent string (library version).
	 */
	static const QString getUserAgent( void );

	QString getAuthenticationUsername( void ) const { return authentication_username; }
	QString getAuthenticationPassword( void ) const { return authentication_password; }
	QString getSipProxy( void ) const { return proxy; }
	SipUser *findUser( SipUri uri );
	void updateIdentity( SipUser *user, QString newproxy);
	enum Socket { UDP, TCP };
	bool isTcpSocket( void ) const { return SocketMode == TCP; }
	void setSocketMode( Socket socket ) { SocketMode = socket; }
	QString getSipProxySrv( QString dname );
	bool isLooseRoute( void ) const { return looseRoute; }
	bool isStrictRoute( void ) const { return strictRoute; }
	void sendTestMessage( QString sendaddr, unsigned int port, QString msg );
	void setTest( bool on ) { testOn = on; }
	void setUseProxyDial( bool on ) { useProxyDial = on; }
	QString getNAPTR( QString strUri );
	QString getSRV( QString naptr );

private slots:
	void callMemberUpdated( void );

signals:
	/**
	 * Triggered when a new SIP call is created from an incoming message.
	 */
	void incomingCall( SipCall *, QString );

	void hideCallWidget( SipCall * );

	/**
	 * Triggered when a call is deleted or added to the internal list of
	 * calls, or when a call has its type changed.
	 */
	void callListUpdated( void );

	void incomingInstantMessage( SipMessage * );
	void incomingNotify( SipMessage * );
	void incomingSubscribe( SipCallMember *, bool );
	void incomingTestMessage();
	void updateSubscribes( void );

private:
	QList<SipUser> users;
	QList<SipCall> calls;

	QString authentication_username;
	QString authentication_password;
	Socket SocketMode;
	int clilen, newsockfd;
	struct sockaddr_in cli_addr;

	// Our proxy, if applicable.
	QString proxy;
	unsigned int proxyport;
	bool useProxyDial;
	bool useExplicitProxy;
	SipUri sipProxy;
	QString sipProxySrv;
	QString sipProxyName;

	SipUri contacturi;

	// Call forwarding
	bool fwmode;
	SipUri forwarduri;
	QString fwbody;

	// Busy
	bool busymode;
	QString busybody;

	// Max-forwards
	int maxforwards;

	// Via hide mode
	HideViaMode hidemode;

	// Default user mode
	bool defusermode;

	// The Default User
	SipUser *defuser;

	// No socket list (yet).
	UDPMessageSocket listener;
	TCPMessageSocket TCP_listener;

	// Log stuff to a file
	int loggerfd;

	// Audit pending messages (retransmissions).
	void auditPending( void );

	bool setupSocketStuff( unsigned int newListenport );
	void setupContactUri( void );
	void incomingMessage( int socketfd );
	void parseMessage( QString fullmessage );
	void sendQuickResponse( SipMessage *origmessage, const SipStatus &status,
		const QString &body = QString::null,
		const MimeContentType &bodytype = MimeContentType::null );

	void sendAck( SipMessage *origmessage );

	// These methods are for SipUser
	void addUser( SipUser *user );
	void deleteUser( SipUser *user );

	// These methods are for SipCall
	void addCall( SipCall *call );
	void deleteCall( SipCall *call );

	bool sendRequest( SipMessage *msg, bool contact = true,
		const SipUri &regProxy = SipUri::null,
		TCPMessageSocket **mss = 0 );

	void sendResponse( SipMessage *msg, bool contact = true );
	void sendRaw( SipMessage *msg );
	void callTypeUpdated( void );
	QString getResSearch( QString dname, int type, bool UDP );
	QString messageCID;
	QString subscribeCID;
	bool useStunProxy;
	SipUri stunProxy;
	QString messageCSeq;
	QString subscribeCSeq;
	bool looseRoute;
	bool strictRoute;

	bool testOn;

};

#endif // SIPCLIENT_H_INCLUDED
