#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <kconfig.h>
#include <kglobal.h>
#include <dspdiverter.h>
#include <dspfulldiverter.h>
#include <dspoutrtp.h>
#include <dspoutoss.h>
#include <dissipate2/sipprotocol.h>
#include <dissipate2/sipcall.h>
#include <artsc/artsc.h>
#include <callaudio.h>

CallAudio::CallAudio( void )
{
	input = 0;
	output = 0;
	curcall = 0;
	curmember = 0;
	useoss = true;
	setOSSFilename( QString::null );
	setVideoSW( QString::null );
	local.setIpAddress( "0.0.0.0" );
	KConfig *config = KGlobal::config();
	config->setGroup( "DSP" );
	numfrags = config->readNumEntry( "NumberOfFragment", 8 );
	size = config->readNumEntry( "SizeOfFragment", 7 );
	payload = config->readNumEntry( "SizeOfPayload", 80 );
	codec = codecILBC;
	rtpCodec = codecUnknown;
	videoCodec = codecUnknown;
	videoRtpCodec = codecUnknown;
	bodyMask = QString::null;
	useStun = false;
}

CallAudio::~CallAudio( void )
{
	if( input ) delete input;
	if( output ) delete output;
}

void CallAudio::useOSS( void )
{
	if( useoss == true ) return;
	useoss = true;
	renegotiateCall();
}

void CallAudio::setOSSFilename( const QString &devname )
{
	if( devname == QString::null ) {
		ossfilename = "/dev/dsp";
	} else {
		ossfilename = devname;
	}
}

void CallAudio::setVideoSW( const QString &sw )
{
	if( sw == QString::null ) {
		videoSW = "vic";
	} else {
		videoSW = sw;
	}
}

void CallAudio::sendToRemote( void )
{
	QString hostname = remote.getIpAddress();
	unsigned int portnum = remote.getPort();
	if( hostname == QString::null || portnum == 0 ) {
		printf( "CallAudio: SendToRemote called but we don't have a valid session description yet\n" );
		return;
	}
	if( remote.isOnHold() ) {
		printf( "CallAudio: Remote is currently putting us on hold, waiting patiently\n" );
		return;
	}
	printf( "CallAudio: Sending to remote site %s:%d\n", hostname.latin1(), portnum );
	if( output ) {
		stopSending();
	}
	DspOut *out;
	DspOut *in;
	if( useoss ) {
		printf( "CallAudio: Opening OSS device %s for Input \n", ossfilename.latin1() );
		DspOutRtp *outrtp = new DspOutRtp( getRtpCodec(), getRtpCodecNum(), hostname );
		outrtp->setPortNum( portnum );
		outrtp->openDevice( DspOut::WriteOnly );
		outrtp->setPayload( payload );
		DspOutOss *inaudio = new DspOutOss( ossfilename );
		inaudio->setSize( size );
		inaudio->setNumfrags( numfrags );
		inaudio->setSampleRate( 8000 );
		KConfig *config = KGlobal::config();
		config->setGroup( "General" );
		if( !inaudio->openDevice( DspOut::ReadOnly ) ) {
			printf( "** sendToRemote: openDevice Failed.\n" );
		}
		inaudio->readBuffer();
		out = outrtp;
		in = inaudio;
		printf( "CallAudio: Creating OSS->RTP Diverter\n" );
		output = new DspFullDiverter( in, out );
	}
	output->run();
	if( curcall->getCallType() == SipCall::videoCall ) {
		printf( "CallAudio: Opening SW for video input and output\n" );
		pidVideo = fork();
		if( !pidVideo ) {
			KConfig *config = KGlobal::config();
			config->setGroup( "Video" );
			QString videoSW = config->readEntry( "videoSW", "/usr/local/bin/vic" );
			QString videoCodec = getVideoRtpCodecName();
			QString SW = videoSW;
			if( SW.contains( "/" ) ) {
				SW = SW.mid( SW.findRev( '/' ) + 1 );
			}
			QString videoSWParam = hostname + "/" + QString::number( remote.getVideoPort() ) + "/";
			videoSWParam += videoCodec + "/16/" + QString::number( local.getVideoPort() );

			printf( "CallAudio: execlp( %s, %s, %s, 0)\n", videoSW.latin1(), SW.latin1(), videoSWParam.latin1() );

			execlp( videoSW.latin1(), SW.latin1(), videoSWParam.latin1(), 0 );

			printf("error executing " + videoSW + "\n");
			exit(1);
		}
	}
}

void CallAudio::stopSending( void )
{
	if( output ) delete output;
	output = 0;
	remote = SdpMessage::null;
}

SdpMessage CallAudio::listenForRtp( void )
{
	printf( "CallAudio: listening for incomming RTP\n" );
	if( input ) {
		stopListening();
	}
	DspOut *out;
	DspOut *in;
	local.setIpAddress( Sip::getLocalAddress() );
	local.setName( "The Funky Flow" );
	if( useoss ) {
		printf( "CallAudio: Opening OSS device %s for Output\n", ossfilename.latin1() );
		DspOutRtp *inrtp = new DspOutRtp( getRtpCodec(), getRtpCodecNum() );
		inrtp->setPayload( payload );
		if( useStun ) {
			inrtp->setStunSrv( stunSrv );
		}
		inrtp->openDevice( DspOut::ReadOnly );
		local.setPort( inrtp->getPortNum() );
		local.setVideoPort( inrtp->getVideoPortNum() );
		DspOutOss *outoss = new DspOutOss( ossfilename );
		outoss->setSize( size );
		outoss->setNumfrags( numfrags );
		outoss->setSampleRate( 8000 );
		if( !outoss->openDevice( DspOut::WriteOnly )){
			printf("** listenForRtp: openDevice Failed.\n");
		}
		in = inrtp;
		out = outoss;
		printf( "CallAudio: Creating RTP->OSS Diverter\n" );
		input = new DspDiverter( in, out );
	}
	input->run();

	return local;
}

void CallAudio::stopListening( void )
{
	if( input ) delete input;
	input = 0;
	local.setIpAddress( "0.0.0.0" );
	local.setName( "Idle" );
	local.setPort( 0 );
	local.setVideoPort( 0 );
}

void CallAudio::saveAudioSettings( void )
{
	KConfig *config = KGlobal::config();
	config->setGroup( "Audio" );
	config->writeEntry( "oss-filename", getOSSFilename() );
	config->sync();
}

void CallAudio::readAudioSettings( void )
{
	KConfig *config = KGlobal::config();
	config->setGroup( "Audio" );
	useOSS();
	setOSSFilename( config->readEntry( "oss-filename" ) );
}

void CallAudio::readVideoSettings( void )
{
	KConfig *config = KGlobal::config();
	config->setGroup( "Video" );
	setVideoSW( config->readEntry( "videoSW", "/usr/local/bin/vic"  ) );
}

void CallAudio::setCurrentCall( SipCall *newcall )
{
	curcall = newcall;
	statusUpdated();
}

void CallAudio::attachToCallMember( SipCallMember *newmember )
{
	if( curmember ) {
		detachAndHold();
	}
	curmember = newmember;
	connect( curmember, SIGNAL( statusUpdated() ), this, SLOT( memberStatusUpdated() ) );
	statusUpdated();
}

void CallAudio::detachAndHold( void )
{
	if( local.isOnHold() ) {
		printf( "CallAudio: Call already on hold\n" );
	} else {
		toggleOnHold();
	}
}

void CallAudio::toggleOnHold( void )
{
	if( local.isOnHold() ) {
		printf( "CallAudio: Resuming call\n" );
		curmember->requestInvite(
			listenForRtp().message( getRtpCodec(), getVideoRtpCodec(), getBodyMask() ),
			MimeContentType( "application/sdp" ) );
	} else {
		printf( "CallAudio: Putting call on hold\n" );
		local.setIpAddress( "0.0.0.0" );
		local.setName( "Whoa there dewd" );
		local.setPort( 0 );
		local.setVideoPort( 0 );
		curmember->requestInvite(
			local.message( getRtpCodec(), getVideoRtpCodec(), getBodyMask() ),
			MimeContentType( "application/sdp" ) );
		detachFromCall();
	}
}

bool CallAudio::checkCodec( SipCallMember *member )
{
	bool status = true;
	QString mstr = member->getSessionDescription();
	rtpCodec = codecUnknown;
	videoRtpCodec = codecUnknown;
	if( mstr.contains( "m=audio" ) ) {
		QString ilbc = "";
		QString m = mstr.mid( mstr.find( "m=audio" ) );
		m = m.left( m.find( "\n" ) );
		m = m.mid( m.find( "RTP/AVP" ) + 7 );
		m += ' ';
		if( mstr.lower().contains( "ilbc/8000" ) ) {
			ilbc = mstr.mid( mstr.lower().find( "ilbc/8000" ) - 7, 6 );
			if( ilbc.contains( ":" ) ) {
				ilbc = ilbc.mid( ilbc.find( ":" ) + 1 );
			}
			ilbc = ilbc.simplifyWhiteSpace();
		}
		int posULAW = m.find( " 0 " );
		int posGSM = m.find( " 3 " );
		int posILBC = m.lower().find( " " + ilbc + " " );
		if( posULAW < 0 ) posULAW = 101;
		if( posGSM < 0 ) posGSM = 101;
		if( posILBC < 0 ) posILBC = 101;
		if( posULAW < posGSM && posULAW < posILBC ) {
			rtpCodec = codecULAW;
			rtpCodecNum = 0;
		} else if( posGSM < posULAW && posGSM < posILBC ) {
			rtpCodec = codecGSM;
			rtpCodecNum = 3;
		} else if( posILBC < posULAW && posILBC < posGSM ) {
			rtpCodec = codecILBC;
			rtpCodecNum = ilbc.toInt();
		}
	}
	if( mstr.contains( "m=video" ) ) {
		QString ilbc = "";
		QString m = mstr.mid( mstr.find( "m=video" ) );
		m = m.left( m.find( "\n" ) );
		m = m.mid( m.find( "RTP/AVP" ) + 7 );
		m += ' ';
		int posH261 = m.find( " 31 " );
		int posH263 = m.find( " 34 " );
		if( posH261 < 0 ) posH261 = 101;
		if( posH263 < 0 ) posH263 = 101;
		if( posH261 < posH263 ) {
			videoRtpCodec = codecH261;
			videoRtpCodecNum = 31;
		} else if( posH263 < posH261 ) {
			videoRtpCodec = codecH263;
			videoRtpCodecNum = 34;
		}
	}

	if( rtpCodec == codecILBC ) {
		printf( "CallAudio: Using iLBC for output\n" );
	} else if( rtpCodec == codecGSM ) {
		printf( "CallAudio: Using GSM for output\n" );
	} else if( rtpCodec == codecULAW ) {
		printf( "CallAudio: Using G711u for output\n" );
	} else if( rtpCodec == codecUnknown ) {
		status = false;
	}
	if( rtpCodec != codecUnknown ) {
		if( videoRtpCodec == codecH261 ) {
			printf( "CallAudio: Using H261 for video output\n" );
		} else if( videoRtpCodec == codecH263 ) {
			printf( "CallAudio: Using H263 for video output\n" );
		} else {
			if( curcall ) {
				curcall->setCallType( SipCall::StandardCall ); 
			}
		}
	}
	
	return status;
}

void CallAudio::memberStatusUpdated( void )
{
	SdpMessage sdpm;
	SdpMessage rsdp;
	sdpm.parseInput( curmember->getSessionDescription() );
	if( checkCodec( curmember ) ) {
		if( curmember->getStatus() == SipCallMember::RequestingReInvite ) {
			if( sdpm.isOnHold() ) {
				rsdp.setName( "Accepting on hold" );
				rsdp.setIpAddress( "0.0.0.0" );
				rsdp.setPort( 0 );
				rsdp.setVideoPort( 0 );
				curmember->acceptInvite(
					rsdp.message( getRtpCodec(), getVideoRtpCodec(), getBodyMask() ),
					MimeContentType( "application/sdp" ) );
				stopListening();
				stopSending();
			} else {
				curmember->acceptInvite(
					listenForRtp().message( getRtpCodec(), getVideoRtpCodec(), getBodyMask() ),
					MimeContentType( "application/sdp" ) );
			}
		}
		if( sdpm != remote ) {
			remote = sdpm;
			if( !sdpm.isOnHold() ) {
				sendToRemote();
			}
			statusUpdated();
		}
	}
}

void CallAudio::detachFromCall( void )
{
	if( curmember ) {
		disconnect( curmember, 0, this, 0 );
	}
	curmember = 0;
	stopListening();
	stopSending();
	setCurrentCall( 0 );

	if( pidVideo ) {
		kill( pidVideo, SIGKILL );
		pidVideo = 0;
	}
}

bool CallAudio::isRemoteHold( void )
{
	return remote.isOnHold();
}

void CallAudio::renegotiateCall( void )
{
	if( !curcall ) return;
	stopListening();
	stopSending();
	curmember->requestInvite(
		listenForRtp().message( getRtpCodec(), getVideoRtpCodec(), getBodyMask() ),
		MimeContentType( "application/sdp" ) );
}


bool CallAudio::isAudioOn( void )
{
	return (input || output );
}

codecType CallAudio::getRtpCodec( void )
{
	if( rtpCodec != codecUnknown ) {
		return rtpCodec;
	} else {
		return codec;
	}
}

int CallAudio::getRtpCodecNum( void )
{
	int c;
	if( rtpCodec != codecUnknown ) {
		c = rtpCodecNum;
	} else {
		switch( codec ) {
			case codecGSM:
				c = 3;
				break;
			case codecILBC:
				c = 97;
				break;
			case codecULAW:
			default:
				c = 0;
				break;
		}
	}
	return c;
}

codecType CallAudio::getVideoRtpCodec( void )
{
	if( videoRtpCodec != codecUnknown ) {
		return videoRtpCodec;
	} else {
		if( curcall->getCallType() == SipCall::videoCall ) {
			return videoCodec;
		} else {
			return codecUnknown;
		}
	}
}

int CallAudio::getVideoRtpCodecNum( void )
{
	int c;
	if( curcall ) {
		if( curcall->getCallType() != SipCall::videoCall ) {
			return -1;
		}
	}
	if( videoRtpCodec != codecUnknown ) {
		c = videoRtpCodecNum;
	} else {
		switch( videoCodec ) {
			case codecH261:
				c = 31;
				break;
			case codecH263:
				c = 34;
				break;
			default:
				c = -1;
				break;
		}
	}
	return c;
}

QString CallAudio::getVideoRtpCodecName( void )
{
	QString c;
	if( curcall->getCallType() != SipCall::videoCall ) {
		return "";
	}
	if( videoRtpCodec != codecUnknown ) {
		switch( videoRtpCodec ) {
			case codecH263:
				c = "h263";
				break;
			default:
			case codecH261:
				c = "h261";
				break;
		}
	} else {
		switch( videoCodec ) {
			case codecH263:
				c = "h263";
				break;
			default:
			case codecH261:
				c = "h261";
				break;
		}
	}
	return c;
}

void CallAudio::setStunSrv( QString newStunSrv )
{
	useStun = true;
	stunSrv = newStunSrv;
}

#include "callaudio.moc"
