#ifndef DSPACTION_H_INCLUDED
#define DSPACTION_H_INCLUDED

class AudioBuffer;

/**
 * @short Base class for DSP Actions
 * @author Billy Biggs (vektor@div8.net)
 *
 * DspAction is the base class for actions to be executed by a
 * DspActionGenerator object. Each action implements a function to
 * produce playable audio buffers, and a function to indicate when it
 * is finished producing buffers.
 */
class DspAction
{
public:
	DspAction( void );
	virtual ~DspAction( void );

	/**
	 * This must return false until the last buffer has been
	 * grabbed using process().
	 */
	virtual bool completed( void ) = 0;

	/**
	 * This gives the caller the next audio buffer to be played.
	 */
	virtual AudioBuffer *process( void ) = 0;

	/**
	 * Check if the action is 'loopable'. If it is, then
	 * @ref setLoop (loop) can be used to have the action
	 * keep sending out buffers.
	 */
	bool isLoopable( void ) { return isloopable; }

	/**
	 * If an action is 'loopable', this sets its loopability.
	 * If @param loop is true, the action will loop until
	 * @ref #stop () is called.
	 */
	void setLoop( bool loop );

	/**
	 * If the action is in the repeat state (looping), it will
	 * stop after processing one more buffer.
	 */
	void stop( void );

protected:
	bool isloopable;
	bool done;
};

/**
 * A simple silence action.  Returns blankness until done.  Loopable.
 *
 * This action is very useful for timed pauses, but doesn't do this
 * intelligently (yet).  It will create buffers of the given size.
 */
class DspSilenceAction : public DspAction
{
public:
	/**
	 * Creates @param length milliseconds of silence at a
	 * sampling rate of @param samplerate.  Default samplerate
	 * is 22050.  Set @param loop to true to have the silence
	 * continue until stopped.
	 */
	DspSilenceAction( unsigned int length, bool loop = false,
				unsigned int samplerate = 22050 );

	/**
	 * Destructor for the Silence Action.
	 */
	~DspSilenceAction( void );

	/**
	 * Returns true if there are no more buffers available.
	 */
	bool completed( void );

	/**
	 * Returns the next buffer.  Currently returns the entire
	 * specified duration, with more available if looping is on.
	 */
	AudioBuffer *process( void );

private:
	unsigned int len;
};

/**
 * Simple DspAction to produce buffers of two pure sine wave tones added
 * together.
 */
class DspDualToneAction : public DspAction
{
public:
	/**
	 * Create a dual tone action of @param length milliseconds at a
	 * sampling rate of @param samplerate.  Volume is a scaling from
	 * 0 to 100.
	 */
	DspDualToneAction( unsigned int tone1, unsigned int tone2,
		unsigned int length = 50, bool repeat = false,
		unsigned int volume = 100, unsigned int samplerate = 22050 );

	/**
	 * Destructor for this action.
	 */
	~DspDualToneAction( void );

	/**
	 * Returns true if there are no more buffers available.
	 */
	bool completed( void );

	/**
	 * Returns the next buffer.  Currently returns the entire
	 * specified duration, with more available if looping is on.
	 */
	AudioBuffer *process( void );

private:
	bool initial;
	unsigned int len;
	unsigned int freq1;
	unsigned int freq2;
	unsigned int vol;
	unsigned int rate;
	unsigned int wave_index;
};

#endif // DSPACTION_H_INCLUDED
