#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <defines.h>
#include <kglobal.h>
#include <kconfig.h>
#include <dspoutoss.h>

DspOutOss::DspOutOss( const QString &fileName )
	: DspOut()
{
	filename = fileName;
	setDeviceName( "oss" );
	audio_fd = -1;
}

DspOutOss::~DspOutOss( void )
{
	close( audio_fd );
}

bool DspOutOss::openDevice( DeviceMode mode )
{
	int oflag = (mode == ReadOnly) ? O_RDONLY : O_WRONLY;
	if( devstate == DeviceOpened ) {
		lasterror = "Device Already Open";
		printf( "ERROR: %s\n", lasterror.ascii() );
		return false;
	}

	// Open immediately or else
	audio_fd = open( filename.ascii(), oflag | O_NONBLOCK );
	if( audio_fd == -1 ) {
		lasterror = "Open Failed";
		printf( "ERROR: %s\n", lasterror.ascii() );
		return false;
	}

	// Remove O_NONBLOCK
	int flags = fcntl( audio_fd, F_GETFL );
	flags &= ~O_NONBLOCK;
	fcntl( audio_fd, F_SETFL, flags );

	// keep fragsize less than 20ms !!
//	int frag = ( ( ( numfrags - 1 ) << 16 ) | size );
	int frag = ( ( 32767 << 16 ) | size );
	printf( "Frag bitmap: %x\n", (unsigned)frag);

	if( ioctl( audio_fd, SNDCTL_DSP_SETFRAGMENT, &frag ) ) {
		lasterror = QString( "SETFRAG" ) + QString( strerror( errno ) );
		printf( "ERROR: %s\n", lasterror.ascii() );
		return false;
	}

	int format = AFMT_S16_LE;

	if( ioctl( audio_fd, SNDCTL_DSP_SETFMT, &format ) == -1 ) {
		lasterror = QString( "SETFMT" ) + QString( strerror( errno ) );
		printf( "ERROR: %s\n", lasterror.ascii() );
		return false;
	}

	if( format != AFMT_S16_LE ) {
		lasterror = "Format not supported";
		printf( "ERROR: %s\n", lasterror.ascii() );
		return false;
	}

	int channels = 1;

	if( ioctl( audio_fd, SNDCTL_DSP_CHANNELS, &channels ) == -1 ) {
		lasterror = QString( "DSP_STEREO" ) + QString( strerror( errno ) );	
		printf( "ERROR: %s\n", lasterror.ascii() );
		return false;
	}

	if( channels != 1 ) {
		lasterror = "Unsupported Number of Channels";
		printf( "ERROR: %s\n", lasterror.ascii() );
		return false;
	}

	rate = 8000;

	if( ioctl( audio_fd, SNDCTL_DSP_SPEED, &rate ) == -1 ) {
		lasterror = QString( "DSP_SPEED" ) + QString( strerror( errno ) );
		printf( "ERROR: %s\n", lasterror.ascii() );
		return false;
	}

	audio_buf_info info;

	if( mode == ReadOnly ) {
		if( ioctl( audio_fd, SNDCTL_DSP_GETISPACE, &info ) == -1 ) {
			lasterror = QString( "GETISPACE" ) + QString( strerror( errno ) );
			printf( "ERROR: %s\n", lasterror.ascii() );
			return false;
		}
	} else {	
		if( ioctl( audio_fd, SNDCTL_DSP_GETOSPACE, &info ) == -1 ) {
			lasterror = QString( "GETOSPACE" ) + QString( strerror( errno ) );
			printf( "ERROR: %s\n", lasterror.ascii() );
			return false;
		}
	}

	totalfrags = info.fragments;

	audio_buf.resize( info.fragsize * sizeof( sample_t ) );
	output_buf.resize( info.fragsize * sizeof( short ) );

	printf( "DspOutOss: Fragments %d, fragstotal %d, fragsize %d\n", info.fragments, info.fragstotal, info.fragsize );

	lasterror = QString::null;
	devstate = DeviceOpened;
	return true;
}

bool DspOutOss::closeDevice( void )
{
	if( devstate != DeviceOpened ) {
		lasterror = "Device Not Open";
		return false;
	}
	close( audio_fd );
	devstate = DeviceClosed;
	return true;
}

unsigned int DspOutOss::writableFragments( void )
{
	struct timeval timeout;
	audio_buf_info info;
	fd_set write_fds;
	if( devstate != DeviceOpened ) {
		return 0;
	}
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
	FD_ZERO( &write_fds );
	FD_SET( audio_fd, &write_fds );
	if( select( audio_fd + 1, NULL, &write_fds, NULL, &timeout ) == -1 ) {
		return 0;
	}
	if( !FD_ISSET( audio_fd, &write_fds ) ) {
		return 0;
	}
	if( ioctl( audio_fd, SNDCTL_DSP_GETOSPACE, &info ) == -1 ) {
		lasterror = strerror( errno );
		return 0;
	}
	return info.fragments;
}

bool DspOutOss::writeBuffer( void )
{
	if( devstate != DeviceOpened ) {
		lasterror = "Device Not Open";
		return false;
	}
	sample_t *inbuf;
	short *outbuf;
	sample_t iterate;
	unsigned int i;

	inbuf = (sample_t *) audio_buf.getData();
	outbuf = (short *) output_buf.getData();

	for( i = 0; i < ( audio_buf.getSize() / sizeof( sample_t ) ); i++ ) {
		iterate = *inbuf;
		inbuf++;
		if( iterate > 32767.0 ) { iterate = 32767.0; }
		if( iterate < -32767.0 ) { iterate = -32767.0; }
		*outbuf = (short) iterate;
		outbuf++;
	}
	for(;;) {
		if( write( audio_fd, output_buf.getData(), output_buf.getSize() ) != -1 ) {
			break;
		}
	}
	return true;
}

bool DspOutOss::setSampleRate( int newrate )
{
	if( devstate == DeviceOpened ) {
		lasterror = "Can't reset rate while device open";
		return false;
	}
	rate = newrate;
	return true;
}

unsigned int DspOutOss::readableFragments( void )
{
	struct timeval timeout;
	audio_buf_info info;
	fd_set read_fds;

	if( devstate != DeviceOpened ) {
		return 0;
	}
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
	FD_ZERO( &read_fds );
	FD_SET( audio_fd, &read_fds );
	if( select( audio_fd + 1, &read_fds, NULL, NULL, &timeout ) == -1 ) {
		return 0;
	}
	if( !FD_ISSET( audio_fd, &read_fds ) ) {
		return 0;
	}
	if( ioctl( audio_fd, SNDCTL_DSP_GETISPACE, &info ) == -1 ) {
		lasterror = strerror( errno );
		return 0;
	}
	return info.fragments;
}

bool DspOutOss::readBuffer( void )
{
	if( devstate != DeviceOpened ) {
		lasterror = "Device Not Open";
		return false;
	}
	sample_t *outbuf;
	short *inbuf;
	unsigned int i;

	outbuf = (sample_t *) audio_buf.getData();
	read( audio_fd, output_buf.getData(), output_buf.getSize() );
	inbuf = (short *) output_buf.getData();
	for( i = 0; i < ( output_buf.getSize() / sizeof( short ) ); i++ ) {
		*outbuf = (sample_t) (*inbuf);
		outbuf++;
		inbuf++;
	}
	return true;
}

