#include <stdio.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qlabel.h>
#include <qframe.h>
#include <qlineedit.h>
#include <kmessagebox.h>
#include <qhbuttongroup.h>
#include <qradiobutton.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <dissipate2/sipcall.h>
#include <dissipate2/sipuser.h>
#include <dissipate2/sipregister.h>
#include <dissipate2/sipclient.h>
#include <kphone.h>
#include <kphoneview.h>
#include <ksipregistrations.h>
#include <kseparator.h>
#include <kconfig.h>
#include <kglobal.h>


KSipIdentityEdit::KSipIdentityEdit( QWidget *parent, const char *name, QObject *reg )
	: KDialogBase( parent, name, false, QString( name ) + "Identity Editor", Ok|Cancel, Ok, true )
{
	QVBox *vbox = makeVBoxMainWidget();
	(void) new QLabel( "Full Name:", vbox );
	fullname = new QLineEdit( vbox );
	fullname->setMinimumWidth( fontMetrics().maxWidth() * 20 );
	(void) new QLabel( "User Part of SIP URL:", vbox );
	username = new QLineEdit( vbox );
	(void) new QLabel( "Host Part of SIP URL:", vbox );
	hostname = new QLineEdit( vbox );
	(void) new QLabel( "Outbound Proxy (optional):", vbox );
	sipProxy = new QLineEdit( vbox );
	(void) new QLabel( "Authentication Username (optional):", vbox );
	sipProxyUsername = new QLineEdit( vbox );
	(void) new QLabel( "q-value between 0.0-1.0 (optional):", vbox );
	qValue = new QLineEdit( vbox );
	autoRegister = new QCheckBox( "Auto Register", vbox );
	labelRegister = new QLabel( "", vbox );
	buttonRegister = new QPushButton( "Register", vbox );
	connect( buttonRegister, SIGNAL( clicked() ), reg, SLOT( changeRegistration() ) );
}

KSipIdentityEdit::~KSipIdentityEdit( void )
{
}

void KSipIdentityEdit::updateState( QString state )
{
	if( state == "off" ) {
		buttonRegister->setText( "Register" );
		buttonRegister->setEnabled( false );
	} else if( state == "Active" ) {
		labelRegister->setText( "Registration : " + state );
		buttonRegister->setText( "Unregister" );
		buttonRegister->setEnabled( true );
	} else {
		labelRegister->setText( "Registration : " + state );
		buttonRegister->setText( "Register" );
		buttonRegister->setEnabled( true );
	}
}

QString KSipIdentityEdit::getFullname( void ) const
{
	return fullname->text();
}

QString KSipIdentityEdit::getUsername( void ) const
{
	return username->text();
}

QString KSipIdentityEdit::getHostname( void ) const
{
	return hostname->text();
}

QString KSipIdentityEdit::getSipProxy( void ) const
{
	return sipProxy->text();
}

QString KSipIdentityEdit::getSipProxyUsername( void ) const
{
	return sipProxyUsername->text();
}

QString KSipIdentityEdit::getQvalue( void ) const
{
	bool ok;
	float f = qValue->text().toFloat( &ok );
	if( ok && f >= 0 && f <= 1 ) {
		return qValue->text();
	}
	return "";
}

QString KSipIdentityEdit::getUri( void ) const
{
	return "\"" + fullname->text() + "\" <sip:" + username->text() + "@" + hostname->text() + ">";
}

void KSipIdentityEdit::setFullname( const QString &newFullname )
{
	fullname->setText( newFullname );
}

void KSipIdentityEdit::setUsername( const QString &newUsername )
{
	username->setText( newUsername );
}

void KSipIdentityEdit::setHostname( const QString &newHostname )
{
	hostname->setText( newHostname );
}

void KSipIdentityEdit::setSipProxy( const QString &newSipProxy )
{
	sipProxy->setText( newSipProxy );
}

void KSipIdentityEdit::setQvalue( const QString &qvalue )
{
	qValue->setText( qvalue );
}

void KSipIdentityEdit::setSipProxyUsername( const QString &newSipProxyUsername )
{
	sipProxyUsername->setText( newSipProxyUsername );
}

void KSipIdentityEdit::slotOk( void )
{
	if( username->text() == "" || hostname->text() == "" ) {
		QDialog::reject();
	} else {
		update();
		QDialog::accept();
	}
}

void KSipIdentityEdit::slotCancel( void )
{
	QDialog::reject();
}

KSipAuthenticationRequest::KSipAuthenticationRequest(
		const QString &server, const QString &sipuri, const QString &caption, QWidget *parent, const char *name )
		: KDialogBase( parent, name, true, caption, Ok|Cancel, Ok, true )
{
	QVBox *vbox = makeVBoxMainWidget();
	(void) new QLabel( "SipUri = " + sipuri, vbox );
	(void) new QLabel( "Server = " + server, vbox );
	(void) new QLabel( "Username:", vbox );
	username = new QLineEdit( vbox );
	(void) new QLabel( "Password:", vbox );
	password = new QLineEdit( vbox );
	password->setEchoMode( QLineEdit::Password );
}

KSipAuthenticationRequest::~KSipAuthenticationRequest( void )
{
}

void KSipAuthenticationRequest::setUsername( const QString &newUsername )
{
	username->setText( newUsername );
}

void KSipAuthenticationRequest::setPassword( const QString &newPassword )
{
	password->setText( newPassword );
}

QString KSipAuthenticationRequest::getUsername( void )
{
	return username->text();
}

QString KSipAuthenticationRequest::getPassword( void )
{
	return password->text();
}

KSipRegistrations::KSipRegistrations( SipClient *client, KPhoneView *phoneView, QWidget *p, const char *prefix )
{
	parent = p;
	c = client;
	v = phoneView;
	userPrefix = prefix;
	useStun = false;

	edit = new KSipIdentityEdit( parent, userPrefix.latin1(), this );
	connect( edit, SIGNAL( update() ), this, SLOT( update() ) );
	sipreg = 0;
	QString label;
	QString uristr;
	QString str;
	SipUri uri;
	KConfig *config = KGlobal::config();

	label = userPrefix + "Local";
	config->setGroup( label );
	expires = config->readNumEntry( "RegistrationExpiresTime", constRegistrationExpiresTime );
	if( expires == 0 ) {
		expires = -1;
	}
	label = userPrefix + "Registration";
	config->setGroup( label );
	if( config->hasKey( "SipUri" ) ) {
		uristr = config->readEntry( "SipUri" );
		uri = SipUri( uristr );
		u = c->findUser( uri );
		if( u == NULL ) {
			u = new SipUser( c, uri );
		}
		stunSrv = "";
		config->setGroup( "STUN" );
		if( config->readEntry( "UseStun", "" ) == "Yes" ) {
			useStun = true;
			stunSrv = config->readEntry( "StunServer", constStunServer );
			if( stunSrv.isEmpty() ) {
				QString dname = u->getMyUri()->getHostname();
				stunSrv = dname;
				QString srv = client->getSRV( QString( "_stun._udp." ) + dname );
				if( !srv.isEmpty() ) {
					stunSrv = srv;
				}
				stunSrv += ":3478";
			}
		}

		config->setGroup( label );
		uristr = "";
		if( config->hasKey( "SipServer" ) ) {
			uristr = config->readEntry( "SipServer" );
		}
		QString qvalue = config->readEntry( "qValue", "" );
		sipreg = new SipRegister( u, SipUri( uristr ), expires, qvalue );
		connect( sipreg, SIGNAL( statusUpdated() ), this, SLOT( registerStatusUpdated() ) );
		v->updateIdentity( u, sipreg );
		str = config->readEntry( "UserName" );
		u->getMyUri()->setProxyUsername( str );
		str = config->readEntry( "AutoRegister" );
		if( str == "Yes" ) {
			autoRegister = true;
			if( useStun ) {
				sipreg->setAutoRegister( true );
			} else {
				sipreg->requestRegister();
			}
		} else {
			autoRegister = false;
			sipreg->setAutoRegister( false );
		}
	} else {
		editRegistration();
	}
}

KSipRegistrations::~KSipRegistrations( void )
{
}

void KSipRegistrations::showIdentity( void )
{
	editRegistration();
}

void KSipRegistrations::save( void )
{
}

void KSipRegistrations::editRegistration( void )
{
	if( sipreg ) {
		setRegisterState();
		edit->setFullname( u->getUri().getFullname() );
		edit->setUsername( u->getUri().getUsername() );
		edit->setHostname( u->getUri().getHostname() );
		edit->setSipProxy( sipreg->getOutboundProxy() );
		edit->setSipProxyUsername( u->getUri().getProxyUsername() );
		edit->setAutoRegister( autoRegister );
		edit->setQvalue( sipreg->getQvalue() );
	} else {
		edit->updateState("off");
		edit->setFullname( "" );
		edit->setUsername( "" );
		edit->setHostname( "" );
		edit->setSipProxy( "" );
		edit->setSipProxyUsername( "" );
		edit->setQvalue( "" );
		edit->setAutoRegister( true );
	}
	edit->show();
}

void KSipRegistrations::update( void )
{
	QString label;
	bool isDiff = false;
	KConfig *config = KGlobal::config();
	label = userPrefix + "Registration";

	QString s = edit->getSipProxy();
	
	config->setGroup( label );
	if( config->readEntry( "SipUri", "" ) != edit->getUri() ||
			config->readEntry( "SipServer", "" ) != edit->getSipProxy() ||
			config->readEntry( "UserName", "" ) != edit->getSipProxyUsername() ||
			config->readEntry( "qValue", "" ) != edit->getQvalue() ) {
		isDiff = true;
	}

	config->writeEntry( "SipUri", edit->getUri() );
	config->writeEntry( "SipServer", edit->getSipProxy() );
	config->writeEntry( "UserName", edit->getSipProxyUsername() );
	if( edit->getAutoRegister() ) {
		autoRegister = true;
		config->writeEntry( "AutoRegister", "Yes");
	} else {
		autoRegister = false;
		config->writeEntry( "AutoRegister", "No");
	}

	s = edit->getSipProxy();


	config->writeEntry( "qValue", edit->getQvalue() );
	config->sync();
	if( !sipreg ) {
		QString uristr = edit->getUri();
		SipUri uri = SipUri( uristr );
		u = c->findUser( uri );
		if( u == NULL ) {
			u = new SipUser( c, uri );
		}
		uristr = edit->getSipProxy();
		QString qvalue = config->readEntry( "qValue", "" );
		sipreg = new SipRegister( u, SipUri( uristr ), expires, qvalue );
		connect( sipreg, SIGNAL( statusUpdated() ), this, SLOT( registerStatusUpdated() ) );
		v->updateIdentity( u, sipreg );
		QString str = edit->getSipProxyUsername();
		u->getMyUri()->setProxyUsername( str );
		if( edit->getAutoRegister() ) {
			autoRegister = true;
			sipreg->requestRegister();
		} else {
			autoRegister = false;
			sipreg->setAutoRegister( false );
		}
		c->updateIdentity( u, sipreg->getOutboundProxy() );
		v->updateIdentity( u, sipreg );
		if( edit->getAutoRegister() ) {
			changeRegistration();
		}
	} else {
		if( isDiff ) {
			QMessageBox::information( parent, "Identity", "Restart Kphone to apply identity changes." );
		}
	}
}

void KSipRegistrations::changeRegistration( void )
{
	if( sipreg ) {
		if( sipreg->getRegisterState() == SipRegister::Connected ) {
			sipreg->requestClearRegistration();
		} else {
			sipreg->updateRegister();
			v->setContactsOnline();
		}
	}
}

void KSipRegistrations::unregAllRegistration( void )
{
	if( sipreg ) {
		if( sipreg->getRegisterState() == SipRegister::Connected ) {
			sipreg->requestClearRegistration();
		}
	}
}

void KSipRegistrations::setRegisterState( void )
{
	switch( sipreg->getRegisterState() ) {
		case SipRegister::NotConnected:
			edit->updateState( "Inactive" );
			break;
		case SipRegister::TryingServer:
		case SipRegister::TryingServerWithPassword:
			edit->updateState( "Trying..." );
			break;
		case SipRegister::AuthenticationRequired:
		case SipRegister::AuthenticationRequiredWithNewPassword:
			edit->updateState( "Auth Required..." );
			break;
		case SipRegister::Connected:
			edit->updateState( "Active" );
			break;
		case SipRegister::Disconnecting:
			edit->updateState( "Unregistering..." );
			break;
		case SipRegister::Reconnecting:
			edit->updateState( "Reregistering..." );
			break;
	}
}

void KSipRegistrations::registerStatusUpdated( void )
{
	setRegisterState();
	if( sipreg->getRegisterState() != SipRegister::AuthenticationRequired &&
			sipreg->getRegisterState() != SipRegister::AuthenticationRequiredWithNewPassword ) {
		return;
	}
	QString authtype;
	switch( sipreg->getAuthenticationType() ) {
		case SipRegister::DigestAuthenticationRequired:
			authtype = "Digest Authentication Request"; break;
		case SipRegister::BasicAuthenticationRequired:
			authtype = "Basic Authentication Request"; break;
		case SipRegister::ProxyDigestAuthenticationRequired:
			authtype = "Proxy Digest Authentication Request"; break;
		case SipRegister::ProxyBasicAuthenticationRequired:
			authtype = "Proxy Basic Authentication Request"; break;
	}
	QString server = sipreg->getServerUri().proxyUri();
	QString sipuri = u->getUri().uri();
	KSipAuthenticationRequest authreq( server, sipuri, userPrefix + authtype );
	QString username = sipreg->getRegisterCall()->getProxyUsername();
	authreq.setUsername( username );
	QString password = sipreg->getRegisterCall()->getPassword();
	if( password.isEmpty() || sipreg->getRegisterState() == SipRegister::AuthenticationRequiredWithNewPassword ) {
		if( authreq.exec() ) {
			if( authreq.getUsername().isEmpty() || authreq.getUsername().isEmpty() ) {
				return;
			}
			sipreg->getRegisterCall()->setProxyUsername( authreq.getUsername() );
			sipreg->getRegisterCall()->setPassword( authreq.getPassword() );
			sipreg->requestRegister( authreq.getUsername(), authreq.getPassword() );
		}
	} else {
		sipreg->requestRegister( username, password );
	}
}

#include "ksipregistrations.moc"
