
/*
 * process_view.cc
 * Copyright (C) 1999 by John Heidemann
 * $Id: process_view.cc,v 1.37 2000/06/04 05:59:56 johnh Exp $
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <stdio.h>  // snprintf

#include "main.hh"
#include "process_view.hh"
#include "process_model.hh"
#include "user.hh"


#if 0
#define PV_AUTOSIZE_DEBUG(X) X
#else /* ! 0 */
#define PV_AUTOSIZE_DEBUG(X) 
#endif /* 0 */


int process_view::memory_sum_t::sum_ = 0;
change_tracking<int> process_view::size_divisor_(30);
int process_view::size_low_water_ = -1;
int process_view::size_high_water_ = -1;
int process_view::show_size_ = 1;
double process_view::size_target_multiplier_ = 0.75;

bool
process_view_ordering::operator()(process_view *a, process_view *b) const
{
	// For some reason we get called with a null...
	// if we do reorders() in age().
	// Recover from that.
	if (!a || !b)
		return false;
	if (a->age_.last() != b->age_.last())
		return a->age_.last() < b->age_.last();
	if (a->size_.last() != b->size_.last())
		return a->size_.last() < b->size_.last();
	// force a total ordering
	return a->pm_->pid() < b->pm_->pid();
}


process_view::process_view_order_t process_view::orders_;


process_view::process_view(process_model *pm) :
	pm_(pm),
	order_(orders_.end())
{
	ENTRY_TRACE(__FILE__,__LINE__);
	b_ = blob::create_real_blob(this);
	init_position();  // must preceed reset_size() or the bigger blob moves off the field
	reset_size();
	time_.tick_set(scale_time(pm_->utime_.last() + pm_->stime_.last()));
	reset_color();
	update();   // set dark percentage
}

process_view::~process_view()
{
	ENTRY_TRACE(__FILE__,__LINE__);
	if (order_ != orders_.end())
		orders_.erase(order_);
	order_ = orders_.end();
	delete b_;
}

void
process_view::reorder()
{
	// optimize for not moving
	process_view_order_t::iterator prev_order_(order_);
	if (order_ != orders_.end()) {
		--prev_order_;
		orders_.erase(order_);
	};
	process_view_order_t::iterator next_order_(order_);
	if (next_order_ != orders_.end())
		++next_order_;

	order_ = orders_.insert(prev_order_, this);
	// If we changed order, tell the blob to redraw.
	// (Boy, it's a shame there isn't an iterator where ++end == end!)
	process_view_order_t::iterator new_prev_order_(order_);
	if (order_ != orders_.end())
		--new_prev_order_;
	process_view_order_t::iterator new_next_order_(order_);
	if (new_next_order_ != orders_.end())
		++new_next_order_;
	if (prev_order_ != new_prev_order_ || next_order_ != new_next_order_)
		b_->reordered();
}

void
process_view::reset_color()
{
	ENTRY_TRACE(__FILE__,__LINE__);
	S_ = 0.6;
	B_ = 0.8 - (pm_->age() * 0.06);
	const double MIN_B = 0.15;
	if (B_ < MIN_B)
		B_ = MIN_B;

	H_ = (pm_->cmd_hash_.last() % 2048) / 2048.0;

	b_->set_hsb(H_, S_, B_);
}

void
process_view::init_position()
{
	ENTRY_TRACE(__FILE__,__LINE__);
	int x = pm_->pid_ % (blob::X_MAX - blob::INITIAL_SLOP);
	int y = pm_->uid_ % (blob::Y_MAX - blob::INITIAL_SLOP);
	b_->set_position(x, y);
}

int
process_view::scale_size(int size)
{
	ENTRY_TRACE(__FILE__,__LINE__);
	return size / size_divisor_.last();
}

int
process_view::scale_time(int time)
{
	ENTRY_TRACE(__FILE__,__LINE__);
	// covert from msec to 100ths of a second
	return time / time_divisor_;
}

void
process_view::reset_size()
{
	int size = scale_size(report_vm.last() != PHYSICAL_MEM ? 
			      pm_->virtual_size_.last() :
			      pm_->resident_.last());
	size = max(size, 3);   // work around not handling small sizes well.
	size_.tick_set(size);
	if (size_.changed())
		b_->set_size(size_.last());

	if (report_vm.last() == BOTH_MEM && pm_->inmem_pct_.changed())
		b_->set_dark_percentage(100 - pm_->inmem_pct_.last());
}

void
process_view::update()
{
	ENTRY_TRACE(__FILE__,__LINE__);
	age_.tick_set(pm_->age());
	if (age_.changed() || pm_->cmd_hash_.changed())
		reset_color();

	if (report_vm.changed() ||
	    size_divisor_.changed() ||
	    (report_vm.last() == BOTH_MEM ?
	     pm_->virtual_size_.changed() || pm_->resident_.changed() :
	     report_vm.last() == VIRTUAL_MEM ?
	     pm_->virtual_size_.changed() : pm_->resident_.changed())) {
		reset_size();
	};

	time_.tick_set(scale_time(pm_->utime_.last() + pm_->stime_.last()));
	if (time_.changed()) {
		int change = time_.change();
		if (change < -400000)  // hack around integer overflow
			change = 0;
		else assert(change > 0);
		b_->move(change);
	};

	if (age_.changed() || size_.changed())
		reorder();
}


void
process_view::info_to_buf(char *buf, int len)
{
	ENTRY_TRACE(__FILE__,__LINE__);
	static char debug_buf[128];
	if (lava_debug) {
		int rs = b_->get_real_size();
		int crs = b_->calc_real_size();
		snprintf(debug_buf, 128, "(pvs=%d, rs=%d, crs=%d)\n", size_.last(), rs, crs);
	} else debug_buf[0] = 0;
	snprintf(buf, len, "user: %s (%d)\npid: %d\ncmd: %s\nnice: %d\nvirt mem: %dKB\nphys mem: %dKB (%d%%)\n%s",
		 uid_to_name(pm_->uid_),
		 pm_->uid_,
		 pm_->pid_,
		 pm_->cmd_.c_str(),
		 pm_->nice_,
		 pm_->virtual_size_.last(),
		 pm_->resident_.last(),
		 pm_->inmem_pct_.last(),
		 debug_buf);
}

process_view *
process_view::superior()
{
	if (order_ == orders_.end())
	    return NULL;
	// I don't know how to check for moving our iterator
	// off the front.  I assumed it would return end(),
	// but maybe not.  The next check is paranoia.
	if (order_ == orders_.begin())
	    return NULL;
	process_view_order_t::iterator superior(order_);
	--superior;
	return superior == orders_.end() ? NULL : *superior;
}

void
process_view::mem_adjust(int dir)
{
	// fix to the inifite growth/shrink problem from Nick Bailey <nick@polonius.demon.co.uk>:
	// always make sure we change by a non-zero value.
        int proposal = dir * int(size_divisor_.last() / size_recalc_divisor_);
	if (proposal == 0)          // Always change by at least +/- 1
		proposal = dir;
	size_divisor_.tick_incr(proposal);
	PV_AUTOSIZE_DEBUG(cout << "mem_adjust dir=" << dir << " size_divisor_ = " << size_divisor_.last() << endl);
	show_size_ += 2;
}

void
process_view::mem_target_adjust(int dir)
{
	int available_area = blob::available_area();
	if (dir)
		size_target_multiplier_ += dir * size_target_multiplier_ / size_recalc_divisor_;
	PV_AUTOSIZE_DEBUG(cout << "mem_target_adjust dir=" << dir << " size_target_multiplier_=" << size_target_multiplier_ << endl);
	show_size_ += 2;
	int desired_area = int(available_area * size_target_multiplier_);
	size_low_water_ = desired_area - 2 * desired_area / size_recalc_divisor_;
	size_high_water_ = desired_area + 2 * desired_area / size_recalc_divisor_;
	// make things larger
	if (dir)
		mem_adjust(dir);
}

void
process_view::autosize()
{
	if (!allow_autosize)
		return;

	// too much stuff on the screen?
	if (size_low_water_ < 0)
		mem_target_adjust(0);
	if (size_divisor_.changed())
		size_divisor_.clear_change();

	if (show_size_) {
		PV_AUTOSIZE_DEBUG(cout << "limits: sum: " << memory_sum_t::sum_ << " in [" << size_low_water_ << "," << size_high_water_ << "]\n");
		show_size_--;
	};
	if (memory_sum_t::sum_ < size_low_water_) {
		mem_adjust(-1);
		blob::splash("mem grow\n");
	} else if (memory_sum_t::sum_ > size_high_water_) {
		mem_adjust(1);
		blob::splash("mem shrink\n");
	};
}

int
process_view::pid()
{
	// This function can't be inlined because of recursion between
	// the class definitions.
	return pm_->pid();
}
