//
//  Little cms
//  Copyright (C) 1998-2000 Marti Maria
//
// THIS SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
// EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
// WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
//
// IN NO EVENT SHALL MARTI MARIA BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
// INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
// OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
// WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF
// LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
// OF THIS SOFTWARE.
//
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// Example: Closest neutral-to


#include "lcms.h"


static
double DistanceOf(LPcmsCIExyY a, LPcmsCIExyY b)
{
       double dx, dy;

       dx = fabs(b -> x - a -> x);
       dy = fabs(b -> y - a -> y);

       return sqrt(dx*dx + dy*dy);
}


static
void XYZ2xyY(LPcmsCIExyY Dest, CONST LPcmsCIEXYZ Source)
{
       double ISum;

       ISum = 1./(Source -> X + Source -> Y + Source -> Z);

       Dest -> x = (Source -> X) * ISum;
       Dest -> y = (Source -> Y) * ISum;
       Dest -> Y = Source -> Y;
}


void xyY2XYZ(LPcmsCIEXYZ Dest, CONST LPcmsCIExyY Source)
{

        Dest -> X = (Source -> x / Source -> y) * Source -> Y;
        Dest -> Y = Source -> Y;
        Dest -> Z = ((1 - Source -> x - Source -> y) / Source -> y) * Source -> Y;
}





static
int ClosestNeutral(LPcmsCIEXYZ XYZ)
{
       cmsCIExyY Chromacity, Tried;
       int TempMin, TempMax, Temp, BestTemp;
       double DistanceMin, Distance;

       XYZ2xyY(&Chromacity, XYZ);

       TempMin = 4000;      // Lowest value lcms does accept
       TempMax = 25000;     // Higest value lcms does accept
       DistanceMin = 1E20;  // A big value
       BestTemp = -1;

       for (Temp = TempMin; Temp < TempMax; Temp++)
       {
              cmsWhitePointFromTemp(Temp, &Tried);

              Distance    = DistanceOf(&Tried, &Chromacity);

              if (Distance < DistanceMin) {
                     DistanceMin = Distance;
                     BestTemp = Temp;
                     }
        }

       return BestTemp;
}


static
double f(double t)
{

       if (t <= 0.008856)
                     return 7.787*t + (16./116.);
       else
                     return pow(t, 1.0/3.0);

}

static
void XYZ2Lab(double X, double Y, double Z, double *L, double *a, double *b)
{
       double x, y, z;
       double fx, fy, fz;



       if (X==0 && Y==0 && Z==0) {

                     *L = 0;
                     *a = *b = -1;
                     return;
       }

       // PCS is in D50


       x = (X) / 0.964294;
       y = (Y);
       z = (Z) / 0.825104;


       fx = f(x);
       fy = f(y);
       fz = f(z);

       *L = 116.* fy - 16.;

       *a = 500.*(fx - fy);
       *b = 200.*(fy - fz);

}



int main (int argc, char *argv[])
{

       cmsCIEXYZ xyz, Media;
       cmsCIExyY MediaChromacity;
       double dn, L, a, b;

       printf("Closest-neutral-to ver 1.0\n\n");

       printf("X? "); scanf("%lf", &xyz.X);
       printf("Y? "); scanf("%lf", &xyz.Y);
       printf("Z? "); scanf("%lf", &xyz.Z);

       printf("XYZ=(%g,%g,%g)\n", xyz.X, xyz.Y,xyz.Z);

       dn = ClosestNeutral(&xyz);

       printf("Closest black body locus point Is D%d\n", (int) (dn / 100));

       cmsWhitePointFromTemp(dn, &MediaChromacity);
       MediaChromacity.Y = xyz.Y;
       xyY2XYZ(&Media, &MediaChromacity);

       printf("XYZ=(%g,%g,%g) ", Media.X, Media.Y, Media.Z);
       XYZ2Lab(Media.X, Media.Y, Media.Z, &L, &a, &b);
       printf("Lab is (%g, %g, %g)", L, a, b);
       return 0;

}
