
/* Little cms */
/* Copyright (C) 1998-2000 Marti Maria */

/* THIS SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, */
/* EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY */
/* WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. */

/* IN NO EVENT SHALL MARTI MARIA BE LIABLE FOR ANY SPECIAL, INCIDENTAL, */
/* INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, */
/* OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, */
/* WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF */
/* LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE */
/* OF THIS SOFTWARE. */


/* This library is free software; you can redistribute it and/or */
/* modify it under the terms of the GNU Lesser General Public */
/* License as published by the Free Software Foundation; either */
/* version 2 of the License, or (at your option) any later version. */

/* This library is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU */
/* Lesser General Public License for more details. */

/* You should have received a copy of the GNU Lesser General Public */
/* License along with this library; if not, write to the Free Software */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "lcms.h"

/* Gamma handling. I'm encoding gamma as 0..ffff value */

LPGAMMATABLE LCMSEXPORT cmsAllocGamma(int nEntries);
void         LCMSEXPORT cmsFreeGamma(LPGAMMATABLE Gamma);
LPGAMMATABLE LCMSEXPORT cmsBuildGamma(int nEntries, double Gamma);
LPGAMMATABLE LCMSEXPORT cmsReverseGamma(int nResultSamples, LPGAMMATABLE InGamma);
LPGAMMATABLE LCMSEXPORT cmsJoinGamma(LPGAMMATABLE InGamma,
                                LPGAMMATABLE OutGamma);

LPGAMMATABLE cdecl cmsScaleGamma(LPGAMMATABLE Input, Fixed32 Factor);

/* default gamma function */

/* Implementing gamma as a two-stages funtion. */
/* I chose the cutoff between linear-exponential stage on */

/* (breakin, breakout)=(0.081, 0.018) */

/* The linear segment minimizes the effect of sensor noise in */
/* practical devices. */

#define PHOTOSHOP_COMPATIBILITY    1

#if 1 /* def PHOTOSHOP_COMPATIBLITY --- Buggy 24/5/2000 */

static
double FGamma(double R, double x)
{
       return pow(R, x);
}

#else

static
double FGamma(double R, double x)
{
       double startin, endin;
       double startout, endout;
       double breakin, breakout;
       double a, c;


       startin = 0; endin = 1.0;
       startout= 0; endout = 1.0;

       breakin  = 0.081;
       breakout = 0.018;

       if (R >= startin && R <= breakin)
       {
              return (breakout-startout)*(R-startin)/(breakin-startin) + startout;
       }
       else
       {
              a = (endout-breakout)/(1-pow((breakin-startin)/(endin-startin), x));
              c = endout - a;

              return a*pow((R-startin)/(endin-startin), x)+c;
       }
}

#endif

LPGAMMATABLE LCMSEXPORT cmsAllocGamma(int nEntries)
{
       LPGAMMATABLE p;
       size_t size;

       size = sizeof(GAMMATABLE) + (sizeof(WORD) * (nEntries-1));

       p = (LPGAMMATABLE) malloc(size);
       if (!p) return NULL;

       p -> nEntries = nEntries;
       ZeroMemory(p -> GammaTable, nEntries * sizeof(WORD));

       return p;
}

void LCMSEXPORT cmsFreeGamma(LPGAMMATABLE Gamma)
{
       free(Gamma);
}




/* Build a gamma table based on gamma constant */

LPGAMMATABLE LCMSEXPORT cmsBuildGamma(int nEntries, double Gamma)
{
       LPGAMMATABLE p;
       LPWORD Table;
       int i;
       double R, Val;

       p = cmsAllocGamma(nEntries);
       if (!p) return NULL;

       Table = p -> GammaTable;
       if (Gamma == 0.0)
       {
              ZeroMemory(Table, nEntries*sizeof(WORD));
              return p;
       }


       for (i=0; i < nEntries; i++)
       {
              R   = (double) i / (nEntries-1);
              Val = FGamma(R, Gamma);

              Table[i] = (WORD) floor(Val * 0xFFFFL);
       }

       return p;
}


/* Handle gamma using interpolation tables. The resulting curves can become */
/* very stange, but are pleasent to eye. */

LPGAMMATABLE LCMSEXPORT cmsJoinGamma(LPGAMMATABLE InGamma,
                          LPGAMMATABLE OutGamma)
{
       register int i;
       L16PARAMS L16In, L16Out;
       LPWORD InPtr, OutPtr;
       LPGAMMATABLE p;

       p = cmsAllocGamma(256);
       if (!p) return NULL;

       cmsCalcL16Params(InGamma -> nEntries, &L16In);
       InPtr  = InGamma -> GammaTable;

       cmsCalcL16Params(OutGamma -> nEntries, &L16Out);
       OutPtr = OutGamma-> GammaTable;

       for (i=0; i < 256; i++)
       {
              WORD wValIn, wValOut;

              wValIn  = cmsLinearInterpLUT16(RGB_8_TO_16(i), InPtr, &L16In);
              wValOut = cmsReverseLinearInterpLUT16(wValIn, OutPtr, &L16Out);

              p -> GammaTable[i] = wValOut;
       }

       return p;
}

/* Reverse a gamma table */

LPGAMMATABLE LCMSEXPORT cmsReverseGamma(int nResultSamples, LPGAMMATABLE InGamma)
{
       register int i;
       L16PARAMS L16In;
       LPWORD InPtr;
       LPGAMMATABLE p;

       p = cmsAllocGamma(nResultSamples);
       if (!p) return NULL;

       cmsCalcL16Params(InGamma -> nEntries, &L16In);
       InPtr  = InGamma -> GammaTable;

       for (i=0; i < nResultSamples; i++)
       {
              WORD wValIn, wValOut;

              wValIn = (WORD) ((int) ((int) i * 0xFFFFL) / (nResultSamples - 1));

              wValOut = cmsReverseLinearInterpLUT16(wValIn, InPtr, &L16In);
              p -> GammaTable[i] = wValOut;
       }

       return p;
}

LPGAMMATABLE cmsScaleGamma(LPGAMMATABLE Input, Fixed32 Factor)
{
       LPGAMMATABLE Output;
       int i, nEntries;


       nEntries = Input -> nEntries;
       Output = cmsAllocGamma(nEntries);
       if (!Output) return NULL;

       for (i=0; i < nEntries; i++)
              Output->GammaTable[i] = FixedScale(Input->GammaTable[i], Factor);

       return Output;
}

