
/* Little cms */
/* Copyright (C) 1998-2000 Marti Maria */

/* THIS SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, */
/* EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY */
/* WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. */

/* IN NO EVENT SHALL MARTI MARIA BE LIABLE FOR ANY SPECIAL, INCIDENTAL, */
/* INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, */
/* OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, */
/* WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF */
/* LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE */
/* OF THIS SOFTWARE. */


/* This library is free software; you can redistribute it and/or */
/* modify it under the terms of the GNU Lesser General Public */
/* License as published by the Free Software Foundation; either */
/* version 2 of the License, or (at your option) any later version. */

/* This library is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU */
/* Lesser General Public License for more details. */

/* You should have received a copy of the GNU Lesser General Public */
/* License along with this library; if not, write to the Free Software */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "lcms.h"


/* Conversions */

void LCMSEXPORT cmsXYZ2xyY(LPcmsCIExyY Dest, const LPcmsCIEXYZ Source)
{
       double ISum;

       ISum = 1./(Source -> X + Source -> Y + Source -> Z);

       Dest -> x = (Source -> X) * ISum;
       Dest -> y = (Source -> Y) * ISum;
       Dest -> Y = Source -> Y;
}


void LCMSEXPORT cmsxyY2XYZ(LPcmsCIEXYZ Dest, const LPcmsCIExyY Source)
{

        Dest -> X = (Source -> x / Source -> y) * Source -> Y;
        Dest -> Y = Source -> Y;
        Dest -> Z = ((1 - Source -> x - Source -> y) / Source -> y) * Source -> Y;
}



/* Obtains WhitePoint from Temperature */

BOOL LCMSEXPORT cmsWhitePointFromTemp(int TempK, LPcmsCIExyY WhitePoint)
{
       double x, y;
       double T, T2, T3;
       /* double M1, M2; */


       /* No optimization provided. */

       T = TempK;
       T2 = T*T;            /* Square */
       T3 = T2*T;           /* Cube */

       /* For correlated color temperature (T) between 4000K and 7000K: */

       if (T >= 4000. && T <= 7000.)
       {
              x = -4.6070*(1E9/T3) + 2.9678*(1E6/T2) + 0.09911*(1E3/T) + 0.244063;
       }
       else
              /* or for correlated color temperature (T) between 7000K and 25000K: */

       if (T > 7000.0 && T <= 25000.0)
       {
              x = -2.0064*(1E9/T3) + 1.9018*(1E6/T2) + 0.24748*(1E3/T) + 0.237040;
       }
       else {
              cmsSignalError(LCMS_ERRC_ABORTED, "cmsWhitePointFromTemp: invalid temp");
              return FALSE;
              }

       /* Obtain y(x) */

       y = -3.000*(x*x) + 2.870*x - 0.275;

       /* wave factors (not used, but here for futures extensions) */

       /* M1 = (-1.3515 - 1.7703*x + 5.9114 *y)/(0.0241 + 0.2562*x - 0.7341*y); */
       /* M2 = (0.0300 - 31.4424*x + 30.0717*y)/(0.0241 + 0.2562*x - 0.7341*y); */



       /* Fill WhitePoint struct */

       WhitePoint -> x = x;
       WhitePoint -> y = y;
       WhitePoint -> Y = 1.0;

       return TRUE;
}

/* Build a White point, primary chromas transfer matrix from RGB to CIE XYZ */
/* This is just an approximation, I am not handling all the non-linear */
/* aspects of the RGB to XYZ process, and assumming that the gamma correction */
/* has transitive property in the tranformation chain. */

/* the alghoritm: */

/* - First I build the absolute conversion matrix using */
/* primaries in XYZ. This matrix is next inverted */
/* - Then I eval the source white point across this matrix */
/* obtaining the coeficients of the transformation */
/* - Then, I apply these coeficients to the original matrix */
/* - Last, I adapt the resulting matrix to D50 PCS */


BOOL LCMSEXPORT cmsBuildRGB2XYZtransferMatrix(LPMAT3 r, LPcmsCIExyY WhitePt,
                                            LPcmsCIExyYTRIPLE Primrs)
{
        VEC3 WhitePoint, Coef;
        MAT3 Result, Primaries;
        double xn, yn;
        double xr, yr;
        double xg, yg;
        double xb, yb;


        xn = WhitePt -> x;
        yn = WhitePt -> y;
        xr = Primrs -> Red.x;
        yr = Primrs -> Red.y;
        xg = Primrs -> Green.x;
        yg = Primrs -> Green.y;
        xb = Primrs -> Blue.x;
        yb = Primrs -> Blue.y;


        /* Build Primaries matrix */

        VEC3init(&Primaries.v[0], xr,        xg,         xb);
        VEC3init(&Primaries.v[1], yr,        yg,         yb);
        VEC3init(&Primaries.v[2], (1-xr-yr), (1-xg-yg),  (1-xb-yb));


        /* Result = Primaries ^ (-1) inverse matrix */

        if (!MAT3inverse(&Primaries, &Result))
                        return FALSE;


        VEC3init(&WhitePoint, xn/yn, 1.0, (1.0-xn-yn)/yn);

        /* Across inverse primaries ... */

        MAT3eval(&Coef, &Result, &WhitePoint);

        /* Give us the Coefs, then I build transformation matrix */

        VEC3init(&r -> v[0], Coef.n[VX]*xr,          Coef.n[VY]*xg,          Coef.n[VZ]*xb);
        VEC3init(&r -> v[1], Coef.n[VX]*yr,          Coef.n[VY]*yg,          Coef.n[VZ]*yb);
        VEC3init(&r -> v[2], Coef.n[VX]*(1.0-xr-yr), Coef.n[VY]*(1.0-xg-yg), Coef.n[VZ]*(1.0-xb-yb));


        /* Finally, adapt to D50 */

        cmsAdaptMatrixToD50(r, WhitePt);

        return TRUE;
}





/* This procedure implements the Lam & Rigg Bradford chromatic adaptation. */

static
void BradfordLamRiggChromaticAdaptation(LPMAT3 Conversion,
                                        LPcmsCIEXYZ SourceWhitePoint,
                                        LPcmsCIEXYZ DestWhitePoint)

{
       MAT3 LamRigg   = {{ /* Bradford matrix */
                     {{  0.8951,  0.2664, -0.1614 }},
                     {{ -0.7502,  1.7135,  0.0367 }},
                     {{  0.0389, -0.0685,  1.0296 }}
                     }};
       MAT3 LamRigg_1 = {{ /* |Bradford|^^-1 */
                     {{  0.9870, -0.1471,  0.1600 }},
                     {{  0.4323,  0.5184,  0.0493 }},
                     {{ -0.0085,  0.0400,  0.9685 }},
                     }};

        VEC3 ConeSourceXYZ, ConeSourceRGB;
        VEC3 ConeDestXYZ, ConeDestRGB;
        MAT3 Cone, Tmp;

        VEC3init(&ConeSourceXYZ, SourceWhitePoint -> X,
                                 SourceWhitePoint -> Y,
                                 SourceWhitePoint -> Z);

        VEC3init(&ConeDestXYZ,   DestWhitePoint -> X,
                                 DestWhitePoint -> Y,
                                 DestWhitePoint -> Z);

        MAT3eval(&ConeSourceRGB, &LamRigg, &ConeSourceXYZ);
        MAT3eval(&ConeDestRGB,   &LamRigg, &ConeDestXYZ);

        /* Build matrix */

        VEC3init(&Cone.v[0], ConeDestRGB.n[0]/ConeSourceRGB.n[0],    0.0,  0.0);
        VEC3init(&Cone.v[1], 0.0,   ConeDestRGB.n[1]/ConeSourceRGB.n[1],   0.0);
        VEC3init(&Cone.v[2], 0.0,   0.0,   ConeDestRGB.n[2]/ConeSourceRGB.n[2]);


        /* Normalize */
        MAT3per(&Tmp, &Cone, &LamRigg);
        MAT3per(Conversion, &LamRigg_1, &Tmp);

}


BOOL cmsAdaptMatrixToD50(LPMAT3 r, LPcmsCIExyY SourceWhitePt)
{
        cmsCIEXYZ Dn;
        cmsCIEXYZ D50 = {0.964294, 1.000000, 0.825104};
        MAT3 Bradford;
        MAT3 Tmp;

        cmsxyY2XYZ(&Dn, SourceWhitePt);

        BradfordLamRiggChromaticAdaptation(&Bradford, &Dn, &D50);
        Tmp = *r;
        MAT3per(r, &Bradford, &Tmp);

        return TRUE;
}

BOOL cmsAdaptMatrixFromD50(LPMAT3 r, LPcmsCIExyY DestWhitePt)
{
        cmsCIEXYZ Dn;
        cmsCIEXYZ D50 = {0.964294, 1.000000, 0.825104};
        MAT3 Bradford;
        MAT3 Tmp;

        cmsxyY2XYZ(&Dn, DestWhitePt);

        BradfordLamRiggChromaticAdaptation(&Bradford, &D50, &Dn);
        Tmp = *r;
        MAT3per(r, &Bradford, &Tmp);

        return TRUE;
}


/* Adapts a color to a given illuminant. Original color is expected to have */
/* a SourceWhitePt white point. (Currently uses a von-kryes simplification */
/* of Bradford transform). */

BOOL LCMSEXPORT cmsAdaptToIlluminant(LPcmsCIEXYZ Result, LPcmsCIExyY SourceWhitePt, LPcmsCIExyY Illuminant, LPcmsCIEXYZ Value)
{
        cmsCIEXYZ Dn, Dill;
        MAT3 Bradford;
        VEC3 In, Out;

        cmsxyY2XYZ(&Dn,   SourceWhitePt);
        cmsxyY2XYZ(&Dill, Illuminant);

        BradfordLamRiggChromaticAdaptation(&Bradford, &Dn, &Dill);

        VEC3init(&In, Value -> X, Value -> Y, Value -> Z);
        MAT3eval(&Out, &Bradford, &In);

        Result -> X = Out.n[0];
        Result -> Y = Out.n[1];
        Result -> Z = Out.n[2];

        return TRUE;
}


