
/* Little cms */
/* Copyright (C) 1998-1999 Marti Maria */

/* THIS SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, */
/* EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY */
/* WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. */

/* IN NO EVENT SHALL MARTI MARIA BE LIABLE FOR ANY SPECIAL, INCIDENTAL, */
/* INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, */
/* OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, */
/* WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF */
/* LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE */
/* OF THIS SOFTWARE. */


/* This library is free software; you can redistribute it and/or */
/* modify it under the terms of the GNU Lesser General Public */
/* License as published by the Free Software Foundation; either */
/* version 2 of the License, or (at your option) any later version. */

/* This library is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU */
/* Lesser General Public License for more details. */

/* You should have received a copy of the GNU Lesser General Public */
/* License along with this library; if not, write to the Free Software */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* Test Suite for Little cms */


#include "lcms.h"
#include <time.h>
#include <stdio.h>
#include <stdlib.h>

#define DIRSEP	"/"

#define PREC  20

#define TYPE_XYZA_16            (COLORSPACE_SH(PT_XYZ)|CHANNELS_SH(3)|BYTES_SH(2)|EXTRA_SH(1))
#define TYPE_LABA_16            (COLORSPACE_SH(PT_Lab)|CHANNELS_SH(3)|BYTES_SH(2)|EXTRA_SH(1))

typedef struct {BYTE r, g, b, a;}   Scanline_rgb1;
typedef struct {WORD r, g, b, a;}   Scanline_rgb2;



#ifndef LCMS_DLL


/* Simpler fixed-point math */

static
void TestFixedPoint(void)
{
       Fixed32 a, b, c, d, e;
       double f;

       a = DOUBLE_TO_FIXED(1.12345);
       b = DOUBLE_TO_FIXED(2.56789);

       c = FixedMul(a, b);
       e = FixedDiv(a, b);

       d = (c & 0xffff);
       f = ((double) d / 0xffff) * 10000.0;

       printf("Testing fixed point: 2.8848960205 = %d.%d\n", FIXED_TO_INT(c), (int) f);

       d = (e & 0xffff);
       f = ((double) d / 0xffff) * 10000.0;

       printf("\t\t0.437499269828536 = %d.%d\n", FIXED_TO_INT(e), (int) f);
}



static
int TestFixedScaling(void)
{
       int i, j, nfl, nfx;
       double FloatFactor;
       Fixed32 FixedFactor;



       printf("Testing fixed scaling...");

       for (j=5; j<100; j++)
       {
       FloatFactor = (double) j / 100.0 ;
       FloatFactor = FIXED_TO_DOUBLE(DOUBLE_TO_FIXED(FloatFactor));
       FixedFactor = DOUBLE_TO_FIXED(FloatFactor);

       for (i=0; i < 0x10000L; i++)
              {
                     nfl = (WORD) ((double) i * FloatFactor);
                     nfx = FixedScale((WORD) i, FixedFactor);

                     if (nfl != nfx) {
                            printf("Failed!\ni=%x (%d), float=%x, fixed=%x", i, i, nfl, nfx);
                            return 0;
                            }
              }

       }

       printf ("Ok!\n");
       return 1;
}


/* Linear interpolation test. Here I check the cmsLinearInterpLUT16 */
/* Tables are supposed to be monotonic, but the algorithm works on */
/* non-monotonic as well. */

static
int TestLinearInterpolation(int lExhaustive)
{
       static WORD Tab[4098];
       int j, i, k;
       L16PARAMS p;
       int n;
       clock_t time;

       printf("Testing linear interpolation ...");

       /* First I will check exact values. Since prime factors of 65535 (FFFF) are, */
       
       /* 0xFFFF = 1 * 3 * 5 * 17 * 257 */
       
       /* I test tables of 2, 4, 6, and 18 points, that will be exact. */
       /* Then, a table of 3 elements are tested. Error must be < 1 */
       /* Since no floating point is involved, This will be a measure of speed. */


       /* Perform 10 times, so i can measure average times */

       time = clock();
       for (j=0; j < 10; j++)
       {

       /* 2 points - exact */

       Tab[0] = 0;
       Tab[1] = 0xffffU;

       cmsCalcL16Params(2, &p);

       for (i=0; i <= 0xffffL; i++)
       {
              n = cmsLinearInterpLUT16((WORD) i, Tab, &p);
              if (n != i)
                     {
                     printf("Error in Linear interpolation (2p): Must be i=%x, But is n=%x\n", i, n);
                     return 0;
                     }

       }


       /* 3 points - Here the error must be <= 1, since */
       /* 2 == (3 - 1)  is not a factor of 0xffff */

       Tab[0] = 0;
       Tab[1] = 0x7FFF;
       Tab[2] = 0xffffU;

       cmsCalcL16Params(3, &p);

       for (i=0; i <= 0xffffL; i++)
       {
              n = cmsLinearInterpLUT16((WORD) i, Tab, &p);
              if (abs(n - i) > 1)
                     {
                     printf("Error in Linear interpolation (3p): Must be i=%x, But is n=%x\n", i, n);
                     return 0;
                     }

       }


       /* 4 points - exact */

       Tab[0] = 0;
       Tab[1] = 0x5555U;
       Tab[2] = 0xAAAAU;
       Tab[3] = 0xffffU;

       cmsCalcL16Params(4, &p);

       for (i=0; i <= 0xffffL; i++)
       {
              n = cmsLinearInterpLUT16((WORD) i, Tab, &p);
              if (n != i) {
                     printf("Error in Linear interpolation (4p): Must be i=%x, But is n=%x\n", i, n);
                     return 0;
                     }

       }


       /* 6 - points */

       Tab[0] = 0;
       Tab[1] = 0x3333U;
       Tab[2] = 0x6666U;
       Tab[3] = 0x9999U;
       Tab[4] = 0xCCCCU;
       Tab[5] = 0xFFFFU;

       cmsCalcL16Params(6, &p);

       for (i=0; i <= 0xffffL; i++)
       {
              n = cmsLinearInterpLUT16((WORD) i, Tab, &p);
              if (n != i) {
                     printf("Error in Linear interpolation (6p): Must be i=%x, But is n=%x\n", i, n);
                     return 0;
                     }

       }


       /* 18 points */

       for (i=0; i < 18; i++)
              Tab[i] = (WORD) (0x0f0fU*i);

       cmsCalcL16Params(18, &p);

       for (i=0; i <= 0xffffL; i++)
       {
              n = cmsLinearInterpLUT16((WORD) i, Tab, &p);
              if (n != i) {
                     printf("Error in Linear interpolation (18p): Must be i=%x, But is n=%x\n", i, n);
                     return 0;
                     }
       }
       }



       printf("Ok! %d tics\n", clock() - time);

       /* Now test descending tables */
       printf("Testing descending tables (linear interpolation) ...");

       /* 2 points - exact */

       Tab[1] = 0;
       Tab[0] = 0xffffU;

       cmsCalcL16Params(2, &p);

       for (i=0xffffL; i > 0; --i)
       {
              n = cmsLinearInterpLUT16((WORD) i, Tab, &p);
              if ((0xffffL - n) != i) {

                     printf("Error in Linear interpolation (descending) (2p): Must be i=%x, But is n=%x\n", i, 0xffffl - n);
                     return 0;
                     }
       }


       /* 3 points - Here the error must be <= 1, since */
       /* 2 = (3 - 1)  is not a factor of 0xffff */

       Tab[2] = 0;
       Tab[1] = 0x7FFF;
       Tab[0] = 0xffffU;

       cmsCalcL16Params(3, &p);

       for (i=0xffffL; i > 0; --i)
       {
              n = cmsLinearInterpLUT16((WORD) i, Tab, &p);
              if (abs((0xffffL - n) - i) > 1) {

                     printf("Error in Linear interpolation (descending) (3p): Must be i=%x, But is n=%x\n", i, n);
                     return 0;
                     }
       }


       /* 4 points - exact */

       Tab[3] = 0;
       Tab[2] = 0x5555U;
       Tab[1] = 0xAAAAU;
       Tab[0] = 0xffffU;

       cmsCalcL16Params(4, &p);

       for (i=0xffffL; i > 0; --i)
       {
              n = cmsLinearInterpLUT16((WORD) i, Tab, &p);
              if ((0xffffL - n) != i) {

                     printf("Error in Linear interpolation (descending) (4p): Must be i=%x, But is n=%x\n", i, n);
                     return 0;
                     }
       }


       /* 6 - points */

       Tab[5] = 0;
       Tab[4] = 0x3333U;
       Tab[3] = 0x6666U;
       Tab[2] = 0x9999U;
       Tab[1] = 0xCCCCU;
       Tab[0] = 0xFFFFU;

       cmsCalcL16Params(6, &p);

       for (i=0xffffL; i > 0; --i)
       {
              n = cmsLinearInterpLUT16((WORD) i, Tab, &p);
              if ((0xffffL - n) != i) {
                     printf("Error in Linear interpolation (descending) (6p): Must be i=%x, But is n=%x\n", i, n);
                     return 0;
                     }

       }


       /* 18 points */

       for (i=0; i < 18; i++)
              Tab[17-i] = (WORD) (0x0f0fU*i);

       cmsCalcL16Params(18, &p);

       for (i=0xffffL; i > 0; --i)
       {
              n = cmsLinearInterpLUT16((WORD) i, Tab, &p);
              if ((0xffffL - n) != i) {

                     printf("Error in Linear interpolation (descending) (18p): Must be i=%x, But is n=%x\n", i, n);
                     return 0;
                     }
       }

       printf("Ok!\n");

       if (!lExhaustive) return 1;

       printf("Now, checking interpolation errors for tables of n elements ...\n");

       for (j=10; j < 4096; j ++)
       {
       printf("%d\r", j);

       for (i=0; i <= j; i++)
              {
              Tab[i] = floor((((double) i / ((double) j-1)) * 65535.0) + .5);
              }

       k =0;
       cmsCalcL16Params(j, &p);
       for (i=0; i <= 0xffffL; i++)
       {
              n = cmsLinearInterpLUT16((WORD) i, Tab, &p);
              if (n != i) k++;

       }
       if (k > 0) printf("\r%d: %d errors\n", j, k);
       }

       return 1;
}



static
int IsGood(const char *frm, WORD in, WORD out)
{
       if ((abs(in - out) > 2))
              {
              printf("error %s %x - %x\n", frm, in, out);
              return 0;
              }

       return 1;
}


/* TODO: Buggy */

static
int Test3D(void)
{

       int i;

       /* cLutPoints  = 2 */
       /* outputChann = 3 */

       /* OutputChan   1      2        3         X     Y     Z */
       /* ---    ---      --- */
       WORD Tab[24] = {  0,     0,       0,      /* 0     0     0 */
                         0,     0x5555,  0xffff, /* 0     0     1 */
                         0,     0x5555,  0xffff, /* 0     1     0 */
                         0,     0x5555,  0xffff, /* 0     1     1 */
                         0,     0,       0,      /* 1     0     0 */
                         0,     0x5555,  0xffff, /* 1     0     1 */
                         0,     0x5555,  0xffff, /* 1     1     0 */
                         0,     0x5555,  0xffff  /* 1     1     1 */
                      };


       WORD Input[3], Output[3];
       L16PARAMS p;


       printf("Testing 3D interpolation...");
       cmsCalcCLUT16Params(2, 3, 3, &p);

       for (i=0x0; i <= 0xFFFFL; i ++)
       {
       Input[0] = (WORD) 123;      /* Decoupled */
       Input[1] = (WORD) i;        /* First one channel */
       Input[2] = (WORD) 0;



       cmsTrilinearInterp16(Input, Output, Tab, &p);

       if (!IsGood("Channel 1 (I)", (WORD)     0, Output[0])) return 0;
       if (!IsGood("Channel 2 (I)", (WORD) (i/3), Output[1])) return 0;
       if (!IsGood("Channel 3 (I)", (WORD)     i, Output[2])) return 0;


       /* Then, other channel */

       Input[0] = (WORD) 242;      /* Decoupled */
       Input[1] = (WORD) 0;        /* First one channel */
       Input[2] = (WORD) i;


       cmsTrilinearInterp16(Input, Output, Tab, &p);

       if (!IsGood("Channel 1 (II)", (WORD)     0, Output[0])) return 0;
       if (!IsGood("Channel 2 (II)", (WORD) (i/3), Output[1])) return 0;
       if (!IsGood("Channel 3 (II)", (WORD)     i, Output[2])) return 0;



       Input[0] = (WORD) 321;      /* Decoupled */
       Input[1] = (WORD) i;        /* First one channel */
       Input[2] = (WORD) i;


       cmsTrilinearInterp16(Input, Output, Tab, &p);


       if (!IsGood("Channel 1 (III)", 0, Output[0])) return 0;
       }

       printf("Ok!\n");

       return 1;
}



static
void PrintMatrix(LPMAT3 lpM)
{
       int i, j;

       for (i=0; i < 3; i++) {
              printf ("[ ");
              for (j=0; j < 3; j++)
                     {
                            printf("%g  ", (*lpM).v[i].n[j]);
                     }
              printf("]\n");
       }

       printf("\n");

}


static
int TestMatrixCreation(void)
{
       MAT3 Mat;
       int rc;

       cmsCIExyY WhitePt =  {0.3127, 0.3290, 1.0};
       cmsCIExyYTRIPLE Primaries = {
                                   {0.6400, 0.3300, 1.0},
                                   {0.3000, 0.6000, 1.0},
                                   {0.1500, 0.0600, 1.0}
                                   };

       printf("Emulating sRGB from primaries & White point:\n\n");

       rc = cmsBuildRGB2XYZtransferMatrix(&Mat,
                                          &WhitePt,
                                          &Primaries);

       if (rc < 0)
       {
       printf("TestMatrixCreation failed, rc = %d\n", rc);
       return 0;
       }

       printf("sRGB final matrix is:\n"
       "[ 0.436066  0.385147  0.143066 ]\n"
       "[ 0.222488  0.716873  0.060608 ]\n"
       "[ 0.013916  0.097076  0.714096 ]\n");

       printf("\nlcms calculated matrix is:\n");

       PrintMatrix(&Mat);
       return 1;

}



/*

       Used for debug purposes
*/

static
void AdaptationMatrixTest(void)
{
       cmsCIExyY D65 = {0.3127, 0.329001, 1.0};   /* D65 */
       MAT3 sRGB, TosRGB;


       VEC3init(&sRGB.v[0], 0.4124,  0.3576,  0.1805);
       VEC3init(&sRGB.v[1], 0.2126,  0.7152,  0.0722);
       VEC3init(&sRGB.v[2], 0.0193,  0.1192,  0.9505);

       cmsAdaptMatrixToD50(&sRGB, &D65);
       printf("Adaptation matrix D65 -> D50 (to PCS)\n");
       PrintMatrix(&sRGB);

       MAT3inverse(&sRGB, &TosRGB);
       printf("inverse\n");
       PrintMatrix(&TosRGB);

       cmsAdaptMatrixFromD50(&TosRGB, &D65);
       printf("adaptated to D65\n");
       PrintMatrix(&TosRGB);
}

#endif



static
double VecDist(Scanline_rgb2 *bin, Scanline_rgb2 *bout)
{
       double rdist, gdist, bdist;

       rdist = fabs(bout -> r - bin -> r);
       gdist = fabs(bout -> g - bin -> g);
       bdist = fabs(bout -> b - bin -> b);

       return (sqrt((rdist*rdist + gdist*gdist + bdist*bdist)));
}



typedef struct _Stats {
                      double n, x, y, x2, y2, xy;
                      double Peak;
                      } STATS, FAR* LPSTATS;

static void ClearStats(LPSTATS p)
{
       p -> x = p -> y = p -> x2 = p -> y2 = p -> xy
       = p -> Peak = 0.0;
}

static double Std(LPSTATS p)
{
       return sqrt((p->n*p->x2 - p->x * p->x) / (p->n*(p->n-1)));
}



static
void PrintStatistics(clock_t time, LPSTATS Stats)
{

       clock_t diff;
       double a;

       diff = clock() - time;
       a = (double) diff / CLK_TCK;

       /* These are statistics of 16 bit, so divide */
       /* by 256 to get dE relative to 8 bits */

       printf("\rdE : mean=%g, SD=%g, max=%g : ",
                     (Stats->x / Stats -> n) / 256.,
                     (Std(Stats)),
                     Stats -> Peak / 256.);

       printf("%d tics, %g sec.\n", diff, a);

}



/* Perform sampling in the full spectrum & acotate error. */
/* I choose red for the lowest incidence in eye. */
/* Green is most lightful, eye is most accurate on blue. */

static
int TestFullSpectrum(cmsHTRANSFORM xform, int nRedInterv)
{
       int r, g, b;
       double err;
       Scanline_rgb2 *bin, *bout;
       STATS Stats;
       clock_t t;


       bin  = (Scanline_rgb2 *) malloc(256*sizeof(Scanline_rgb2));
       bout = (Scanline_rgb2 *) malloc(256*sizeof(Scanline_rgb2));


       ClearStats(&Stats);

       Stats.x = 0.0; Stats.n = 0.0; /* GCC BUG HERE!!!! */

       t = clock();

       for (r=0; r < 256; r+= nRedInterv)
       {
              printf("\r%02x:", r);

              for (g=0; g < 256; g++)
                     {

                            for (b=0; b < 256; b++)
                            {
                            bin[b].r = r << 8;          /* For L 0nly to 0xFF00 */
                            bin[b].g = RGB_8_TO_16(g);
                            bin[b].b = RGB_8_TO_16(b);
                            }

                            cmsDoTransform(xform, bin, bout, 256);

                            /* I'm using b as index */

                            for (b=0; b < 256; b ++)
                            {
                                   /* I measure the error using vector distance */

                                   err = VecDist(bin+b, bout+b);
                                   Stats.x += (double) err;
                                   Stats.x2 += (double) err * err;
                                   Stats.n += 1.0;
                                   if (err > Stats.Peak)
                                          Stats.Peak = err;


                                   if (err > 0x1000L)
                                   {
                                          printf("Coarse error! : In=(%x,%x,%x) Out=(%x,%x,%x)\n",
                                                        bin[b].r, bin[b].g, bin[b].b,
                                                        bout[b].r, bout[b].g, bout[b].b);
                                          return 0;
                                   }
                            }

                     }

       }


       PrintStatistics(t, &Stats);
       free(bin);
       free(bout);

       return 1;
}




static
int TestInducedError(const char *cProfile, DWORD Type)
{
       cmsHPROFILE *In, *Out;
       cmsHTRANSFORM xform;
       int nMaxError;

       In  = cmsOpenProfileFromFile(cProfile, "rb");
       Out = cmsOpenProfileFromFile(cProfile, "rb");

       printf("Error Induced by the CMM due to roundoff (dE)\n");
       printf("Testing %s:\n", cmsTakeProductName(In));
       xform = cmsCreateTransform(In,  Type,
                                  Out, Type,
                                  INTENT_RELATIVE_COLORIMETRIC, 0);

       printf("!");
       assert(xform);

       nMaxError = TestFullSpectrum(xform, 31);

       cmsDeleteTransform(xform);
       cmsCloseProfile(In);
       cmsCloseProfile(Out);

       return nMaxError;
}


static
double ConvertL(WORD v)
{
       int fix32;

       fix32 = v;
       return (double)fix32/652.800;    /* 0xff00/100.0 */
}


static
double Convertab(WORD v)
{
       int fix32;


       fix32 = v;

       return ((double)fix32/256.0)-128.0;
}


#define BASE  255

static
int CompareTransforms(cmsHTRANSFORM xform1, cmsHTRANSFORM xform2, int nRedInterv)
{
       int r, g, b;
       double err;
       Scanline_rgb2 *bin, *bout1, *bout2;
       clock_t time;
       STATS Stats;
       int OutOfGamut = 0;


       bin   = (Scanline_rgb2 *) malloc(256*sizeof(Scanline_rgb2));
       bout1 = (Scanline_rgb2 *) malloc(256*sizeof(Scanline_rgb2));
       bout2 = (Scanline_rgb2 *) malloc(256*sizeof(Scanline_rgb2));


       ClearStats(&Stats);
       time = clock();

       for (r=0; r < BASE; r+= nRedInterv)
       {
              printf("\r%02x:", r);

              for (g=0; g < BASE; g++)
                     {
                            /* I will test random LSB */

                            for (b=0; b < BASE; b++)     /* 256 */
                            {

                            bin[b].r = RGB_8_TO_16(r);
                            bin[b].g = RGB_8_TO_16(g);
                            bin[b].b = RGB_8_TO_16(b);
                            }

                            cmsDoTransform(xform1, bin, bout1, 256);
                            cmsDoTransform(xform2, bin, bout2, 256);

                            /* I'm using b as index */

                            for (b=0; b < BASE; b ++) {

                                   /* I measure the error using vector distance */
                                   /* Only if encodable values */

                              if (bout1[b].r != 0xffff && bout1[b].g != 0xffff && bout1[b].b != 0xffff)
                              {

                                   err = VecDist(bout1+b, bout2+b);


                                   if (err > 0x1000L)
                                   {


                                          printf("Coarse error: In=(%x,%x,%x) Out1=(%g,%g,%g) Out2=(%g,%g,%g)\n",
                                                        bin[b].r, bin[b].g, bin[b].b,
                                                        ConvertL(bout1[b].r), Convertab(bout1[b].g), Convertab(bout1[b].b),
                                                        ConvertL(bout2[b].r), Convertab(bout2[b].g), Convertab(bout2[b].b));
                                          return 0;

                                   }

                                   else
                                   {
                                   Stats.x += (double) err;
                                   Stats.x2 += (double) err * err;
                                   Stats.n += 1.0;
                                   if (err > Stats.Peak)
                                          Stats.Peak = err;
                                   }
                              } else
                                   OutOfGamut++;
                            }

                     }

       }


       /* PrintStatistics(time, &Stats); */
       printf("\rOk!. Out of encodeable representation=%d\n\n", OutOfGamut);

       free(bin);
       free(bout1);
       free(bout2);

       return 1;
}


static
int TestToLab(void)
{
       cmsHPROFILE In1, In2, Out1, Out2;
       cmsHTRANSFORM xform1, xform2;
       int nMaxErr;

       printf("Testing CIELab conversion:\n");

       /* sRGB is a simpler, public domain XYZ PCS profile */
       /* sRGBSpac comes with Win95 Platform SDK, in the public domain. */
       /* (not latest revisions) */

       /* Using XYZ identity as output profile, I'm forcing an */
       /* implicit Lab -> XYZ conversion */
       /* xform2 is 8 bits - LUT based, and PCS is L*a*b */

       In1   = cmsOpenProfileFromFile(".." DIRSEP "profiles" DIRSEP "sRGB Color Space Profile.icm", "rb");
       Out1  = cmsOpenProfileFromFile(".." DIRSEP "profiles" DIRSEP "lcmsXYZi.icm", "rb");

       In2   = cmsOpenProfileFromFile(".." DIRSEP "profiles" DIRSEP "sRGBSpac.icm", "rb");
       Out2  = cmsOpenProfileFromFile(".." DIRSEP "profiles" DIRSEP "lcmsXYZi.icm", "rb");

       printf("1=%s\n",  cmsTakeProductName(In1));
       printf("2=%s\n",  cmsTakeProductName(In2));

       /* Since LUT is 8-bits width, */
       xform1 = cmsCreateTransform(In1, TYPE_RGBA_16, Out1, TYPE_XYZA_16, 0, cmsFLAGS_NOTPRECALC|cmsFLAGS_MATRIXINPUT);
       xform2 = cmsCreateTransform(In2, TYPE_RGBA_16, Out2, TYPE_XYZA_16, 0, cmsFLAGS_NOTPRECALC);

       nMaxErr = CompareTransforms(xform1, xform2, 31);

       cmsDeleteTransform(xform1);
       cmsCloseProfile(In1);
       cmsCloseProfile(Out1);

       cmsDeleteTransform(xform2);
       cmsCloseProfile(In2);
       cmsCloseProfile(Out2);

       return nMaxErr;
}



/* --------------------------------------------------------- */






static
int TestPreview(void)
{
       cmsHPROFILE In, Out, Proof;
       cmsHTRANSFORM xform;
       int nMaxErr;

       printf("Testing preview\n");

       In  = cmsOpenProfileFromFile(".." DIRSEP "profiles" DIRSEP "lcmsLABi.icm", "rb");
       Out = cmsOpenProfileFromFile(".." DIRSEP "profiles" DIRSEP "lcmsLABi.icm", "rb");
       Proof = cmsOpenProfileFromFile(".." DIRSEP "profiles" DIRSEP "lcmsLABi.icm", "rb");

       xform = cmsCreateProofingTransform(In, TYPE_LABA_16, Out, TYPE_LABA_16, Proof, 0, 0, 0);

       nMaxErr = TestFullSpectrum(xform, 31);

       cmsDeleteTransform(xform);
       cmsCloseProfile(In);
       cmsCloseProfile(Out);
       cmsCloseProfile(Proof);

       return nMaxErr;
}



int main(int argc, char *argv[])
{
       int lExhaustive = 0;


       printf("little cms testbed. ver 1.2 build %s %s\n\n", __DATE__, __TIME__);



#ifndef LCMS_DLL
       TestFixedPoint();
       if (!TestFixedScaling()) return 1;
       if (!TestLinearInterpolation(lExhaustive)) return 1;
#endif

#ifdef USE_FLOAT
       Test3D();
#endif

#ifndef LCMS_DLL
       if (!TestMatrixCreation()) return 1;
#endif

       if (!TestToLab()) return 1;
       if (!TestInducedError(".." DIRSEP "profiles" DIRSEP "lcmsLABi.icm", TYPE_LABA_16)) return 1;
       if (!TestPreview()) return 1;

       printf("\nSuccess.\n");

       return 0;

}
