# libezget.tcl --
#
#    libcisco - Configuration management API for Cisco networking equipment
#    Copyright (C) June 2002  Andy Ziegelbein
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Please send any questions or comments to andy@packetz.org.
#
#
# This file contains the Tcl code for the ezget functions.
#
# The "ezget" family of functions provide a means for getting information from
# a target device.  In order for a function to be in the "ez" family of 
# functions, it must adhere to the following three (3) criteria:
#
#       (1) Provide an OS layer of abstraction.
#       (2) Provide a "command-type" layer of abstraction.
#       (3) Silently return from the function with a success result even if
#           the "command-type" is not applicable for the OS of the target
#           device (e.g. setting a banner message is possible under Catalyst
#           code and IOS, but not under a 1900).
#
# In addition, the following rules apply to procedures contained within this
# file:
#
# Arguments   : ez procedures MAY take a variable number of arguments.
# Options     : ez procedures MAY take optional values.
# SessionId   : ez procedures MUST support an implicit SessionId, which is
#                provided by the 'last_spawn_id" element of the 'state'
#                namespace array variable.
# Relationship: The following list defines the relationship between ez
#                procedures and other procedures defined within this package:
#
#                       Kernel - MUST NOT call kernel procedures.
#                       Exported Kernel - MUST NOT call exported kernel procs.
#                       User - SHOULD call one (1) or more user procedures.
#                       Exported User - MAY call other exported user procs.
#                       Package Support - MAY call package support procedures.
#                       Tcl - MAY call Tcl commands.
#                       Expect - MUST NOT call Expect commands.
#
# RCS|SCCS: %Z% %M% %I% %E% %U%

package provide libcisco 1.0


# libcisco::ezzget --
#
#       This procedure is used to launch the appropriate procedure from the
#       "ezget" family of procedures.  In order to work with the TCL-DP
#       package, the ezget procedure must return the result of the ezget
#       command instead of using a pass-by-reference model using upvar.
#
# Arguments:
#       Cmd             string.  The type of ezset command from the family of
#                        ezset commands.
#       args            The remaining arguments are dependpent upon the Cmd
#                        selected.  See the appropriate proc for additional
#                        arguments types.
#
# Results
#       On success, 0.
#       A short textual message on error

proc ::libcisco::ezget { Cmd args } {
    switch -- $Cmd {
        config {
            RetOnErr [ EzGetConfig $args ] Result
        }
        default {
            return "errBadCmdType"
        }
    }

    return $Result
}


# libcisco::EzGetConfig --
#
#       The EzGetConfig procedure is used to retrieve the configuration from
#       the target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       ConfigType      (optional) string.  The type of config to retrieve.
#                        Valid options are "primary" and "secondary".  What
#                        this translates to on the device is dependent upon
#                        the target device's OS.  The matrix is as follows:
#           
#                                     primary             secondary
#                               IOS   running-config      startup-config
#                               XDI   all                 non-default
#                               1900  running-config      (na)
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::EzGetConfig { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
        set ConfigType [ lindex $args 1 ]
    } else {
        set SessionId $state(last_spawn_id)
        set ConfigType [ lindex $args 0 ]
    }

    # If a ConfigType has not been specified, default it to primary.

    if { ! [ string length $ConfigType ] } {
        set ConfigType "primary"
    }

    # Launch the appropriate procedure.

    switch -- $state($SessionId,Os),$ConfigType {
        ios,primary {
            RetOnErr [ UsrGetConfigIos $SessionId running-config ] ConfigList
        }
        ios,secondary {
            RetOnErr [ UsrGetConfigIos $SessionId startup-config ] ConfigList
        }
        xdi,primary {
            RetOnErr [ UsrGetConfigXdi $SessionId all ] ConfigList
        }
        xdi,secondary {
            RetOnErr [ UsrGetConfigXdi $SessionId nondefault ] ConfigList
        }
        1900,primary {
            RetOnErr [ UsrGetConfig1900 $SessionId running-config ] ConfigList
        }
        1900,secondary {
            # The running-config is the only valid option on the 1900.  Hence,
            # we will silently ignore this request and return success.

            return 0
        }
        default {
            return "errConfigType"
        }
    }

    return $ConfigList
}
