# libezset.tcl --
#
#    libcisco - Configuration management API for Cisco networking equipment
#    Copyright (C) June 2002  Andy Ziegelbein
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Please send any questions or comments to andy@packetz.org.
#
#
# This file contains the Tcl code for the ezset functions.
#
# The "ezset" family of functions provide a means for setting config values on
# a target device.  In order for a function to be in the "ez" family of 
# functions, it MUST adhere to the following three (3) criteria:
#
#       (1) Provide an OS layer of abstraction.
#       (2) Provide a "command-type" layer of abstraction.
#       (3) Silently return from the function with a success result even if
#           the "command-type" is not applicable for the OS of the target
#           device (e.g. setting a banner message is possible under Catalyst
#           code and IOS, but not under a 1900).
#
# In addition, the following rules apply to procedures contained within this
# file:
#
# Arguments   : ez procedures MAY take a variable number of arguments.
# Options     : ez procedures MAY take optional values.
# SessionId   : ez procedures MUST support an implicit SessionId, which is
#                provided by the 'last_spawn_id" element of the 'state'
#                namespace array variable.
# Relationship: The following list defines the relationship between ez
#                procedures and other procedures defined within this package:
#
#                       Kernel - MUST NOT call kernel procedures.
#                       Exported Kernel - MUST NOT call exported kernel procs.
#                       User - SHOULD call one (1) or more user procedures.
#                       Exported User - MAY call other exported user procs.
#                       Package Support - MAY call package support procedures.
#                       Tcl - MAY call Tcl commands.
#                       Expect - MUST NOT call Expect commands.
#
# RCS|SCCS: %Z% %M% %I% %E% %U%

package provide libcisco 1.0


# libcisco::ezset --
#
#       This procedure is used to launch the appropriate procedure from the
#       "ezset" family of procedures.
#
# Arguments:
#       Cmd             string.  The type of ezset command from the family of
#                        ezset commands.
#       args            The remaining arguments are dependpent upon the
#                        'type' selected.  See the appropriate proc for
#                        additional arguments types.
#
# Results
#       0 on success
#       A short textual message on error

proc ::libcisco::ezset { Cmd args } {
    switch -- $Cmd {
        password {
            RetOnErr [ EzSetPassword $args ]
        }
        banner {
            RetOnErr [ EzSetBanner $args ]
        }
        hostname {
            RetOnErr [ EzSetHostname $args ]
        }
        contact {
            RetOnErr [ EzSetContact $args ]
        }
        location {
            RetOnErr [ EzSetLocation $args ]
        }
        community {
            RetOnErr [ EzSetCommunity $args ]
        }
        ippermit {
            RetOnErr [ EzSetIpPermit $args ]
        }
        acl {
            RetOnErr [ EzSetAcl $args ]
        }
        default {
            return "errBadCmdType"
        }
    }

    return 0
}

proc ::libcisco::EzSetPassword { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
        set PwType     [ lindex $args 1 ]
        set CurrentPw  [ lindex $args 2 ]
        set NewPw      [ lindex $args 3 ]
    } else {
        set SessionId $state(last_spawn_id)
        set PwType     [ lindex $args 0 ]
        set CurrentPw  [ lindex $args 1 ]
        set NewPw      [ lindex $args 2 ]
    }

    # Check to make sure NewPw is not null.

    if { ! [ string length $NewPw ] } {
        RetOnErr "errTooFewArgs"
    }

    # Launch the appropriate procedure.

    switch -- $state($SessionId,Os),$PwType {
        ios,login {
            RetOnErr [ UsrGetConfigIos $SessionId running-config ] Config
            RetOnErr [ UsrGetLineRangesIos\
                    $SessionId $Config {vty con aux} ] LineList
            foreach Line $LineList {
                RetOnErr [ UsrSetPasswordLineIos $SessionId $Line $NewPw ]
            }
        }
        ios,enable {
            RetOnErr [ UsrSetPasswordEnableIos $SessionId secret $NewPw ]
        }
        xdi,login {
            RetOnErr [ UsrSetPasswordXdi $SessionId login $CurrentPw $NewPw ]
        }
        xdi,enable {
            RetOnErr [ UsrSetPasswordXdi $SessionId enable $CurrentPw $NewPw ]
        }
        1900,login {
            RetOnErr [ UsrSetPassword1900 $SessionId login $NewPw ]
        }
        1900,enable {
            RetOnErr [ UsrSetPassword1900 $SessionId enable $NewPw ]
        }
        default {
            return "errPwType"
        }
    }

    return 0
}


proc ::libcisco::EzSetBanner { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { file } { boolean } { 0 }
    }

    RetOnErr [ GetOpts $OptList $args OptValue ArgsAfterOpts ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId  [ lindex $ArgsAfterOpts 0 ]
        set Banner     [ lindex $ArgsAfterOpts 1 ]
    } else {
        set SessionId $state(last_spawn_id)
        set Banner     [ lindex $ArgsAfterOpts 0 ]
    }

    # Determine whether or not the Banner variable is a filename or a Tcl
    # list containing the banner and set the BannerList accordingly.

    if { $OptValue(file) } {
        RetOnErr [ AsciiFileToList $Banner ] BannerList
    } else {
        set BannerList $Banner
    }

    if { ! [ llength $BannerList ] } {
        RetOnErr "errEmptyBanner"
    }

    # Launch the appropriate procedure.

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ UsrSetBannerIos $SessionId login $BannerList ]
        }
        xdi {
            RetOnErr [ UsrSetBannerMotdXdi $SessionId $BannerList ]
        }
        1900 {
            # The 1900 doesn't support a login banner.
        }
        default {
            return "errUnknownOs"
        }
    }

    return 0
}


proc ::libcisco::EzSetHostname { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
        set Hostname   [ lindex $args 1 ]
    } else {
        set SessionId $state(last_spawn_id)
        set Hostname   [ lindex $args 0 ]
    }

    if { ! [ string length $Hostname ] } {
        RetOnErr "errEmptyHostname"
    }

    # IOS-based devices and 1900s can support up to 29 characters, but a
    # switch running Catalyst Code will support only 20 characters for the
    # prompt and 256 characters for the system name.  LCD = 20.

    if { [ string length $Hostname ] > 20 } {
        RetOnErr "errNameTooLong"
    }

    # Launch the appropriate procedure.

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ UsrSetHostnameIos $SessionId $Hostname ]
        }
        xdi {
            RetOnErr [ UsrSetPromptXdi $SessionId $Hostname ]
            RetOnErr [ UsrSetSystemNameXdi $SessionId $Hostname ]
        }
        1900 {
            RetOnErr [ UsrSetHostnameIos $SessionId $Hostname ]
        }
        default {
            return "errUnknownOs"
        }
    }

    return 0
}


proc ::libcisco::EzSetContact { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
        set Contact    [ lindex $args 1 ]
    } else {
        set SessionId $state(last_spawn_id)
        set Contact    [ lindex $args 0 ]
    }

    if { ! [ string length $Contact ] } {
        RetOnErr "errEmptyContact"
    }

    # 1900s can support no more than 220 characters for the contact field.
    # IOS and XDI-based devices will support 255 characters.  LCD = 220.

    if { [ string length $Contact ] > 220 } {
        RetOnErr "errContactTooLong"
    }

    # Launch the appropriate procedure.

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ UsrSetSnmpContactIos $SessionId $Contact ]
        }
        xdi {
            RetOnErr [ UsrSetSnmpContactXdi $SessionId $Contact ]
        }
        1900 {
            RetOnErr [ UsrSetSnmpContact1900 $SessionId $Contact ]
        }
        default {
            return "errUnknownOs"
        }
    }

    return 0
}


proc ::libcisco::EzSetLocation { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
        set Location   [ lindex $args 1 ]
    } else {
        set SessionId $state(last_spawn_id)
        set Location   [ lindex $args 0 ]
    }

    if { ! [ string length $Location ] } {
        RetOnErr "errEmptyLocation"
    }

    # 1900s can support no more than 220 characters for the location field.
    # IOS and XDI-based devices will support 255 characters.  LCD = 220.

    if { [ string length $Location ] > 220 } {
        RetOnErr "errLocationTooLong"
    }

    # Launch the appropriate procedure.

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ UsrSetSnmpLocationIos $SessionId $Location ]
        }
        xdi {
            RetOnErr [ UsrSetSnmpLocationXdi $SessionId $Location ]
        }
        1900 {
            RetOnErr [ UsrSetSnmpLocation1900 $SessionId $Location ]
        }
        default {
            return "errUnknownOs"
        }
    }

    return 0
}


proc ::libcisco::EzSetCommunity { args } {
    variable state
    variable config

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { clear } { boolean } { 0 }
    }

    RetOnErr [ GetOpts $OptList $args OptValue ArgsAfterOpts ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId     [ lindex $ArgsAfterOpts 0 ]
        set StringType    [ string tolower [ lindex $ArgsAfterOpts 1 ] ]
        set StringAclList [ lrange $ArgsAfterOpts 2 end ]
    } else {
        set SessionId $state(last_spawn_id)
        set StringType    [ string tolower [ lindex $ArgsAfterOpts 0 ] ]
        set StringAclList [ lrange $ArgsAfterOpts 1 end ]
    }

    # XDI-based devices will support a community string up to 20 characters. 
    # 1900s will support a community string up to 31 characters.  IOS-based
    # devices will support up to 220 characters.  LCD = 20.

    foreach StringAclPair $StringAclList {
        set StringAclSplit [ split $StringAclPair , ]
        set String         [ lindex $StringAclSplit 0 ]

        if { [ string length $String ] > 20 } {
            RetOnErr "errStringTooLong"
        }
    }

    # Launch the appropriate procedure.

    switch -- $state($SessionId,Os) {
        ios {
            # Populate the config array if it isn't already.

            if { ! [ info exists config($SessionId) ] } {
                RetOnErr [ UsrGetConfigIos $SessionId running-config ]
            }

            # Clear the current strings of the specified type.

            if { $OptValue(clear) } {
                if { [ info exists config($SessionId,SnmpStringCfgList) ] } {
                    foreach StringCfg $config($SessionId,SnmpStringCfgList) {
                        set String [ string tolower [ lindex $StringCfg 0 ] ]
                        set Type   [ string tolower [ lindex $StringCfg 1 ] ]

                        if { [ string match $Type $StringType ] } {
                            RetOnErr [ UsrSetSnmpCommunityIos $SessionId\
                                    remove $Type $String "" ]
                        }
                    }
                }
            }

            # Set the new community strings (if any).

            foreach StringAclPair $StringAclList {
                set StringAclSplit [ split $StringAclPair , ]
                set String         [ lindex $StringAclSplit 0 ]
                set Acl            [ lindex $StringAclSplit 1 ]

                RetOnErr [ UsrSetSnmpCommunityIos $SessionId add\
                            $StringType $String $Acl ]
            }
        }
        xdi {
            # Only the first string in the list will be used as CatCode based
            # switches support only one community string per type.  If the 
            # string is empty, the specified string type will be cleared.

            set StringAclPair  [ lindex $StringAclList 0 ]
            set StringAclSplit [ split $StringAclPair , ]
            set String         [ lindex $StringAclSplit 0 ]
            set Acl            [ lindex $StringAclSplit 1 ]

            switch -- $StringType {
                ro {
                    RetOnErr [ UsrSetSnmpCommunityXdi $SessionId\
                            read-only $String ]
                }
                rw {
                    RetOnErr [ UsrSetSnmpCommunityXdi $SessionId\
                            read-write $String ]
                    RetOnErr [ UsrSetSnmpCommunityXdi $SessionId\
                            read-write-all $String ]
                }
                default {
                    RetOnErr "errUnknownType"
                }
            }
        }
        1900 {
            # Populate the config array if it isn't already.

            if { ! [ info exists config($SessionId) ] } {
                RetOnErr [ UsrGetConfig1900 $SessionId running-config ]
            }

            # Clear the current strings of the specified type.

            if { $OptValue(clear) } {
                if { [ info exists config($SessionId,SnmpStringCfgList) ] } {
                    foreach StringCfg $config($SessionId,SnmpStringCfgList) {
                        set String [ lindex $StringCfg 0 ]
                        set Type   [ lindex $StringCfg 1 ]

                        if { [ string match $Type $StringType ] } {
                            RetOnErr [ UsrSetSnmpCommunity1900 $SessionId\
                                    remove $Type $String ]
                        }
                    }
                }
            }

            # Set the new community strings (if any).

            set Count 0
            foreach StringAclPair $StringAclList {
                set StringAclSplit [ split $StringAclPair , ]
                set String         [ lindex $StringAclSplit 0 ]

                RetOnErr [ UsrSetSnmpCommunity1900 $SessionId add\
                        $StringType $String ]

                # The 1900 will support only four community strings.

                incr Count
                if { $Count == 4 } {
                    break
                }
            }
        }
        default {
            return "errUnknownOs"
        }
    }

    return 0
}

proc ::libcisco::EzSetIpPermit { args } {
    variable state
    variable config

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { file } { boolean } { 0 }
    }

    RetOnErr [ GetOpts $OptList $args OptValue ArgsAfterOpts ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId  [ lindex $ArgsAfterOpts 0 ]
        set PermitList [ lindex $ArgsAfterOpts 1 ]
    } else {
        set SessionId $state(last_spawn_id)
        set PermitList [ lindex $ArgsAfterOpts 0 ]
    }

    # Determine whether or not the PermitList variable is a filename or a Tcl
    # list containing the permit statements and set the PermitList accordingly.

    if { $OptValue(file) } {
        RetOnErr [ AsciiFileToList $PermitList ] PermitList
    }

    if { ! [ llength $PermitList ] } {
        RetOnErr "errEmptyPermit"
    }

    # Cleanup the list removing leading/trailing spaces and empty lines.

    foreach Line $PermitList { 
        if { [ string length $Line ] } {
            lappend NewPermitList [ string trim $Line ]
        }
    }

    set PermitList $NewPermitList

    switch -- $state($SessionId,Os) {
        ios {
            # IOS devices do not support a permit list.
        }
        xdi {
            # Populate the config array if it isn't already.

            if { ! [ info exists config($SessionId) ] } {
                RetOnErr [ UsrGetConfigXdi $SessionId all ]
            }

            if { ! [ info exists config($SessionId,Version) ] } {
                RetOnErr "errCodeVer"
            }

            set Version  [ string trim $config($SessionId,Version) "()" ]
            set VerList  [ split $Version ".(" ]
            set MajorVer [ lindex $VerList 0 ]
            set MinorVer [ lindex $VerList 1 ]

            if { ($MajorVer == 5 && $MinorVer >= 4) || ($MajorVer > 5) } {
                RetOnErr [ EzSetIpPermit54Xdi $SessionId $PermitList ]
            } else {
                RetOnErr [ EzSetIpPermit10Xdi $SessionId $PermitList ]
            }
        }
        1900 {
            # 1900s do not support a permit list.
        }
        default {
            return "errUnknownOs"
        }
    }

    return 0
}

proc ::libcisco::EzSetIpPermit10Xdi { SessionId PermitList } {
    variable state
    variable config
    
    # If the existing list matches the new list, do nothing.

    if { [ info exists config($SessionId,PermitList) ] } {
        if { [ PermitMatch $config($SessionId,PermitList) $PermitList ] } {
            return 0
        } else {
            RetOnErr [ UsrSetIpPermitXdi $SessionId disable ]
            RetOnErr [ UsrDoClearIpPermitXdi $SessionId all ]

            foreach NewEntry $PermitList {
                RetOnErr [ UsrSetIpPermitXdi $SessionId "$NewEntry" ]
            }

            # If the permit list was enabled before, re-enable it.

            if { [ info exists config($SessionId,IpPermitTelnet) ] } {
                if { $config($SessionId,IpPermitTelnet) } {
                    UsrSetIpPermitXdi $SessionId enable
                }
            }
        }
    } else {
        RetOnErr [ UsrSetIpPermitXdi $SessionId disable ]
        RetOnErr [ UsrDoClearIpPermitXdi $SessionId all ]

        foreach NewEntry $PermitList {
            RetOnErr [ UsrSetIpPermitXdi $SessionId "$NewEntry" ]
        }

        # If the permit list was enabled before, re-enable it.

        if { [ info exists config($SessionId,IpPermitTelnet) ] } {
            if { $config($SessionId,IpPermitTelnet) } {
                UsrSetIpPermitXdi $SessionId enable
            }
        }
    }

    return 0
}

proc ::libcisco::EzSetIpPermit54Xdi { SessionId PermitList } {
    variable state
    variable config

    # If the existing list matches the new list, do nothing.

    if { [ info exists config($SessionId,PermitList) ] } {
        if { [ PermitMatch $config($SessionId,PermitList) $PermitList ] } {
            return 0
        } else {
            RetOnErr [ UsrSetIpPermitXdi $SessionId "disable" ]
            RetOnErr [ UsrDoClearIpPermitXdi $SessionId all ]

            foreach NewEntry $PermitList {
                RetOnErr [ UsrSetIpPermitXdi $SessionId "$NewEntry" ]
            }

            # If the permit list was enabled before, re-enable it.

            if { [ info exists config($SessionId,IpPermitTelnet) ] } {
                if { $config($SessionId,IpPermitTelnet) } {
                    UsrSetIpPermitXdi $SessionId "enable telnet"
                }
            }

            if { [ info exists config($SessionId,IpPermitSnmp) ] } {
                if { $config($SessionId,IpPermitSnmp) } {
                    UsrSetIpPermitXdi $SessionId "enable snmp"
                }
            }
        }
    } else {
        RetOnErr [ UsrSetIpPermitXdi $SessionId "disable telnet" ]
        RetOnErr [ UsrDoClearIpPermitXdi $SessionId all ]

        foreach NewEntry $PermitList {
            RetOnErr [ UsrSetIpPermitXdi $SessionId "$NewEntry" ]
        }

        # If the permit list was enabled before, re-enable it.

        if { [ info exists config($SessionId,IpPermitTelnet) ] } {
            if { $config($SessionId,IpPermitTelnet) } {
                UsrSetIpPermitXdi $SessionId "enable telnet"
            }
        }

        if { [ info exists config($SessionId,IpPermitSnmp) ] } {
            if { $config($SessionId,IpPermitSnmp) } {
                UsrSetIpPermitXdi $SessionId "enable snmp"
            }
        }
    }

    return 0
}

proc ::libcisco::EzSetAcl { args } {
    variable state
    variable config

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { file } { boolean } { 0 }
    }

    RetOnErr [ GetOpts $OptList $args OptValue ArgsAfterOpts ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId [ lindex $ArgsAfterOpts 0 ]
        set AclNum    [ lindex $ArgsAfterOpts 1 ]
        set AclList   [ lindex $ArgsAfterOpts 2 ]
    } else {
        set SessionId $state(last_spawn_id)
        set AclNum    [ lindex $ArgsAfterOpts 0 ]
        set AclList   [ lindex $ArgsAfterOpts 1 ]
    }

    # Determine whether or not the AclList variable is a filename or a Tcl
    # list containing the entries and set the AclList accordingly.

    if { $OptValue(file) } {
        RetOnErr [ AsciiFileToList $AclList ] AclList
    }

    if { ! [ llength $AclList ] } {
        RetOnErr "errEmptyAcl"
    }

    # Cleanup the list removing leading/trailing spaces and empty lines.

    foreach Line $AclList { 
        if { [ string length $Line ] } {
            lappend NewAclList [ string trim $Line ]
        }
    }

    set AclList $NewAclList

    switch -- $state($SessionId,Os) {
        ios {
            if { ! [ info exists config($SessionId) ] } {
                RetOnErr [ UsrGetConfigIos $SessionId running-config ]
            }

            if { [ info exists config($SessionId,Acl,$AclNum) ] } {
                if { [ AclMatch $config($SessionId,Acl,$AclNum) $AclList ] } {
                    return 0
                } else {
                    RetOnErr [ UsrDoClearNumAclIos $SessionId $AclNum ]
                    RetOnErr [ UsrSetNumAclIos $SessionId $AclNum $AclList ]
                }
            } else {
                RetOnErr [ UsrSetNumAclIos $SessionId $AclNum $AclList ]
            }
        }
        xdi {
            # CatCode doesn't have an access-list.
        }
        1900 {
            # 1900s do not support an access-list.
        }
        default {
            return "errUnknownOs"
        }
    }

    return 0
}
