# libkernel.tcl --
#
#    libcisco - Configuration management API for Cisco networking equipment
#    Copyright (C) June 2002  Andy Ziegelbein
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Please send any questions or comments to andy@packetz.org.
#
#
# This file contains the Tcl code for the kernel procedures.  Kernel procedures
# provide all of the communication contructs for communicating with and
# network devices (routers & switches) via a telnet or ssh session.
#
# The following rules apply to procedures contained within this file:
#
# Arguments   : Kernel procedues SHOULD take a fixed number of arguments.
# Options     : Kernel procedures MUST NOT take options.
# SessionId   : Kernel procedures MUST be given an explicit SessionId.
# Relationship: The following list defines the relationship between kernel
#                procedures, and other procedures defined within this package:
#
#                       Kernel - MAY call other kernel procedures.
#                       Exported Kernel - MUST NOT call exported kernel
#                               procedures.
#                       User - MUST NOT call user procedures.
#                       Exported User - MUST NOT call exported user procedures.
#                       Package Support - MAY call package support procedures.
#                       Tcl - MAY call Tcl commands.
#                       Expect - SHOULD call one (1) or more Expect commands.
#
# RCS|SCCS: %Z% %M% %I% %E% %U%

package provide libcisco 1.0


proc ::libcisco::SessionCloseTelnet { SessionId Retries timeout } {
    variable prompt

    RetOnErr [ ExpectInit $SessionId ]

    # Set boolean value of LoggedOut to False

    set LoggedOut 0
    set Attempts  0

    set PatList {
        timeout {
            SessionCleanup $SessionId close_wait
            RetOnErr "errTimeout"
        }
    }
    set PatList [ subst $PatList ]
    expect_before -brace $PatList

    # Attempt a clean logout.

    while { $Attempts < $Retries } {
        exp_send "exit\r"
        incr Attempts

        expect {
            eof {
                SessionCleanup $SessionId wait
                set LoggedOut 1
                break
            }
            -nocase -re "exit\[\r\n]{1,3}session disconnected" {
                LoggerMark $SessionId ""
                LoggerMark $SessionId "# Sending SIGHUP"
                SessionCleanup $SessionId close_wait
                set LoggedOut 1
                break
            }
            -nocase -re "exit\[\r\n]{1,3}session $" {
                # It's possible for the string "session " to be misinterpreted
                # as a valid xdi,user prompt, so we need this pattern to
                # handle this case.

                LoggerMark $SessionId ""
                LoggerMark $SessionId "# Sending SIGHUP"
                SessionCleanup $SessionId close_wait
                set LoggedOut 1
                break
            }
            -nocase -re "invalid input detected(.*)\[#>]$" {
                RetOnErr "errInvalidInput"
            }
            -nocase -re $prompt(ios,user) {
            }
            -nocase -re $prompt(ios,enable) {
            }
            -nocase -re $prompt(xdi,user) {
            }
            -nocase -re $prompt(xdi,enable) {
            }
        }

    }; # end of while

    if { ! $LoggedOut } {
        # A clean logout did not happen; therefore, send a SIGHUP.

        SessionCleanup $SessionId close_wait
    }

    return 0
}


# libcisco::ExpectInit --
#
#       The ExpectInit proc contains the standard initialization code for 
#       ALL expect commands used in this file.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       args            string.  Each remaining argument is treated as a local,
#                        namespace, or global variable to be set in the calling
#                        proc's context.  These arguments are in the form
#                        of 'variable=value' for local variables.  Global and
#                        namespace variables are not set in this proc, but
#                        rather are simply brought into the calling proc's
#                        context.  Namespace variables are brought into the
#                        current context with 'variable=variable_name' and 
#                        global variables with 'global=variable_name'.
#
# Results:
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::ExpectInit { SessionId args } {
    variable prompt
    set spawn_id $SessionId

    # Verify that the SessionId is valid.

    RetOnErr [ ValidateSessionId $SessionId ]

    # Set the spawn_id in the calling proc and bring the prompt namespace
    # variable into the calling proc's context.

    uplevel "set spawn_id $SessionId"

    # Install the log handler.

    uplevel "LoggerInstall $SessionId"

    # Set any local, namespace, or global variables.

    foreach Arg $args {
        set VarPair [ split $Arg "=" ]
        
        if { [ llength $VarPair ] != 2 } {
            RetOnErr "errBadArg"
        } else {
            set Variable [ lindex $VarPair 0 ]
            set Value    [ lindex $VarPair 1 ]
         
            switch -- $Variable {
                variable {
                    uplevel "variable $Value"
                }
                global {
                    uplevel "global $Value"
                }
                default {
                    uplevel "set $Variable $Value"
                }
            }
        }
    }

    # Set the default expect_before pattern.

    set PatList {
        timeout {
            expect *
            RetOnErr "errTimeout"
        }
        eof {
            SessionCleanup $SessionId wait
            RetOnErr "errEof"
        }
        -nocase -re "invalid input detected(.*)$prompt(ios,enable)" {
            RetOnErr "errInvalidInput"
        }
        -nocase -re "incomplete command(.*)$prompt(ios,enable)" {
            RetOnErr "errIncompleteCmd"
        }
        -nocase -re "ambiguous command(.*)$prompt(ios,enable)" {
            RetOnErr "errAmbiguousCmd"
        }
        -nocase -re "unknown command" {
            RetOnErr "errUnknownCmd"
        }
        -nocase -re "usage:(.*) $" {
            RetOnErr "errUsage"
        }
        -nocase -re "non-designated router(.*)$prompt(ios,enable)" {
            RetOnErr "errNonDesgRtr"
        }
        -nocase -re "--more--" {
            exp_send " "
            exp_continue
        }
    }

    expect_before -brace $PatList

    return 0
}


# libcisco::Login --
#
#       The Login procedure will attempt to login to the target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Os              string.  The expected OS of the target device.  Valid
#                        options are 'ios', 'xdi', '1900', and 'auto'.
#       LinePw          string.  The login password used as long as a username
#                        prompt was not detected first.
#       Username        string.  The login username.
#       UserPw          string.  The login password to use if a username prompt
#                        is detected.
#       Retries         unsigned integer.  The maximum number of times a
#                        password will be attempted.
#       UserLoginPrompt string.  A regular expression used to match the
#                        username login prompt.
#       PasswordPrompt  string.  A regular expression used to match the
#                        password prompt.
#       timeout         unsigned integer.  The timeout value used by Expect.
#
# Results:
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::Login { SessionId Os LinePw Username UserPw Retries\
        UserLoginPrompt PasswordPrompt timeout } {
    variable prompt
    variable state

    # Check to ensure a valid OS has been passed in.

    if { [ lsearch -exact "ios xdi 1900 auto" $Os ] == -1 } {
        RetOnErr "errUnknownOs"
    }

    # Ensure the UserLoginPrompt and PasswordPrompt are lowercase for -nocase
    # comparison.

    set UserLoginPrompt [ string tolower $UserLoginPrompt ]
    set PasswordPrompt  [ string tolower $PasswordPrompt  ]

    # Certain characters must be escaped as they have special meaning to
    # the regular expression pattern matcher.

    set UserLoginPrompt [ EscapeChars $UserLoginPrompt regexp ]
    set PasswordPrompt  [ EscapeChars $PasswordPrompt regexp ]

    set LoginAttempts 0

    # Set the initial password to be that of the line password; this password
    # will be used unless a username prompt is detected.

    set Password $LinePw

    RetOnErr [ ExpectInit $SessionId ]

    # Set the recurring expect before pattern.

    set PatList {
        timeout {
            SessionCleanup $SessionId close_wait
            RetOnErr "errTimeout"
        }
        eof {
            SessionCleanup $SessionId wait
            RetOnErr "errEof"
        }
    }
    set PatList [ subst $PatList ]
    expect_before -brace $PatList

    set DetectedOs ""

    # The following expect block will loop until either (1) an error occurs,
    # (2) we successfully login, (3) the retry limit is met or exceeded
    # without a successful login, or (4) the remote end closes the connection.

    expect {
        -nocase -re "selection: {0,2}$" {
            if { ! [ string match $Os 1900 ] && ! [ string match $Os auto ] } {
                RetOnErr "errUnexpectedOs"
            }

            set DetectedOs 1900

            exp_send "k"
            exp_continue
        }
        -nocase -re "${UserLoginPrompt} ?$" {
            if { [ string match $Os "1900" ] &&\
                    ! [ string match $DetectedOs 1900 ] } {
                RetOnErr "errUnexpectedOs"
            }

            # The username prompt has been detected; therefore, we must change
            # the password to be that which is associated with the username.

            set Password $UserPw

            exp_send "$Username\r"
            exp_continue
        }
        -nocase -re "${UserLoginPrompt} ?t$" {
            if { [ string match $Os "1900" ] &&\
                    ! [ string match $DetectedOs 1900 ] } {
                RetOnErr "errUnexpectedOs"
            }

            # Note: For some reason the letter 't' will appear as the first
            #       letter of the username.  This is an IOS bug.  To 
            #       compensate, we will simply preceed our username with a 
            #       backspace.

            set Password $UserPw

            exp_send "\b$Username\r"
            exp_continue
        }
        -nocase -re "unable to contact server.*${PasswordPrompt} ?$" {
            # If a AAA server is down, a CatCode switch will still prompt
            # for a username.  When the device times out, it will print an
            # "unable to contact server" message at which point it will
            # usually (depends on config) revert to a local password.

            set Password $LinePw

            exp_send "$Password\r"
            exp_continue
        }
        -nocase -re "unable to contact server.*password: ?$" {
            # If a AAA server is down, a CatCode switch will still prompt
            # for a username.  When the device times out, it will print an
            # "unable to contact server" message at which point it will
            # usually (depends on config) revert to a local password.

            set Password $LinePw

            exp_send "$Password\r"
            exp_continue
        }
        -nocase -re "${PasswordPrompt} ?$" {
            if { [ string match $Os "1900" ] &&\
                    ! [ string match $DetectedOs 1900 ] } {
                RetOnErr "errUnexpectedOs"
            }

            # This is the only place we check to see if we have met our retry
            # limit.  It's not perfect, but it should still work well.

            if { $LoginAttempts < $Retries } {
                exp_send "$Password\r"
                incr LoginAttempts
                exp_continue
            } else {
                SessionCleanup $SessionId close_wait
                RetOnErr "errLoginFailed"
            }
        }
        -nocase -re $prompt(ios,user) {
            if { ! [ string match $DetectedOs 1900 ] } {
                set DetectedOs "ios"
            }

            set state($SessionId,Os)   $DetectedOs
            set state($SessionId,Mode) user
        }
        -nocase -re $prompt(ios,enable) {
            if { ! [ string match $DetectedOs 1900 ] } {
                set DetectedOs "ios"
            }

            set state($SessionId,Os)   $DetectedOs
            set state($SessionId,Mode) enable
        }
        -nocase -re $prompt(xdi,user) {
            set DetectedOs "xdi"

            set state($SessionId,Os)   $DetectedOs
            set state($SessionId,Mode) user
        }
        -nocase -re $prompt(xdi,enable) {
            set DetectedOs "xdi"

            set state($SessionId,Os)   $DetectedOs
            set state($SessionId,Mode) enable
        }

    }; # end of expect

    if { ! [ string match $DetectedOs $Os ] && ! [ string match $Os auto ] } {
        RetOnErr "errUnexpectedOs"
    }

    return 0
}


# libcisco::LoginModule --
#
#       The LoginModule procedure will attempt to login to module on an
#       already connected device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Module          string.  The module number.
#       Os              string.  The expected OS of the module.  Valid
#                        options are 'ios' and 'auto'.
#       LinePw          string.  The login password used as long as a username
#                        prompt was not detected first.
#       Username        string.  The login username.
#       UserPw          string.  The login password to use if a username prompt
#                        is detected.
#       Retries         unsigned integer.  The maximum number of times a
#                        password will be attempted.
#       UserLoginPrompt string.  A regular expression used to match the
#                        username login prompt.
#       PasswordPrompt  string.  A regular expression used to match the
#                        password prompt.
#       timeout         unsigned integer.  The timeout value used by Expect.
#
# Results:
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::LoginModule { SessionId Module Os LinePw Username UserPw\
        Retries UserLoginPrompt PasswordPrompt timeout } {
    variable prompt
    variable state

    # Check to ensure a valid OS has been passed in.

    if { [ lsearch -exact "ios auto" $Os ] == -1 } {
        RetOnErr "errUnknownOs"
    }

    # Ensure the UserLoginPrompt and PasswordPrompt are lowercase for -nocase
    # comparison.

    set UserLoginPrompt [ string tolower $UserLoginPrompt ]
    set PasswordPrompt  [ string tolower $PasswordPrompt  ]

    # Certain characters must be escaped as they have special meaning to
    # the regular expression pattern matcher.

    set UserLoginPrompt [ EscapeChars $UserLoginPrompt regexp ]
    set PasswordPrompt  [ EscapeChars $PasswordPrompt regexp ]

    set LoginAttempts 0

    # Set the initial password to be that of the line password; this password
    # will be used unless a username prompt is detected.

    set Password $LinePw

    RetOnErr [ ExpectInit $SessionId ]

    set DetectedOs ""

    RetOnErr [ GetCurrentPrompt $SessionId ] Prompt

    # The following expect block will loop until either (1) an error occurs,
    # (2) we successfully login, (3) the retry limit is met or exceeded
    # without a successful login, or (4) the remote end closes the connection.

    exp_send "session $Module\r"

    expect {
        -nocase -re "module number must be in the range" {
            RetOnErr "errBadModNum"
        }
        -nocase -re "module $Module is not installed" {
            RetOnErr "errBadModNum"
        }
        -nocase -re "feature not supported" {
            RetOnErr "errModNotSupp"
        }
        -nocase -re "unable to tunnel" {
            RetOnErr "errTunnelErr"
        }
        -nocase -re "${UserLoginPrompt} ?$" {
            # The username prompt has been detected; therefore, we must change
            # the password to be that which is associated with the username.

            set Password $UserPw

            exp_send "$Username\r"
            exp_continue
        }
        -nocase -re "${PasswordPrompt} ?$" {
            # This is the only place we check to see if we have met our retry
            # limit.  It's not perfect, but it should still work well.

            if { $LoginAttempts < $Retries } {
                exp_send "$Password\r"
                incr LoginAttempts
                exp_continue
            } else {
                # We don't want to leave the device in a state where it is 
                # waiting for a password, so  we will attempt up to 5 times
                # to return back to the normal system prompt by entering a
                # blank password.

                exp_send "\r"
                for { set Attempts 0 } { $Attempts < 5 } { incr Attempts } {
                    expect {
                        -nocase -re "$PasswordPrompt ?$" {
                            exp_send "\r"
                        }
                        -nocase -re "password: ?$" {
                            exp_send "\r"
                        }
                        -nocase -re $Prompt {
                            break
                        }
                    }
                }
                RetOnErr "errModLoginFailed"
            }
        }
        -nocase -re "bad passwords.*$Prompt" {
            RetOnErr "errModLoginFailed"
        }
        -nocase -re $prompt(ios,user) {
            set DetectedOs "ios"

            set state($SessionId,Os)   $DetectedOs
            set state($SessionId,Mode) user
        }
        -nocase -re $prompt(ios,enable) {
            set DetectedOs "ios"

            set state($SessionId,Os)   $DetectedOs
            set state($SessionId,Mode) enable
        }
    }; # end of expect

    if { ! [ string match $DetectedOs $Os ] && ! [ string match $Os auto ] } {
        RetOnErr "errUnexpectedOs"
    }

    return 0
}


# libcisco::SpawnTelnet --
#
#       The SpawnTelnet procedure will open a telnet session to the target
#       device.  This is a switch/router OS agnostic function.
#
# Arguments:
#       Host            string.  The hostname or IP address of the destination
#                        device.
#       TcpPort         unsigned integer.  The destination TCP port for the
#                        telnet session.
#       timeout         string.  The timeout value used by expect.
#
# Results:
#       On success, a valid Expect spawn_id.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SpawnTelnet { Host TcpPort timeout } {
    variable state

    # Open telnet session.

    if { [ catch "spawn telnet $Host $TcpPort" ] } {
        RetOnErr "errOpenFailed"
    }

    # Update the state array variable.

    set state($spawn_id,Host) $Host
    set state($spawn_id,TcpPort) $TcpPort

    return $spawn_id
}


# libcisco::PostSpawnCheck --
#
#       The PostSpawnCheck procedure will check the connected device to see if
#       a valid network device login prompt has been detected.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       UserLoginPrompt string.  The expected username prompt.
#       PasswordPrompt  string.  The expected password prompt.
#       timeout         string.  The timeout value used by expect.
#
# Results:
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::PostSpawnCheck { SessionId UserLoginPrompt PasswordPrompt\
        timeout } {
    set spawn_id $SessionId
    variable prompt
    variable state

    # Clear any expect_before or expect_after patterns that may have been
    # left over from a prior errored out call to a proc with the same spawn_id.

    expect_before
    expect_after

    # Certain characters must be escaped as they have special meaning to
    # the regular expression pattern matcher.

    set UserLoginPrompt [ EscapeChars $UserLoginPrompt regexp ]
    set PasswordPrompt  [ EscapeChars $PasswordPrompt regexp ]

    # Set the recurring expect patterns.

    set PatList {
        timeout {
            SessionCleanup $SessionId close_wait
            RetOnErr "errTimeout"
        }
        eof {
            SessionCleanup $SessionId wait
            RetOnErr "errEof"
        }
    }
    set PatList [ subst $PatList ]
    expect_before -brace $PatList

    if { ! [ string match $state($SessionId,TcpPort) 23 ] } {
        # If a non-standard port is used, we will assume that we are connecting
        # to the console and we will assume that we need to hit 'enter' to
        # activate the line, or at least get feedback for a login prompt.

        sleep 2
        exp_send "\r"
    }

    # The following expect block will loop until either (1) an error occurs,
    # (2) we successfully login, (3) the retry limit is met or exceeded
    # without a successful login, or (4) the remote end closes the connection.

    expect {
        -nocase -notransfer -re "access not permitted" {
            SessionCleanup $SessionId wait
            RetOnErr "errOpenFailed"
        }
        -nocase -notransfer -re "no route to host" {
            SessionCleanup $SessionId wait
            RetOnErr "errOpenFailed"
        }
        -nocase -notransfer -re "connection refused" {
            SessionCleanup $SessionId wait
            RetOnErr "errConnRefused"
        }
        -nocase -notransfer -re "unable to connect"  {
            SessionCleanup $SessionId wait
            RetOnErr "errOpenFailed"
        }
        -nocase -notransfer -re "password required, but none set" {
            SessionCleanup $SessionId wait
            RetOnErr "errOpenFailed"
        }
        -nocase -notransfer -re "aironet" {
            SessionCleanup $SessionId close_wait
            RetOnErr "errAironetLogin"
        }
        -nocase -notransfer -re "hp jet" {
            SessionCleanup $SessionId close_wait
            RetOnErr "errHpJetLogin"
        }
        -nocase -notransfer -re "irix(.*)login:" {
            SessionCleanup $SessionId close_wait
            RetOnErr "errIrixLogin"
        }
        -nocase -notransfer -re "aix(.*)login:" {
            SessionCleanup $SessionId close_wait
            RetOnErr "errAixLogin"
        }
        -nocase -notransfer -re "sunos(.*)login:" {
            SessionCleanup $SessionId close_wait
            RetOnErr "errSunOsLogin"
        }
        -nocase -notransfer -re "hp-ux(.*)login:" {
            SessionCleanup $SessionId close_wait
            RetOnErr "errHpUxLogin"
        }
        -nocase -notransfer -re "linux(.*)login:" {
            SessionCleanup $SessionId close_wait
            RetOnErr "errLinuxLogin"
        }
        -nocase -notransfer -re "selection: {0,2}$" {
        }
        -nocase -notransfer -re "${UserLoginPrompt} ?t$" {
        }
        -nocase -notransfer -re "${UserLoginPrompt} ?$" {
        }
        -nocase -notransfer -re "${PasswordPrompt} ?$" {
        }
        -nocase -notransfer -re "password: ?$" {
        }
        -nocase -notransfer -re "$prompt(xdi,user)" {
        }
        -nocase -notransfer -re "$prompt(xdi,enable)" {
        }
        -nocase -notransfer -re "$prompt(ios,user)" {
        }
        -nocase -notransfer -re "$prompt(ios,enable)" {
        }

    }; # end of expect

    return 0
}


# libcisco::SendCmd --
#
#       The SendCmd proc will send one or more commands from either
#       User or Privileged EXEC mode on the target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Cmd             string.  The command to be issued.
#       PatList         string list.  A list of valid expect patterns and 
#                        actions.
#       timeout         string.  The timeout value used by expect.
#
# Results
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SendCmd { SessionId Cmd PatList timeout } {
    # Initialize the procedure.

    RetOnErr [ ExpectInit $SessionId ]
    RetOnErr [ GetCurrentPrompt $SessionId ] Prompt

    # Setup a trace to save the output from the command to CmdOut.

    trace variable expect_out(buffer) w SendCmdUpdateCmdOut

    # Certain characters must be escaped as they have special meaning to
    # the regular expression pattern matcher.  We don't, however, want the
    # text that is sent via exp_send to be escaped.

    set MatchCmd [ EscapeChars $Cmd regexp ]

    # Set the default pattern list if one has not been specified.

    if { [ llength $PatList ] == 0 } {
        set PatList {
            -nocase -re "\r|\n" {
                exp_continue
            }
            -nocase -re "$Prompt" {
            }
        }
    } elseif { [ string match "nofeedback" $PatList ] } {
        set PatList {
            -nocase -re "${MatchCmd}\[\r\n]{1,3}$Prompt" {
            }
            -nocase -re "${MatchCmd}\[\r\n]{1,3}\[^\r\n]*.*$Prompt" {
                RetOnErr "errUnexpFeedback"
            }
        }
    }

    # Save the current value for match_max and set a new large value.

    set SavedMatchMax [ match_max ]
    match_max 65536

    exp_send "$Cmd\r"
    expect -brace $PatList

    # Restore the saved value for match_max.

    match_max $SavedMatchMax

    # Remove the trace for expect_out(buffer).  The trace appears to be
    # deleted automatically once the variable goes out of scope, but just to
    # be safe, we'll explicitly delete it here.

    trace vdelete expect_out(buffer) w SendCmdUpdateCmdOut

    return $CmdOut
}


# libcisco::SetModeXXX --
#
#       The SetMode family of procedures will place the target device into the
#       appropriate EXEC or configuration mode.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       (ModeInfo)      string.  Some procedures require mode specific
#                        information.  See the procedure for the variable
#                        name.
#
# Results:
#       0 on success.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SetModeEnableToGlobal { SessionId } {
    variable prompt
    variable state

    RetOnErr [ ExpectInit $SessionId timeout=10 ]

    exp_send "configure terminal\r"

    expect {
        -nocase -re $prompt(ios,global) {
            set state($SessionId,Mode) global
        }
    }

    return 0
}

proc ::libcisco::SetModeConfigToEnable { SessionId } {
    variable prompt
    variable state

    RetOnErr [ ExpectInit $SessionId timeout=10 ]

    exp_send "end\r"

    expect {
        -nocase -re $prompt(ios,enable) {
            set state($SessionId,Mode) enable
          
            if { [ info exists state($SessionId,ModeInfo) ] } {
                unset state($SessionId,ModeInfo)
            }
        }
    }

    return 0
}

proc ::libcisco::SetModeConfigToLine { SessionId Line } {
    variable prompt
    variable state

    RetOnErr [ ExpectInit $SessionId timeout=10 ]

    # Set line to all lowercase in accordance with the standard which requires
    # all state information to be in lowercase.

    set Line [ string tolower $Line ]

    switch -- $state($SessionId,Mode) {
        global {
        }
        line {
            # We are already in line configuration mode.  Check to see if the
            # requested lines to be configured match those that we are already
            # configuring.

            if { [ string match $state($SessionId,ModeInfo) $Line ] } {
                return 0
            }
        }
        interface {
        }
        default {
            RetOnErr "errUndefinedMode"
        }
    }

    exp_send "line $Line\r"

    expect {
        -nocase -re $prompt(ios,line) {
            set state($SessionId,Mode) line
            set state($SessionId,ModeInfo) $Line
        }
    }

    return 0
}

proc ::libcisco::SetModeConfigToInterface { SessionId Interface } {
    variable prompt
    variable state

    RetOnErr [ ExpectInit $SessionId timeout=10 ]

    # Set interface to all lowercase in accordance with the standard which
    # requires all state information to be in lowercase.

    set Interface [ string tolower $Interface ]

    switch -- $state($SessionId,Mode) {
        global {
        }
        line {
        }
        interface {
            # We are already in interface configuration mode.  Check to see if
            # the requested interface to be configured matches that which we
            # are already configuring.

            if { [ string match $state($SessionId,ModeInfo) $Interface ] } {
                return 0
            }
        }
        default {
            RetOnErr "errUndefinedMode"
        }
    }

    exp_send "interface $Interface\r"

    expect {
        -nocase -re $prompt(ios,interface) {
            set state($SessionId,Mode) interface
            set state($SessionId,ModeInfo) $Interface
        }
    }

    return 0
}

proc ::libcisco::SetModeConfigToGlobal { SessionId } {
    variable prompt
    variable state

    RetOnErr [ ExpectInit $SessionId timeout=10 ]

    exp_send "exit\r"

    expect {
        -nocase -re $prompt(ios,global) {
            set state($SessionId,Mode) global

            if { [ info exists state($SessionId,ModeInfo) ] } {
                unset state($SessionId,ModeInfo)
            }
        }
    }

    return 0
}


# libcisco::SessionEnableXXX --
#
#       The SessionEnableXXX family of procedures will place the target device
#       into privileged EXEC, or enable mode.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       PwList          string list.  A list of one or more passwords.
#       Retries         unsigned integer.  The number of times each password
#                        in PwList will be attempted before moving on to the
#                        next password.
#       PasswordPrompt  string.  A regular expression used to match the
#                        password prompt.
#       timeout         unsigned integer.  The timeout value used by Expect.
#
# Results:
#       0 on success.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SessionEnableIos { SessionId PwList Username Retries\
        UserLoginPrompt PasswordPrompt timeout } {
    variable prompt

    RetOnErr [ ExpectInit $SessionId ]

    # Certain characters must be escaped as they have special meaning to
    # the regular expression pattern matcher.

    set PasswordPrompt [ EscapeChars $PasswordPrompt regexp ]

    foreach Password $PwList {
        exp_send "\r"

        for { set Attempts 0 } { $Attempts <= $Retries } { incr Attempts } {

            expect {
                -nocase -re $prompt(ios,user) {
                    if { $Attempts < $Retries } {
                        exp_send "enable\r"
                        exp_continue
                    }
                }
                -nocase -re "$UserLoginPrompt ?$" {
                    if { [ string length $Username ] } {
                        exp_send "$Username\r"
                        exp_continue
                    } else {
                        RetOnErr "errEnableFailed"
                    }
                }
                -nocase -re $prompt(ios,enable) {
                    return 0
                }
                -nocase -re "$PasswordPrompt ?$" {
                    if { $Attempts < $Retries } {
                        exp_send "$Password\r"
                    }
                }
    
            }; # end of expect

        }; # end of for

    }; # end of foreach

    # At this point int the procedure, the password has been attempted
    # without success.

    # The following code is necessary in order to ensure that the procedure
    # leaves the device in User mode and not waiting at a "password:" prompt.
    # If after 5 attempts it fails to return to User mode, it will simply
    # give up.

    exp_send "\r"
    for { set Attempts 0 } { $Attempts < 5 } { incr Attempts } {

        expect {
            -nocase -re $prompt(ios,user) {
                break
            }
            -nocase -re "$PasswordPrompt ?$" {
                exp_send "\r"
            }
            -nocase -re "password: ?$" {
                exp_send "\r"
            }
        }
    }

    RetOnErr "errEnableFailed"
}

proc ::libcisco::SessionEnableXdi { SessionId PwList Username Retries\
       UserLoginPrompt PasswordPrompt timeout } {
    variable prompt

    RetOnErr [ ExpectInit $SessionId ]

    # Certain characters must be escaped as they have special meaning to
    # the regular expression pattern matcher.

    set PasswordPrompt [ EscapeChars $PasswordPrompt regexp ]

    foreach Password $PwList {
        exp_send "\r"

        for { set Attempts 0 } { $Attempts <= $Retries } { incr Attempts } {

            expect {
                -nocase -re $prompt(xdi,user) {
                    if { $Attempts < $Retries } {
                        exp_send "enable\r"
                        exp_continue
                    }
                }
                -nocase -re "$UserLoginPrompt ?$" {
                    if { [ string length $Username ] } {
                        exp_send "$Username\r"
                        exp_continue
                    } else {
                        RetOnErr "errEnableFailed"
                    }
                }
                -nocase -re $prompt(xdi,enable) {
                    return 0
                }
                -nocase -re "$PasswordPrompt ?$" {
                    if { $Attempts < $Retries } {
                        exp_send "$Password\r"
                    }
                }

            }; # end of expect

        }; # end of for

    }; # end of foreach

    # At this point int the procedure, all passwords have been attempted
    # without success.

    # The following code is necessary in order to ensure that the procedure
    # leaves the device in User mode and not waiting at a "password:" prompt.
    # If after 5 attempts it fails to return to User mode, it will simply
    # give up.

    exp_send "\r"
    for { set Attempts 0 } { $Attempts < 5 } { incr Attempts } {

        expect {
            -nocase -re $prompt(xdi,user) {
                break
            }
            -nocase -re "$PasswordPrompt ?$" {
                exp_send "\r"
            }
        }
    }

    RetOnErr "errEnableFailed"
}


# libcisco::SessionCleanup
#
#       The SessionCleanup procedure is used to free up system resources after
#       a spawned process has terminated.
#
# Arguments:
#       SessionId   string.  The unique ID of the session.
#       CleanupType (optional) string.  The type of cleanup to perform.
#
# Results:
#       None

proc ::libcisco::SessionCleanup { SessionId { CleanupType close_wait } } {
    variable state
    set spawn_id $SessionId

    # Log the remaining contents of the expect_out buffer and close the
    # logging channels.

    LoggerInstall $SessionId
    catch [ expect * ]
    LoggerChannelsClose $SessionId

    # Free up the memory allocated to the namespace state variable and
    # any user-defined variables following the same naming convention.

    if { [ array exists state ] } {
        # Cleanup any user-defined state arrays for this session.

        if { [ info exists state($SessionId,ArrayList) ] } {
            foreach ArrayName $state($SessionId,ArrayList) {
                if { [ array exists $ArrayName ] } {
                    upvar #0 $ArrayName UserArray
set ElementList [ array names UserArray $SessionId,* ]
set ElementList [ lsort $ElementList ]
                    foreach Element $ElementList {
#puts "Unsetting $ArrayName\($Element\): $UserArray($Element)"
                        unset UserArray($Element)
                    }
                }
            }
        }

        # Cleanup the state array elements for this session.

        foreach Element [ array names state ${SessionId},* ] {
#puts "Unsetting state\($Element\): $state($Element)"
            unset state($Element)
        }
    }

    # Set the CleanupType to lower case for a case insensitive comparison.

    set CleanupType [ string tolower $CleanupType ]

    switch $CleanupType {
        wait {
            # Cleanup resources allocated to the spawned telnet session.

            # Remove the defunct zombie process (reaper).

            wait
        }
        close_wait {
            # Terminate the spawned telnet session and cleanup resources
            # allocated to the session.

            # Send a SIGHUP to the spawned process.

            close

            # Remove the defunct zombie process (reaper).

            wait
        }
        default {
            # Terminate the spawned telnet session and cleanup resources
            # allocated to the session.

            # Send a SIGHUP to the spawned process.

            close

            # Remove the defunct zombie process (reaper).

            wait
        }
    }

    return
}


# libcisco::ValidateSessionId --
#
#       The ValidateSessionId command will verify that the passed in SessionId
#       is a valid spawn_id.  If it not, the procedure will return an error.
#
# Arguments:
#       SessionId   string.  The unique ID of the session.
#
# Results:
#       0 on success (valid SessionId).
#       On error, a short text message beginning with the string "err".

proc ::libcisco::ValidateSessionId { SessionId } {
    # Okay, this is not very elegant, but its the best thing I could find to
    # test if a spawn_id is valid while having minimal to no impact on the
    # spawned process.

    if { [ catch [ parity -i $SessionId 1 ] ] } {
        RetOnErr "errInvalidId"
    } else {
        # The SessionId is a valid spawn_id.

        return 0
    }
}


# libcisco::SendCmdUpdateCmdOut --
#
#       The SendCmdUpdateCmdOut procedure is called by the SendCmd procedures.
#       It is used as a callback command for a variable trace.
#
# Arguments:
#       varName         string.  The name of the variable being traced.
#       Index           string.  Array index name (not applicable).
#       Op              string.  Operation performed on the traced variable.
#
# Results:
#       Appends the contents of the expect_out(buffer) to a variable in the
#       calling procedure by the name of CmdOut.

proc ::libcisco::SendCmdUpdateCmdOut { varName Index Op } {
    uplevel {append CmdOut $expect_out(buffer)}

    return
}


# libcisco::LoggerInstall --
#
#       The LoggerInstall procedure installs a trace for expect_out(buffer) in
#       the calling procedure.  This is the method employed to log output as
#       it is the ONLY method that will work with multiple spawned processes.
#
# Arguments:
#       SessionId   string.  The unique ID of the session.
#
# Results:
#       None

proc ::libcisco::LoggerInstall { SessionId } {
    uplevel {trace variable expect_out(buffer) w LogExpectOut}

    return
}


# libcisco::LoggerChannelsOpen --
#
#       The LoggerChannelsOpen procedure will open a channel identifier for
#       each file requested.  It will also mark the log file with a 
#       LOG BEGIN timestamp.
#
# Arguments:
#       SessionId          string.  The unique ID of the session.
#       StdOutFlag         boolean.  On (1) indicates that output should be
#                           logged to the stdout stream.
#       FileOverWriteList  string list.  A list of filenames to be opened for
#                           logging in overwrite mode.
#       FileAppendList     string list.  A list of filenames to be opened for
#                           logging in append mode.
#
# Results:
#       0 on success.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::LoggerChannelsOpen { SessionId StdOutFlag FileOverwriteList\
        FileAppendList } {
    variable state

    # Add the stdout stream to the LogIdList if requested.

    if { $StdOutFlag } {
        lappend state($SessionId,LogIdList) stdout
    }

    # Open all requested files for overwrite.

    foreach FileName $FileOverwriteList {
        if { [ catch { open $FileName w } FileId ] } {
            RetOnErr "errLogFile"
        } else {
            lappend state($SessionId,LogIdList) $FileId
        }
    }

    # Open all requested files for appending.

    foreach FileName $FileAppendList {
        if { [ catch { open $FileName a } FileId ] } {
            RetOnErr "errLogFile"
        } else {
            lappend state($SessionId,LogIdList) $FileId
        }
    }

    # Mark the start time for logging.

    set CurrentTime [ clock format [ clock seconds ] ]

    lappend LogHeaderList ""
    lappend LogHeaderList "#"
    lappend LogHeaderList "# BEGIN LOG: $CurrentTime"
    lappend LogHeaderList "#"
    lappend LogTrailerList ""

    foreach LogLine $LogHeaderList {
        LoggerMark $SessionId $LogLine
    }
  
    return 0
}


# libcisco::LoggerChannelsClose --
#
#       The LoggerChannelsClose procedure will close all open logging channels
#       (except stdout).  It will also mark the log file with a END OF LOG
#       timestamp.
#
# Arguments:
#       SessionId          string.  The unique ID of the session.
#
# Results:
#       Always returns 0.

proc ::libcisco::LoggerChannelsClose { SessionId } {
    variable state

    # If there are no channels in the list, there is nothing to do.

    if { ! [ info exists state($SessionId,LogIdList) ] } {
        return 0
    }

    # Mark the start time for logging.

    set CurrentTime [ clock format [ clock seconds ] ]

    lappend LogTrailerList ""
    lappend LogTrailerList "#"
    lappend LogTrailerList "# END OF LOG: $CurrentTime"
    lappend LogTrailerList "#"
    lappend LogTrailerList "" 

    foreach LogLine $LogTrailerList {
        LoggerMark $SessionId $LogLine
    }

    # Close all channels except the stdout channel.

    foreach ChannelId $state($SessionId,LogIdList) {
        if { [ string match $ChannelId stdout ] } {
            continue
        } else {
            close $ChannelId
        }
    }

    return 0
}


# libcisco::LoggerMark --
#
#       The LoggerMark procedure will update all open logging channels with
#       one or more lines of text.  All channels except the stdout stream
#       will receive this output.
#
# Arguments:
#       SessionId          string.  The unique ID of the session.
#       LogText            string list.  A text string to be output to the
#                           logging channels.
#
# Results:
#       Always returns 0.

proc ::libcisco::LoggerMark { SessionId LogText } {
    variable state

    if { [ info exists state($SessionId,LogIdList) ] } {
        foreach ChannelId $state($SessionId,LogIdList) {
            if { [ string match $ChannelId stdout ] } {
                # The stdout stream is handled differently.  We send only raw
                # output to the stdout stream with no logging messages.

                continue
            } else {
                puts $ChannelId $LogText
            }
        }
    }

    return 0
}


# libcisco::LogExpectOut --
#
#       The LogExpectOut procedure logs any data written to the
#       expect_out(buffer) variable to the channel identifier stored in the
#       'SessionId,LogIdList' element of the 'state' namespace variable.
#
# Arguments:
#       varName         string.  The name of the variable being traced.
#       Index           string.  Array index name (not applicable).
#       Op              string.  Operation performed on the traced variable.
# 
# Results:
#       Writes the contents of expect_out(buffer) from the calling procedure
#       to the log channel identifier.

proc ::libcisco::LogExpectOut { varName Index Op } {
    variable state
    upvar spawn_id SessionId
    upvar {expect_out(buffer)} ExpectOut

    if { [ info exists state($SessionId,LogIdList) ] } {
        foreach LogId $state($SessionId,LogIdList) {
            puts -nonewline $LogId $ExpectOut
        }
    }

    return
}


# libcisco::GetCurrentPrompt --
#
#       The GetCurrentPrompt procedure will return a regular expression
#       that can be used to match the current prompt of the target device.
#
# Argumetns:
#       SessionId       string.  The unique ID of the session.
#
# Results:
#       On success, a regular expression is returned.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::GetCurrentPrompt { SessionId } {
    variable state
    variable prompt

    if { ! [ info exists state($SessionId,Os) ] ||\
            ! [ info exists state($SessionId,Mode) ] } {
        RetOnErr "errGCPInternal1"
    }

    set Os $state($SessionId,Os)
    set Mode $state($SessionId,Mode)

    if { ! [ info exists prompt($Os,$Mode) ] } {
        RetOnErr "errGCPInternal2"
    }

    return $prompt($Os,$Mode)
}
