# libusrdo.tcl --
#
#    libcisco - Configuration management API for Cisco networking equipment
#    Copyright (C) June 2002  Andy Ziegelbein
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Please send any questions or comments to andy@packetz.org.
#
#
# This file contains the Tcl code for the user functions.
#
# The user functions provide operating-system specific functions that perform
# typically a single task on a target device (e.g. add NTP server).  In order
# for a function to be defined in this file, it MUST adhere to the following
# two (2) criteria:
#
#       (1) The procedure must be written for a specific operating system.
#       (2) The procedure should be written to perform a single task.
#
# In addition, the following rules apply to procedures contained within this
# file:
#
# Arguments   : User procedures MUST NOT take a variable number of arguments.
# Options     : User procedures MUST NOT take optional values.
# SessionId   : User procedures MUST be given an explicit SessionId.
# Relationship: The following list defines the relationship between user
#                procedures and other procedures defined within this package:
#
#                       Kernel - MUST NOT call kernel procedures.
#                       Exported Kernel - SHOULD call one (1) or more kernel
#                               procedures.
#                       User - MAY call other user procedures.
#                       Exported User - MUST NOT call exported user procs.
#                       Package Support - MAY call package support procedures.
#                       Tcl - MAY call Tcl commands.
#                       Expect - MUST NOT call Expect commands.
#
# RCS|SCCS: %Z% %M% %I% %E% %U%

package provide libcisco 1.0


# libcisco::UsrDoCopyRunStartIos
#
#       The UsrDoCopyRunStartIos procedure will copy the running-config to
#       the startup-config.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrDoCopyRunStartIos { SessionId } {
    variable state

    # Check to ensure that the session is not in user mode.

    if { [ string match $state($SessionId,Mode) user ] } {
        return "errWrongMode"
    }

    set Cmd "copy running-config startup-config"

    set PatList {
        -nocase -re "open failed.*$Prompt" {
            RetOnErr "errOpenFailed"
        }
        -nocase -re "retry.*$Prompt" {
            RetOnErr "errSaveFailed"
        }
        -nocase -re "destination filename.*" {
            exp_send "\r"
            exp_continue
        }
        -nocase -re "overwrite the previous nvram configuration" {
            # This error occurs subsequent to a code upgrade when the 
            # configuration in NVRAM was written by an earlier version of
            # code.
        
            exp_send "y\r"
            exp_continue
        }
        -nocase -re "ok.*$Prompt" {
        }
    }

    RetOnErr [ sendCmd exec -timeout 45 -patlist $PatList $SessionId $Cmd ]

    return 0
}


# libcisco::UsrDoClearCountersXXX
#
#       The UsrDoClearCountersXXX procedures will clear the counters on the
#       target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrDoClearCountersIos { SessionId } {
    variable state

    # Check to ensure that the session is not in user mode.

    if { [ string match $state($SessionId,Mode) user ] } {
        return "errWrongMode"
    }

    set Cmd "clear counters"

    set PatList {
        -nocase -re "clear.* counters on all interfaces" {
            exp_send "\r"
            exp_continue
        }
        -nocase -re "$Prompt" {
        }
    }

    RetOnErr [ sendCmd exec -timeout 15 -patlist $PatList $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrDoClearCountersXdi { SessionId } {
    variable state

    # Check to ensure that the session is not in user mode.

    if { [ string match $state($SessionId,Mode) user ] } {
        return "errWrongMode"
    }

    set Cmd "clear counters"

    set PatList {
        -nocase -re "do you want to continue.* ?" {
            exp_send "y\r"
        }
        -nocase -re "failed" {
            RetOnErr "errClearFailed"
        }
        -nocase -re "mac and port counters clear.*$Prompt" {
        }
    }

    RetOnErr [ sendCmd exec -timeout 15 -patlist $PatList $SessionId $Cmd ]

    return 0
}


# libcisco::UsrDoDebugIos --
#
#       The UsrDoDebugIos and UsrDoUndebugIos procedures are used to enable
#       and disable debuging on IOS-based devices.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       ArgList         string list.  A list of arguments to be concatenated
#                        into a single string and then evaluated on the target
#                        device as a command.
#
# Results:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrDoDebugIos { SessionId ArgList } {
    variable state

    # Check to ensure that the session is not in user mode.

    if { [ string match $state($SessionId,Mode) user ] } {
        return "errWrongMode"
    }

    set Cmd "debug"

    foreach Argument $ArgList {
        append Cmd " $Argument"
    }

    RetOnErr [ sendCmd exec $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrDoUndebugIos { SessionId ArgList } {
    variable state

    # Check to ensure that the session is not in user mode.

    if { [ string match $state($SessionId,Mode) user ] } {
        return "errWrongMode"
    }

    set Cmd "undebug"

    foreach Argument $ArgList {
        append Cmd " $Argument"
    }

    RetOnErr [ sendCmd exec $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrDoClearIpPermitXdi { SessionId Arg } {
    variable state

    # Check to ensure that the session is not in user mode.

    if { [ string match $state($SessionId,Mode) user ] } {
        return "errWrongMode"
    }

    set Cmd "clear ip permit $Arg"

    set PatList {
        -nocase -re "not in IP permit list" {
            RetOnErr "errNothingToClear"
        }
        -nocase -re "unknown host.*$Prompt" {
            RetOnErr "errUnknownHost"
        }
        -nocase -re "usage:.*$Prompt" {
            RetOnErr "errUsage"
        }
        -nocase -re "$Cmd\[\r\n\]{1,3}$Prompt" {
        }
        -nocase -re "cleared from.*permit list.*$Prompt" {
        }
        -nocase -re "permit list cleared.*$Prompt" {
        }
    }
    
    RetOnErr [ sendCmd exec -timeout 15 -patlist $PatList $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrDoClearNumAclIos { SessionId AclNum } {
    variable state

    # Check to ensure that the session is not in user mode.

    if { [ string match $state($SessionId,Mode) user ] } {
        return "errWrongMode"
    }

    set Cmd "no access-list $AclNum"

    RetOnErr [ sendCmd global $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrDoClearIpAclIos { SessionId Type AclName } {
    variable state

    # Check to ensure that the session is not in user mode.

    if { [ string match $state($SessionId,Mode) user ] } {
        return "errWrongMode"
    }

    set Cmd "no ip access-list $Type $AclName"

    RetOnErr [ sendCmd global $SessionId $Cmd ]

    return 0
}
