# libusrget.tcl --
#
#    libcisco - Configuration management API for Cisco networking equipment
#    Copyright (C) June 2002  Andy Ziegelbein
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Please send any questions or comments to andy@packetz.org.
#
#
# This file contains the Tcl code for the user functions.
#
# The user functions provide operating-system specific functions that perform
# typically a single task on a target device (e.g. add NTP server).  In order
# for a function to be defined in this file, it MUST adhere to the following
# two (2) criteria:
#
#       (1) The procedure must be written for a specific operating system.
#       (2) The procedure should be written to perform a single task.
#
# In addition, the following rules apply to procedures contained within this
# file:
#
# Arguments   : User procedures MUST NOT take a variable number of arguments.
# Options     : User procedures MUST NOT take optional values.
# SessionId   : User procedures MUST be given an explicit SessionId.
# Relationship: The following list defines the relationship between user
#                procedures and other procedures defined within this package:
#
#                       Kernel - MUST NOT call kernel procedures.
#                       Exported Kernel - SHOULD call one (1) or more kernel
#                               procedures.
#                       User - MAY call other user procedures.
#                       Exported User - MUST NOT call exported user procs.
#                       Package Support - MAY call package support procedures.
#                       Tcl - MAY call Tcl commands.
#                       Expect - MUST NOT call Expect commands.
#
# RCS|SCCS: %Z% %M% %I% %E% %U%

package provide libcisco 1.0


# libcisco::UsrGetConfigXXX
#
#       The UsrGetConfig procedure is used to retrieve the configuration from
#       the target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       ConfigType      string.  The type of config to retrieve.
#
# Retsults:
#       On success, the configuration of the target device is returned.
#       A short text message on error.

proc ::libcisco::UsrGetConfigIos { SessionId ConfigType } {
    variable state

    # Check to ensure that the session is not in user mode.

    if { [ string match $state($SessionId,Mode) user ] } {
        return "errWrongMode"
    }

    switch -regexp -- $ConfigType {
        run.* {
            set Cmd "show running-config"
        }
        start.* {
            set Cmd "show startup-config"
        }
        default {
            return "errBadConfig"
        }
    }

    RetOnErr [ sendCmd exec $SessionId $Cmd ] Result

    set StartIndex [ string first "!" $Result ]
    set StopIndex  [ string last "end" $Result ]

    if { ( $StartIndex != -1 ) && ( $StopIndex != -1 ) } {
        set StopIndex [ expr { $StopIndex + 2 } ]
        set Config [ string range $Result $StartIndex $StopIndex ]
    }

    switch -regexp -- $ConfigType {
        run.* {
            RetOnErr [ UsrPopulateConfigArrayIos $SessionId $Config ]
        }
    }

    return $Config
}

proc ::libcisco::UsrGetConfigXdi { SessionId ConfigType } {
    variable state

    # Check to ensure that the session is not in user mode.

    if { [ string match $state($SessionId,Mode) user ] } {
        return "errWrongMode"
    }

    switch -regexp -- $ConfigType {
        all {
            set Cmd "show config all"
        }
        non.* {
            set Cmd "show config"
        }
        default {
            return "errBadConfig"
        }
    }

    RetOnErr [ sendCmd exec $SessionId $Cmd ] Result

    set StartIndex [ string first "begin" $Result ]
    set StopIndex  [ string last "end" $Result ]

    if { ( $StartIndex != -1 ) && ( $StopIndex != -1 ) } {
        set StopIndex [ expr { $StopIndex + 2 } ]
        set Config [ string range $Result $StartIndex $StopIndex ]
    }

    switch -regexp -- $ConfigType {
        all {
            RetOnErr [ UsrPopulateConfigArrayXdi $SessionId $Config ]
        }
    }

    return $Config
}

proc ::libcisco::UsrGetConfig1900 { SessionId ConfigType } {
    variable state

    # Check to ensure that the session is not in user mode.

    if { [ string match $state($SessionId,Mode) user ] } {
        return "errWrongMode"
    }

    switch -regexp -- $ConfigType {
        run.* {
            set Cmd "show running-config"
        }
        default {
            return "errBadConfig"
        }
    }

    RetOnErr [ sendCmd exec $SessionId $Cmd ] Result

    set Result     [ RemoveMorePrompts $Result ]

    set StartIndex [ string first "!" $Result ]
    set StopIndex  [ string last "end" $Result ]

    if { ( $StartIndex != -1 ) && ( $StopIndex != -1 ) } {
        set StopIndex [ expr { $StopIndex + 2 } ]
        set Config [ string range $Result $StartIndex $StopIndex ]
    }

    switch -regexp -- $ConfigType {
        run.* {
            RetOnErr [ UsrPopulateConfigArray1900 $SessionId $Config ]
        }
    }

    return $Config
}


# UsrGetLineRangesIos --
#
#       The UsrGetLineRangesIos proc will get a list of valid line ranges for
#       the specified line type(s).  The result will be stored in a variable
#       that is passed in by reference.   The result will be of type list in
#       the following format:
#
#               { {vty 0 4} {con 0} ... }
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Config          string.  An IOS configuration file as returned
#                        by the UsrGetConfig procedures.
#       LineTypeList    string.  A list of regular expressions where each
#                        expression is used to match a desired line type.
#                        Typical line types are aux, con, vty, tty.
#
# Results:
#       On success, a list of matching line types.  If no matching line
#       types are found, an empry string is returned.

proc ::libcisco::UsrGetLineRangesIos { SessionId Config LineTypeList } {
    variable state

    # Convert the configuration file into a list for easier parsing.

    set ConfigList [ SavedResultToList $Config ]

    foreach LineType $LineTypeList {
        # In the config file, there is no line type of 'tty.'  These lines
        # are simply indicated with a line followed immediately by their
        # numeric line range (e.g. line 1 8).

        if { [ string match $LineType "tty" ] } {
            set ConfigLineType ""
        } else {
            set ConfigLineType $LineType
        }

        foreach ConfigLine $ConfigList {
            if { [ regexp -nocase -- \
                    "^line ${ConfigLineType}( *(\[0-9]+))( (\[0-9]+))*"\
                    $ConfigLine Discard Discard StartLine Discard EndLine ] } {
                if { [ info exists LowestLine ] } {
                    if { $StartLine < $LowestLine } {
                        set LowestLine $StartLine
                    }
                    if { [ string length $EndLine ] } {
                        if { $EndLine > $HighestLine } {
                            set HighestLine $EndLine
                        }
                    }
                } else {
                    set LowestLine $StartLine

                    if { [ string length $EndLine ] } {
                        set HighestLine $EndLine
                    }
                }
                unset StartLine
                unset EndLine
            }
        }

        # Set a variable to hold the line range.

        if { [ info exists LowestLine ] } {
            if { [ info exists HighestLine ] } {
                set LineRange "$LineType $LowestLine $HighestLine"
                unset LowestLine
                unset HighestLine
            } else {
                set LineRange "$LineType $LowestLine"
                unset LowestLine
            }
        } else {
            # Nothing was found.

            continue
        }

        # If the line range does not already exist in the list, add it.

        if { [ info exists LinesList ] } {
            if { [ lsearch -regexp $LinesList $LineRange ] == -1 } {
                lappend LinesList $LineRange
            }
        } else {
            lappend LinesList $LineRange
        }
    }

    if { [ info exists LinesList ] } {
        return $LinesList
    } else {
        return
    }
}


# libcisco::UsrPopulateConfigArrayIos --
#
#       The UsrPopulateConfigArrayXXX procedures will populate the
#       config array variable.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Config          string.  An IOS configuration file as returned
#                        by the UsrGetConfig procedures.
#
# Results:
#       Sets the 'config' array variables.

proc ::libcisco::UsrPopulateConfigArrayIos { SessionId Config } {
    variable config

    # Convert the configuration file into a list for easier parsing.

    set ConfigList [ SavedResultToList $Config ]

    # Clear any pre-existing config array data.
  
    foreach Element [ array names config $SessionId,* ] {
        unset config($Element)
    }

    set config($SessionId) 1
    RetOnErr [ UsrPopulateConfigArraySystemIos $SessionId $ConfigList ]
    RetOnErr [ UsrPopulateConfigArrayLineIos $SessionId $ConfigList ]

    RetOnErr [ session array $SessionId config ]

    return 0
}

proc ::libcisco::UsrPopulateConfigArraySystemIos { SessionId ConfigList } {
    variable config

    # Set known logging defaults (verified in 10.3 & 12.2).

    set config($SessionId,LogStatus)      "1"
    lappend config($SessionId,LogCfgBuf)  "1" "debugging" "default"
    lappend config($SessionId,LogCfgMon)  "1" "debugging"
    lappend config($SessionId,LogCfgCon)  "1" "debugging"
    lappend config($SessionId,LogCfgTrap) "1" "informational"
    set config($SessionId,LogFacility)    "local7"

    # Set known service defaults (verified in 11.3 & 12.2).

    set config($SessionId,SvcFinger)        "1"
    set config($SessionId,SvcLineNumber)    "0"
    set config($SessionId,SvcPwEncrypt)     "0"
    set config($SessionId,SvcUdpSmallSvr)   "0"
    set config($SessionId,SvcTcpSmallSvr)   "0"
    set config($SessionId,SvcTcpSmallSvr)   "0"
    set config($SessionId,NtpAuthenticate)  "0"
    lappend config($SessionId,SvcTimeLogCfg)    "0" {}
    lappend config($SessionId,SvcTimeDebugCfg)  "0" {}

    lappend config($SessionId,ClockTimeZoneCfg)   "0" {} {} {}
    lappend config($SessionId,ClockSummerTimeCfg) "0" {} {} {}

    foreach ConfigLine $ConfigList {
        set ConfigLine [ string trimleft $ConfigLine ]

        switch -regexp -- $ConfigLine {
            "^!.*" {
                continue
            }
            "^version.*" {
                regexp "^version (.*)" $ConfigLine Str Version
                set config($SessionId,Version) $Version
            }
            "^hostname .*" {
                regexp "^hostname (.*)" $ConfigLine Str Hostname
                set config($SessionId,Hostname) $Hostname
            }
            "^no logging on" {
                set config($SessionId,LogStatus) 0
            }
            "^logging (\[0-9\.]+)" {
                regexp "^logging (\[0-9\.]+)" $ConfigLine Str LogServer
                lappend config($SessionId,LogServerList) $LogServer
            }
            "^logging source-interface .*" {
                regexp "^logging source-interface (.*)" $ConfigLine Str LogIf
                set config($SessionId,LogSourceIf) $LogIf
            }
            "^logging buffered .*" {
                regexp "^logging buffered (\[0-9]*) ?(\[a-z]*)" $ConfigLine\
                        Str BufferSize Level

                if { [ info exists BufferSize ] } {
                    if { [ string length $BufferSize ] } {
                        set config($SessionId,LogCfgBuf)\
                                [ lreplace $config($SessionId,LogCfgBuf) 2 2\
                                $BufferSize ]
                    }
                }
                if { [info exists Level ] } {
                    if { [ string length $Level ] } {
                        set config($SessionId,LogCfgBuf)\
                                [ lreplace $config($SessionId,LogCfgBuf) 1 1\
                                $Level ]
                    }
                }
            }
            "^no logging buffered" {
                set config($SessionId,LogCfgBuf) [ list 0 "" "" ]
            }
            "^logging monitor .*" {
                regexp "^logging monitor (.*)" $ConfigLine Str Level
                set config($SessionId,LogCfgMon) [ list "1" $Level ]
            }
            "^no logging monitor" {
                set config($SessionId,LogCfgMon) [ list 0 "" ]
            }
            "^logging console .*" {
                regexp "^logging console (.*)" $ConfigLine Str Level
                set config($SessionId,LogCfgCon) [ list "1" $Level ]
            }
            "^no logging console" {
                set config($SessionId,LogCfgCon) [ list 0 "" ]
            }
            "^logging trap .*" {
                regexp "^logging trap (.*)" $ConfigLine Str Level
                set config($SessionId,LogCfgTrap) [ list "1" $Level ]
            }
            "^no logging trap" {
                set config($SessionId,LogCfgTrap) [ list 0 "" ]
            }
            "^logging facility .*" {
                regexp "^logging facility (.*)" $ConfigLine Str Facility
                set config($SessionId,LogFacility) $Facility
            }
            "^ip domain-name .*" {
                regexp "^ip domain-name (.*)" $ConfigLine Str DnsDomainName
                set config($SessionId,DnsDomainName) $DnsDomainName
            }
            "^ip domain-list .*" {
                regexp "^ip domain-list (.*)" $ConfigLine Str SearchDomain
                lappend config($SessionId,DnsDomainList) $SearchDomain
            }
            "^ip name-server .*" {
                regexp "^ip name-server (.*)" $ConfigLine Str DnsServer
                lappend config($SessionId,DnsServerList) $DnsServer
            }
            "^snmp-server location .*" {
                regexp "^snmp-server location (.*)" $ConfigLine Str SnmpLoc
                set config($SessionId,SnmpLocation) $SnmpLoc
            }
            "^snmp-server contact .*" {
                regexp "^snmp-server contact (.*)" $ConfigLine Str SnmpContact
                set config($SessionId,SnmpContact) $SnmpContact
            }
            "^snmp-server chassis-id .*" {
                regexp "^snmp-server chassis-id (.*)" $ConfigLine Str SnmpChId
                set config($SessionId,SnmpChassisId) $SnmpChId
            }
            "^snmp-server community.*" {
                set VarList "String Type View Acl"

                regexp "^snmp-server community (\[^ ]*)" $ConfigLine Str String
                regexp ".* view (\[^ ]*)" $ConfigLine Str View
                regexp ".* (\[ROWrow]{2}) ?(\[^ ]*)" $ConfigLine Str Type Acl
                set Type [ string tolower $Type ]

                foreach Var $VarList {
                    if { ! [ info exists $Var ] } {
                        set $Var ""
                    }
                }

                lappend config($SessionId,SnmpStringList)\
                        $String $Type $View $Acl
            }
            "^ntp server .*" {
                regexp "^ntp server\
                        (\[0-9\.]+)( version \[1-9])*( key \[0-9]+)*(\
                        source \[^ ]+)*( prefer)*" $ConfigLine Str NtpServer\
                        NtpVersion NtpKeyNum NtpSourceIf NtpPrefer
                regexp "(\[1-9]+)" $NtpVersion Str NtpVersion
                regexp "(\[0-9]+)" $NtpKeyNum Str NtpKeyNum
                regexp " source (\[^ ]+)" $NtpSourceIf Str NtpSourceIf
                if { [ string length $NtpPrefer ] } {
                    set NtpPrefer "1"
                } else {
                    set NtpPrefer "0"
                }

                lappend NtpServerCfg $NtpServer $NtpVersion $NtpKeyNum\
                        $NtpSourceIf $NtpPrefer
                lappend config($SessionId,NtpServerCfg) $NtpServerCfg
                unset NtpServerCfg
            }
            "^ntp authentication-key .*" {
                regexp "^ntp authentication-key (\[0-9\]+) md5 (.*)"\
                        $ConfigLine Str NtpKeyNum NtpKeyValue

                if { [ llength $NtpKeyValue ] == 2 } {
                    set EncryptedPw [ lindex $NtpKeyValue 0 ]
 
                    RetOnErr [ DecryptCiscoPassword $EncryptedPw ] NtpKeyValue

                    lappend NtpKeyCfg $NtpKeyNum $NtpKeyValue
                    lappend config($SessionId,NtpKeyListCfg) $NtpKeyCfg
                    unset NtpKeyCfg
                }
            }
            "^ntp authenticate" {
                set config($SessionId,NtpAuthenticate) "1"
            }
            "^no ntp authenticate" {
                set config($SessionId,NtpAuthenticate) "0"
            }
            "^banner login \\^C.*(\\^C)*" {
                regexp "banner login \\^C(\[^\\^C]*)(\\^C)?" $ConfigLine Str\
                        FirstLine EndDelimiter

                lappend config($SessionId,LoginBannerList) $FirstLine

                if { ! [ string length $EndDelimiter ] } {
                    set BannerType "LoginBannerList"
                }
            }
            "^.*\\^C" {
                if { [ info exists BannerType ] } {
                    regexp "(.*)\\^C" $ConfigLine Str LastLine
                    lappend config($SessionId,$BannerType) $LastLine

                    unset BannerType
                }
            }
            "^service finger" {
                set config($SessionId,SvcFinger) "1"
            }
            "^no service finger" {
                set config($SessionId,SvcFinger) "0"
            }
            "^service linenumber" {
                set config($SessionId,SvcLineNumber) "1"
            }
            "^no service linenumber" {
                set config($SessionId,SvcLineNumber) "0"
            }
            "^no ip finger" {
                set config($SessionId,SvcFinger) "0"
            }
            "^service password-encryption" {
                set config($SessionId,SvcPwEncrypt) "1"
            }
            "^no service password-encryption" {
                set config($SessionId,SvcPwEncrypt) "0"
            }
            "^service udp-small-servers.*" {
                set config($SessionId,SvcUdpSmallSvr) "1"
            }
            "^no service udp-small-servers.*" {
                set config($SessionId,SvcUdpSmallSvr) "0"
            }
            "^service tcp-small-servers.*" {
                set config($SessionId,SvcTcpSmallSvr) "1"
            }
            "^no service tcp-small-servers.*" {
                set config($SessionId,SvcTcpSmallSvr) "0"
            }
            "^service timestamps debug uptime" {
                set config($SessionId,SvcTimeDebugCfg)\
                        [ list "1" [ list uptime ] ]
            }
            "^service timestamps debug datetime.*" {
                set Msec         0
                set LocalTime    0
                set ShowTimeZone 0

                if { [ string match *msec* $ConfigLine ] } {
                    set Msec 1
                }
                if { [ string match *localtime* $ConfigLine ] } {
                    set LocalTime 1
                }
                if { [ string match *show-timezone* $ConfigLine ] } {
                    set ShowTimeZone 1
                }

                set config($SessionId,SvcTimeDebugCfg) [ list\
                        "1" [ list datetime $Msec $LocalTime $ShowTimeZone ] ]
            }
            "^service timestamps log uptime" {
                set config($SessionId,SvcTimeLogCfg)\
                        [ list "1" [ list uptime ] ]
            }
            "^service timestamps log datetime.*" {
                set Msec         0
                set LocalTime    0
                set ShowTimeZone 0

                if { [ string match *msec* $ConfigLine ] } {
                    set Msec 1
                }
                if { [ string match *localtime* $ConfigLine ] } {
                    set LocalTime 1
                }
                if { [ string match *show-timezone* $ConfigLine ] } {
                    set ShowTimeZone 1
                }

                set config($SessionId,SvcTimeLogCfg) [ list\
                        "1" [ list datetime $Msec $LocalTime $ShowTimeZone ] ]
            }
            "^clock timezone.*" {
                regexp "^clock timezone (\[^ ]*) (\[^ ]*) ?(\[^ ]*)"\
                        $ConfigLine Str TimeZone HourOffset MinuteOffset

                set config($SessionId,ClockTimeZoneCfg)\
                        [ list 1 $TimeZone $HourOffset $MinuteOffset ]
            }
            "^clock summer-time.*recurring.*" {
                set VarList "StartWeek StartDay StartMonth StartTime EndWeek\
                        EndDay EndMonth EndTime MinuteOffset"

                regexp "^clock summer-time (\[^ ]*) recurring\
                        ?(\[^ ]*) ?(\[^ ]*) ?(\[^ ]*) ?(\[^ ]*)\
                        ?(\[^ ]*) ?(\[^ ]*) ?(\[^ ]*) ?(\[^ ]*) ?(\[^ ]*)"\
                        $ConfigLine Str TimeZone StartWeek StartDay StartMonth\
                        StartTime EndWeek EndDay EndMonth EndTime MinuteOffset

                foreach Var $VarList {
                    if { ! [ info exists $Var ] } {
                        set $Var ""
                    }
                }

                if { ! [ string length $MinuteOffset ] } {
                    # The default minute offset is 60.

                    set MinuteOffset "60"
                }

                set TypeCfg [ list $StartWeek $StartDay $StartMonth $StartTime\
                        $EndWeek $EndDay $EndMonth $EndTime $MinuteOffset ]
                set config($SessionId,ClockSummerTimeCfg)\
                        [ list "1" $TimeZone "recurring" $TypeCfg ]
            }
            "^clock summer-time.*date.*" {
                set VarList "StartDate StartMonth StartYear StartTime EndDate\
                        EndMonth EndYear EndTime MinuteOffset"

                regexp "^clock summer-time (\[^ ]*) date\
                        ?(\[^ ]*) ?(\[^ ]*) ?(\[^ ]*) ?(\[^ ]*) ?(\[^ ]*)\
                        ?(\[^ ]*) ?(\[^ ]*) ?(\[^ ]*) ?(\[^ ]*) ?(\[^ ]*)"\
                        $ConfigLine Str TimeZone StartMonth StartDate\
                        StartYear StartTime EndMonth EndDate EndYear EndTime\
                        MinuteOffset

                foreach Var $VarList {
                    if { ! [ info exists $Var ] } {
                        set $Var ""
                    }
                }

                if { ! [ string length $MinuteOffset ] } {
                    # The default minute offset is 60.

                    set MinuteOffset "60"
                }

                # Although the IOS supports two input methods, it appears
                # to always store the date in "Month Date Year" format.

                set TypeCfg [ list $StartMonth $StartDate $StartYear\
                        $StartTime $EndMonth $EndDate $EndYear $EndTime\
                        $MinuteOffset ]
                set config($SessionId,ClockSummerTimeCfg)\
                        [ list "1" $TimeZone "date" $TypeCfg ]
            }
            "^access-list.*" {
                regexp "^access-list (\[^ ]+) (.*)" $ConfigLine Str Acl Entry
                set Entry [ string trim $Entry ]
                lappend config($SessionId,Acl,$Acl) $Entry
            }
            default {
                if { [ info exists BannerType ] } {
                    lappend config($SessionId,$BannerType) $ConfigLine
                }
                continue
            }
        }
    }

    return
}

proc ::libcisco::UsrPopulateConfigArrayLineIos { SessionId ConfigList } {
    variable config

    foreach ConfigLine $ConfigList {
        set ConfigLine [ string trimleft $ConfigLine ]

        switch -regexp -- $ConfigLine {
            "^!" {
                if { [ info exists LineRange ] } {
                    lappend config($SessionId,LinePwCfgList) $LinePwCfg
                    lappend config($SessionId,LineExecToCfgList)\
                        $LineExecToCfg
                    lappend config($SessionId,LineLoginCfgList) $LineLoginCfg

                    unset LineRange
                }
            }
            "^line .*" {
                if { [ info exists LineRange ] } {
                    lappend config($SessionId,LinePwCfgList) $LinePwCfg
                    lappend config($SessionId,LineExecToCfgList)\
                        $LineExecToCfg
                    lappend config($SessionId,LineLoginCfgList) $LineLoginCfg

                    unset LineRange
                }

                regexp "^line (.*)" $ConfigLine Str LineRange

                # Initialize line configuration defaults.

                set LinePwCfg [ list $LineRange "" ]
                set LineExecToCfg [ list $LineRange 10 0 ]
                set LineLoginCfg [ list $LineRange 0 "" ]
            }
            "^password .*" {
                regexp "^password (.*)" $ConfigLine Str LinePw

                if { [ llength $LinePw ] == 2 } {
                    set LinePw [ lindex $LinePw 1 ]

                    RetOnErr [ DecryptCiscoPassword $LinePw ] LinePw
                }
                set LinePwCfg [ list $LineRange $LinePw ]
            }
            "^exec-timeout .*" {
                regexp "^exec-timeout (\[0-9\]+) (\[0-9\]+)" $ConfigLine Str\
                        TimeoutMin TimeoutSec

                set LineExecToCfg [ list $LineRange $TimeoutMin $TimeoutSec ]
            }
            "^login.*" {
                regexp "^login ?(.*)" $ConfigLine Str Method

                set LineLoginCfg [ list $LineRange 1 $Method ]
            }
            "^access-class .*" {
                regexp "^access-class (\[^ \]+) (\[^ \]+)" $ConfigLine Str\
                        AclNum Direction

                set LineAclCfg [ list $LineRange $AclNum $Direction ]
                lappend config($SessionId,LineAclCfgList) $LineAclCfg
            }
            default {
                continue
            }
        }
    }

    return
}

proc ::libcisco::UsrPopulateConfigArray1900 { SessionId Config } {
    variable config

    # Convert the configuration file into a list for easier parsing.

    set ConfigList [ SavedResultToList $Config ]

    # Clear any pre-existing config array data.
  
    foreach Element [ array names config $SessionId,* ] {
        unset config($Element)
    }

    set config($SessionId) 1
    set config($SessionId,ConsoleTimeout) 0

    foreach ConfigLine $ConfigList {
        set ConfigLine [ string trimleft $ConfigLine ]
        switch -regexp -- $ConfigLine {
            "^!" {
            }
            "^hostname \".*\"" {
                regexp "^hostname \"(.*)\"" $ConfigLine Str Hostname
                set config($SessionId,Hostname) $Hostname
            }
            "^ip domain-name +\".*\"" {
                regexp "^ip domain-name +\"(.*)\"" $ConfigLine Str Domain
                set config($SessionId,DnsDomainName) $Domain
            }
            "^ip name-server .*" {
                regexp "^ip name-server (.*)" $ConfigLine Str DnsServer
                lappend config($SessionId,DnsServerList) $DnsServer
            }
            "^snmp-server location \".*\"" {
                regexp "^snmp-server location \"(.*)\"" $ConfigLine Str SnmpLoc
                set config($SessionId,SnmpLocation) $SnmpLoc
            }
            "^snmp-server contact \".*\"" {
                regexp "^snmp-server contact \"(.*)\"" $ConfigLine Str Contact
                set config($SessionId,SnmpContact) $Contact
            }
            "^snmp-server community.*" {
                regexp "^snmp-server community \"(.*)\" (.*)" $ConfigLine Str\
                        String Type
                set StringType [ string tolower $StringType ]
                lappend config($SessionId,SnmpStringList) $String $Type
            }
            "^time-out.*" {
                regexp "^time-out (.*)" $ConfigLine Str Timeout
                set config($SessionId,ConsoleTimeout) $Timeout
            }
            default {
            }
        }
    }

    RetOnErr [ session array $SessionId config ]

    return 0
}

proc ::libcisco::UsrPopulateConfigArrayXdi { SessionId Config } {
    variable config

    # Convert the configuration file into a list for easier parsing.

    set ConfigList [ SavedResultToList $Config ]

    # Clear any pre-existing config array data.

    catch { array unset config $SessionId,* }

    set config($SessionId,SnmpRmon) "0"
    set config($SessionId,IpPermitTelnet) "0"
    set config($SessionId,IpPermitSnmp) "0"

    foreach ConfigLine $ConfigList {
        set ConfigLine [ string trimleft $ConfigLine ]
        switch -regexp -- $ConfigLine {
            "^!" {
            }
            "^#version.*" {
                regexp "^#version (.*)" $ConfigLine Str Version
                set config($SessionId,Version) $Version
            }
            "^set system name.*" {
                regexp "^set system name (.*)" $ConfigLine Str Hostname
                set config($SessionId,Hostname) $Hostname
            }
            "^set prompt.*" {
                regexp "^set prompt (.*)" $ConfigLine Str Prompt
                set config($SessionId,Prompt) $Prompt
            }
            "^set ip dns domain.*" {
                regexp "^set ip dns domain (.*)" $ConfigLine Str DnsDomainName
                set config($SessionId,DnsDomainName) $DnsDomainName
            }
            "^set ip dns server.*" {
                regexp "^set ip dns server (\[^ ]*)" $ConfigLine Str DnsServer
                lappend config($SessionId,DnsServerList) $DnsServer
            }
            "^set system location.*" {
                regexp "^set system location (.*)" $ConfigLine Str Location
                set config($SessionId,SnmpLocation) $Location
            }
            "^set system contact.*" {
                regexp "^set system contact (.*)" $ConfigLine Str Contact
                set config($SessionId,SnmpContact) $Contact
            }
            "^set snmp community read-only.*" {
                regexp "^set snmp community read-only (.*)" $ConfigLine\
                        Str String
                set String [ string trim $String ]
                lappend config($SessionId,SnmpStringList) $String ro
            }
            "^set snmp community read-write .*" {
                regexp "^set snmp community read-write (.*)" $ConfigLine\
                        Str String
                set String [ string trim $String ]
                lappend config($SessionId,SnmpStringList) $String rw
            }
            "^set snmp community read-write-all .*" {
                regexp "^set snmp community read-write-all (.*)" $ConfigLine\
                        Str String
                set String [ string trim $String ]
                lappend config($SessionId,SnmpStringList) $String rwa
            }
            "^set snmp rmon enable.*" {
                set config($SessionId,SnmpRmon) 1
            }
            "^set snmp rmon disable.*" {
                set config($SessionId,SnmpRmon) 0
            }
            "^set ip permit enable telnet" {
                set config($SessionId,IpPermitTelnet) 1
            }
            "^set ip permit disable telnet" {
                set config($SessionId,IpPermitTelnet) 0
            }
            "^set ip permit enable snmp" {
                set config($SessionId,IpPermitSnmp) 1
            }
            "^set ip permit disable snmp" {
                set config($SessionId,IpPermitSnmp) 0
            }
            "^set ip permit disable" {
                set config($SessionId,IpPermitTelnet) 0
                set config($SessionId,IpPermitSnmp) 0
            }
            "^set ip permit enable" {
                set config($SessionId,IpPermitTelnet) 1
                set config($SessionId,IpPermitSnmp) 1
            }
            "^set ip permit.*" {
                regexp "^set ip permit (.*)" $ConfigLine Str PermitListEntry
                set PermitListEntry [ string trim $PermitListEntry ]
                lappend config($SessionId,PermitList) $PermitListEntry
            }
            default {
            }
        }
    }

    RetOnErr [ session array $SessionId config ]

    return 0
}
