# libekern.tcl --
#
#    libcisco - Configuration management API for Cisco networking equipment
#    Copyright (C) June 2002  Andy Ziegelbein
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Please send any questions or comments to andy@packetz.org.
#
#
# This file contains the Tcl code for the exported kernel procedures.  Exported
# kernel procedures provide the user with an interface to the core functions
# of this package.
#
# The following rules apply to procedures contained within this file:
#
# Arguments   : Exported procedures MAY take a variable number of arguments.
# Options     : Exported procedures MAY take options.
# SessionId   : Exported procedures MUST support an implicit SessionId, which
#                is provided by the 'last_spawn_id' element of the 'state'
#                namespace array variable.
# Relationship: The following list defines the relationship between exported
#                kernel procedures and other procedures defined within this
#                package:
#                
#                       Kernel - SHOULD call one (1) or more kernel procedures.
#                       Exported Kernel - MAY call other exported kernel
#                               procedures.
#                       User   - MUST NOT call user procedures.
#                       Exported User - MUST NOT call exported user procedures.
#                       Package Support - MAY call package support procedures.
#                       Tcl - MAY call Tcl commands.
#                       Expect - MUST NOT call Expect commands.
#
# RCS|SCCS: %Z% %M% %I% %E% %U%

package provide libcisco 1.0


# libcisco::sendCmd --
#
#       This procedure will launch the appropriate procedure from the
#       "Send Command" family of procedures.
#
# Arguments:
#       Type            string.  The type of send command from the family of
#                        send commands.
#       args            The remaining arguments are dependpent upon the
#                        Type selected.  See the appropriate proc for
#                        additional arguments types.
#
# Results
#       On success, the result of the command is returned.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::sendCmd { Type args } {
    variable state

    switch -- $Type {
        exec {
            RetOnErr [ SendCmdExec $args ] Result
        }
        global {
            RetOnErr [ SendCmdGlobal $args ] Result
        }
        line {
            RetOnErr [ SendCmdLine $args ] Result
        }
        interface {
            RetOnErr [ SendCmdInterface $args ] Result
        }
        default {
            RetOnErr "errBadCmdType"
        }
    }

    return $Result
}


# libcisco::session --
#
#       The session procedure is used to open and close sessions with Cisco
#       network devices.  It is also used to change the mode of an established
#       session to enable mode.
#
# Arguments:
#       Option          string.  The name of the session option to execute.
#                        Valid options are as follow:
#               
#                               open    Open a remote session
#                               close   Close a remote session
#                               enable  Change the session to enable mode
#
#       args            The remaining arguments are dependent upon the
#                        type of command selected.
#
# Results
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::session { Option args } {
    variable state

    switch -- $Option {
        open {
            RetOnErr [ SessionOpen $args ] Result

            # The return value of open session is handled differently than any
            # of the other functions.  This function must return an expect
            # spawn_id in order to work as an RPC server.  The RPC server
            # returns only the return value of the function to the client.
            # It does not allow for passing variables by reference.

            return $Result
        }
        enable {
            RetOnErr [ SessionEnable $args ]
        }
        logger {
            RetOnErr [ SessionLogger $args ]
        }
        module {
            RetOnErr [ SessionModule $args ]
        }
        array {
            RetOnErr [ SessionArray $args ]
        }
        close {
            RetOnErr [ SessionClose $args ]
        }
        default {
            RetOnErr "errBadSessionCmd"
        }
    }

    return 0
}



# SessionOpen --
#
#       The SessionOpen procedure is used to launch the appropriate proc to
#       establish a remote session with the target host device.  As of
#       02/15/02, the only valid type is 'telnet.'  Possible future session
#       types may include 'ssh' or 'rlogin.'
#
# Arguments:
#       args       A variable number of arguments.  Options are extracted
#                  by GetOpts.  Remaining arguments are as follows:
#       
#                  Host     The hostname or IP address of a target device.
#                  LinePw   The line password on the target device.
#                  Username The username to use should a "username" prompt
#                            be detected.
#                  UserPw   The password to use should a "username" prompt
#                            be detected.
#
# Results
#       On success, the value of a unique session identifier.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SessionOpen { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { type }       { string }  { telnet }
        { os }         { string }  { auto }
        { retries }    { string }  { 3 }
        { userprompt } { string }  { username: }
        { pwprompt }   { string }  { password: }
        { timeout }    { string }  { 30 }
        { logfile }    { string }  {}
        { logappend }  { string }  {}
        { verbose }    { boolean } { 0 }
        { port }       { string }  {}
    }

    RetOnErr [ GetOpts $OptList $args OptValue ArgsAfterOpts ]

    set Host     [ lindex $ArgsAfterOpts 0 ]
    set LinePw   [ lindex $ArgsAfterOpts 1 ]
    set Username [ lindex $ArgsAfterOpts 2 ]
    set UserPw   [ lindex $ArgsAfterOpts 3 ]

    switch -- $OptValue(type) {
        telnet {
            RetOnErr [ SessionOpenTelnet $Host $OptValue(port) $OptValue(os)\
                    $LinePw $Username $UserPw $OptValue(retries)\
                    $OptValue(userprompt) $OptValue(pwprompt)\
                    $OptValue(timeout) $OptValue(verbose) $OptValue(logfile)\
                    $OptValue(logappend) ] SessionId
        }
        default {
            RetOnErr "errBadSessionType"
        }
    }

    set state($SessionId,LinePw) $LinePw
    set state($SessionId,Username) $Username
    set state($SessionId,UserPw) $UserPw
    set state($SessionId,SessionType) $OptValue(type)
    set state($SessionId,Archive) 0
    set state(last_spawn_id) $SessionId

    return $SessionId
}



# libcisco::SessionOpenTelnet --
#
#       The SessionOpenTelnet proc will attempt to login to the specified 
#       device using the telnet program.  Following a successful login, the 
#       proc will disable the -more- prompts.  This is required in order for 
#       the other procs in this package to work properly.
#
# Arguments:
#       Host            string.  The IP address or hostname of a target device.
#       TcpPort         unsigned integer.  The TCP port to use for the telnet
#                        session.
#       OsType          string.  The type of operating system that the target
#                        device is running (ios, xdi, 1900, auto)
#       LinePw          string.  The line password (usu. stored in the local
#                        config) to be used if a $UserLoginPrompt prompt is not
#                        detected.
#       Username        string.  The username to be sent in response to the
#                        $UserLoginPrompt prompt.
#       UserPw          string.  The password to be used for login if a
#                        $UserLoginPrompt prompt is detected.
#       Retries         unsigned integer.  The maximum number of login 
#                        attempts.
#       UserLoginPrompt string.  The expected username prompt.
#       PasswordPrompt  string.  The expected password prompt.
#       timeout         string.  The timeout value used by expect.
#       StdOut Flag     boolean.  Determines whether output from the spawned
#                        process is sent to stdout or not.
#       LogFile         string.  The name of a log file to be opened in 
#                        overwrite mode.
#       LogAppend       string.  The name of a log file to be opened in
#                        append mode.
#
# Results:
#       On success, the value of a unique session identifier.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SessionOpenTelnet { Host TcpPort OsType LinePw Username\
        UserPw Retries UserLoginPrompt PasswordPrompt timeout StdOutFlag\
        LogFile LogAppend } {
    if { ! [ string length $TcpPort ] } {
        set TcpPort 23
    }

    RetOnErr [ SpawnTelnet $Host $TcpPort $timeout ] SessionId

    RetOnErr [ LoggerChannelsOpen $SessionId $StdOutFlag $LogFile $LogAppend ]

    RetOnErr [ PostSpawnCheck $SessionId $UserLoginPrompt $PasswordPrompt\
            $timeout ]

    RetOnErr [ Login $SessionId $OsType $LinePw $Username $UserPw $Retries\
            $UserLoginPrompt $PasswordPrompt $timeout ]

    RetOnErr [ LoginPostProcess $SessionId ]

    return $SessionId
}


# libcisco::LoginPostProcess --
#
#       The LoginPostProcess proc will peform initialization functions for the
#       EXEC session.
#
# Arguments:
#       SessionId       string.  The name of a variable in the calling 
#                        procedure which holds the spawn_id of the telnet
#                        session.
#
# Results:
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::LoginPostProcess { SessionId } {
    variable state

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ LoginPostProcessIos $SessionId ]
        }
        xdi {
            RetOnErr [ LoginPostProcessXdi $SessionId ]
        }
        1900 {
        }
        default {
            RetOnErr "errBadOsType"
        }
    }

    # It is possible for a session to go straight into enable mode following
    # a successful login.  Since the SessionEnable procedures have their own
    # LoginPostProcess functions, we should check here to see if the login took
    # us straight to enable mode.  If so, we will call the LoginPostProcess
    # procedure.

    if { [ string match $state($SessionId,Mode) "enable" ] } {
        RetOnErr [ SessionEnablePostProcess $SessionId ]
    }

    return 0
}

proc ::libcisco::LoginPostProcessIos { SessionId } {
    variable state

    # We want to turn of the -more- prompts as these aren't very useful when
    # when a script is interpreting the results.  In fact, they get in the way
    # and result in ugly code (see the 1900s where we can't turn of -more-
    # prompts.  Hence, it is VERY desirable to turn these off.

    RetOnErr [ sendCmd exec -patlist nofeedback $SessionId\
            "terminal length 0" ]

    # Very long commands will not carry forward to the next line.  Instead, the
    # terminal editing feature (enabled by default) will scroll the text at
    # the beginning of the command off the left hand side of the screen or
    # prompt.  This causes problems for the pattern matcher when it is
    # expecting to see the whole command in the pattern.  Hence, we want to
    # turn off this feature.

    RetOnErr [ sendCmd exec -patlist nofeedback $SessionId\
            "terminal no editing" ]

    return 0
}

proc ::libcisco::LoginPostProcessXdi { SessionId } {
    variable state

    set PatList {
        -nocase -re "unknown command(.*) $" {
            RetOnErr "errUnknownCmd"
        }
        -nocase -re "screen length for(.*)$Prompt" {
        }
    }

    RetOnErr [ sendCmd exec -patlist $PatList $SessionId "set length 0" ]

    return 0
}


# libcisco::SessionClose --
#
#       The SessionClose procedure will close the specified session with the
#       target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Retries         unsigned integer.  The maximum number of times to 
#                        issue the exit command before resorting to a
#                        close_wait (SIGHUP).  Defaults to 4.  Setting Retries
#                        to 0 will disable clean disconnects causing the
#                        procedure to simply send a SIGHUP.
#
# Results:
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SessionClose { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { retries } { string } { 4 }
        { timeout } { string } { 15 }
    }
    GetOpts $OptList $args OptValue ArgsAfterOpts

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId [ lindex $ArgsAfterOpts 0 ]
    } else {
        set SessionId $state(last_spawn_id)
    }

    # Drop down to the appropriate EXEC mode before issuing the command to
    # logout.

    switch -regexp -- $state($SessionId,Mode) {
        "user|enable" {
            # Do nothing.
        }
        default {
            RetOnErr [ SetMode $SessionId enable ]
        }
    }

    switch -- $state($SessionId,SessionType) {
        telnet {
            RetOnErr [ SessionCloseTelnet $SessionId $OptValue(retries)\
                    $OptValue(timeout) ]
        }
        default {
            RetOnErr "errBadSessionType"
        }
    }

    return 0
}


# libcisco::SendCmdExec --
#
#       The SendCmdExec proc will send one or more commands from either
#       User or Privileged EXEC mode on the target device.
#
# Arguments:
#       args       A variable number of arguments.  Options are extracted
#                  by GetOpts.
#
# Results
#       On success, the result of the command is returned.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SendCmdExec { args } {
    variable state
    variable prompt

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList { 
        { patlist } { string } {}
        { timeout } { string } { 10 }
    }
    GetOpts $OptList $args OptValue ArgsAfterOpts

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId [ lindex $ArgsAfterOpts 0 ]
        set Cmd       [ lindex $ArgsAfterOpts 1 ]
    } else {
        set SessionId $state(last_spawn_id)
        set Cmd       [ lindex $ArgsAfterOpts 0 ]
    }

    # Set the current mode to a user or privileged EXEC mode.

    switch -regexp -- $state($SessionId,Mode) {
        "user|enable" {
            # do nothing
        }
        default {
            RetOnErr [ SetMode $SessionId enable ]
        }
    }

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ SendCmd $SessionId $Cmd $OptValue(patlist)\
                    $OptValue(timeout) ] Result
        }
        xdi {
            RetOnErr [ SendCmd $SessionId $Cmd $OptValue(patlist)\
                    $OptValue(timeout) ] Result
        }
        1900 {
            RetOnErr [ SendCmd $SessionId $Cmd $OptValue(patlist)\
                    $OptValue(timeout) ] Result
        }
        default {
            RetOnErr "errBadOsType"
        }
    }

    return $Result
}


# libcisco::SendCmdGlobal --
#
#       The SendCmdGlobal proc will send a configuration command in global
#       config mode on the target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Cmd             string.  The global configuration command to send.
#                        The command should not be passed in with a trailing \r
#                        or \n as this will automatically be appended.
#       PatList         string list.  A list of valid expect patterns and 
#                        actions.
#       timeout         string.  The timeout value used by expect.
#
# Results
#       On success, the result of the command is returned.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SendCmdGlobal { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { patlist } { string } {}
        { timeout } { string } { 10 }
    }
    GetOpts $OptList $args OptValue ArgsAfterOpts

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId [ lindex $ArgsAfterOpts 0 ]
        set Cmd       [ lindex $ArgsAfterOpts 1 ]
    } else {
        set SessionId $state(last_spawn_id)
        set Cmd       [ lindex $ArgsAfterOpts 0 ]
    }

    RetOnErr [ SetMode $SessionId global ]

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ SendCmd $SessionId $Cmd $OptValue(patlist)\
                    $OptValue(timeout) ] Result
        }
        1900 {
            RetOnErr [ SendCmd $SessionId $Cmd $OptValue(patlist)\
                    $OptValue(timeout) ] Result
        }
        default {
            RetOnErr "errCmdWrongOs"
        }
    }

    set state($SessionId,Archive) 1

    return $Result
}


# libcisco::SendCmdLine --
#
#       The SendCmdLine proc will send a configuration command to a line or
#       range of lines.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Cmd             string.  The line configuration.  The command
#                        should not be passed in with a trailing \r or \n as
#                        this will automatically be appended by this proc.
#       Line            string.  The line to be configured.
#       PatList         string list.  A list of valid expect patterns and 
#                        actions.
#       timeout         string.  The timeout value used by expect.
#
# Results
#       On success, the result of the command is returned.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SendCmdLine { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { patlist } { string } {}
        { timeout } { string } { 10 }
    }
    GetOpts $OptList $args OptValue ArgsAfterOpts

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId [ lindex $ArgsAfterOpts 0 ]
        set Line      [ lindex $ArgsAfterOpts 1 ]
        set Cmd       [ lindex $ArgsAfterOpts 2 ]
    } else {
        set SessionId $state(last_spawn_id)
        set Line      [ lindex $ArgsAfterOpts 0 ]
        set Cmd       [ lindex $ArgsAfterOpts 1 ]
    }

    RetOnErr [ SetMode $SessionId line $Line ]

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ SendCmd $SessionId $Cmd $OptValue(patlist)\
                    $OptValue(timeout) ] Result
        }
        default {
            RetOnErr "errCmdWrongOs"
        }
    }

    set state($SessionId,Archive) 1

    return $Result
}


# libcisco::SendCmdInterface --
#
#       The SendCmdInterface proc will send a configuration command to a line
#       or range of lines.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Cmd             string.  The line configuration(s).  The command
#                        should not be passed in with a trailing \r or \n as
#                        this will automatically be appended by this proc.
#       Interface       string.  The line to be configured.
#       PatList         string list.  A list of valid expect patterns and 
#                        actions.
#       timeout         string.  The timeout value used by expect.
#
# Results
#       On success, the result of the command is returned.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SendCmdInterface { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { patlist } { string } {}
        { timeout } { string } { 10 }
    }
    GetOpts $OptList $args OptValue ArgsAfterOpts

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId [ lindex $ArgsAfterOpts 0 ]
        set Interface [ lindex $ArgsAfterOpts 1 ]
        set Cmd       [ lindex $ArgsAfterOpts 2 ]
    } else {
        set SessionId $state(last_spawn_id)
        set Interface [ lindex $ArgsAfterOpts 0 ]
        set Cmd       [ lindex $ArgsAfterOpts 1 ]
    }

    RetOnErr [ SetMode $SessionId interface $Interface ]

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ SendCmd $SessionId $Cmd $OptValue(patlist)\
                    $OptValue(timeout) ] Result
        }
        default {
            RetOnErr "errCmdWrongOs"
        }
    }

    set state($SessionId,Archive) 1

    return $Result
}


# libcisco::SessionEnable --
#
#       The SessionEnable proc will put the device into enable mode.  If the
#       device is an IOS based switch or router, it will place the device
#       in the default enable mode--level 15.
#
# Arguments:
#       args       A variable number of arguments.  Options are extracted
#                  by GetOpts.  Remaining arguments are as follows:
#
#                  SessionId  A unique session identifier.
#                  PwList     A list of enable passwords.
#
# Results:
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SessionEnable { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { retries }    { string } { 3 }
        { username }   { string } {}
        { userprompt } { string } { username: }
        { pwprompt }   { string } { password: }
        { timeout }    { string } { 30 }
    }
    GetOpts $OptList $args OptValue ArgsAfterOpts

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId [ lindex $ArgsAfterOpts 0 ]
        set PwList [ lindex $ArgsAfterOpts 1 ]
    } else {
        set SessionId $state(last_spawn_id)
        set PwList [ lindex $ArgsAfterOpts 0 ]
    }

    # Check to see if the session is already in privileged EXEC mode.

    if { ! [ string match $state($SessionId,Mode) user ] } {
        return 0
    }

    # Ensure the PasswordPrompt is lowercase for -nocase comparison.

    set OptValue(pwprompt) [ string tolower $OptValue(pwprompt) ]

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ SessionEnableIos $SessionId $PwList $OptValue(username)\
                    $OptValue(retries) $OptValue(userprompt)\
                    $OptValue(pwprompt) $OptValue(timeout) ]
        }
        xdi {
            RetOnErr [ SessionEnableXdi $SessionId $PwList $OptValue(username)\
                    $OptValue(retries) $OptValue(userprompt)\
                    $OptValue(pwprompt) $OptValue(timeout) ]
        }
        1900 {
            RetOnErr [ SessionEnableIos $SessionId $PwList $OptValue(username)\
                    $OptValue(retries) $OptValue(userprompt)\
                    $OptValue(pwprompt) $OptValue(timeout) ]
        }
        default {
            RetOnErr "errUnknownOs"
        }
    }

    set state($SessionId,Mode) enable

    # Perform post-enable mode initialization if necessary.

    RetOnErr [ SessionEnablePostProcess $SessionId ]

    return 0
}


# libcisco::SessionEnablePostProcess --
#
#       The SessionEnablePostProcess proc will peform initialization functions
#       for SessionEnable procedures.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#
# Results:
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SessionEnablePostProcess { SessionId } {
    variable state

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ SessionEnablePostProcessIos $SessionId ]
        }
        xdi {
        }
        1900 {
        }
        default {
            RetOnErr "errUndefinedOs"
        }
    }

    return 0
}

proc ::libcisco::SessionEnablePostProcessIos { SessionId } {
    variable state

    # Yet another nuisance to the pattern matcher which is useful when humans
    # are interacting with the device, but not very useful to scripts are the
    # "Configured from console by vty0..." messages.  These only appear on
    # enable EXEC sessions.

    RetOnErr [ sendCmd exec $SessionId "terminal no monitor" ]

    return 0
}


# libcisco::SessionLogger --
#
#       The SessionLogger procedure will call the kernel LoggerMark procedure
#       to update all logging channels with the passed in text.
#
# Arguments:
#       args       A variable number of arguments.  Options are extracted
#                  by GetOpts.  Remaining arguments are as follows:
#
#                  SessionId  A unique session identifier.
#                  LogText    A line of text to be sent to the logging
#                              channels.
#
# Results:
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SessionLogger { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId [ lindex $args 0 ]
        set LogText   [ lindex $args 1 ]
    } else {
        set SessionId $state(last_spawn_id)
        set LogText   [ lindex $args 0 ]
    }

    RetOnErr [ LoggerMark $SessionId $LogText ]

    return 0
}


# libcisco::SessionModule --
#
#       The SessionModule procedure will session to the specified module and
#       login.
#
# Arguments:
#       args       A variable number of arguments.  Options are extracted
#                  by GetOpts.  Remaining arguments are as follows:
#
#                  SessionId  string.  A unique session identifier.
#                  Module     string.  The target module number.
#                  LinePw     string.  The line password on the target device.
#                  Username   string. The username to use should a "username"
#                              prompt be detected.
#                  UserPw     string. The password to use should a "username"
#                              prompt be detected.
#
# Results:
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SessionModule { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { type }       { string }  { telnet }
        { os }         { string }  { auto }
        { retries }    { string }  { 3 }
        { userprompt } { string }  { username: }
        { pwprompt }   { string }  { password: }
        { timeout }    { string }  { 30 }
    }

    RetOnErr [ GetOpts $OptList $args OptValue ArgsAfterOpts ]

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId [ lindex $ArgsAfterOpts 0 ]
        set Module    [ lindex $ArgsAfterOpts 1 ]
        set LinePw    [ lindex $ArgsAfterOpts 2 ]
        set Username  [ lindex $ArgsAfterOpts 3 ]
        set UserPw    [ lindex $ArgsAfterOpts 4 ]
    } else {
        set SessionId $state(last_spawn_id)
        set Module    [ lindex $ArgsAfterOpts 0 ]
        set LinePw    [ lindex $ArgsAfterOpts 1 ]
        set Username  [ lindex $ArgsAfterOpts 2 ]
        set UserPw    [ lindex $ArgsAfterOpts 3 ]
    }

    # Walk backwards through the arguments to see which were left unspecified
    # (blank).  If an argument was not specified, we will use the same
    # arguments that were used with the call to "session open".

    if { ! [ string length $UserPw ] } {
        set UserPw $state($SessionId,UserPw)
        if { ! [ string length $Username ] } {
            set Username $state($SessionId,Username)
            if { ! [ string length $LinePw ] } {
                set LinePw $state($SessionId,LinePw)
            }
        }
    }

    RetOnErr [ LoginModule $SessionId $Module $OptValue(os) $LinePw $Username\
            $UserPw $OptValue(retries) $OptValue(userprompt)\
            $OptValue(pwprompt) $OptValue(timeout) ]

    RetOnErr [ LoginPostProcess $SessionId ]

    return 0
}


# libcisco::SessionArray --
#
#       The SessionArray procedure will update the ArrayList element of
#       of the state array variable.
#
# Arguments:
#       args       A variable number of arguments.  Options are extracted
#                  by GetOpts.  Remaining arguments are as follows:
#
#                  SessionId  string.  A unique session identifier.
#                  ArrayName  string.  The name of the array to be added.
#
# Results:
#       Adds an array name to the ArrayList element of the state array 
#       variable if it doesn't exist already.

proc ::libcisco::SessionArray { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId [ lindex $args 0 ]
        set ArrayName [ lindex $args 1 ]
    } else {
        set SessionId $state(last_spawn_id)
        set ArrayName [ lindex $args 0 ]
    }

    if { ! [ string match [ string range $ArrayName 0 1 ] "::" ] } {
        # The passed in array name will be interpreted relative to the calling
        # procedures namespace.

        set CallingNamespace [ uplevel 2 { namespace current } ]
        set CallingNamespace [ string trimright $CallingNamespace : ]

        set ArrayName "${CallingNamespace}::${ArrayName}"
    }

    if { [ info exists state($SessionId,ArrayList) ] } {
        # Check to see if the variable has already been added to the list.

        if { [ lsearch -exact\
                $state($SessionId,ArrayList) $ArrayName ] == -1 } {
            # The variable doesn't already exist, so add it.

            lappend state($SessionId,ArrayList) $ArrayName
        }
    } else {
        lappend state($SessionId,ArrayList) $ArrayName
    }

    return 0
}


# libcisco::SetMode --
#
#       The SetMode procedure puts the connected device in the specified
#       mode.  The target device must already have a privileged EXEC session
#       established with the target device.
#
# Arguments:
#       SessionId    string.  The unique ID of the session.
#       DesiredMode  string.  The mode desired for the connected device.  The
#                     following mode types are accepted:
#
#                            enable
#                            global
#                            line (requires additional arguments)
#                            interface (requires additional arguments)
#
#       args         (optional) Remaining arguments are dependent upon
#                     the desired mode.  The following mode types require
#                     additional arguments as follows:
#
#                            line
#                                    Line      The specific line to configure
#                                               in line configuration mode.
#                            interface
#                                    Interface The specific interface to
#                                               configure in interface
#                                               configuration mode.
# Results
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::SetMode { SessionId DesiredMode args } {
    variable state

    if { ! [ info exists state($SessionId,Mode) ] } {
        RetOnErr "errUnknownMode"
    }

    # The following switch statement lists only valid mode switches.  Any other
    # mode switches will be caught by the default statment which will return
    # an error.

    switch -regexp --\
            "$state($SessionId,Os),$state($SessionId,Mode) $DesiredMode" {
        "ios,user user" {
        }
        "ios,user.*" {
            RetOnErr "errNotFromUser"
        }
        "ios,enable enable" {
            # Nothing to do.
        }
        "ios,enable global" {
            RetOnErr [ SetModeEnableToGlobal $SessionId ]
        }
        "ios,enable line" {
            set Line [ lindex $args 0 ]

            RetOnErr [ SetModeEnableToGlobal $SessionId ]
            RetOnErr [ SetModeConfigToLine $SessionId $Line ]
        }
        "ios,enable interface" {
            set Interface [ lindex $args 0 ]

            RetOnErr [ SetModeEnableToGlobal $SessionId ]
            RetOnErr [ SetModeConfigToInterface $SessionId $Interface ]
        }
        "ios,global enable" {
            RetOnErr [ SetModeConfigToEnable $SessionId ]
        }
        "ios,global global" {
            # Nothing to do.
        }
        "ios,global line" {
            set Line [ lindex $args 0 ]

            RetOnErr [ SetModeConfigToLine $SessionId $Line ]
        }
        "ios,global interface" {
            set Interface [ lindex $args 0 ]

            RetOnErr [ SetModeConfigToInterface $SessionId $Interface ]
        }
        "ios,line enable" {
            RetOnErr [ SetModeConfigToEnable $SessionId ]
        }
        "ios,line global" {
            RetOnErr [ SetModeConfigToGlobal $SessionId ]
        }
        "ios,line line" {
            set Line [ lindex $args 0 ]

            RetOnErr [ SetModeConfigToLine $SessionId $Line ]
        }
        "ios,line interface" {
            set Interface [ lindex $args 0 ]

            RetOnErr [ SetModeConfigToInterface $SessionId $Interface ]
        }
        "ios,interface enable" {
            RetOnErr [ SetModeConfigToEnable $SessionId ]
        }
        "ios,interface global" {
            RetOnErr [ SetModeConfigToGlobal $SessionId ]
        }
        "ios,interface line" {
            set Line [ lindex $args 0 ]

            RetOnErr [ SetModeConfigToLine $SessionId $Line ]
        }
        "ios,interface interface" {
            set Interface [ lindex $args 0 ]

            RetOnErr [ SetModeConfigToInterface $SessionId $Interface ]
        }
        "xdi,user user" {
        }
        "xdi,user.*" {
            RetOnErr "errNotFromUser"
        }
        "xdi,enable enable" {
            # Nothing to do.
        }
        "1900,user user" {
        }
        "1900,user.*" {
            RetOnErr "errNotFromUser"
        }
        "1900,enable enable" {
            # Nothing to do.
        }
        "1900,enable global" {
            RetOnErr [ SetModeEnableToGlobal $SessionId ]
        }
        "1900,enable interface" {
            set Interface [ lindex $args 0 ]

            RetOnErr [ SetModeEnableToGlobal $SessionId ]
            RetOnErr [ SetModeConfigToInterface $SessionId $Interface ]
        }
        "1900,global enable" {
            RetOnErr [ SetModeConfigToEnable $SessionId ]
        }
        "1900,global global" {
            # Nothing to do.
        }
        "1900,global interface" {
            set Interface [ lindex $args 0 ]

            RetOnErr [ SetModeConfigToInterface $SessionId $Interface ]
        }
        "1900,interface enable" {
            RetOnErr [ SetModeConfigToEnable $SessionId ]
        }
        "1900,interface global" {
            RetOnErr [ SetModeConfigToGlobal $SessionId ]
        }
        "1900,interface interface" {
            set Interface [ lindex $args 0 ]

            RetOnErr [ SetModeConfigToInterface $SessionId $Interface ]
        }
        default {
            RetOnErr "errUnknownMode"
        }
    }

    return 0
}
