# libezdo.tcl --
#
#    libcisco - Configuration management API for Cisco networking equipment
#    Copyright (C) June 2002  Andy Ziegelbein
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Please send any questions or comments to andy@packetz.org.
#
#
# This file contains the Tcl code for the ezdo functions.
#
# The "ezdo" family of functions provide a means for performing a specific task
# on a target device.  In order for a function to be in the "ez" family of 
# functions, it must adhere to the following three (3) criteria:
#
#       (1) Provide an OS layer of abstraction.
#       (2) Provide a "command-type" layer of abstraction.
#       (3) Silently return from the function with a success result even if
#           the "command-type" is not applicable for the OS of the target
#           device (e.g. setting a banner message is possible under Catalyst
#           code and IOS, but not under a 1900).
#
# In addition, the following rules apply to procedures contained within this
# file:
#
# Arguments   : ez procedures MAY take a variable number of arguments.
# Options     : ez procedures MAY take optional values.
# SessionId   : ez procedures MUST support an implicit SessionId, which is
#                provided by the 'last_spawn_id" element of the 'state'
#                namespace array variable.
# Relationship: The following list defines the relationship between ez
#                procedures and other procedures defined within this package:
#
#                       Kernel - MUST NOT call kernel procedures.
#                       Exported Kernel - MUST NOT call exported kernel procs.
#                       User - SHOULD call one (1) or more user procedures.
#                       Exported User - MAY call other exported user procs.
#                       Package Support - MAY call package support procedures.
#                       Tcl - MAY call Tcl commands.
#                       Expect - MUST NOT call Expect commands.
#
# RCS|SCCS: %Z% %M% %I% %E% %U%

package provide libcisco 1.0


# libcisco::ezzdo --
#
#       This procedure is used to launch the appropriate procedure from the
#       "ezdo" family of procedures.
#
# Arguments:
#       Cmd             string.  The type of ezdo command from the family of
#                        ezdo commands.
#       args            The remaining arguments are dependpent upon the Cmd
#                        selected.  See the appropriate proc for additional
#                        arguments types.
#
# Results
#       On success, 0.
#       A short textual message on error

proc ::libcisco::ezdo { Cmd args } {
    switch -- $Cmd {
        savecfg {
            RetOnErr [ EzDoSaveConfig $args ]
        }
        clearcntrs {
            RetOnErr [ EzDoClearCounters $args ]
        }
        debug {
            RetOnErr [ EzDoDebug $args ]
        }
        undebug {
            RetOnErr [ EzDoUndebug $args ]
        }
        default {
            return "errBadCmdType"
        }
    }

    return 0
}


# libcisco::EzDoSaveConfig --
#
#       The EzDoSaveConfig procedure will save the configuration in RAM to 
#       NVRAM.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::EzDoSaveConfig { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { force } { boolean } { 0 }
    }

    RetOnErr [ GetOpts $OptList $args OptValue ArgsAfterOpts ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId  [ lindex $ArgsAfterOpts 0 ]
    } else {
        set SessionId $state(last_spawn_id)
    }

    # Launch the appropriate procedure.

    switch -- $state($SessionId,Os) {
        ios {
            if { $state($SessionId,Archive) == 1 || $OptValue(force) == 1 } {
                RetOnErr [ UsrDoCopyRunStartIos $SessionId ]
            }
        }
        xdi {
            # Switches based on Catalyst code automatically save their
            # config in RAM to NVRAM.
        }
        1900 {
            # 1900s automatically save their configuration from RAM to NVRAM.
        }
        default {
            return "errUnknownOs"
        }
    }

    return 0
}


# libcisco::EzDoClearCounters --
#
#       The EzDoClearCounters procedure will clear all counters on the target
#       device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::EzDoClearCounters { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
    } else {
        set SessionId $state(last_spawn_id)
    }

    # Launch the appropriate procedure.

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ UsrDoClearCountersIos $SessionId ]
        }
        xdi {
            RetOnErr [ UsrDoClearCountersXdi $SessionId ]
        }
        1900 {
            RetOnErr [ UsrDoClearCountersIos $SessionId ]
        }
        default {
            return "errUnknownOs"
        }
    }

    return 0
}


# libcisco::EzDoDebug --
#
#       The EzDoDebug and EzDoUndebug procedures will enable and disable
#       debugging on the target device.  The main reason these two procedures
#       were not combined into one procedure with a simple option to either
#       enable or disable debugging was to prevent accidents.  For instance,
#       if the intent was to disable all debugging with a 'no debug all' and
#       the switch was inadvertently left off to negate the command, the
#       result would be a 'debug all' which would most likely render the
#       device useless.  For the psychological comfort of the user and to
#       prevent mistakes, these two commands were semantically separated.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::EzDoDebug { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
        set ArgList    [ lrange $args 1 end ]
    } else {
        set SessionId $state(last_spawn_id)
        set ArgList    [ lrange $args 0 end ]
    }

    # Launch the appropriate procedure.

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ UsrDoDebugIos $SessionId $ArgList ]
        }
        xdi {
            # Catalyst code doesn't support this feature.
        }
        1900 {
            # Catalyst code doesn't support this feature.
        }
        default {
            return "errUnknownOs"
        }
    }

    return 0
}

proc ::libcisco::EzDoUndebug { args } {
    variable state

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
        set ArgList    [ lrange $args 1 end ]
    } else {
        set SessionId $state(last_spawn_id)
        set ArgList    [ lrange $args 0 end ]
    }

    # Launch the appropriate procedure.

    switch -- $state($SessionId,Os) {
        ios {
            RetOnErr [ UsrDoUndebugIos $SessionId $ArgList ]
        }
        xdi {
            # Catalyst code doesn't support this feature.
        }
        1900 {
            # Catalyst code doesn't support this feature.
        }
        default {
            return "errUnknownOs"
        }
    }

    return 0
}
