#!/bin/sh
# -*- tcl -*- \
exec tclsh "$0" "$@"

global state
set state(Ver)                   "1.3"
set state(LibInstallDirList)     "/usr/local/lib /usr/lib /usr/lib/tcl8.3"
set state(ManPageInstallDirList) "/usr/local/man /usr/man /usr/share/man"

proc main { argv0 argv } {
    # Get defaults to save the user some typing.

    GetDefaultLibInstallDir
    GetDefaultManPageInstallDir
    GetDefaultHtmlDocsInstallDir

    # Get user input.

    GetInstallOptions
    puts "---"

    GetInstallDirs
    puts "---"

    # Install the appropriate files.

    InstallFiles
    puts "---"

    # Test the library installation.
    TestLibraryInstallation

    return
}

proc XCopy { Src Dest {WildCard *} } {
    file mkdir $Dest
    foreach File [glob [file join $Src $WildCard]] {
        set Base [file tail $File]
        set Sub  [file join $Dest $Base]

        if {[file isdirectory $File]} then {
            file mkdir  $Sub
            XCopy $File $Sub
        } else {
            file copy -force $File $Sub
        }
    }

    return
}

proc GetDefaultLibInstallDir {} {
    global state
    global auto_path

    set DefaultDir ""

    foreach Dir $state(LibInstallDirList) {
        if { [ lsearch -exact $auto_path $Dir ] != -1 } {
            set DefaultDir $Dir
            break
        }
    }

    set state(DefaultLibInstallDir) "$DefaultDir/libcisco$state(Ver)"

    return
}

proc GetDefaultManPageInstallDir {} {
    global env
    global state

    # Try man -w, man --path, and then look to the MANPATH env variable.

    if { [ catch { exec man -w } Result ] } {
        if { [ catch { exec man --path } Result ] } {
            if { [ info exists env(MANPATH) ] } {
                set Result $env(MANPATH)
            } else {
                set state(DefaultManPageInstallDir) ""
                return
            }
        }
    }

    set ManPathList [ split $Result : ]
    set DefaultDir  [ lindex $ManPathList 0 ]

    foreach Dir $state(ManPageInstallDirList) {
        if { [ lsearch -exact $ManPathList $Dir ] != -1 } {
            set DefaultDir $Dir
            break
        }
    }

    set state(DefaultManPageInstallDir) "$DefaultDir/mann"

    return
}

proc GetDefaultHtmlDocsInstallDir {} {
    global state

    set state(DefaultHtmlDocsInstallDir) "/tmp/libcisco$state(Ver)"

    foreach RootDir "var opt usr usr/local home" {
        foreach ApacheDir "apache www http" {
            foreach HtmlDir "htdocs/doc htdocs/docs html/doc html/docs\
                    htdocs html" {
                if { [file exists "/$RootDir/$ApacheDir/$HtmlDir"] &&\
                        [file isdirectory "/$RootDir/$ApacheDir/$HtmlDir"] } {
                    set state(DefaultHtmlDocsInstallDir)\
                            "/$RootDir/$ApacheDir/$HtmlDir/libcisco$state(Ver)"
                    return
                }
            }
        }
    }

    return
}

proc GetInstallOptions {} {
    global state

    puts -nonewline "Install man pages (y/N)? "
    flush stdout
    gets stdin state(ManPageResponse)

    puts -nonewline "Install HTML documentation (y/N)? "
    flush stdout
    gets stdin state(HtmlDocsResponse)

    return
}

proc GetInstallDirs {} {
    global state

    puts -nonewline "Library install dir\
            \[$state(DefaultLibInstallDir)\]: "
    flush stdout
    gets stdin state(LibInstallDir)

    if { [ string length $state(LibInstallDir) ] == 0 } {
        set state(LibInstallDir) $state(DefaultLibInstallDir)
    }

    switch -regexp -- $state(ManPageResponse) {
        "y|Y|yes|YES|Yes" {
            puts -nonewline "Manpage install dir\
                    \[$state(DefaultManPageInstallDir)\]: "
            flush stdout
            gets stdin state(ManPageInstallDir)

            if { [ string length $state(ManPageInstallDir) ] == 0 } {
                set state(ManPageInstallDir) $state(DefaultManPageInstallDir)
            }
        }
    }

    switch -regexp -- $state(HtmlDocsResponse) {
        "y|Y|yes|YES|Yes" {
            puts -nonewline "HTML docs install dir\
                    \[$state(DefaultHtmlDocsInstallDir)\]: "
            flush stdout
            gets stdin state(HtmlDocsInstallDir)

            if { [ string length $state(HtmlDocsInstallDir) ] == 0 } {
                set state(HtmlDocsInstallDir) $state(DefaultHtmlDocsInstallDir)
            }
        }
    }

    return
}

proc InstallFiles {} {
    global state

    # We need to make sure the current direcotry is the same directory that
    # the install script is located in.

    set Dir [ file dirname [ info script ] ]
    cd $Dir

    InstallLibraryFiles

    switch -regexp -- $state(ManPageResponse) {
        "y|Y|yes|YES|Yes" {
            InstallManPages
        }
    }

    switch -regexp -- $state(HtmlDocsResponse) {
        "y|Y|yes|YES|Yes" {
            InstallHtmlDocs
        }
    }

    return
}

proc InstallLibraryFiles {} {
    global state

    puts "Installing libcisco into $state(LibInstallDir)"

    if { [ catch { XCopy [file join lib] $state(LibInstallDir) } Result ] } {
        puts stderr $Result
        exit 1
    }

    return
}

proc InstallManPages {} {
    global state

    puts "Installing manpages into $state(ManPageInstallDir)"

    if { [ catch { XCopy [file join doc] $state(ManPageInstallDir) *.n }\
             Result ] } {
        puts stderr $Result
        exit 1
    }
    
    if { [ catch { file copy -force -- [file join doc man.macros]\
            $state(ManPageInstallDir) } Result ] } {
        puts stderr $Result
        exit 1
    }
    
    return
}

proc InstallHtmlDocs {} {
    global state

    puts "Installing HTML docs into $state(HtmlDocsInstallDir)"

    if { [ catch { XCopy [file join doc] $state(HtmlDocsInstallDir) *.html }\
             Result ] } {
        puts stderr $Result
        exit 1
    }
    
    return
}

proc TestLibraryInstallation {} {
    global state

    puts -nonewline "Testing library installation..."
    flush stdout

    if { [ catch { package require -exact libcisco $state(Ver) } Result ] } {
        puts "failed"
        puts stderr $Result
        exit 1
    } else {
        puts "ok"
    }

    return
}

main $argv0 $argv
