# init.tcl --
#
#    libcisco - Configuration management API for Cisco networking equipment
#    Copyright (C) June 2002  Andy Ziegelbein
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Please send any questions or comments to andy@packetz.org.
#
# RCS|SCCS: %Z% %M% %I% %E% %U%

package provide libcisco 1.3
package require Tcl 8.3
package require Expect 5.3

namespace eval libcisco {
    # Declare namespace variables.

    variable prompt

    # The state variable is used to hold all application state information
    # in an array.  Values set by functions include the following:
    #
    #        last_spawn_id        The spawn_id of the last spawned process
    #                              that established a successful login.
    #
    #                              (set by Login kernel proc)
    #
    #        spawn_id,Host        The hostname or IP address of the system.
    #
    #                              (set by SpawnTelnet kernel proc)
    #
    #        spawn_id,TcpPort     The destination host's TCP port number.
    #
    #                              (set by SpawnTelnet kernel proc)
    #
    #        spawn_id,Os          The operating system of the connected device.
    #                              Set by OpenSession procs.  Valid OS' are:
    #
    #                                     ios
    #                                     xdi (a.k.a. CatCode)
    #                                     1900
    #
    #                              (set by Login kernel proc)
    #
    #        spawn_id,TacLogin    A boolean value representing whether or not
    #                              server username authentication is being
    #                              performed (e.g. via TACACS+/RADIUS).
    #
    #                              (set by Login/LoginModule kernel proc)
    #
    #        spawn_id,LinePw      The line password associated with the
    #                              session.
    #
    #                              (set by SessionOpen exp kernel proc)
    #
    #        spawn_id,Username    The username associated with session.
    #
    #                              (set by SessionOpen exp kernel proc)
    #
    #        spawn_id,UserPw      The username's password associated with the
    #                              session.
    #
    #                              (set by SessionOpen exp kernel proc)
    #
    #        spawn_id,Mode        The current mode of the target device.  Set
    #                              by SendCmd procs.  Valid values are:
    #
    #                                     user
    #                                     enable
    #                                     global
    #                                     line
    #                                     interface
    #
    #                              (set by Login & SetMode kernel procs)
    #
    #        spawn_id,ModeInfo    Mode specific information (if applicable).
    #                              If spawn_id,Mode is 'line', then this
    #                              variable will hold the current line(s)
    #                              being configured.  If spawn_id,Mode is
    #                              'interface', then this variable will hold
    #                              the current interface(s) being configured.
    #
    #                              (set by SetMode kernel procs)
    #        
    #        spawn_id,SessionType The type of session established.  Set by 
    #                              OpenSession proc.  Valid values are:
    #
    #                                     telnet
    #                                     ssh (future)
    #
    #                               (set by SessionOpen exp kernel proc)
    #
    #        spawn_id,LogIdList    A list of channel identifiers used for
    #                               logging the output of a session.
    #
    #                               (set by LoggerChannelsOpen kernel proc)
    #
    #        spawn_id,ArrayList    A list of arrays set outside the
    #                               kernel that follow the same element
    #                               naming convention as the state variable
    #                               which should be cleaned up when the
    #                               session is closed.
    #
    #                               (set by SessionArray kernel proc)
    #
    #        spawn_id,Archive      A boolean value indicating whether or not
    #                               the configuration on the target device
    #                               has been altered.  The archive variable
    #                               is set to 0 after a session has been
    #                               opened and will be set to 1 if a config
    #                               change is made.
    #
    #                               (set by SessionOpen, SendCmdGlobal,
    #                                SendCmdLine, and SendCmdInterface 
    #                                kernel procs)

    variable state

    # The ErrorHandling element is used to control how errors are handled
    # by procedures in the package.  Valid options are as follows:
    #
    #       return - the procedure simply returns a string beginning with "err"
    #                followed by a short text description of the error.
    #       exit   - the procedure will print a short error message beginning
    #                with "err" to the stderr stream, and then exit with an
    #                exit code of 1 (exit 1).
    #       error  - raise a Tcl error condtion

    set state(ErrorHandling) "exit"

    # The prompt array is an array of the different types of prompts.  These
    # prompts are all regular expressions.  The following prompts must be in
    # lowercase in order to work with the -nocase comparisons in the libraries.

    set prompt(ios,user)         "\[^ \r\n]*>$"
    set prompt(ios,enable)       "\[^ \r\n]*#$"
    set prompt(ios,global)       "\[^ \r\n]*.config.#$"
    set prompt(ios,line)         "\[^ \r\n]*.config-l\[ine]{0,3}.#$"
    set prompt(ios,interface)    "\[^ \r\n]*.config-(if?|s.*).#$"
    set prompt(ios,acl)          "\[^ \r\n]*.config-\[^ \r\n]*#$"
    set prompt(xdi,user)         "\n(\[^ ]+)(\[^:)]) $"
    set prompt(xdi,enable)       "\[^ \r\n]* .enable. $"
    set prompt(1900,user)        "\[^ \r\n]*>$"
    set prompt(1900,enable)      "\[^ \r\n]*#$"
    set prompt(1900,global)      "\[^ \r\n]*.config.#$"
    set prompt(1900,interface)   "\[^ \r\n]*.config-if?.#$"

    # The config variable holds device specific configurations information
    # in a Tcl array.  Its contents are set by the UsrPopulateConfigArrayXXX
    # user procedures.  The following elements will be set:
    #
    #        spawn_id                  Always set to 1.
    #                                          
    #    IOS Code Elements:
    #        spawn_id,Version          The version of code.
    #        spawn_id,Hostname         The hostname of the device.
    #        spawn_id,LogStatus        The status of the logging process.
    #        spawn_id,LogServerList    A list of the logging servers.
    #        spawn_id,LogSourceIf      The logging source-interface.
    #        spawn_id,LogCfgBuf        The logging buffer config where elements
    #                                   are ordered as follows:
    #                                          Level (emergencies, alerts,
    #                                              critical, error, warnings,
    #                                              notifications,
    #                                              informational, debugging)
    #                                          BufferSize
    #        spawn_id,LogCfgMon        The logging monitor config where
    #                                   elements are ordered as follows:
    #                                          Status (0=off, 1=on)
    #                                          Level
    #        spawn_id,LogCfgCon        The logging monitor config where
    #                                   elements are ordered as follows:
    #                                          Status (0=off, 1=on)
    #                                          Level
    #        spawn_id,LogCfgTrap       The logging monitor config where
    #                                   elements are ordered as follows:
    #                                          Status (0=off, 1=on)
    #                                          Level
    #        spawn_id,LogFacility      The logging facility (syslog).
    #        spawn_id,DnsDomainName    The DNS domain to which the device
    #                                   belongs.
    #        spawn_id,DnsDomainList    The DNS domain search list.
    #        spawn_id,DnsServerList    The DNS server list.
    #        spawn_id,SnmpLocation     The SNMP system location.
    #        spawn_id,SnmpContact      The SNMP system contact.
    #        spawn_id,SnmpChassisId    The SNMP chassis ID.
    #        spawn_id,SnmpStringList   A list of SNMP community strings.  There
    #                                   must be an even number of elements in
    #                                   the list.  Elements are ordered as
    #                                          String Type View Acl ...
    #                                   where type can be "ro", or "rw"
    #        spawn_id,NtpServerCfg     The NTP server list config where
    #                                   elements are ordered as follows:
    #                                          ServerIp
    #                                          Version
    #                                          KeyNum
    #                                          SourceInterface
    #                                          PreferedBoolean
    #        spawn_id,NtpKeyListCfg    The NTP key list config where elements
    #                                   are ordered as follows:
    #                                          KeyNum
    #                                          KeyValue
    #        spawn_id,NtpAuthenticate  A boolean value representing the
    #                                   "ntp authenticate" setting.
    #        spawn_id,BannerLoginList  The login banner as a Tcl list.
    #        spawn_id,SvcFinger        A boolean value representing the 
    #                                   "service finger" setting.
    #        spawn_id,SvcLineNumber    A boolean value representing the 
    #                                   "service linenumber" setting.
    #        spawn_id,SvcPwEncrypt     A boolean value representing the 
    #                                   "service password-encryption" setting.
    #        spawn_id,SvcTcpSmallSvr   A boolean value representing the 
    #                                   "service tcp-small-servers" setting.
    #        spawn_id,SvcUdpSmallSvr   A boolean value representing the 
    #                                   "service udp-small-servers" setting.
    #        spawn_id,SvcTimeLogCfg    The service timestamp configuration for
    #                                   the "service timestamps log" setting:
    #                                          Status (boolean)
    #                                          TimeStampCfg
    #                                              Type (uptime | datetime)
    #                                              Msec (bool, datetime only)
    #                                              Localtime (bool, dt only)
    #                                              ShowTimeZone (bool, dt only)
    #        spawn_id,SvcTimeDebugCfg  The service timestamp configuration for
    #                                   the "service timestamps debug" setting:
    #                                          Status (boolean)
    #                                          TimeStampCfg (uptime|datetime)
    #        spawn_id,ClockTimeZoneCfg The "clock timezone" configuration:
    #                                          Status (boolean)
    #                                          Timezone
    #                                          HourOffset
    #                                          MinuteOffset
    #        spawn_id,ClockSummerTimeCfg The "clock summer-time" configuration:
    #                                          Status (boolean)
    #                                          Timezone
    #                                          Type (recurring | date)
    #                                          TypeCfg
    #                                              -recurring config-
    #                                                  StartWeek
    #                                                  StartDay
    #                                                  StartMonth
    #                                                  StartTime (hh:mm)
    #                                                  EndWeek
    #                                                  EndDay
    #                                                  EndMonth
    #                                                  EndTime (hh:mm)
    #                                                  MinuteOffset
    #                                              -date config-
    #                                                  StartMonth
    #                                                  StartDate
    #                                                  StartYear
    #                                                  StartTime (hh:mm)
    #                                                  EndMonth
    #                                                  EndDate
    #                                                  EndYear
    #                                                  EndTime (hh:mm)
    #                                                  MinuteOffset
    #        spawn_id,LinePwCfgList    The line password configuration:
    #                                          LineRange (e.g. "vty 0 4")
    #                                          Password
    #        spawn_id,ExecTimeoutCfgList A list of exec-timeout configurations
    #                                   where a configuration is ordered as 
    #                                   follows:
    #                                          LineRange (e.g. "vty 0 4")
    #                                          ExecTimeoutMin
    #                                          ExecTimeoutSec
    #        spawn_id,LineLoginCfgList A list of line login configurations:
    #                                          LineRange (e.g. "vty 0 4")
    #                                          Status (boolean)
    #                                          Method (empty|local|tacacs|
    #                                              authentication X)
    #        spawn_id,Acl,$Acl         A list of ACL entries where each element
    #                                   is a separate ACL entry.  The order of
    #                                   the list follows the order of the ACL
    #                                   in the config.  $Acl represents the
    #                                   ACL number or name.
    #
    #    Catalyst Code Elements:
    #        spawn_id,Version          The version of code.
    #        spawn_id,Hostname         The hostname of the device.
    #        spawn_id,Prompt           The prompt of the device.
    #        spawn_id,DnsDomainName    The DNS domain to which the device
    #                                   belongs.
    #        spawn_id,DnsServerList    The DNS server list.
    #        spawn_id,SnmpLocation     The SNMP system location.
    #        spawn_id,SnmpContact      The SNMP system contact.
    #        spawn_id,SnmpStringList   A list of SNMP community strings.  There
    #                                   must be an even number of elements in
    #                                   the list.  Elements are ordered as
    #                                          String Type ... String Type
    #                                   where type can be "ro", "rw", or "rwa"
    #        spawn_id,SnmpRmon         A boolean value representing the status
    #                                   of mini-RMON (0: disabled, 1: enabled)
    #        spawn_id,PermitList       A list of permit list entries.
    #
    #    1900 Code Elements:
    #        spawn_id,Hostname         The hostname of the device.
    #        spawn_id,DnsDomainName    The DNS domain to which the device
    #                                   belongs.
    #        spawn_id,DnsServerList    The DNS server list.
    #        spawn_id,SnmpLocation     The SNMP system location.
    #        spawn_id,SnmpContact      The SNMP system contact.
    #        spawn_id,SnmpStringList   A list of SNMP community strings.  There
    #                                   must be an even number of elements in
    #                                   the list.  Elements are ordered as
    #                                          String Type ... String Type
    #                                   where type can be "ro", or "rw"
    #        spawn_id,ConsoleTimeout   The console timeout period in seconds.
    #

    variable config

    # Turn off output to the user's terminal.

    log_user 0

    namespace export session sendCmd ezget ezset ezdo

}
