# libezset.tcl --
#
#    libcisco - Configuration management API for Cisco networking equipment
#    Copyright (C) June 2002  Andy Ziegelbein
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Please send any questions or comments to andy@packetz.org.
#
#
# This file contains the Tcl code for the ezset functions.
#
# The "ezset" family of functions provide a means for setting config values on
# a target device.  In order for a function to be in the "ez" family of 
# functions, it MUST adhere to the following three (3) criteria:
#
#       (1) Provide an OS layer of abstraction.
#       (2) Provide a "command-type" layer of abstraction.
#       (3) Silently return from the function with a success result even if
#           the "command-type" is not applicable for the OS of the target
#           device (e.g. setting a banner message is possible under Catalyst
#           code and IOS, but not under a 1900).
#
# In addition, the following rules apply to procedures contained within this
# file:
#
# Arguments   : ez procedures MAY take a variable number of arguments.
# Options     : ez procedures MAY take optional values.
# SessionId   : ez procedures MUST support an implicit SessionId.
# Relationship: The following list defines the relationship between ez
#                procedures and other procedures defined within this package:
#
#                       Kernel - MUST NOT call kernel procedures.
#                       Exported Kernel - MUST NOT call exported kernel procs.
#                       User - SHOULD call one (1) or more user procedures.
#                       Exported User - MAY call other exported user procs.
#                       Package Support - MAY call package support procedures.
#                       Tcl - MAY call Tcl commands.
#                       Expect - MUST NOT call Expect commands.
#
# RCS|SCCS: %Z% %M% %I% %E% %U%

package provide libcisco 1.3


# libcisco::ezset --
#
#       This procedure is used to launch the appropriate procedure from the
#       "ezset" family of procedures.
#
# Arguments:
#       Cmd             string.  The type of ezset command from the family of
#                        ezset commands.
#       args            The remaining arguments are dependpent upon the
#                        'type' selected.  See the appropriate proc for
#                        additional arguments types.
#
# Results
#       0 on success
#       A short textual message on error

proc ::libcisco::ezset { Cmd args } {
    switch -- $Cmd {
        password {
            RetOnErr [ EzSetPassword $args ]
        }
        banner {
            RetOnErr [ EzSetBanner $args ]
        }
        hostname {
            RetOnErr [ EzSetHostname $args ]
        }
        contact {
            RetOnErr [ EzSetContact $args ]
        }
        location {
            RetOnErr [ EzSetLocation $args ]
        }
        community {
            RetOnErr [ EzSetCommunity $args ]
        }
        permitlist {
            RetOnErr [ EzSetPermitList $args ]
        }
        ippermit {
            RetOnErr [ EzSetIpPermit $args ]
        }
        acl {
            RetOnErr [ EzSetAcl $args ]
        }
        default {
            RetOnErr "errBadCmdType"
        }
    }

    return 0
}

proc ::libcisco::EzSetPassword { args } {
    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId   [ lindex $args 0 ]
        set PwType      [ lindex $args 1 ]
        set CurrentPw   [ lindex $args 2 ]
        set NewPw       [ lindex $args 3 ]
        set MinArgCount 4
    } else {
        set SessionId   [ session info lastid ]
        set PwType      [ lindex $args 0 ]
        set CurrentPw   [ lindex $args 1 ]
        set NewPw       [ lindex $args 2 ]
        set MinArgCount 3
    }

    # Check to make sure NewPw is not null.

    if { [ llength $args ] < $MinArgCount } {
        RetOnErr "errTooFewArgs"
    }

    # Launch the appropriate procedure.

    switch -- [ session info $SessionId os ],$PwType {
        ios,login {
            RetOnErr [ UsrGetConfigIos $SessionId running-config ] Config
            RetOnErr [ UsrGetLineRangesIos\
                    $SessionId $Config {vty con aux} ] LineList
            foreach Line $LineList {
                RetOnErr [ UsrSetPasswordLineIos $SessionId $Line $NewPw ]
            }
        }
        ios,enable {
            RetOnErr [ UsrSetPasswordEnableIos $SessionId secret $NewPw ]
        }
        xdi,login {
            RetOnErr [ UsrSetPasswordXdi $SessionId login $CurrentPw $NewPw ]
        }
        xdi,enable {
            RetOnErr [ UsrSetPasswordXdi $SessionId enable $CurrentPw $NewPw ]
        }
        1900,login {
            RetOnErr [ UsrSetPassword1900 $SessionId login $NewPw ]
        }
        1900,enable {
            RetOnErr [ UsrSetPassword1900 $SessionId enable $NewPw ]
        }
        default {
            RetOnErr "errPwType"
        }
    }

    return 0
}


proc ::libcisco::EzSetBanner { args } {
    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { file } { boolean } { 0 }
    }

    RetOnErr [ GetOpts $OptList $args OptValue ArgsAfterOpts ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId  [ lindex $ArgsAfterOpts 0 ]
        set Banner     [ lindex $ArgsAfterOpts 1 ]
    } else {
        set SessionId  [ session info lastid ]
        set Banner     [ lindex $ArgsAfterOpts 0 ]
    }

    # Determine whether or not the Banner variable is a filename or a Tcl
    # list containing the banner and set the BannerList accordingly.

    if { $OptValue(file) } {
        RetOnErr [ AsciiFileToList $Banner ] BannerList
    } else {
        set BannerList $Banner
    }

    if { ! [ llength $BannerList ] } {
        RetOnErr "errEmptyBanner"
    }

    # Launch the appropriate procedure.

    switch -- [ session info $SessionId os ] {
        ios {
            RetOnErr [ UsrSetBannerIos $SessionId login $BannerList ]
        }
        xdi {
            RetOnErr [ UsrSetBannerMotdXdi $SessionId $BannerList ]
        }
        1900 {
            # The 1900 doesn't support a login banner.
        }
        default {
            RetOnErr "errUnknownOs"
        }
    }

    return 0
}


proc ::libcisco::EzSetHostname { args } {
    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
        set Hostname   [ lindex $args 1 ]
    } else {
        set SessionId  [ session info lastid ]
        set Hostname   [ lindex $args 0 ]
    }

    if { ! [ string length $Hostname ] } {
        RetOnErr "errEmptyHostname"
    }

    # IOS-based devices and 1900s can support up to 29 characters, but a
    # switch running Catalyst Code will support only 20 characters for the
    # prompt and 256 characters for the system name.  LCD = 20.

    if { [ string length $Hostname ] > 20 } {
        RetOnErr "errNameTooLong"
    }

    # Launch the appropriate procedure.

    switch -- [ session info $SessionId os ] {
        ios {
            RetOnErr [ UsrSetHostnameIos $SessionId $Hostname ]
        }
        xdi {
            RetOnErr [ UsrSetPromptXdi $SessionId $Hostname ]
            RetOnErr [ UsrSetSystemNameXdi $SessionId $Hostname ]
        }
        1900 {
            RetOnErr [ UsrSetHostnameIos $SessionId $Hostname ]
        }
        default {
            RetOnErr "errUnknownOs"
        }
    }

    return 0
}


proc ::libcisco::EzSetContact { args } {
    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
        set Contact    [ lindex $args 1 ]
    } else {
        set SessionId  [ session info lastid ]
        set Contact    [ lindex $args 0 ]
    }

    if { ! [ string length $Contact ] } {
        RetOnErr "errEmptyContact"
    }

    # 1900s can support no more than 220 characters for the contact field.
    # IOS and XDI-based devices will support 255 characters.  LCD = 220.

    if { [ string length $Contact ] > 220 } {
        RetOnErr "errContactTooLong"
    }

    # Launch the appropriate procedure.

    switch -- [ session info $SessionId os ] {
        ios {
            RetOnErr [ UsrSetSnmpContactIos $SessionId $Contact ]
        }
        xdi {
            RetOnErr [ UsrSetSnmpContactXdi $SessionId $Contact ]
        }
        1900 {
            RetOnErr [ UsrSetSnmpContact1900 $SessionId $Contact ]
        }
        default {
            RetOnErr "errUnknownOs"
        }
    }

    return 0
}


proc ::libcisco::EzSetLocation { args } {
    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
        set Location   [ lindex $args 1 ]
    } else {
        set SessionId  [ session info lastid ]
        set Location   [ lindex $args 0 ]
    }

    if { ! [ string length $Location ] } {
        RetOnErr "errEmptyLocation"
    }

    # 1900s can support no more than 220 characters for the location field.
    # IOS and XDI-based devices will support 255 characters.  LCD = 220.

    if { [ string length $Location ] > 220 } {
        RetOnErr "errLocationTooLong"
    }

    # Launch the appropriate procedure.

    switch -- [ session info $SessionId os ] {
        ios {
            RetOnErr [ UsrSetSnmpLocationIos $SessionId $Location ]
        }
        xdi {
            RetOnErr [ UsrSetSnmpLocationXdi $SessionId $Location ]
        }
        1900 {
            RetOnErr [ UsrSetSnmpLocation1900 $SessionId $Location ]
        }
        default {
            RetOnErr "errUnknownOs"
        }
    }

    return 0
}


proc ::libcisco::EzSetCommunity { args } {
    variable config

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { clear } { boolean } { 0 }
    }

    RetOnErr [ GetOpts $OptList $args OptValue ArgsAfterOpts ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId     [ lindex $ArgsAfterOpts 0 ]
        set StringType    [ string tolower [ lindex $ArgsAfterOpts 1 ] ]
        set StringAclList [ lrange $ArgsAfterOpts 2 end ]
    } else {
        set SessionId     [ session info lastid ]
        set StringType    [ string tolower [ lindex $ArgsAfterOpts 0 ] ]
        set StringAclList [ lrange $ArgsAfterOpts 1 end ]
    }

    # XDI-based devices will support a community string up to 20 characters. 
    # 1900s will support a community string up to 31 characters.  IOS-based
    # devices will support up to 220 characters.  LCD = 20.

    foreach StringAclPair $StringAclList {
        set StringAclSplit [ split $StringAclPair , ]
        set String         [ lindex $StringAclSplit 0 ]

        if { [ string length $String ] > 20 } {
            RetOnErr "errStringTooLong"
        }
    }

    # Launch the appropriate procedure.

    switch -- [ session info $SessionId os ] {
        ios {
            # Populate the config array if it isn't already.

            if { ! [ info exists config($SessionId) ] } {
                RetOnErr [ UsrGetConfigIos $SessionId running-config ]
            }

            # Clear the current strings of the specified type.

            if { $OptValue(clear) } {
                if { [ info exists config($SessionId,SnmpStringList) ] } {
                    foreach {String Type View Acl}\
                            $config($SessionId,SnmpStringList) {
                        set Type [ string tolower $Type ]

                        if { [ string match $Type $StringType ] } {
                            RetOnErr [ UsrSetSnmpCommunityIos $SessionId\
                                    remove $Type $String "" ]
                        }
                    }
                }
            }

            # Set the new community strings (if any).

            foreach StringAclPair $StringAclList {
                set StringAclSplit [ split $StringAclPair , ]
                set String         [ lindex $StringAclSplit 0 ]
                set Acl            [ lindex $StringAclSplit 1 ]

                RetOnErr [ UsrSetSnmpCommunityIos $SessionId add\
                            $StringType $String $Acl ]
            }
        }
        xdi {
            # Only the first string in the list will be used as CatCode based
            # switches support only one community string per type.  If the 
            # string is empty, the specified string type will be cleared.

            set StringAclPair  [ lindex $StringAclList 0 ]
            set StringAclSplit [ split $StringAclPair , ]
            set String         [ lindex $StringAclSplit 0 ]
            set Acl            [ lindex $StringAclSplit 1 ]

            if { $OptValue(clear) } {
                switch -- $StringType {
                    ro {
                        RetOnErr [ UsrSetSnmpCommunityXdi $SessionId\
                                read-only "" ]
                    }
                    rw {
                        RetOnErr [ UsrSetSnmpCommunityXdi $SessionId\
                                read-write "" ]
                        RetOnErr [ UsrSetSnmpCommunityXdi $SessionId\
                                read-write-all "" ]
                    }
                    default {
                        RetOnErr "errUnknownType"
                    }
                }
            }

            switch -- $StringType {
                ro {
                    RetOnErr [ UsrSetSnmpCommunityXdi $SessionId\
                            read-only $String ]
                }
                rw {
                    RetOnErr [ UsrSetSnmpCommunityXdi $SessionId\
                            read-write $String ]
                    RetOnErr [ UsrSetSnmpCommunityXdi $SessionId\
                            read-write-all $String ]
                }
                default {
                    RetOnErr "errUnknownType"
                }
            }
        }
        1900 {
            # Populate the config array if it isn't already.

            if { ! [ info exists config($SessionId) ] } {
                RetOnErr [ UsrGetConfig1900 $SessionId running-config ]
            }

            # Clear the current strings of the specified type.

            if { $OptValue(clear) } {
                if { [ info exists config($SessionId,SnmpStringList) ] } {
                    foreach {String Type} $config($SessionId,SnmpStringList) {
                        set Type [ string tolower $Type ]

                        if { [ string match $Type $StringType ] } {
                            RetOnErr [ UsrSetSnmpCommunity1900 $SessionId\
                                    remove $Type $String ]
                        }
                    }
                }
            }

            # Set the new community strings (if any).

            set Count 0
            foreach StringAclPair $StringAclList {
                set StringAclSplit [ split $StringAclPair , ]
                set String         [ lindex $StringAclSplit 0 ]

                RetOnErr [ UsrSetSnmpCommunity1900 $SessionId add\
                        $StringType $String ]

                # The 1900 will support only four community strings.

                incr Count
                if { $Count == 4 } {
                    break
                }
            }
        }
        default {
            RetOnErr "errUnknownOs"
        }
    }

    return 0
}

proc ::libcisco::EzSetPermitList { args } {
    variable config

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { file }   { boolean } { 0 }
        { ifdiff } { boolean } { 0 }
    }

    RetOnErr [ GetOpts $OptList $args OptValue ArgsAfterOpts ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId  [ lindex $ArgsAfterOpts 0 ]
        set PermitList [ lindex $ArgsAfterOpts 1 ]
    } else {
        set SessionId  [ session info lastid ]
        set PermitList [ lindex $ArgsAfterOpts 0 ]
    }

    # Determine whether or not the PermitList variable is a filename or a Tcl
    # list containing the permit statements and set the PermitList accordingly.

    if { $OptValue(file) } {
        RetOnErr [ AsciiFileToList $PermitList ] PermitList
    }

    if { ! [ llength $PermitList ] } {
        RetOnErr "errEmptyPermit"
    }

    # Cleanup the list removing leading/trailing spaces and empty lines.

    foreach Line $PermitList { 
        if { [ string length $Line ] } {
            lappend NewPermitList [ string trim $Line ]
        }
    }

    set PermitList $NewPermitList

    switch -- [ session info $SessionId os ] {
        ios {
            # IOS devices do not support a permit list.
        }
        xdi {
            # Run the "ifdiff" check.

            if { $OptValue(ifdiff) } {
                # Populate the config array if it hasn't been already.

                if { ! [ info exists config($SessionId) ] } {
                    RetOnErr [ UsrGetConfigXdi $SessionId all ]
                }

                if { [ info exists config($SessionId,PermitList) ] } {
                    if { [ PermitMatch $config($SessionId,PermitList)\
                            $PermitList ] } {
                        return 0
                    }
                }
            }

            RetOnErr [ UsrDoClearIpPermitXdi $SessionId all ]

            foreach NewEntry $PermitList {
                RetOnErr [ UsrSetIpPermitXdi $SessionId "$NewEntry" ]
            }
        }
        1900 {
            # 1900s do not support a permit list.
        }
        default {
            RetOnErr "errUnknownOs"
        }
    }

    return 0
}

proc ::libcisco::EzSetIpPermit { args } {
    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $args 0 ] ] } {
        set SessionId  [ lindex $args 0 ]
        set Operator   [ lindex $args 1 ]
        set Operand    [ lindex $args 2 ]
    } else {
        set SessionId  [ session info lastid ]
        set Operator   [ lindex $args 0 ]
        set Operand    [ lindex $args 1 ]
    }

    if { ! [ string length $Operand ] } {
        RetOnErr "errTooFewArgs"
    }

    switch -glob -- [ session info $SessionId os ],$Operator,$Operand {
        ios,*,* {
            # IOS devices do not support a permit list.
        }
        xdi,enable,all {
            RetOnErr [ UsrSetIpPermitXdi $SessionId "enable" ]
        }
        xdi,disable,all {
            RetOnErr [ UsrSetIpPermitXdi $SessionId "disable" ]
        }
        xdi,disable,telnet {
            RetOnErr [ ezget version Ver ]

            if {($Ver(Major) == 5 && $Ver(Minor) >= 4) || ($Ver(Major) > 5)} {
                RetOnErr [ UsrSetIpPermitXdi $SessionId "disable telnet" ]
            } else {
                RetOnErr [ UsrSetIpPermitXdi $SessionId "disable" ]
            }
        }
        xdi,disable,snmp {
            RetOnErr [ ezget version Ver ]

            if {($Ver(Major) == 5 && $Ver(Minor) >= 4) || ($Ver(Major) > 5)} {
                RetOnErr [ UsrSetIpPermitXdi $SessionId "disable snmp" ]
            } else {
                RetOnErr [ UsrSetIpPermitXdi $SessionId "disable" ]
            }
        }
        xdi,enable,telnet {
            RetOnErr [ ezget version Ver ]

            if {($Ver(Major) == 5 && $Ver(Minor) >= 4) || ($Ver(Major) > 5)} {
                RetOnErr [ UsrSetIpPermitXdi $SessionId "enable telnet" ]
            } else {
                RetOnErr [ UsrSetIpPermitXdi $SessionId "enable" ]
            }
        }
        xdi,enable,snmp {
            RetOnErr [ ezget version Ver ]

            if {($Ver(Major) == 5 && $Ver(Minor) >= 4) || ($Ver(Major) > 5)} {
                RetOnErr [ UsrSetIpPermitXdi $SessionId "enable snmp" ]
            } else {
                RetOnErr [ UsrSetIpPermitXdi $SessionId "enable" ]
            }
        }
        1900,*,* {
            # 1900s do not support a permit list.
        }
        default {
            RetOnErr "errBadOsOrOp"
        }
    }

    return 0
}

proc ::libcisco::EzSetAcl { args } {
    variable config

    # Ensure that leading and trailing curly braces are removed from args.

    set args [ join $args ]

    set OptList {
        { file }   { boolean } { 0 }
        { ifdiff } { boolean } { 0 }
    }

    RetOnErr [ GetOpts $OptList $args OptValue ArgsAfterOpts ]

    # Get the remaining arguments.

    if { [ string match "exp*" [ lindex $ArgsAfterOpts 0 ] ] } {
        set SessionId [ lindex $ArgsAfterOpts 0 ]
        set AclNum    [ lindex $ArgsAfterOpts 1 ]
        set AclList   [ lindex $ArgsAfterOpts 2 ]
    } else {
        set SessionId [ session info lastid ]
        set AclNum    [ lindex $ArgsAfterOpts 0 ]
        set AclList   [ lindex $ArgsAfterOpts 1 ]
    }

    # Determine whether or not the AclList variable is a filename or a Tcl
    # list containing the entries and set the AclList accordingly.

    if { $OptValue(file) } {
        RetOnErr [ AsciiFileToList $AclList ] AclList
    }

    if { ! [ llength $AclList ] } {
        RetOnErr "errEmptyAcl"
    }

    # Cleanup the list removing leading/trailing spaces and empty lines.

    foreach Line $AclList { 
        if { [ string length $Line ] } {
            lappend NewAclList [ string trim $Line ]
        }
    }

    set AclList $NewAclList

    switch -- [ session info $SessionId os ] {
        ios {
            if { $OptValue(ifdiff) } {
                if { ! [ info exists config($SessionId) ] } {
                    RetOnErr [ UsrGetConfigIos $SessionId running-config ]
                }

                if { [ info exists config($SessionId,Acl,$AclNum) ] } {
                    if { [ AclMatch $config($SessionId,Acl,$AclNum)\
                            $AclList ] } {
                        return 0
                    } else {
                        RetOnErr [UsrDoClearNumAclIos $SessionId $AclNum]
                        RetOnErr [UsrSetNumAclIos $SessionId $AclNum $AclList]
                    }
                } else {
                    RetOnErr [ UsrSetNumAclIos $SessionId $AclNum $AclList ]
                }
            } else {
                RetOnErr [ UsrDoClearNumAclIos $SessionId $AclNum ]
                RetOnErr [ UsrSetNumAclIos $SessionId $AclNum $AclList ]
            }
        }
        xdi {
            # CatCode doesn't have an access-list.
        }
        1900 {
            # 1900s do not support an access-list.
        }
        default {
            RetOnErr "errUnknownOs"
        }
    }

    return 0
}
