# libusrset.tcl --
#
#    libcisco - Configuration management API for Cisco networking equipment
#    Copyright (C) June 2002  Andy Ziegelbein
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Please send any questions or comments to andy@packetz.org.
#
#
# This file contains the Tcl code for the user functions.
#
# The user functions provide operating-system specific functions that perform
# typically a single task on a target device (e.g. add NTP server).  In order
# for a function to be defined in this file, it MUST adhere to the following
# two (2) criteria:
#
#       (1) The procedure must be written for a specific operating system.
#       (2) The procedure should be written to perform a single task.
#
# In addition, the following rules apply to procedures contained within this
# file:
#
# Arguments   : User procedures MUST NOT take a variable number of arguments.
# Options     : User procedures MUST NOT take optional values.
# SessionId   : User procedures MUST be given an explicit SessionId.
# Relationship: The following list defines the relationship between user
#                procedures and other procedures defined within this package:
#
#                       Kernel - MUST NOT call kernel procedures.
#                       Exported Kernel - SHOULD call one (1) or more kernel
#                               procedures.
#                       User - MAY call other user procedures.
#                       Exported User - MUST NOT call exported user procs.
#                       Package Support - MAY call package support procedures.
#                       Tcl - MAY call Tcl commands.
#                       Expect - MUST NOT call Expect commands.
#
# RCS|SCCS: %Z% %M% %I% %E% %U%

package provide libcisco 1.3

# libcisco::UsrSetPasswordEnableIos
#
#       The UsrSetPasswordEnableIos procedure is used to set the enable password
#       on the target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       PwType          string.  The type of enable password (enable|secret).
#       Password        string.  The new password.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetPasswordEnableIos { SessionId PwType Password } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    switch -regexp -- $PwType {
        enable {
            set Cmd "enable password $Password"
        }
        secret {
            set Cmd "enable secret $Password"
        }
        default {
            RetOnErr "errPwType"
        }
    }

    # Certain characters must be escaped as they have special meaning to
    # the regular expression pattern match

    set MatchCmd [ EscapeChars $Cmd regexp 2 ]

    set PatList {
        -nocase -re "as your enable secret(.*)\$Prompt" {
        }
        -nocase -re "as your enable password(.*)\$Prompt" {
        }
        -nocase -re "${MatchCmd}\\\[\r\n]{1,3}\$Prompt" {
        }
    }
    set PatList [ subst $PatList ]

    RetOnErr [ sendCmd global -patlist $PatList $SessionId $Cmd ]

    return 0
}

# libcisco::UsrSetPasswordLineIos
#
#       The UsrSetPasswordLine procedure is used to set the line password on
#       the target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Line            string.  The line or range of lines to be configured.
#       Password        string.  The new password.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetPasswordLineIos { SessionId Line Password } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    set Cmd "password $Password"

    RetOnErr [ sendCmd line -patlist nofeedback $SessionId $Line $Cmd ]

    return 0
}

# libcisco::UsrSetPasswordXdi
#
#       The UsrSetPasswordXdi procedure is used to set the password on the
#       target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       PwType          string.  The type of enable password (login|enable).
#       CurrentPw       string.  The current password.
#       NewPw           string.  The new password.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetPasswordXdi { SessionId PwType CurrentPw NewPw } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    switch -regexp -- $PwType {
        login {
            set Cmd "set password"
        }
        enable {
            set Cmd "set enablepass"
        }
        default {
            RetOnErr "errBadPwType"
        }
    }

    set PatList {
        -nocase -re "password incorrect(.*)$Prompt" {
            RetOnErr "errBadPw"
        }
        -nocase -re "password unchanged(.*)$Prompt" {
            RetOnErr "errUnchanged"
        }
        -nocase -re "password changed.*$Prompt" {
        }
    }

    set SubstPatList {
        -nocase -re "old password: ?$" {
            exp_send "${CurrentPw}\r"
            exp_continue
        }
        -nocase -re "new password: ?$" {
            exp_send "${NewPw}\r"
            exp_continue
        }
    }

    append PatList [ subst $SubstPatList ]

    RetOnErr [ sendCmd exec -patlist $PatList $SessionId $Cmd ]

    return 0
}

# libcisco::UsrSetPassword1900
#
#       The UsrSetPassword1900 procedure is used to set the password on the
#       target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       PwType          string.  The type of enable password (enable|secret).
#       Password        string.  The new password.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetPassword1900 { SessionId PwType Password } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    switch -regexp -- $PwType {
        login {
            set Cmd "enable password level 1 $Password"
        }
        enable {
            set Cmd "enable password level 15 $Password"
        }
        secret1 {
            set Cmd "enable secret level 1 $Password"
        }
        secret15 {
            set Cmd "enable secret level 15 $Password"
        }
        secret {
            set Cmd "enable secret $Password"
        }
        default {
            RetOnErr "errBadPwType"
        }
    }

    # Certain characters must be escaped as they have special meaning to
    # the regular expression pattern match

    set MatchCmd [ EscapeChars $Cmd regexp 2 ]

    set PatList {
        -nocase -re "invalid password length.*\$Prompt" {
            RetOnErr "errPwLength"
        }
        -nocase -re "${MatchCmd}\\\[\r\n]{1,3}\$Prompt" {
        }
    }
    set PatList [ subst $PatList ]

    RetOnErr [ sendCmd global -patlist $PatList $SessionId $Cmd ]

    return 0
}


# libcisco::UsrSetBannerIos
#
#       The UsrSetBannerIos procedure is used to set a banner message on the
#       target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       BannerType      string.  The type of banner.
#       BannerList      string list.  A list where each element is a separate
#                        line of text to be separated by a newline character.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetBannerIos { SessionId BannerType BannerList } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # Trim any leading or trailing newline and carriage return characters.

    foreach LineOfText $BannerList {
        lappend TempBannerList [ string trim $LineOfText "\r\n" ]
    }
    set BannerList $TempBannerList

    set BannerLength [ llength $BannerList ]

    if { $BannerLength == 0 } {
        RetOnErr "errNoBanner"
    }

    # Get a unique character to be used as the delimiting character and
    # extract the last line from the list--it will be sent separately.

    RetOnErr [ GetUniqueChar $BannerList ] DelimChar

    set FirstLine  [ lindex $BannerList 0 ]
    set LastLine   [ lindex $BannerList end ]
    set LastLine   [ EscapeChars $LastLine { {\"} {\[} {\]} } ]
    set BannerList [ lreplace $BannerList 0 0 ]
    set BannerList [ lreplace $BannerList end end ]

    switch -regexp -- $BannerType {
        exec {
            set Cmd "banner exec $DelimChar"
        }
        incoming {
            set Cmd "banner incoming $DelimChar"
        }
        login {
            set Cmd "banner login $DelimChar"
        }
        motd {
            set Cmd "banner motd $DelimChar"
        }
        prompt-timeout {
            set Cmd "banner prompt-timeout $DelimChar"
        }
        default {
            RetOnErr "errBannerType"
        }
    }

    append Cmd $FirstLine

    if { $BannerLength == 1 } {
        append Cmd $DelimChar

        set PatList "nofeedback"
    } elseif { $BannerLength == 2 } {
        set PatList {
            -nocase -re "enter text message.*\\\[\r\n\\\]{1,3}\$" {
                exp_send "${LastLine}${DelimChar}\\r"
     
                expect {
                    -nocase -re "\$Prompt" {
                    }
                }
            }
        }
    } else {
        set PatList {
            -nocase -re "enter text message.*\\\[\r\n\\\]{1,3}\$" {
                foreach LineOfText {$BannerList} {
                    exp_send "\$LineOfText\r"

                    expect {
                        -nocase -re "\\\[\r\n\\\]{1,3}\$" {
                        }
                        -nocase -re "\$Prompt" {
                            RetOnErr "errIncompBanner"
                        }
                    }

                }
                exp_send "${LastLine}${DelimChar}\r"
                
                expect {
                    -nocase -re "\\\[\r\n\\\]{1,3}\$Prompt" {
                    }
                }
            }
        }
    }

    set PatList [ subst $PatList ]

    RetOnErr [ sendCmd global -patlist $PatList $SessionId $Cmd ]

    return 0
}


# libcisco::UsrSetBannerMotdXdi
#
#       The UsrSetBannerMotdXdi procedure is used to set a banner message on
#       the target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       BannerList      string list.  A list where each element is a separate
#                        line of text to be separated by a newline character.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetBannerMotdXdi { SessionId BannerList } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # Trim any leading or trailing newline and carriage return characters.

    foreach LineOfText $BannerList {
        lappend TempBannerList [ string trim $LineOfText "\r\n" ]
    }
    set BannerList $TempBannerList

    set BannerLength [ llength $BannerList ]

    if { $BannerLength == 0 } {
        RetOnErr "errNoBanner"
    }

    # Get a unique character to be used as the delimiting character and
    # extract the last line from the list--it will be sent separately.

    RetOnErr [ GetUniqueChar $BannerList ] DelimChar

    set FirstLine  [ lindex $BannerList 0 ]
    set LastLine   [ lindex $BannerList end ]
    set LastLine   [ EscapeChars $LastLine { {\"} {\[} {\]} } ]
    set BannerList [ lreplace $BannerList 0 0 ]
    set BannerList [ lreplace $BannerList end end ]

    set Cmd "set banner motd $DelimChar"

    append Cmd $FirstLine

    if { $BannerLength == 1 } {
        append Cmd $DelimChar

        set PatList {
            -nocase -re "motd banner set.*\$Prompt" {
            }
        }
    } elseif { $BannerLength == 2 } {
        set PatList {
            -nocase -re "set banner motd $DelimChar.*\\\[\r\n]{1,3}\$" {
                exp_send "${LastLine}${DelimChar}\r"

                expect {
                    -nocase -re "motd banner set.*\$Prompt" {
                    }
                }
            }
        }
    } else {
        set PatList {
            -nocase -re "set banner motd $DelimChar.*\\\[\r\n]{1,3}\$" {
                foreach LineOfText {$BannerList} {
                    exp_send "\$LineOfText\r"

                    expect {
                        -nocase -re "\\\[\r\n]{1,3}\$" {
                        }
                        -nocase -re "\$Prompt" {
                            RetOnErr "errBannerTooLong"
                        }
                    }

                }
                exp_send "${LastLine}${DelimChar}\r"
                
                expect {
                    -nocase -re "motd banner set.*\$Prompt" {
                    }
                }
            }
        }
    }

    set PatList [ subst $PatList ]

    RetOnErr [ sendCmd exec -patlist $PatList $SessionId $Cmd ]

    return 0
}


# libcisco::UsrSetHostnameIos
#
#       The UsrSetHostnameIos procedure will set the hostname on an IOS-based
#       device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Hostname        string.  The hostname to be set.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetHostnameIos { SessionId Hostname } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # As of IOS v11.2, the hostname is limited to 29 characters.

    if { [ string length $Hostname ] > 29 } {
        RetOnErr "errNameTooLong"
    }	

    set Cmd "hostname $Hostname"

    RetOnErr [ sendCmd global -patlist nofeedback $SessionId $Cmd ]

    return 0
}


# libcisco::UsrSetPromptXdi
#
#       The UsrSetPromptXdi procedure will set the prompt on an XDI-based
#       device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Prompt          string.  The prompt to be set.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetPromptXdi { SessionId Prompt } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # As of CatCode v5.5(5), the prompt is limited to 20 characters.

    if { [ string length $Prompt ] > 20 } {
        RetOnErr "errPromptTooLong"
    }	

    set Cmd "set prompt $Prompt"

    RetOnErr [ sendCmd exec -patlist nofeedback $SessionId $Cmd ]

    return 0
}


# libcisco::UsrSetSystemNameXdi
#
#       The UsrSetSystemNameXdi procedure will set the system name on an
#       XDI-based device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       SysName         string.  The prompt to be set.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetSystemNameXdi { SessionId SysName } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # SNMP field length for the contact field is limited to 255 characters.

    if { [ string length $SysName ] > 255 } {
        RetOnErr "errSysNameTooLong"
    }	

    set Cmd "set system name $SysName"

    set PatList {
        -nocase -re "system name set.*$Prompt" {
        }
    }

    RetOnErr [ sendCmd exec -patlist $PatList $SessionId $Cmd ]

    return 0
}


# libcisco::UsrSetSnmpContactXXX
#
#       The UsrSetSnmpContactXXX procedures will set the system contact on the
#       target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Contact         string.  The contact to be set.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetSnmpContactIos { SessionId Contact } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # SNMP field length for the contact field is limited to 255 characters.

    if { [ string length $Contact ] > 255 } {
        RetOnErr "errContactTooLong"
    }	

    set Cmd "snmp-server contact $Contact"

    RetOnErr [ sendCmd global -patlist nofeedback $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrSetSnmpContactXdi { SessionId Contact } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # As of CatCode v5.5(5), the system contact is limited to 255 characters.

    if { [ string length $Contact ] > 255 } {
        RetOnErr "errContactTooLong"
    }	

    set Cmd "set system contact $Contact"

    set PatList {
        -nocase -re "system contact set.*$Prompt" {
        }
    }

    RetOnErr [ sendCmd exec -patlist $PatList $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrSetSnmpContact1900 { SessionId Contact } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # The 1900 is limited to 220 characters of input from config mode when
    # entering this field.

    if { [ string length $Contact ] > 220 } {
        RetOnErr "errContactTooLong"
    }	

    set Cmd "snmp-server contact \"$Contact\""

    RetOnErr [ sendCmd global $SessionId $Cmd ]

    return 0
}


# libcisco::UsrSetSnmpLocationXXX
#
#       The UsrSetSnmpLocationXXX procedures will set the system location on
#       the target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Location        string.  The location to be set.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetSnmpLocationIos { SessionId Location } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # SNMP field length for the location field is limited to 255 characters.

    if { [ string length $Location ] > 255 } {
        RetOnErr "errLocationTooLong"
    }	

    set Cmd "snmp-server location $Location"

    RetOnErr [ sendCmd global -patlist nofeedback $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrSetSnmpLocationXdi { SessionId Location } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # As of CatCode v5.5(5), the system location is limited to 255 characters.

    if { [ string length $Location ] > 255 } {
        RetOnErr "errLocationTooLong"
    }	

    set Cmd "set system location $Location"

    set PatList {
        -nocase -re "system location set.*$Prompt" {
        }
    }

    RetOnErr [ sendCmd exec -patlist $PatList $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrSetSnmpLocation1900 { SessionId Location } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # The 1900 is limited to 220 characters of input from config mode when
    # entering this field.

    if { [ string length $Location ] > 220 } {
        RetOnErr "errLocationTooLong"
    }	

    set Cmd "snmp-server location \"$Location\""

    RetOnErr [ sendCmd global $SessionId $Cmd ]

    return 0
}


# libcisco::UsrSetSnmpCommunityStringXXX
#
#       The UsrSetSnmpCommunityStringXXX procedures will set the SNMP 
#       community string on the target device.
#
# Arguments:
#       SessionId       string.  The unique ID of the session.
#       Operation       string.  The operation to perform.  Valid values are
#                        'add' and 'remove'.
#       StringType      string.  The type of community string.  Valid value
#                        are 'ro' and 'rw'.
#       String          string.  The SNMP community string.  The keyword
#                        'all' may be specified with the 'remove' operation to
#                        remove all community strings of the specified type.
#       Acl             unsigned integer.  The ACL number associated with
#                        the community string.
#
# Retsults:
#       0 on success.
#       A short text message on error.

proc ::libcisco::UsrSetSnmpCommunityIos { SessionId Operation StringType\
        String Acl } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # An IOS-based device can support up to approximately 220 characters for
    # a community string.

    if { [ string length $String ] > 220 } {
        RetOnErr "errStringTooLong"
    }

    if { ! [ string match $StringType ro ] &&\
            ! [ string match $StringType rw ] } {
        RetOnErr "errBadStringType"
    }

    switch -- $Operation {
        add {
            set Cmd "snmp-server community $String $StringType $Acl"
        }
        remove {
            set Cmd "no snmp-server community $String $StringType $Acl"
        }
        default {
            RetOnErr "errBadOp"
        }
    }
    set Cmd [ string trim $Cmd ]

    set PatList {
        -nocase -re "cannot find community.*$Prompt" {
            RetOnErr "errStringNotSet"
        }
        -nocase -re "$Cmd\[\r\n\]{1,3}$Prompt" {
        }
    }

    RetOnErr [ sendCmd global -patlist $PatList $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrSetSnmpCommunityXdi { SessionId StringType String } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # An XDI-based device can support up to 20 characters for the
    # community string.

    if { [ string length $String ] > 20 } {
        RetOnErr "errStringTooLong"
    }

    if { ! [ string match $StringType read-only ] &&\
            ! [ string match $StringType read-write ] &&\
            ! [ string match $StringType read-write-all ] } {
        RetOnErr "errBadStringType"
    }

    # An empty string will result in the string type being cleared.

    set Cmd [ string trim "set snmp community $StringType $String" ]

    set PatList {
        -nocase -re "community string cleared.*$Prompt" {
        }
        -nocase -re "community string set.*$Prompt" {
        }
    }

    RetOnErr [ sendCmd exec -patlist $PatList $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrSetSnmpCommunity1900 { SessionId Operation StringType\
        String } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    # A 1900 can support up to 31 characters for a community string.

    if { [ string length $String ] > 31 } {
        RetOnErr "errStringTooLong"
    }

    if { ! [ string match $StringType ro ] &&\
            ! [ string match $StringType rw ] } {
        RetOnErr "errBadStringType"
    }

    switch -- $Operation {
        add {
        }
        remove {
            append Cmd "no "
        }
        default {
            RetOnErr "errBadOp"
        }
    }

    append Cmd "snmp-server community $String $StringType"

    set PatList {
        -nocase -re "error: community string not found.*$Prompt" {
            RetOnErr "errStringNotSet"
        }
        -nocase -re "error: maximum number.*$Prompt" {
            RetOnErr "errMaxNumStrings"
        }
        -nocase -re "$Cmd\[\r\n\]{1,3}$Prompt" {
        }
    }

    RetOnErr [ sendCmd global -patlist $PatList $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrSetIpPermitXdi { SessionId Args } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    set Cmd "set ip permit $Args"

    set PatList {
        -nocase -re "entry already exists" {
            RetOnErr "errEntryExists"
        }
        -nocase -re "unknown host" {
            RetOnErr "errUnknownHost"
        }
        -nocase -re "permit list disabled.*$Prompt" {
        }
        -nocase -re "permit list enabled.*$Prompt" {
        }
        -nocase -re "added to.*permit list.*$Prompt" {
        }
    }

    RetOnErr [ sendCmd exec -patlist $PatList $SessionId $Cmd ]

    return 0
}

proc ::libcisco::UsrSetNumAclIos { SessionId AclNum AclList } {
    # Check to ensure that the session is not in user mode.

    if { [ string match [ session info $SessionId mode ] user ] } {
        RetOnErr "errWrongMode"
    }

    foreach Entry $AclList {
        set Cmd "access-list $AclNum $Entry"

        RetOnErr [ sendCmd global $SessionId $Cmd ]
    }

    return 0
}
