#!/bin/sh
# the next line restarts using the tclsh interpreter \
exec tclsh "$0" "$@"

# Tell the interpreter to load the libcisco package.  The interpreter will
# search through the directories and subdirectories of all directories listed
# in the auto_path variable for pkgIndex.tcl files.
package require libcisco

# The procedure names are all contained within a Tcl namespace of "libcisco"
# by default.  We can optionally import these procedures into the global
# namespace with the following command.  This basically means that the rest
# of the script can call the libcisco procedures without the leading
# ::libcisco:: in front of each procedure (e.g. "::libcisco::session" may now
# be called simply as "session".
namespace import ::libcisco::*

# There are three different forms of error-handling.  By default, libcisco
# will call the Tcl command "exit 1" when it encounters an error.  We want
# to handle errors using the Tcl "catch" command instead, so we will set
# the error-handling method to 'error' instead of the default of 'exit'.  This
# will cause the routines to raise a Tcl error condition on error.
set ::libcisco::state(ErrorHandling) "error"

# This script expects that the following login information is passed in on
# the command-line as arguments.  Tcl stores these arguments in a the "argv"
# variable.
set StringType  [ lindex $argv 0 ]
set InputFile   [ lindex $argv 1 ]
set LinePw      [ lindex $argv 2 ]
set EnablePw    [ lindex $argv 3 ]
set TacUsername [ lindex $argv 4 ]
set TacLoginPw  [ lindex $argv 5 ]
set TacEnablePw [ lindex $argv 6 ]

# This isn't absolutely necessary, but we'll use it in our next block of code.
# Tcl stores the name of the script being evaluated in a variable "argv0".
set ScriptName [ file tail $argv0 ]

# This doesn't have to be in here, but it's good coding practice to make sure
# we actually received the expected number of arguments passed in on the
# command-line.  Tcl stores the number of arguments passed in on the command-
# line, excluding the script name, in the "argc" variable.
if { $argc < 7 } {
    puts "Usage: $ScriptName {StringType} {InputFile} {LinePw} {EnablePw}\
                 {TacUsername}"
    puts "       {TacLoginPw} {TacEnablePw}"
    puts ""
    puts "        StringType - must be one of two values, 'ro' or 'rw'"
    puts "        InputFile  - a list of IP addresses, one IP per line"
    exit 1
}

# Open the input file.
set FileId [ open $InputFile r ]

# Read each line of the input file (input file is a list of IP addresses).
while { ! [ eof $FileId ] } {
    # Read in a single line of input from the input file.
    gets $FileId Ip

    # If the line is blank, skip to the next line.
    if { [ string length [ string trim $Ip ] ] == 0 } {
        continue
    }

    # Output some sort of progress indidcator to the user.
    puts -nonewline "( [ format %-15.15s $Ip ] )."
    flush stdout

    # No error checking is performed on the following lines of code.  If the
    # called procedure encounters an error, it will cause the script to exit
    # with a return value of 1.  This is the default error-handling method.

    # Open a session to the target host.  Log the output from the session to
    # getConfig.log.
    if { [ catch { session open -logappend clearSnmp.log $Ip $LinePw\
            $TacUsername $TacLoginPw } Result ] } {
        puts $Result
        continue
    } else {
        puts -nonewline "."
        flush stdout
    }

    # Put the session in privileged EXEC (enable) mode.
    if { [ catch { session enable $EnablePw $TacUsername $TacEnablePw }\
            Result ] } {
        puts $Result
        continue
    } else {
        puts -nonewline "."
        flush stdout
    }

    # Clear the community string of the specified type.
    if { [ catch { ezset community -clear $StringType } Result ] } {
        puts $Result
        continue
    } else {
        puts -nonewline "."
        flush stdout
    }

    # Save the config if it has changed.
    if { [ catch { ezdo savecfg -ifchanged } Result ] } {
        puts $Result
        continue
    } else {
        puts -nonewline "."
        flush stdout
    }

    # Close the session.  This will close the telnet session and cleanup all Tcl
    # resources dedicated to the session.
    if { [ catch { session close } Result ] } {
        puts $Result
    } else {
        puts "ok"
    }
}
