# libsupport.tcl --
#
#    libcisco - Configuration management API for Cisco networking equipment
#    Copyright (C) June 2002  Andy Ziegelbein
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Please send any questions or comments to andy@packetz.org.
#
#
# This file contains the Tcl code that makes up the supporting functions
# for all procedures in the package.  These procedures are not for export.
#
# The following rules apply to procedures contained within this file:
#
# Arguments   : Support procedures SHOULD NOT take a variable number of
#                arguments.
# Options     : Support procedures MUST NOT take options.
# SessionId   : Support procedures MUST NOT take or require a SessionId.
# Relationship: The following list defines the relationship between support
#                procedures and other procedures defined within this package:
#
#                       Kernel - MUST NOT call kernel procedures.
#                       Exported Kernel - MUST NOT call exported kernel procs.
#                       User - MUST NOT call user procedures.
#                       Exported User - MUST NOT call exported user procs.
#                       Package Support - MAY call other package support procs.
#                       Tcl - SHOULD call one (1) or more Tcl commands.
#                       Expect - MUST NOT call Expect commands.
#
# RCS|SCCS: %Z% %M% %I% %E% %U%

package provide libcisco 1.3


# libcisco::RetOnErr --
#
#       The RetOnErr procedure will cause the calling proc to return if the
#       passed in value is anything but 0.
#
# Arguments:
#       ReturnCode       string.  The return value to be evaluated.
#       resultStringName (optional) string.  The name of a variable in the
#                         calling procedure that will hold the value of
#                         ReturnCode.
#
# Results:
#       None

proc ::libcisco::RetOnErr { ReturnCode { resultStringName {} } } {
    variable state

    if { [ string length $resultStringName ] > 0 } {
        upvar $resultStringName Result
        set Result $ReturnCode
    }

    if { [ string match "err" [ string range $ReturnCode 0 2 ] ] } {
        switch -- $state(ErrorHandling) {
            exit {
                puts stderr $ReturnCode
                exit 1
            }
            return {
                return -code return $ReturnCode
            }
            error {
                return -code error $ReturnCode
            }
            default {
                return -code return $ReturnCode
            }
        }
    } else {
        return $ReturnCode
    }
}


# libcisco::GetOpts --
#
#       The GetOpts procedure will, given an option list with default values
#       and types, parse through a list of arguments setting the values to a
#       named array where each element is named with its option name.  The
#       remaining arguments will be set to a variable in the calling
#       procedure.
#
# Arguments:
#       OptionList    list.  A list of option and default value pairs.
#       ArgsList      list.  A list of arguments to be parsed.
#       valArrayName  string.  The name of a variable in the calling procedure
#                      that will hold the values of the parsed options.  If
#                      no value is found for an option, the default value
#                      found in the OptionList will be set.
#       argListName   string.  The name of a variable in the calling procedure
#                      that will hold a list of the remaining arguments.
#
# Returns
#       0 on success
#       On error, a short text message beginning with the string "err".

proc ::libcisco::GetOpts { OptionList ArgsList valArrayName argListName } {
    upvar $valArrayName OptValue
    upvar $argListName RemainingArgsList
    variable state

    # Trim the leading and trailing spaces from the name, type, and value.
    # Read the list of options and their corresponding default values and
    # populate the OptType and OptValue arrays.

    foreach { OptionName OptionType DefaultValue } $OptionList {
        set OptionName            [ string trim $OptionName ]

        set OptType($OptionName)  [ string trim $OptionType ]

        switch -- $OptType($OptionName) {
            "boolean" {
                set OptValue($OptionName) 0
            }
            default {
                set OptValue($OptionName) [ string trim $DefaultValue ]
            }
        }
    }

    # Extract the options and set remaining arguments to RemainingArgsList.

    set ArgCount          [ llength $ArgsList ]
    set RemainingArgsList ""

    for { set i 0 } { $i < $ArgCount } { incr i } {
        set Arg [ lindex $ArgsList $i ]

        if { [ string match $Arg "--" ] } {
            incr i
            set RemainingArgsList [ lrange $ArgsList $i end ]
            break
        } elseif { [ string match [ string range $Arg 0 0 ] "-" ] } {
            set FoundOption [ string trimleft $Arg "-" ]

            if { [ info exists OptType($FoundOption) ] } {
                switch -- $OptType($FoundOption) {
                    boolean {
                        set OptValue($FoundOption) 1
                    }
                    string {
                        incr i
                        set OptValue($FoundOption) [ lindex $ArgsList $i ]
                    }
                    default {
                        RetOnErr "errBadOptType"
                    }
                }
            } else {
                RetOnErr "errUnknownOpt"
            }
        } else {
            set RemainingArgsList [ lrange $ArgsList $i end ]
            break
        }
    }

    return 0
}


# libcisco::EscapeBrackets --
#
#       The EscpaeBrackets procedure will insert backslashes before all open
#       or close brackets in the passed in string.
#
# Arguments:
#       String          string.  The string to be manipulated.
#
# Results:
#       The manipulated string with escaped backslashes is returned.

proc ::libcisco::EscapeBrackets { String } {
    regsub -all -- "\\\[" $String "\\\[" String
    regsub -all -- "\\\]" $String "\\\]" String

    return $String
}

proc ::libcisco::EscapeChars { String CharList {Count 1} } {
    switch -- $CharList {
        tcl {
            set CharList { {\\} {\$} \\[ \\] \" \\( \\) }
        }
        regexp {
            set CharList\
                    { {\\} {\"} {\[} {\]} {\*} {\+} {\^} {\$} {\(} {\)} {\.} }
        }
        default {
        }
    }

    for { set i 0 } { $i < $Count } { incr i } {
        foreach Char $CharList {
            regsub -all -- $Char $String "\\$Char" String
        }
    }

    return $String
}


# libcisco::SavedResultToList --
#
#       The SavedResultToList procedure will convert the returned result from
#       a sendCmd function to a list.
#
# Arguments:
#       SavedResult     string.  The saved result which is to be converted.
#
# Results:
#       A list where each element is a single line of text from the saved
#       result.

proc ::libcisco::SavedResultToList { SavedResult } {
    regsub -all "\r\n" $SavedResult "\r" SavedResult
    regsub -all "\n\r" $SavedResult "\r" SavedResult
    regsub -all "\n" $SavedResult "\r" SavedResult

    set SavedResultList [ split $SavedResult "\r" ]

    return $SavedResultList
}


# libcisco::RemoveMorePrompts --
#
#       The RemoveMorePrompts procedure will remove all more prompts including
#       a leading newline formfeed combination from the passed in string.
#
# Arguments:
#       String          string.  The string to be manipulated.
#
# Results:
#       The passed in string with more prompts removed is returned.

proc ::libcisco::RemoveMorePrompts { String } {
    regsub -all -nocase -- "\[\r\n]{1,2}--more--\[\r\n]{1,3}" $String "" String

    return $String
}


# libcisco::GetUniqueChar --
#
#       GetUniqueChar will read through a list of strings and return a single
#       unique character which is not found in any of the strings.  This
#       procedure is intended to be used with functions which require a unique
#       delimiting character to mark the beginning and ending of a block of
#       text (e.g. banner motd).
#
# Arguments:
#       StringList   string list.  A list of strings to be searched.
#
# Results:
#       On success, a unique character is returned.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::GetUniqueChar { StringList } {
    # All of the characters in the following list have been tested under
    # CatOS v5.5(5) and IOS v12.0(9).

    foreach Char { ~ ` ! @ # % & _ = | ; : ' , < . > 0 1 2 3 4 5 6 7 8 9 \
            a b c d e f g h i j k l m n o p q r s t u v w x y z \
            A B C D E F G H I J K L M N O P Q R S T U V W X Y Z } {
        if { [ lsearch -regexp $StringList $Char ] == -1 } {
             set UniqueChar $Char
             return $UniqueChar
        }
    }

    RetOnErr "errUniqueChar"
}


# libcisco::DecryptCiscoPassword --
#
#       DecryptCiscoPassword will decrypt a Cisco type 7 encrypted password.
#
# Arguments:
#       EncryptedPw   string.  The type 7 encrypted password.
#
# Results:
#       On success, the decrypted password is returned.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::DecryptCiscoPassword { EncryptedPw } {
    lappend SeedTable 0x64 0x73 0x66 0x64 0x3b 0x6b 0x66 0x6f 0x41 0x2c\
            0x2e 0x69 0x79 0x65 0x77 0x72 0x6b 0x6c 0x64 0x4a 0x4b 0x44\
            0x48 0x53 0x55 0x42

    # The string length must be even to be a valid encrypted password.

    if { [ expr [ string length $EncryptedPw ] % 2 ] } {
        RetOnErr "errBadEncryptPw"
    }

    set SeedIndex   [ string range $EncryptedPw 0 1 ]
    set SeedIndex   [ string trimleft $SeedIndex 0 ]
    set EncryptedPw [ string range $EncryptedPw 2 end ]
    set PwLength    [ string length $EncryptedPw ]

    if { [ string length $SeedIndex ] == 0 } {
        set SeedIndex 0
    }

    if { $SeedIndex > 15 } {
        RetOnErr "errBadEncryptPw"
    }

    for { set i 0 } { $i < $PwLength } { incr i 2 } {
        set EncodedHexVal    "0x"
        append EncodedHexVal\
                [ string range $EncryptedPw $i [ expr { $i + 1 } ] ]

        set SeedTableHexVal   [ lindex $SeedTable $SeedIndex ]

        set DecodedDecimalVal [ expr $EncodedHexVal ^ $SeedTableHexVal ]
        append DecryptedPw    [ format %c $DecodedDecimalVal ]

        incr SeedIndex
    }

    return $DecryptedPw
}

proc ::libcisco::PermitMatch { List1 List2 } {
    foreach Entry1 $List1 Entry2 $List2 {
        foreach VarName {Entry1 Entry2} Val [ list $Entry1 $Entry2 ] {
            set Length [ llength $Val ]

            if { $Length == 1 } {
                set $VarName "$Val 255.255.255.255"
            } elseif { $Length == 2 && [ string match "*snmp" $Val ] } {
                set $VarName "[lindex $Val 0] 255.255.255.255 snmp"
            } elseif { $Length == 2 && [ string match "*telnet" $Val ] } {
                set $VarName "[lindex $Val 0] 255.255.255.255 telnet"
            }
        }

        if { ! [ string match $Entry1 $Entry2 ] } {
            return 0
        }
    }

    return 1
}

proc ::libcisco::AclMatch { AclList1 AclList2 } {
    set ExtAcl 0

    foreach Entry $AclList1 {
        if { [ llength $Entry ] > 3 } {
            set ExtAcl 1
            break
        }
    }

    if { $ExtAcl } {
        foreach Entry1 $AclList1 Entry2 $AclList2 {
            foreach VarName {Entry1 Entry2} Val [ list $Entry1 $Entry2 ] {
                regsub -all -- "any" $Val "0.0.0.0 255.255.255.255" Val
                regsub -all -- "\[0-9\.]+ 255\.255\.255\.255" $Val\
                        "0.0.0.0 255.255.255.255" Val

                set $VarName $Val
            }

            # Compare the entries.

            if { ! [ string match $Entry1 $Entry2 ] } {
                return 0
            }
        }
    } else {
        foreach Entry1 $AclList1 Entry2 $AclList2 {
            # Convert all entries to a common format.

            foreach VarName {Entry1 Entry2} Val [ list $Entry1 $Entry2 ] {
                set PoD   [ lindex $Val 0 ]
                set Entry [ lrange $Val 1 end ]

                switch -glob -- $Entry {
                    "any" {
                        set NewEntry "$PoD 0.0.0.0 255.255.255.255"
                    }
                    "* 255.255.255.255" {
                        set NewEntry "$PoD 0.0.0.0 255.255.255.255"
                    }
                    default {
                        if { [ llength $Entry ] == 1 } {
                            set NewEntry "$PoD $Entry 0.0.0.0"
                        } else {
                            set NewEntry "$PoD $Entry"
                        }
                    }
                }

                set $VarName $NewEntry
            }

            # Compare the entries.
            if { ! [ string match $Entry1 $Entry2 ] } {
                return 0
            }
        }
    }

    return 1
}


# libcisco::AsciiFileToList --
#
#       AsciiFileToList will read in an ascii file and return the result
#       as a Tcl list where each element is a line of text.  Lines of
#       text are delimited by the platforms newline sequence of characters.
#
# Arguments:
#       Filename      string.  The fully-qualified path and filename of the
#                      ascii file to be read.
#
# Results:
#       On success, the result is returned as a Tcl list.
#       On error, a short text message beginning with the string "err".

proc ::libcisco::AsciiFileToList { Filename } {
    if { ! [ file exists $Filename ] } {
        RetOnErr "errFileNotFound"
    }

    if { ! [ file readable $Filename ] } {
        RetOnErr "errFileNotReadable"
    }

    if { ! [ file isfile $Filename ] } {
        RetOnErr "errInvalidFile"
    }

    set FileId [ open $Filename r ]

    while { ! [ eof $FileId ] } {
        gets $FileId LineBuffer
        lappend LineBufferList $LineBuffer
    }

    close $FileId

    set LineBufferList [ lreplace $LineBufferList end end ]

    return $LineBufferList
}
