//
//! \file  libecc/bitset.h
//! \brief Fixed size bitset.
//!
//! This header file declares <code>template<unsigned int n> class libecc::bitset</code> for a fixed number of \a n bits.
//
// This file is part of the libecc package.
// Copyright (C) 2002, by
//
// Carlo Wood, Run on IRC <carlo@alinoe.com>
// RSA-1024 0x624ACAD5 1997-01-26                    Sign & Encrypt
// Fingerprint16 = 32 EC A7 B6 AC DB 65 A6  F6 F6 55 DD 1C DC FF 61
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#ifndef LIBECC_BITS_H
#define LIBECC_BITS_H

#ifndef LIBECC_DEBUG_H
#error "You need to include the appropriate debug.h in the source file, before including this header file."
#endif

#include <libecc/config.h>
#include <iosfwd>
#include <cstddef>
#include <cstring>
#include <string>
#include <inttypes.h>
#ifdef __i386__
#include <asm/bitops.h>
#endif
#ifdef CWDEBUG
#include <libcw/cwprint.h>
#endif

#ifdef __i386__
// Assembly code, defined in window.s.
extern "C" int libecc_shift_xorassign(unsigned long* bitset1, unsigned long const* bitset2, int lsb, int msb, int shift);
extern "C" int libecc_shiftwindow_xorassign(unsigned long* bitset1, unsigned long const* bitset2, int lsb, int msb, int shift);
#endif

namespace libecc {

/**\typedef unsigned long bitset_digit_t
 *
 * \brief Internal data type, used to store the bits of class bitset.
 *
 * The number of bits in the builtin-type \c long turns out to be equal to the size
 * of the databus between memory and cpu.
 * The internal type used to store the bits of a bitset is therefore an
 * <code>unsigned long</code> resulting in the most efficient code for bit manipulation.
 */
typedef	unsigned long bitset_digit_t;

//! The number of bits in a single bitset digit.
static unsigned int const bitset_digit_bits = sizeof(bitset_digit_t) * 8;

//! The log2 of the number of bits in a single digit.
#if ECC_BITS == 32
static unsigned int const bitset_digit_bits_log2 = 5;
#elif ECC_BITS == 64
#warning 64 bit code has not been tested.  This warning will remain here until you mail me.
static unsigned int const bitset_digit_bits_log2 = 6;
#elif ECC_BITS == 128
#warning 128 bit code has not been tested.  This warning will remain here until you mail me.
static unsigned int const bitset_digit_bits_log2 = 7;
#endif

// Forward declaration.
template<unsigned int n, bool inverted>
  class bitset_invertible;

/**\class bitset_base bitset.h libecc/bitset.h
 *
 * \brief Base class of libecc::bitset.
 *
 * This class is for internal use only.
 * It contains the actual data, an array of \link libecc::bitset_digit_t bitset_digit_t \endlink,
 * the meaning of which is determined by the derived classes;
 * direct access of the data is therefore useless and prohibited (\link bitset_base::vector vector \endlink is protected).
 */
template<unsigned int n>
  struct bitset_base {
    public:
      // Fix this if you add members in front of vector.
      static size_t const offsetof_vector = 0; //sizeof(bitset_digit_t);

    public:
      //! \brief The number of bits in the bitset.
      //!
      //! This constant is simply equal to \a n.
      static unsigned int const number_of_bits = n;

      //! The minimum number of digits needed to store \a n bits.
      static unsigned int const digits = (n - 1) / bitset_digit_bits + 1;

      // ! The number of valid bits in the most significant digit.
      static unsigned int const number_of_valid_bits = bitset_digit_bits - (digits * bitset_digit_bits - n);

      //! A mask marking the valid bits in the most significant digit.
      static bitset_digit_t const valid_bits =
	  ((static_cast<bitset_digit_t>(1) << (number_of_valid_bits - 1)) << 1) - 1;

      //! True when not all bits in the most significant digit are valid.
      static bool const has_excess_bits = ((digits * bitset_digit_bits) != n);

    protected:
      //! \brief The actual bits, stored as an array of \link bitset::digits digits \endlink
      //! number of "digits" of type libecc::bitset_digit_t.
      //!
      //! The first digit is the <em>least significant</em> digit: printing is done from high index to low index.
      //! The bits in the array can represent the actual bits or the inverted bits of the bitset, depending on
      //! the type of the derived class.
      //bitset_digit_t empty_space1;
      bitset_digit_t vector[digits];
      //bitset_digit_t empty_space2;

      template<unsigned int n1, bool i1, unsigned int n2, bool i2>
	friend bool operator==(bitset_invertible<n1, i1> const&, bitset_invertible<n2, i2> const&);

      template<unsigned int n1, bool i1, unsigned int n2, bool i2>
	friend bool operator!=(bitset_invertible<n1, i1> const&, bitset_invertible<n2, i2> const&);

    public:
      // Digit representation
      bitset_digit_t& rawdigit(unsigned int d) { return this->vector[d]; }
      bitset_digit_t const* digits_ptr(void) const { return this->vector; }
      bitset_digit_t* digits_ptr(void) { return this->vector; }
      uint32_t digit32(unsigned int d32) const
      {
	if (sizeof(bitset_digit_t) == 4)
	  return this->vector[d32];
	else
	{
	  unsigned int d = d32 / (sizeof(bitset_digit_t) / 4);
	  unsigned int r = d32 % (sizeof(bitset_digit_t) / 4);
	  return (this->vector[d] >> (r * 32)) & 0xffffffff;
	}
      }
  };

#ifndef HIDE_FROM_DOXYGEN
namespace Operator {

#ifdef __i386__

//
// Due to heavily broken inlining heuristics of gcc,
// we are forced to use macros to do the assembly
// inlining.  There is no alternative, I tried everything.
//

#define LIBECC_INVERT "xorl $-1,%%eax\n\t"
#define LIBECC_NOTHING ""

#define LIBECC_ASMLOOP(destination, source, count, OPCODE, OPTIONAL_INVERT)	\
  do {							\
    int __d0, __d1, __d2;				\
    __asm__ __volatile__ (				\
      "\n1:\n\t"					\
	"movl (%5,%%ecx,4),%%eax\n\t"			\
	OPTIONAL_INVERT					\
	OPCODE " %%eax,(%4,%%ecx,4)\n\t"		\
	"decl %%ecx\n\t"				\
	"jnz 1b"					\
	: "=c" (__d0), "=&r" (__d1), "=&r" (__d2)	\
	: "0" (count),					\
	  "1" ((destination) - 1),			\
	  "2" ((source) - 1)				\
	: "%eax", "memory"				\
    );							\
  } while(0)

#define LIBECC_ASMLOOP_BODY(OPCODE, destination, source, count, inverted) \
  do { \
    if (inverted) \
      LIBECC_ASMLOOP(destination, source, count, OPCODE, LIBECC_INVERT); \
    else \
      LIBECC_ASMLOOP(destination, source, count, OPCODE, LIBECC_NOTHING); \
  } while(0)

#define LIBECC_ASMLOOP2(destination, source1, source2, count, OPCODE, OPTIONAL_INVERT1, OPCODE2, OPTIONAL_INVERT3)		\
  do {								\
    int __d0, __d1, __d2, __d3;					\
    __asm__ __volatile__ (					\
      "\n1:\n\t"						\
	"movl (%6,%%ecx,4),%%eax\n\t"				\
	OPTIONAL_INVERT1					\
	OPCODE2 " (%7,%%ecx,4),%%eax\n\t"			\
	OPTIONAL_INVERT3					\
	OPCODE " %%eax,(%5,%%ecx,4)\n\t"			\
	"decl %%ecx\n\t"					\
	"jnz 1b"						\
	: "=c" (__d0), "=&r" (__d1), "=&r" (__d2), "=&r" (__d3)	\
	: "0" (count),						\
	  "1" ((destination) - 1),				\
	  "2" ((source1) - 1),					\
	  "3" ((source2) - 1)					\
	: "%eax", "memory"					\
    );								\
  } while(0)

#define LIBECC_ASMLOOP2_BODY(OPCODE, OPERATOR, inverted1, inverted2, destination, source1, source2, count) \
  do { \
    if (OPERATOR::id == libecc::Operator::bitsetAND::id) \
    { \
      if (inverted1 && inverted2) \
	LIBECC_ASMLOOP2(destination, source1, source2, count, OPCODE, LIBECC_NOTHING, "orl", LIBECC_INVERT); \
      else if (inverted1) \
	LIBECC_ASMLOOP2(destination, source1, source2, count, OPCODE, LIBECC_INVERT, "andl", LIBECC_NOTHING); \
      else if (inverted2) \
	LIBECC_ASMLOOP2(destination, source2, source1, count, OPCODE, LIBECC_INVERT, "andl", LIBECC_NOTHING); \
      else \
	LIBECC_ASMLOOP2(destination, source1, source2, count, OPCODE, LIBECC_NOTHING, "andl", LIBECC_NOTHING); \
    } \
    else if (OPERATOR::id == libecc::Operator::bitsetOR::id) \
    { \
      if (inverted1 && inverted2) \
	LIBECC_ASMLOOP2(destination, source1, source2, count, OPCODE, LIBECC_NOTHING, "andl", LIBECC_INVERT); \
      else if (inverted1) \
	LIBECC_ASMLOOP2(destination, source1, source2, count, OPCODE, LIBECC_INVERT, "orl", LIBECC_NOTHING); \
      else if (inverted2) \
	LIBECC_ASMLOOP2(destination, source2, source1, count, OPCODE, LIBECC_INVERT, "orl", LIBECC_NOTHING); \
      else \
	LIBECC_ASMLOOP2(destination, source1, source2, count, OPCODE, LIBECC_NOTHING, "orl", LIBECC_NOTHING); \
    } \
    else if (OPERATOR::id == libecc::Operator::bitsetXOR::id) \
    { \
      if (inverted1 == inverted2) \
	LIBECC_ASMLOOP2(destination, source1, source2, count, OPCODE, LIBECC_NOTHING, "xorl", LIBECC_NOTHING); \
      else \
	LIBECC_ASMLOOP2(destination, source1, source2, count, OPCODE, LIBECC_NOTHING, "xorl", LIBECC_INVERT); \
    } \
  } while(0)

#define LIBECC_INLINE_ASMLOOP(ID, destination, source, count, inverted) \
    do { \
      if (ID == libecc::Operator::bitsetAssign::id) \
      { \
	LIBECC_ASMLOOP_BODY("movl", destination, source, count, inverted); \
      } \
      else if (ID == libecc::Operator::bitsetANDAssign::id) \
      { \
	LIBECC_ASMLOOP_BODY("andl", destination, source, count, inverted); \
      } \
      else if (ID == libecc::Operator::bitsetORAssign::id) \
      { \
	LIBECC_ASMLOOP_BODY("orl", destination, source, count, inverted); \
      } \
      else if (ID == libecc::Operator::bitsetXORAssign::id) \
      { \
	LIBECC_ASMLOOP_BODY("xorl", destination, source, count, inverted); \
      } \
    } while(0)

#define LIBECC_INLINE_ASMLOOP2(ID, OPERATOR, inverted1, inverted2, destination, source1, source2, count) \
    do { \
      if (ID == libecc::Operator::bitsetAssign::id) \
      { \
	LIBECC_ASMLOOP2_BODY("movl", OPERATOR, inverted1, inverted2, destination, source1, source2, count); \
      } \
      else if (ID == libecc::Operator::bitsetANDAssign::id) \
      { \
	LIBECC_ASMLOOP2_BODY("andl", OPERATOR, inverted1, inverted2, destination, source1, source2, count); \
      } \
      else if (ID == libecc::Operator::bitsetORAssign::id) \
      { \
	LIBECC_ASMLOOP2_BODY("orl", OPERATOR, inverted1, inverted2, destination, source1, source2, count); \
      } \
      else if (ID == libecc::Operator::bitsetXORAssign::id) \
      { \
	LIBECC_ASMLOOP2_BODY("xorl", OPERATOR, inverted1, inverted2, destination, source1, source2, count); \
      } \
    } while(0)

#define LIBECC_ASMSHIFTRIGHT0(OP1)			\
	      __asm__ __volatile__ (			\
		  "movl 4(%%esi),%%edx\n\t"		\
		  "shrl $1,%%edx\n\t"			\
		  OP1					\
		  "movl %%edx,4(%%edi)"			\
		: "=&S" (ptr1),				\
		  "=&D" (ptr2)				\
		: "0" (&this->vector[initial - count]),	\
		  "1" (&result.vector[initial - count])	\
		: "%eax", "%edx", "memory", "cc"	\
	      )

#if 0	// NOT faster.
#define LIBECC_OP1_SOURCE "(%%esi)"
#define LIBECC_OP2_SOURCE "(%%esi)"
#define LIBECC_OP1_DESTINATION "(%%edi)"
#define LIBECC_OP2_DESTINATION "(%%edi)"
#define LIBECC_WORKREG "%%eax"
#define LIBECC_CLOBBER "%eax"
#define LIBECC_INIT "std\n\t"
#define LIBECC_LOAD1 "lodsl\n\t"
#define LIBECC_SHIFT "shrl $1," LIBECC_WORKREG "\n\t"
#define LIBECC_STORE1 "stosl\n\t"
#define LIBECC_LOAD2 "lodsl\n\t"
#define LIBECC_ROTATE "rcrl $1," LIBECC_WORKREG "\n\t"
#define LIBECC_STORE2 "stosl\n\t"
#define LIBECC_DEINIT "cld\n\t"
#define LIBECC_RIGHT_PRESERVE_CF(OPERAND) "setc %%edx\n\t" OPERAND "\n\tbt $0,%%edx\n\t"
#define LIBECC_PRESERVE_CF_CLOBBER LIBECC_CLOBBER,"%edx"
#define LIBECC_INITIAL_ESI &this->vector[initial]
#define LIBECC_INITIAL_EDI &result.vector[initial]
#else
#define LIBECC_OP1_SOURCE "4(%%esi,%%ecx,4)"
#define LIBECC_OP2_SOURCE "(%%esi,%%ecx,4)"
#define LIBECC_OP1_DESTINATION "4(%%edi,%%ecx,4)"
#define LIBECC_OP2_DESTINATION "(%%edi,%%ecx,4)"
#define LIBECC_WORKREG "%%edx"
#define LIBECC_CLOBBER "%edx"
#define LIBECC_INIT
#define LIBECC_LOAD1 "movl " LIBECC_OP1_SOURCE "," LIBECC_WORKREG "\n\t"
#define LIBECC_SHIFT "shrl $1," LIBECC_WORKREG "\n\t"
#define LIBECC_STORE1 "movl " LIBECC_WORKREG "," LIBECC_OP1_DESTINATION "\n\t"
#define LIBECC_LOAD2 "movl " LIBECC_OP2_SOURCE "," LIBECC_WORKREG "\n\t"
#define LIBECC_ROTATE "rcrl $1," LIBECC_WORKREG "\n\t"
#define LIBECC_STORE2 "movl " LIBECC_WORKREG "," LIBECC_OP2_DESTINATION "\n\t"
#define LIBECC_DEINIT
#define LIBECC_RIGHT_PRESERVE_CF(OPERAND) "lahf\n\t" OPERAND "\n\tsahf\n\t"
#define LIBECC_PRESERVE_CF_CLOBBER LIBECC_CLOBBER,"%eax"
#define LIBECC_INITIAL_ESI &this->vector[initial - count]
#define LIBECC_INITIAL_EDI &result.vector[initial - count]
#endif

#define LIBECC_ASMSHIFTRIGHT1(OP1, OP2, CLOBBER)	\
	      __asm__ __volatile__ (			\
		  LIBECC_INIT				\
		  LIBECC_LOAD1				\
		  LIBECC_SHIFT				\
		  OP1					\
		  LIBECC_STORE1				\
		"1:\n\t"				\
		  LIBECC_LOAD2				\
		  LIBECC_ROTATE				\
		  OP2					\
		  LIBECC_STORE2				\
		  "decl %%ecx\n\t"			\
		  "jnz 1b\n\t"				\
		  LIBECC_DEINIT				\
		: "=&S" (ptr1),				\
		  "=&D" (ptr2),				\
		  "=&c" (c)				\
		: "0" (LIBECC_INITIAL_ESI),		\
		  "1" (LIBECC_INITIAL_EDI),		\
		  "2" (count - 1)			\
		: "memory", "cc", CLOBBER		\
	      )

#define LIBECC_ASMSHIFTLEFT(OP1, OP2, FINISH)		\
	      __asm__ __volatile__ (			\
		  "movl -4(%%esi),%%edx\n\t"		\
		  "shll $1,%%edx\n\t"			\
		  OP1					\
		  "movl %%edx,-4(%%edi)\n\t"		\
		  "jecxz 2f\n"				\
		"1:\n\t"				\
		  "movl (%%esi),%%edx\n\t"		\
		  "rcll $1,%%edx\n\t"			\
		  OP2					\
		  "movl %%edx,(%%edi)\n\t"		\
		  "leal 4(%%esi),%%esi\n\t"		\
		  "leal 4(%%edi),%%edi\n\t"		\
		  "decl %%ecx\n\t"			\
		  "jnz 1b\n"				\
		"2:\n\t"				\
		  FINISH				\
		: "=&S" (ptr1),				\
		  "=&D" (ptr2),				\
		  "=&c" (c)				\
		: "0" (&this->vector[initial + 1]),	\
		  "1" (&result.vector[initial + 1]),	\
		  "2" (count - 1),			\
		  "i" (valid_bits)			\
		: "%eax", "%edx", "memory", "cc"	\
	      )

#define LIBECC_ASMSHIFTLEFT_FINISH(OP)			\
		  "movl (%%esi),%%edx\n\t"		\
		  "rcll $1,%%edx\n\t"			\
		  "andl %6,%%edx\n\t"			\
		  OP					\
		  "movl %%edx,(%%edi)"

#define LIBECC_LEFT_PRESERVE_CF(OPERAND) "lahf\n\t" OPERAND "\n\tsahf\n\t"

#endif // __i386__

// Functors.

// Functor for '&'.
struct bitsetAND {
  static int const id = 1;
  static inline bitset_digit_t exec(bitset_digit_t digit1, bitset_digit_t digit2) { return digit1 & digit2; }
  template<bool inverted1, bool inverted2>
   struct execbool {
     static bool const value = inverted1 && inverted2;
   };
};

// Functor for '|'.
struct bitsetOR {
  static int const id = 2;
  static inline bitset_digit_t exec(bitset_digit_t digit1, bitset_digit_t digit2) { return digit1 | digit2; }
  template<bool inverted1, bool inverted2>
   struct execbool {
     static bool const value = inverted1 || inverted2;
   };
};

// Functor for '^'.
struct bitsetXOR {
  static int const id = 3;
  static inline bitset_digit_t exec(bitset_digit_t digit1, bitset_digit_t digit2) { return digit1 ^ digit2; }
  template<bool inverted1, bool inverted2>
   struct execbool {
     static bool const value = (inverted1 != inverted2);
   };
};

// Functor for '='.
struct bitsetAssign {
  static int const id = 1;
  static bool const sets_excess_bits = true;
  static bool const with_zero_sets_zero = true;
  static bool const with_ones_sets_ones = true;
  static bool const with_ones_inverts = false;
  static inline void exec(bitset_digit_t& out, bitset_digit_t in) { out = in; }
};

// Functor for '&='.
struct bitsetANDAssign {
  static int const id = 2;
  static bool const sets_excess_bits = false;
  static bool const with_zero_sets_zero = true;
  static bool const with_ones_sets_ones = false;
  static bool const with_ones_inverts = false;
  static inline void exec(bitset_digit_t& out, bitset_digit_t in) { out &= in; }
};

// Functor for '|='.
struct bitsetORAssign {
  static int const id = 3;
  static bool const sets_excess_bits = true;
  static bool const with_zero_sets_zero = false;
  static bool const with_ones_sets_ones = true;
  static bool const with_ones_inverts = false;
  static inline void exec(bitset_digit_t& out, bitset_digit_t in) { out |= in; }
};

// Functor for '^='.
struct bitsetXORAssign {
  static int const id = 4;
  static bool const sets_excess_bits = true;
  static bool const with_zero_sets_zero = false;
  static bool const with_ones_sets_ones = false;
  static bool const with_ones_inverts = true;
  static inline void exec(bitset_digit_t& out, bitset_digit_t in) { out ^= in; }
};

template<unsigned int n, bool inverted1, bool inverted2, typename OP>
  struct bitsetExpression
  {
    bitset_invertible<n, inverted1> const& first;
    bitset_invertible<n, inverted2> const& second;
    OP op;
    bitsetExpression(bitset_invertible<n, inverted1> const& arg1, bitset_invertible<n, inverted2> const& arg2) : first(arg1), second(arg2) { }
  };

} // namespace Operator
#endif // HIDE_FROM_DOXYGEN

/**\class bitset_invertible bitset.h libecc/bitset.h
 *
 * \brief A bitset with a fixed number of bits, having possibly an infinite number of leading virtual 1's.
 *
 * When dealing with expressions, short bitsets are assumed to have leading zeros.
 * This class however can represent an 'inverted' bitset, meaning that it has an
 * infinite number of leading ones.
 *
 * The use of this class should be restricted to (optimized out) temporaries in expressions
 * like <code>a = ~b;</code> or <code>a |= b & ~c</code>.  The user should
 * not use this class directly.
 *
 * \see libecc::operator~(bitset_invertible<n, inverted> const& bits)
 *
 * \param n The number of bits in the bitset.
 * \param inverted True when each internal bit has the opposite meaning.
 */
template<unsigned int n, bool inverted>
  class bitset_invertible : public bitset_base<n> {
    public:
      // Constructors
      bitset_invertible(void);
      template<typename Expression>
	explicit bitset_invertible<n, inverted>(Expression const&);

    protected:
      template<typename OP, unsigned int x, bool invertedx>
	void assign(bitset_invertible<x, invertedx> const&, OP);
      template<typename OP1, unsigned int x, bool inverted1, bool inverted2, typename OP2>
	void assign(Operator::bitsetExpression<x, inverted1, inverted2, OP2> const&, OP1);

    private:
      template<unsigned int m, bool i>
	friend class bitset_invertible;
      template<unsigned int m, int DIRECTION>
	friend class bitset_iterator;

    public:
      // Printing
      void base2_print_on(std::ostream& os) const;

      // Digit access.
      bitset_digit_t digit(unsigned int d) const { assert(!inverted); return inverted ? ~this->vector[d] : this->vector[d]; }
  };

/** \brief Print bitset base 2.
 *
 * For debugging purposes.
 *
 * For example (assuming you are using libcwd):
 *
 * \code
 *   bitset<n> btst;
 *   Dout(dc::notice, cwprint_using(btst, &bitset::base2_print_on));
 * \endcode
 */
template<unsigned int n, bool inverted>
  void
  bitset_invertible<n, inverted>::base2_print_on(std::ostream& os) const
  {
    // Binary representation
    for (int d = bitset_base<n>::digits - 1; d >= 0; --d)
      for (bitset_digit_t mask = (~static_cast<bitset_digit_t>(0) >> 1) + 1; mask != 0; mask >>= 1)
	if (d != bitset_base<n>::digits - 1 || (mask & bitset_base<n>::valid_bits))
	  if ((this->digit(d) & mask) == inverted)
	    os << '0';
	  else
	    os << '1';
  }

// Default constructor.
template<unsigned int n, bool inverted>
  inline
  bitset_invertible<n, inverted>::bitset_invertible(void)
  {
    if (has_excess_bits)
      this->vector[digits - 1] = 0; 			// Reset the excess bits!
  }

// Copy constructor.
template<unsigned int n, bool inverted>
  template<typename Expression>
    inline
    bitset_invertible<n, inverted>::bitset_invertible(Expression const& expr)
    {
      this->assign(expr, Operator::bitsetAssign());
    }

//
// Assignment function.
// This function handles:
//
// a = b;
// a = ~b;
// a &= b;
// a &= ~b;
// a |= b;
// a |= ~b;
// a ^= b;
// a ^= ~b;
//
// ASSIGNMENT_OPERATOR is one of bitsetAssign, bitsetANDAssign, bitsetORAssign or bitsetXORAssign.
//
template<unsigned int n, bool inverted>
  template<typename ASSIGNMENT_OPERATOR, unsigned int x, bool inverted_argument>
    inline void
    bitset_invertible<n, inverted>::assign(bitset_invertible<x, inverted_argument> const& bits, ASSIGNMENT_OPERATOR)
    {
      // Handle excess digits.
      if (digits > bitset_base<x>::digits)
      {
	if (!inverted_argument)
	{
	  // Fill excess digits with 0's when needed.
	  if (ASSIGNMENT_OPERATOR::with_zero_sets_zero)
	  {
#ifdef __i386__
	    int __d0, __d1;
	    __asm__ __volatile__ (
		"std; rep; stosl"
		: "=m" (this->vector[digits - 1]), "=&c" (__d0), "=&D" (__d1)
		: "a" (0), "1" (digits - bitset_base<x>::digits), "2" (&this->vector[digits - 1])
		: "memory"
	    );
#else
	    unsigned int count = digits - bitset_base<x>::digits;
	    do { this->vector[count + bitset_base<x>::digits - 1] = 0; } while (--count);
#endif
	  }
	}
	else
	{
	  // Fill excess digits with 1's when needed.
	  if (ASSIGNMENT_OPERATOR::with_ones_sets_ones)
	  {
#ifdef __i386__
	    int __d0, __d1;
	    __asm__ __volatile__ (
		"movl %6,4(%%edi); std; rep; stosl"
		: "=m" (this->vector[digits - 2]), "=&c" (__d0), "=&D" (__d1)
		: "a" (-1), "1" (digits - bitset_base<x>::digits - 1), "2" (&this->vector[digits - 2]), "i" (valid_bits)
		: "memory"
	    );
#else
	    this->vector[digits - 1] = valid_bits;
	    unsigned int count = digits - bitset_base<x>::digits;
	    while(--count) { this->vector[count + bitset_base<x>::digits - 1] = ~static_cast<bitset_digit_t>(0); }
#endif
	  }
	  // Or invert them.
	  if (ASSIGNMENT_OPERATOR::with_ones_inverts)
	  {
#ifdef __i386__
	    int __d0, __d1;
	    __asm__ __volatile__ (
		"xorl %5,(%2)\n\t"
		"jecxz 2f\n\t"
		"subl %6,%2\n"
	"1:\n\t"
		"xorl $-1,(%2,%%ecx,4)\n\t"
		"decl %%ecx\n\t"
		"jnz 1b\n"
	"2:"
		: "=m" (this->vector[digits - 1]), "=&c" (__d0), "=&r" (__d1)
		: "1" (digits - bitset_base<x>::digits - 1), "2" (&this->vector[digits - 1]),
		  "i" (valid_bits), "i" ((digits - bitset_base<x>::digits) * 4)
		: "memory", "cc"
	    );
#else
	    this->vector[digits - 1] ^= valid_bits;
	    unsigned int count = digits - bitset_base<x>::digits;
	    while(--count) { this->vector[count + bitset_base<x>::digits - 1] ^= ~static_cast<bitset_digit_t>(0); }
#endif
	  }
	}
      }

      // Handle other digits.
#ifdef __i386__
      LIBECC_INLINE_ASMLOOP(ASSIGNMENT_OPERATOR::id,
          this->vector, bits.digits_ptr(),
	  ((digits > bitset_base<x>::digits) ? bitset_base<x>::digits : digits),
	  inverted_argument);
#else
      unsigned int d;
      if (digits > bitset_base<x>::digits)
	d = bitset_base<x>::digits - 1;
      else
	d = digits - 1;
      ASSIGNMENT_OPERATOR::exec(this->vector[d], inverted_argument ? ~*(bits.digits_ptr() + d) : bits.digit(d));
      while(d) { --d; ASSIGNMENT_OPERATOR::exec(this->vector[d], inverted_argument ? ~*(bits.digits_ptr() + d) : bits.digit(d)); }
#endif

      // Reset excess bits if needed.
      if (((!inverted_argument && x > n) ||
	    (inverted_argument && digits <= bitset_base<x>::digits)) &&
	  has_excess_bits && ASSIGNMENT_OPERATOR::sets_excess_bits)
        this->vector[digits - 1] &= valid_bits;
    }

//
// Assignment function for expressions.
// This function handles:
//
// a = b & c;
// a = b & ~c;
// a = ~b & c;
// a = ~b & ~b;
// a = b | c;
// a = b | ~c;
// a = ~b | c;
// a = ~b | ~b;
// a = b ^ c;
// a = b ^ ~c;
// a = ~b ^ c;
// a = ~b ^ ~b;
// a &= b & c;
// a &= b & ~c;
// a &= ~b & c;
// a &= ~b & ~b;
// a &= b | c;
// a &= b | ~c;
// a &= ~b | c;
// a &= ~b | ~b;
// a &= b ^ c;
// a &= b ^ ~c;
// a &= ~b ^ c;
// a &= ~b ^ ~b;
// a |= b & c;
// a |= b & ~c;
// a |= ~b & c;
// a |= ~b & ~b;
// a |= b | c;
// a |= b | ~c;
// a |= ~b | c;
// a |= ~b | ~b;
// a |= b ^ c;
// a |= b ^ ~c;
// a |= ~b ^ c;
// a |= ~b ^ ~b;
// a ^= b & c;
// a ^= b & ~c;
// a ^= ~b & c;
// a ^= ~b & ~b;
// a ^= b | c;
// a ^= b | ~c;
// a ^= ~b | c;
// a ^= ~b | ~b;
// a ^= b ^ c;
// a ^= b ^ ~c;
// a ^= ~b ^ c;
// a ^= ~b ^ ~b;
//
// ASSIGNMENT_OPERATOR is one of bitsetAssign, bitsetANDAssign, bitsetORAssign or bitsetXORAssign.
// OPERATOR is one of bitsetAND, bitsetOR or bitsetXOR.
//
template<unsigned int n, bool inverted>
  template<typename ASSIGNMENT_OPERATOR, unsigned int x, bool inverted1, bool inverted2, typename OPERATOR>
    inline void
    bitset_invertible<n, inverted>::assign(Operator::bitsetExpression<x, inverted1, inverted2, OPERATOR> const& expr, ASSIGNMENT_OPERATOR)
    {
      static bool const argument_has_leading_ones = OPERATOR::template execbool<inverted1, inverted2>::value;

      // Handle excess digits.
      if (digits > bitset_base<x>::digits)
      {
	if (!argument_has_leading_ones)
	{
	  // Fill excess digits with 0's when needed.
	  if (ASSIGNMENT_OPERATOR::with_zero_sets_zero)
	  {
#ifdef __i386__
	    int __d0, __d1;
	    __asm__ __volatile__ (
		"std; rep; stosl"
		: "=m" (this->vector[digits - 1]), "=&c" (__d0), "=&D" (__d1)
		: "a" (0), "1" (digits - bitset_base<x>::digits), "2" (&this->vector[digits - 1])
		: "memory"
	    );
#else
	    unsigned int count = digits - bitset_base<x>::digits;
	    do { this->vector[count + bitset_base<x>::digits - 1] = 0; } while (--count);
#endif
	  }
	}
	else
	{
	  // Fill excess digits with 1's when needed.
	  if (ASSIGNMENT_OPERATOR::with_ones_sets_ones)
	  {
#ifdef __i386__
	    int __d0, __d1;
	    __asm__ __volatile__ (
		"movl %6,4(%%edi); std; rep; stosl"
		: "=m" (this->vector[digits - 2]), "=&c" (__d0), "=&D" (__d1)
		: "a" (-1), "1" (digits - bitset_base<x>::digits - 1), "2" (&this->vector[digits - 2]), "i" (valid_bits)
		: "memory"
	    );
#else
	    this->vector[digits - 1] = valid_bits;
	    unsigned int count = digits - bitset_base<x>::digits;
	    while(--count) { this->vector[count + bitset_base<x>::digits - 1] = ~static_cast<bitset_digit_t>(0); }
#endif
	  }
	  // Or invert them.
	  if (ASSIGNMENT_OPERATOR::with_ones_inverts)
	  {
#ifdef __i386__
	    int __d0, __d1;
	    __asm__ __volatile__ (
		"xorl %5,(%2)\n\t"
		"jecxz 2f\n\t"
		"subl %6,%2\n"
	"1:\n\t"
		"xorl $-1,(%2,%%ecx,4)\n\t"
		"decl %%ecx\n\t"
		"jnz 1b\n"
	"2:"
		: "=m" (this->vector[digits - 1]), "=&c" (__d0), "=&r" (__d1)
		: "1" (digits - bitset_base<x>::digits - 1), "2" (&this->vector[digits - 1]),
		  "i" (valid_bits), "i" ((digits - bitset_base<x>::digits) * 4)
		: "memory", "cc"
	    );
#else
	    this->vector[digits - 1] ^= valid_bits;
	    unsigned int count = digits - bitset_base<x>::digits;
	    while(--count) { this->vector[count + bitset_base<x>::digits - 1] ^= ~static_cast<bitset_digit_t>(0); }
#endif
	  }
	}
      }

      // Handle other digits.
#ifdef __i386__
      LIBECC_INLINE_ASMLOOP2(ASSIGNMENT_OPERATOR::id, OPERATOR, inverted1, inverted2,
	  this->vector, expr.first.digits_ptr(), expr.second.digits_ptr(),
	  ((digits > bitset_base<x>::digits) ? bitset_base<x>::digits : digits));
#else
      unsigned int d;
      if (digits < bitset_base<x>::digits)
	d = digits - 1;
      else
	d = bitset_base<x>::digits - 1;
      for(;;)
      {
	ASSIGNMENT_OPERATOR::exec(this->vector[d],
	    OPERATOR::exec(inverted1 ? ~*(expr.first.digits_ptr() + d) : expr.first.digit(d),
			   inverted2 ? ~*(expr.second.digits_ptr() + d) : expr.second.digit(d)));
	if (!d)
	  break;
	--d;
      }
#endif

      // Reset excess bits if needed.
      if (((!argument_has_leading_ones && x > n) ||
	    (argument_has_leading_ones && digits <= bitset_base<x>::digits)) &&
	  has_excess_bits && ASSIGNMENT_OPERATOR::sets_excess_bits)
        this->vector[digits - 1] &= valid_bits;
    }

/** \brief Invert a bitset.
 *
 * This operator is just a cast and doesn't actually do anything
 * until you assign the expression that it is part of to another bitset.
 *
 * For example:
 *
 * \code
 * x = ~y;
 * x ^= ~y & ~z;
 * \endcode
 *
 * Both cases will run over the digits of bitset x just once
 * and apply the given formula (and any like it) directly on
 * the digits of the bitsets involved.
 *
 * Inversion can also be used together with the equivalence
 * operators.  For example:
 *
 * \code
 * if (x == ~y) ...
 * \endcode
 *
 * will only run once over the digits of both bitset and compare
 * the digits once by one using <code>x.digit(d) == ~y.digit(d)</code>.
 */
template<unsigned int n, bool inverted>
  inline bitset_invertible<n, !inverted> const&
  operator~(bitset_invertible<n, inverted> const& bits)
  {
    return *reinterpret_cast<bitset_invertible<n, !inverted> const*>(&bits);
  }

//------------------------------------------------------------------------------------------------------------------
// Iterators
//

/**\class bitset_index bitset.h libecc/bitset.h
 *
 * \brief A bitset index.
 *
 * This class represents a bit index, the distance between the bit refered to and the least significant bit.
 * It does not specify which bitset instance or even the size of the bitset.&nbsp;
 * The index can contain positive and negative values, an offset of -1 represents the singular 'rend()' const_reverse_iterator.&nbsp;
 * Using an index that is too large or too small for the bitset that it is being used with will leads to undefined behaviour.
 */
class bitset_index {
#ifdef __i386__
  protected:
    int M_index;			//!< The bit number that this index points to.

  public:
    // Accessors.
    int get_index(void) const;

#else
  protected:
    int M_digit;			//!< The digit that this index points to.
    bitset_digit_t M_mask;		//!< The digit bitmask of the bit that this index points to.

  public:
    // Accessors.
    int get_digit(void) const;
    bitset_digit_t get_mask(void) const;
#endif

  public:
    // Automatic conversion.
    operator unsigned int () const;

  public:
    // Equality Comparable.
    friend bool operator==(bitset_index const& i1, bitset_index const& i2);
    friend bool operator!=(bitset_index const& i1, bitset_index const& i2);

  protected:
    // Bidirectional.
    void left(void);
    void right(void);

    // Random access.
    void left(int n);
    void right(int n);
    friend int subtract(bitset_index const& i1, bitset_index const& i2);

  protected:
    // Constructors.
    bitset_index(void);
    bitset_index(bitset_index const& index);
    bitset_index(int bit);

  public:
    friend std::ostream& operator<<(std::ostream& os, bitset_index const& i);
};

#ifdef __i386__
/**
 * Accessor for the current bit index.
 */
inline int
bitset_index::get_index(void) const
{
  return M_index;
}

/**
 * Automatic conversion.  Only valid when index points to a real bit (has a positive value).
 */
inline
bitset_index::operator unsigned int () const
{
  return M_index;
}

#else

/**
 * Accessor for the current digit index.
 */
inline int
bitset_index::get_digit(void) const
{
  return M_digit;
}

/**
 * Accessor for the current digit mask.
 */
inline bitset_digit_t
bitset_index::get_mask(void) const
{ 
  return M_mask;
}

#ifndef HIDE_FROM_DOXYGEN
#include <strings.h>	// Needed for ffs.
inline
bitset_index::operator unsigned int () const
{
  return M_digit * bitset_digit_bits + ::ffs(M_mask) - 1;
}
#endif

#endif

/**
 * \brief Equality operator.
 */
inline bool
operator==(bitset_index const& i1, bitset_index const& i2)
{
#ifdef __i386__
  return (i1.M_index == i2.M_index);
#else
  return (i1.M_digit == i2.M_digit && i1.M_mask == i2.M_mask);
#endif
}

/**
 * \brief Inequality operator.
 */
inline bool
operator!=(bitset_index const& i1, bitset_index const& i2)
{
#ifdef __i386__
  return (i1.M_index != i2.M_index);
#else
  return (i1.M_digit != i2.M_digit || i1.M_mask != i2.M_mask);
#endif
}

/**
 * \brief Construct an uninitialized bitset index.
 */
inline
bitset_index::bitset_index(void)
{
}

/**
 * \brief Copy constructor.
 */
inline
bitset_index::bitset_index(bitset_index const& index) :
#ifdef __i386__
    M_index(index.M_index)
#else
    M_digit(index.M_digit), M_mask(index.M_mask)
#endif
{ 
}

/**
 * \brief Construct a bitset index that points to bit number \a bit.
 *
 * \a bit may be -1 (one before the start) which is equivalent with
 * \link bitset::rend rend() \endlink or --\link bitset::begin begin() \endlink.
 */
inline
bitset_index::bitset_index(int bit) :
#ifdef __i386__
    M_index(bit)
#else
    // If bit == -1 then M_digit should become -1 and M_mask 0x80000000.
    M_digit(bit >> bitset_digit_bits_log2),
    M_mask(static_cast<bitset_digit_t>(1) << ((unsigned int)bit & (bitset_digit_bits - 1)))
#endif
{
}

/**
 * \brief Move one bit left in the bitset.
 */
inline void
bitset_index::left(void)
{
#ifdef __i386__
  ++M_index;
#else
  if ((M_mask <<= 1) == 0)
  {
    ++M_digit;
    M_mask = 1;
  }
#endif
}

/**
 * \brief Move one bit right in the bitset.
 */
inline void
bitset_index::right(void)
{
#ifdef __i386__
  --M_index;
#else
  if ((M_mask >>= 1) == 0)
  {
    --M_digit;
    M_mask = static_cast<bitset_digit_t>(1) << (bitset_digit_bits - 1);
  }
#endif
}

/**
 * \brief Move \a n bits left in the bitset.
 */
inline void
bitset_index::left(int n)
{
#ifdef __i386__
  M_index += n;
#else
  int const digit_shift = n >> bitset_digit_bits_log2;
  int const mask_shift = n & (bitset_digit_bits - 1);
  M_digit += digit_shift;
  if (mask_shift)
  {
    bitset_digit_t new_mask = M_mask << mask_shift;
    if (new_mask == 0)
    {
      ++M_digit;
      new_mask = M_mask >> (bitset_digit_bits - mask_shift);
    }
    M_mask = new_mask;
  }
#endif
}

/**
 * \brief Move \a n bits right in the bitset.
 */
inline void
bitset_index::right(int n)
{
#ifdef __i386__
  M_index -= n;
#else
  int const digit_shift = n >> bitset_digit_bits_log2;
  int const mask_shift = n & (bitset_digit_bits - 1);
  M_digit -= digit_shift;
  if (mask_shift)
  {
    bitset_digit_t new_mask = M_mask >> mask_shift;
    if (new_mask == 0)
    {
      --M_digit;
      new_mask = M_mask << (bitset_digit_bits - mask_shift);
    }
    M_mask = new_mask;
  }
#endif
}

enum {
  forwards_iterating,
  backwards_iterating
};

/**\class bitset_index_iterator bitset.h libecc/bitset.h
 *
 * \brief A bitset index with increment/decrement methods.
 *
 * This class is a direction oriented \ref bitset_index with methods to advance and backup the index in a bitset.
 *
 * \param DIRECTION Either libecc::forwards_iterating (normal iterator) or libecc::backwards_iterating (reverse iterator).
 */
template<int DIRECTION>
  class bitset_index_iterator : public bitset_index {
    public:
      // LessThan Comparable.
      friend bool operator< <>(bitset_index_iterator const& i1, bitset_index_iterator const& i2);
      friend bool operator> <>(bitset_index_iterator const& i1, bitset_index_iterator const& i2);
      friend bool operator<= <>(bitset_index_iterator const& i1, bitset_index_iterator const& i2);
      friend bool operator>= <>(bitset_index_iterator const& i1, bitset_index_iterator const& i2);

    public:
      // Bidirectional.
      void increment(void);
      void decrement(void);

      // Random access.
      void increment(int n);
      void decrement(int n);

      friend int operator-(bitset_index const& i1, bitset_index const& i2);

    public:
      // Constructors.
      bitset_index_iterator(void);
      bitset_index_iterator(bitset_index_iterator const& index);
      bitset_index_iterator(int bit);
  };

/**
 * \brief Construct an uninitialized iterator bitmask.
 */
template<int DIRECTION>
  inline
  bitset_index_iterator<DIRECTION>::bitset_index_iterator(void)
  {
  }

/**
 * \brief Copy constructor.
 */
template<int DIRECTION>
  inline
  bitset_index_iterator<DIRECTION>::bitset_index_iterator(bitset_index_iterator<DIRECTION> const& index) :
      bitset_index(index)
  { 
  }

/**
 * \brief Construct an iterator bitmask that points to bit number \a bit.
 *
 * \a bit may be -1 (one before the start) which is equivalent with
 * \link bitset::rend rend() \endlink or --\link bitset::begin begin() \endlink.
 */
template<int DIRECTION>
  inline
  bitset_index_iterator<DIRECTION>::bitset_index_iterator(int bit) : bitset_index(bit)
  {
  }

/**
 * \brief Less.
 */
template<int DIRECTION>
  inline bool
  operator<(bitset_index_iterator<DIRECTION> const& i1, bitset_index_iterator<DIRECTION> const& i2)
  {
#ifdef __i386__
    if (DIRECTION == forwards_iterating)
      return (i1.M_index < i2.M_index);
    else
      return (i1.M_index > i2.M_index);
#else
    if (DIRECTION == forwards_iterating)
      return (i1.M_digit < i2.M_digit || (i1.M_digit == i2.M_digit && i1.M_mask < i2.M_mask));
    else
      return (i1.M_digit > i2.M_digit || (i1.M_digit == i2.M_digit && i1.M_mask > i2.M_mask));
#endif
  }

/**
 * \brief Greater.
 */
template<int DIRECTION>
  inline bool
  operator>(bitset_index_iterator<DIRECTION> const& i1, bitset_index_iterator<DIRECTION> const& i2)
  {
#ifdef __i386__
    if (DIRECTION == forwards_iterating)
      return (i1.M_index > i2.M_index);
    else
      return (i1.M_index < i2.M_index);
#else
    if (DIRECTION == forwards_iterating)
      return (i1.M_digit > i2.M_digit || (i1.M_digit == i2.M_digit && i1.M_mask > i2.M_mask));
    else
      return (i1.M_digit < i2.M_digit || (i1.M_digit == i2.M_digit && i1.M_mask < i2.M_mask));
#endif
  }

/**
 * \brief Less or equal.
 */
template<int DIRECTION>
  inline bool
  operator<=(bitset_index_iterator<DIRECTION> const& i1, bitset_index_iterator<DIRECTION> const& i2)
  {
#ifdef __i386__
    if (DIRECTION == forwards_iterating)
      return (i1.M_index <= i2.M_index);
    else
      return (i1.M_index >= i2.M_index);
#else
    if (DIRECTION == forwards_iterating)
      return (i1.M_digit <= i2.M_digit || (i1.M_digit == i2.M_digit && i1.M_mask <= i2.M_mask));
    else
      return (i1.M_digit >= i2.M_digit || (i1.M_digit == i2.M_digit && i1.M_mask >= i2.M_mask));
#endif
  }

/**
 * \brief Greater or equal.
 */
template<int DIRECTION>
  inline bool
  operator>=(bitset_index_iterator<DIRECTION> const& i1, bitset_index_iterator<DIRECTION> const& i2)
  {
#ifdef __i386__
    if (DIRECTION == forwards_iterating)
      return (i1.M_index >= i2.M_index);
    else
      return (i1.M_index <= i2.M_index);
#else
    if (DIRECTION == forwards_iterating)
      return (i1.M_digit >= i2.M_digit || (i1.M_digit == i2.M_digit && i1.M_mask >= i2.M_mask));
    else
      return (i1.M_digit <= i2.M_digit || (i1.M_digit == i2.M_digit && i1.M_mask <= i2.M_mask));
#endif
  }

/**
 * \brief Advance to the next bit.
 */
template<int DIRECTION>
  inline void
  bitset_index_iterator<DIRECTION>::increment(void)
  {
    if (DIRECTION == forwards_iterating)
      left();
    else
      right();
  }

/**
 * \brief Backup to the previous bit.
 */
template<int DIRECTION>
  inline void
  bitset_index_iterator<DIRECTION>::decrement(void)
  {
    if (DIRECTION == forwards_iterating)
      right();
    else
      left();
  }

/**
 * \brief Advance to the next bit.
 */
template<int DIRECTION>
  inline void
  bitset_index_iterator<DIRECTION>::increment(int n)
  {
    if (DIRECTION == forwards_iterating)
      left(n);
    else
      right(n);
  }

/**
 * \brief Backup to the previous bit.
 */
template<int DIRECTION>
  inline void
  bitset_index_iterator<DIRECTION>::decrement(int n)
  {
    if (DIRECTION == forwards_iterating)
      right(n);
    else
      left(n);
  }

template<int DIRECTION>
  inline int
  operator-(bitset_index_iterator<DIRECTION> const& i1, bitset_index_iterator<DIRECTION> const& i2)
  {
    return (DIRECTION == forwards_iterating) ? subtract(i1, i2) : subtract(i2, i1);
  }

/**\class bitset_iterator bitset.h libecc/bitset.h
 *
 * \brief A non-mutable bitset-iterator.
 *
 * This class represents a 'pointer' to a bit in a given bitset instance.
 * However, <code>operator*</code> does not return <code>bool</code> but returns a \ref bitset_digit_t
 * containing zero when the bit is zero and a mask matching the position of the bit in the respective
 * digit when the bit is set.
 */
template<unsigned int n, int DIRECTION>
  class bitset_iterator : public bitset_index_iterator<DIRECTION>,
                          public std::iterator<std::random_access_iterator_tag, bitset_digit_t,
			                       int, bitset_digit_t*, bitset_digit_t&> {
    protected:
      bitset_invertible<n, false> const* M_bitset_ptr;
    public:
      // Default Constructible
      bitset_iterator(void);

      // Assignable
      bitset_iterator(bitset_iterator const& iter);
      bitset_iterator& operator=(bitset_iterator const& iter);
      bitset_iterator& operator=(bitset_index_iterator<DIRECTION> const& index);

      // Dereferencable
      bitset_digit_t operator*() const;

      // Bi-directional iterator
      //
      bitset_iterator& operator++();
      bitset_iterator operator++(int);
      bitset_iterator& operator--();
      bitset_iterator operator--(int);

      // Random Access Iterator
      //
      bitset_iterator& operator+=(int n);
      friend bitset_iterator operator+ <>(bitset_iterator const& i, int n);
      friend bitset_iterator operator+ <>(int n, bitset_iterator const& i);
      bitset_iterator& operator-=(int n);
      friend bitset_iterator operator- <>(bitset_iterator const& i, int n);
      friend bitset_iterator operator- <>(int n, bitset_iterator const& i);
      bitset_digit_t operator[](int n) const;

      // Special
      //
      bitset_iterator(bitset_invertible<n, false> const* bitset_ptr, int bit);
      void find1(void);

#ifndef __i386__
    private:
      void find1_forward(void);
      void find1_backward(void);
#endif
  };

#ifndef __i386__
template<unsigned int n, int DIRECTION>
  void
  bitset_iterator<n, DIRECTION>::find1_forward(void)
  {
    if (M_digit < (int)bitset_base<n>::digits - 1 || (M_mask & bitset_base<n>::valid_bits))
    {
      register bitset_digit_t mask = M_mask;
      while(!(M_bitset_ptr->digit(M_digit) & mask)) 
      {
	if ((mask <<= 1))
	  continue;
	mask = 1;
	while(++M_digit < (int)bitset_base<n>::digits)
	  if (M_bitset_ptr->digit(M_digit))
	    break;
	if (M_digit == (int)bitset_base<n>::digits)
	{
	  M_digit = (n >> bitset_digit_bits_log2);
	  mask = static_cast<bitset_digit_t>(1) << (n & (bitset_digit_bits - 1));
	  break;
	}
      }
      M_mask = mask;
    }
  }

template<unsigned int n, int DIRECTION>
  void
  bitset_iterator<n, DIRECTION>::find1_backward(void)
  {
    LibEccDout(dc::bitsetfind1, "Entering find1_backward with: " << 
	libcw::debug::cwprint_using(*M_bitset_ptr, &bitset_invertible<n, false>::base2_print_on));
    LibEccDout(dc::bitsetfind1, "Input: " << *this);
    if (M_digit >= 0)
    {
      register bitset_digit_t mask = M_mask;
      if (!(M_bitset_ptr->digit(M_digit) & (mask | (mask - 1))))
      {
	mask = 1 << (bitset_digit_bits - 1);
	do
	{
	  if (--M_digit < 0)
	  {
	    M_mask = mask;
	    return;
	  }
	}
	while (!M_bitset_ptr->digit(M_digit));
      }
      while(!(M_bitset_ptr->digit(M_digit) & mask)) 
	mask >>= 1;
      M_mask = mask;
    }
    LibEccDout(dc::bitsetfind1|flush_cf, "Output: " << *this);
  }
#endif

/**
 * \brief Return the first position for which the bit is set.
 */
template<unsigned int n, int DIRECTION>
  inline void
  bitset_iterator<n, DIRECTION>::find1(void)
  {
    LibEccDout(dc::bitsetfind1, "Input: " << *this);
    LibEccDout(dc::bitsetfind1, "Entering find1 with: " << 
	libcw::debug::cwprint_using(*M_bitset_ptr, &bitset_invertible<n, false>::base2_print_on));
#ifdef __i386__
    int bit_index, digit, ptr;
    if (DIRECTION == backwards_iterating)
    {
      __asm__ __volatile__ (
	  // %eax: M_index (input) and work variable.
	  // %ecx: bit_index (work variable).
	  // %edi: &M_bitset_ptr->digit(M_index/32 - 1) (input) and ptr (work variable).
	  // %2  : digit (work variable).

	  // Make a copy of M_index into %ecx, last time we used M_index.
	  "movl %%eax,%%ecx\n\t"
	  // Set %eax to its correct value by deviding it by 32.
	  "sarl $5,%%eax\n\t"
	  // Set bit_index to its correct value by taking the modulo 32 of it.
	  "andl $31,%%ecx\n\t"
	  // If M_index is -1, do nothing.
          "js 1f\n\t"
	  // Copy the most significant digit, the digit with the
	  // bit at which we will start the search, into %2.
	  // digit = M_bitset_ptr->digit(%eax)
	  "movl 4(%%edi),%2\n\t"
	  // Shift this digit left until the first bit comes at position 31.
	  // bit_index = 31 - bit_index
	  "xorl $31,%%ecx\n\t"
	  // digit <<= bit_index
	  "shll %%cl,%2\n\t"
	  // See http://fatphil.org/x86/pentopt/19.html, chapter 19.3 for why we need the orl.
	  // This is not needed for the athlon for that reason, but we also need this orl
	  // for the case that %cl equals zero in which case the ZF is cleared!
	  "orl %2,%2\n\t"			
	  // If there is no bit set in the current digit, goto digit_search.
	  "jz 5f\n\t"
	  // Search for the (next) most significant bit set.
	  // This is slow on i[345]86: 11 + 2*n cycles; only generates 2 uops on a pentium though.
	  "bsrl %2,%2\n\t"
	  // Correct M_index to point to this bit.
	  // %eax <<= 5
	  "sall $5,%%eax\n\t"
	  // %2 -= bit_index
	  "subl %%ecx,%2\n\t"
	  // M_index = %eax + %2
	  "addl %2,%%eax\n\t"
	  // Done.
	  // goto exit
	  "jmp 1f\n\t"
	  ".align 16\n"
       "7:\n\t"				// Main loop.
       	  "movl (%%edi),%2\n\t"
	  "subl $4,%%edi\n\t"
	  "testl %2,%2\n\t"
	  "jnz 6f\n\t"
       "5:\n\t"				// digit_search:
	  "decl %%eax\n\t"
	  // Repeat main loop.
	  "jns 7b\n"	
       "4:\n\t"				// reached_end:
          // No set bit found at all.  Set M_index to -1.
	  "movl $-1,%%eax\n\t"
	  // Done.
	  // goto exit
	  "jmp 1f\n"
          // Search for the most significant bit set in this digit.
       "6:\n\t"
          "bsrl %2,%2\n\t"
	  // Correct M_index to point to this bit.
	  "sall $5,%%eax\n\t"
	  "addl %2,%%eax\n"
       "1:"				// exit:

	  : "=a" (M_index), "=&c" (bit_index), "=r" (digit), "=&D" (ptr)
	  : "0" (M_index), "3" (M_bitset_ptr->digits_ptr() - 1 + (M_index >> 5))
	  : "cc"
      );
    }
    else
    {
      __asm__ __volatile__ (
	  // %eax: M_index (input) and work variable.
	  // %ecx: bit_index (work variable).
	  // %edi: &M_bitset_ptr->digit(M_index/32 + 1) (input) and ptr (work variable).
	  // %2  : digit (work variable).

	  // if (M_index >= n) then goto exit
	  "cmpl %6,%%eax\n\t"
          "jge 1f\n\t"
	  // Make a copy of M_index into %ecx, last time we used M_index.
	  "movl %%eax,%%ecx\n\t"
	  // Set %eax to its correct value by deviding it by 32.
	  "sarl $5,%%eax\n\t"
	  // Set bit_index to its correct value by taking the modulo 32 of it.
	  "andl $31,%%ecx\n\t"
	  // Copy the least significant digit, the digit with the
	  // bit at which we will start the search, into %2.
	  // digit = M_bitset_ptr->digit(%eax)
	  "movl -4(%%edi),%2\n\t"
	  // Shift this digit right until the first bit comes at position 0.
	  // digit >>= bit_index
	  "shrl %%cl,%2\n\t"
	  // See http://fatphil.org/x86/pentopt/19.html, chapter 19.3 for why we need the orl.
	  // This is not needed for the athlon for that reason, but we also need this orl
	  // for the case that %cl equals zero in which case the ZF is cleared!
	  "orl %2,%2\n\t"			
	  // If there is no bit set in the current digit, goto digit_search.
	  "jz 5f\n\t"
	  // Search for the (next) most significant bit set.
	  // This is slow on i[345]86: 11 + 2*n cycles; only generates 2 uops on a pentium though.
	  "bsfl %2,%2\n\t"
	  // Correct M_index to point to this bit.
	  // %eax <<= 5
	  "sall $5,%%eax\n\t"
	  // %2 -= bit_index
	  "addl %%ecx,%2\n\t"
	  // M_index = %eax + %2
	  "addl %2,%%eax\n\t"
	  // Done.
	  // goto exit
	  "jmp 1f\n\t"
	  ".align 16\n"
       "7:\n\t"				// Main loop.
       	  "movl (%%edi),%2\n\t"
	  "addl $4,%%edi\n\t"
	  "testl %2,%2\n\t"
	  "jnz 6f\n\t"
       "5:\n\t"				// digit_search:
	  "incl %%eax\n\t"
	  "cmpl %7, %%eax\n\t"
	  // Repeat main loop.
	  "jnz 7b\n"		
       "4:\n\t"				// reached_end:
          // No set bit found at all.  Set M_index to n.
	  "movl %6,%%eax\n\t"
	  // Done.
	  // goto exit
	  "jmp 1f\n"
          // Search for the least significant bit set in this digit.
       "6:\n\t"
          "bsfl %2,%2\n\t"
	  // Correct M_index to point to this bit.
	  "sall $5,%%eax\n\t"
	  "addl %2,%%eax\n"
       "1:"				// exit:

	  : "=a" (M_index), "=&c" (bit_index), "=r" (digit), "=&D" (ptr)
	  : "0" (M_index), "3" (M_bitset_ptr->digits_ptr() + 1 + (M_index >> 5)), "i" (n), "i" (bitset_base<n>::digits)
	  : "cc"
      );
    }
#else
    if (DIRECTION == forwards_iterating)
      find1_forward();
    else
      find1_backward();
#endif
    LibEccDout(dc::bitsetfind1|flush_cf, "Output: " << *this);
    return;
  }


/**
 * \brief Construct uninitialized bitset iterator.
 */
template<unsigned int n, int DIRECTION>
  inline
  bitset_iterator<n, DIRECTION>::bitset_iterator(void)
  {
  }

/**
 * \brief Construct a bitset iterator that points to bit \a bit of bitset \a bitset_ptr.
 */
template<unsigned int n, int DIRECTION>
  inline
  bitset_iterator<n, DIRECTION>::bitset_iterator(bitset_invertible<n, false> const* bitset_ptr, int bit) :
      bitset_index_iterator<DIRECTION>(bit), M_bitset_ptr(bitset_ptr)
  {
  }

/**
 * \brief Copy constructor.
 */
template<unsigned int n, int DIRECTION>
  inline
  bitset_iterator<n, DIRECTION>::bitset_iterator(bitset_iterator const& iter) :
      bitset_index_iterator<DIRECTION>(iter), M_bitset_ptr(iter.M_bitset_ptr)
  {
  }

/**
 * \brief Assignment operator.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>&
  bitset_iterator<n, DIRECTION>::operator=(bitset_iterator<n, DIRECTION> const& iter)
  {
#ifdef __i386__
    M_index = iter.M_index;
#else
    M_digit = iter.M_digit;
    M_mask = iter.M_mask;
#endif
    M_bitset_ptr = iter.M_bitset_ptr;
    return *this;
  }

template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>&
  bitset_iterator<n, DIRECTION>::operator=(bitset_index_iterator<DIRECTION> const& index)
  {
#ifdef __i386__
    M_index = index.get_index();
#else
    M_digit = index.get_digit();
    M_mask = index.get_mask();
#endif
    return *this;
  }

/**
 * \brief Preincrement.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>&
  bitset_iterator<n, DIRECTION>::operator++()
  {
    increment();
    return *this;
  }

/**
 * \brief Postincrement.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>
  bitset_iterator<n, DIRECTION>::operator++(int)
  {
    bitset_iterator prev(*this);
    increment();
    return prev;
  }

/**
 * \brief Predecrement.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>&
  bitset_iterator<n, DIRECTION>::operator--()
  {
    decrement();
    return *this;
  }

/**
 * \brief Postdecrement.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>
  bitset_iterator<n, DIRECTION>::operator--(int)
  {
    bitset_iterator prev(*this);
    decrement();
    return prev;
  }

/**
 * \brief Access bit as zero or bitmask.
 *
 * Returns 0 if the current bit is not set, otherwise returns the bit mask of that bit for the current digit.
 * For example, if this iterator points to bit 130 and that bit is set, then <code>*iter</code> will return 4.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_digit_t
  bitset_iterator<n, DIRECTION>::operator*() const
  {
#ifdef __i386__
    register unsigned int M_digit = M_index;
    register bitset_digit_t M_mask = 1;
    register unsigned short shift = M_index & (ECC_BITS - 1);
    M_digit >>= bitset_digit_bits_log2;
    M_mask <<= shift;
#endif
    return (M_bitset_ptr->digit(M_digit) & M_mask);
  }

/**
 * \brief Iterator addition.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>&
  bitset_iterator<n, DIRECTION>::operator+=(int d)
  {
    increment(d);
    return *this;
  }

/**
 * \brief Iterator subtraction.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>&
  bitset_iterator<n, DIRECTION>::operator-=(int d)
  {
    decrement(d);
    return *this;
  }

/**
 * \brief Iterator addition.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>
  operator+(bitset_iterator<n, DIRECTION> const& i, int d)
  {
    bitset_iterator<n, DIRECTION> result(i);
    return result += d;
  }

/**
 * \brief Iterator addition.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>
  operator+(int d, bitset_iterator<n, DIRECTION> const& i)
  {
    bitset_iterator<n, DIRECTION> result(i);
    return result += d;
  }

/**
 * \brief Iterator subtraction.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>
  operator-(bitset_iterator<n, DIRECTION> const& i, int d)
  {
    bitset_iterator<n, DIRECTION> result(i);
    return result -= d;
  }

/**
 * \brief Iterator subtraction.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_iterator<n, DIRECTION>
  operator-(int d, bitset_iterator<n, DIRECTION> const& i)
  {
    bitset_iterator<n, DIRECTION> result(i);
    return result -= d;
  }

/**
 * \brief Element operator.
 */
template<unsigned int n, int DIRECTION>
  inline bitset_digit_t
  bitset_iterator<n, DIRECTION>::operator[](int d) const
  {
    return *(*this + d);
  }

//--------------------------------------------------------------------------------------------------------------------------
/**\class bitset bitset.h libecc/bitset.h
 *
 * \brief A bitset with a fixed number of bits.
 *
 * \param n The number of bits in the bitset.
 */
template<unsigned int n>
  class bitset : public bitset_invertible<n, false> {
    public:
      // Constructors.
      bitset(void);
      bitset(std::string const&);
      explicit bitset(bitset_digit_t low_bits);
      // This definition must be here, inside the template class declaration because
      // otherwise the compiler (2.95 - 3.1) barfs.
      /**
       * \brief Construct a bitset directly from an array of \ref bitset_digit_t.
       *
       * The excess bits of the most significant digit (if any) must be zero.
       */
      bitset(bitset_digit_t const (&v)[bitset_base<n>::digits])
      {
#if ECC_DEBUG
	assert( (v[digits - 1] & ~valid_bits) == 0 );
#endif
	std::memcpy(this->vector, v, sizeof(this->vector));
      }

      // Copy constructors.
      template<unsigned int m, bool inverted>
	bitset(bitset_invertible<m, inverted> const&);
      template<unsigned int m, bool i1, bool i2, typename OP>
	bitset(Operator::bitsetExpression<m, i1, i2, OP> const& expr);

      // Assignment operators.
      bitset& operator=(bitset const&);
      template<typename Expression>
	bitset& operator=(Expression const&);

      // Perform AND, OR, XOR operations
      template<typename Expression>
	bitset& operator&=(Expression const&);
      template<typename Expression>
	bitset& operator|=(Expression const&);
      template<typename Expression>
	bitset& operator^=(Expression const&);

    public:
      // Shift bitset left or right and perform operation with result.
      template<unsigned int shift, class DIRECTION, class OPERATION>
	void shift_op(bitset& result) const;

      // Slower functions for shifting an arbitrary distance.
      bitset& operator<<=(unsigned int shift);
      bitset& operator>>=(unsigned int shift);

      // Rotate left or right
      template<unsigned int shift, class DIRECTION>			// Return a copy rotated `shift' bits in `DIRECTION'.
	void rotate(bitset& result) const;

      // Single bit operations
      bool test(size_t n) const;					// Return true if bit `n' is set.
      bool odd(void) const;						// Return true if the vector has an odd number of bits set.
      void set(size_t n);						// Set bit `n'.
      void clear(size_t n);						// Clear bit `n'.
      void flip(size_t n);						// Toggle bit `n'.

      // Single bit operations using iterators.
      bool test(bitset_index const& index) const;			// Return true if bit refered to by `index' is set.
      void set(bitset_index const& index);				// Set bit refered to by `index'.
      void clear(bitset_index const& index);				// Clear bit refered to by `index'.
      void flip(bitset_index const& index);				// Toggle bit refered to by `index'.

      // Single bit operations at a constant position
      template<unsigned int pos>
	bool test(void) const;						// Return true if bit `pos' is set.
      template<unsigned int pos>
	void set(void);							// Set bit `pos'.
      template<unsigned int pos>
	void clear(void);						// Clear bit `pos'.
      template<unsigned int pos>
	void flip(void);						// Toggle bit `pos'.

      // Other functions
      bitset& reset(void);
      void setall(void);
      bool any(void) const;

    public:
      // Iterator stuff.
      //! Non-mutable iterator.
      typedef bitset_iterator<n, forwards_iterating> const_iterator;
      //! Reverse non-mutable iterator.
      typedef bitset_iterator<n, backwards_iterating> const_reverse_iterator;
      //! Return an iterator to the first bit.
      const_iterator begin(void) const { return const_iterator(this, 0); }
      //! Return an iterator one past the last bit.
      const_iterator end(void) const { return const_iterator(this, n); }
      //! Return a reverse iterator to bit n-1.
      const_reverse_iterator rbegin(void) const { return const_reverse_iterator(this, n - 1); }
      //! Return a reverse iterator one before the first bit.
      const_reverse_iterator rend(void) const { return const_reverse_iterator(this, -1); }

      template<unsigned int m>
	void xor_with_zero_padded(bitset<m> const& bitset, int lsb, int msb, int shift);
  };

/**
 * \brief Shift bitset left or right and XOR with this bitset.
 *
 * \param bitset The input bitset.
 * \param lsb The Least Significant set bit of \a bitset.
 * \param msb The Most Significant set bit in \a bitset.
 * \param shift The distance to shift to the left, may be negative in which case \a bitset is shifted right.
 *
 * Bit \a lsb up till and including bit \a msb of \a bitset are shifted
 * left over a distance \a shift (or shifted right if \a shift is negative)
 * and then bitwise exclusive-OR-ed with this bitset.  No range checking for
 * this bitset is being done; it is the responsibility of the caller to make
 * sure that it will fit.  Moreover, it is assumed that all bits in the same
 * digit but left of \a msb and and right of \a lsb are zero and that that
 * there is at least one fully zeroed digit left and right of the bitset.
 */
template<unsigned int n>
  template<unsigned int m>
#ifdef __i386__
  inline void
#endif
  bitset<n>::xor_with_zero_padded(bitset<m> const& bitset, int lsb, int msb, int shift)
  {
#ifdef __i386__
    libecc_shift_xorassign(vector, bitset.digits_ptr(), lsb, msb, shift);
#else
    int digit1 = lsb >> bitset_digit_bits_log2;
    int digit2 = msb >> bitset_digit_bits_log2;
    bitset_digit_t d1 = 0;
    if (shift < 0)
    {
      unsigned int bitshift = (-shift) & (bitset_digit_bits - 1);
      unsigned int digitshift = (-shift) >> bitset_digit_bits_log2;
      if (bitshift == 0)
      {
	for (int digit = digit2; digit >= digit1; --digit)
	{
	  bitset_digit_t d2 = bitset.digit(digit);
	  vector[digit - digitshift] ^= d2;
	  d1 = d2;
	}
      }
      else
      {
	for (int digit = digit2; digit >= digit1; --digit)
	{
	  bitset_digit_t d2 = bitset.digit(digit);
	  vector[digit - digitshift] ^= (d2 >> bitshift) | (d1 << (32 - bitshift));
	  d1 = d2;
	}
	vector[digit1 - 1 - digitshift] ^= (d1 << (32 - bitshift));
      }
    }
    else if (shift > 0)
    {
      unsigned int bitshift = shift & (bitset_digit_bits - 1);
      unsigned int digitshift = shift >> bitset_digit_bits_log2;
      if (bitshift == 0)
      {
	for (int digit = digit1; digit <= digit2; ++digit)
	{
	  bitset_digit_t d2 = bitset.digit(digit);
	  vector[digit + digitshift] ^= d2;
	  d1 = d2;
	}
      }
      else
      {
	for (int digit = digit1; digit <= digit2; ++digit)
	{
	  bitset_digit_t d2 = bitset.digit(digit);
	  vector[digit + digitshift] ^= (d2 << bitshift) | (d1 >> (32 - bitshift));
	  d1 = d2;
	}
	vector[digit2 + 1 + digitshift] ^= (d1 >> (32 - bitshift));
      }
    }
    else
    {
      for (int digit = digit1; digit <= digit2; ++digit)
	vector[digit] ^= bitset.digit(digit);
    }
#endif
  }

/**
 * \brief Construct an uninitialized bitset of \a n bits.
 */
template<unsigned int n>
  inline
  bitset<n>::bitset(void)
  {
  }

/**
 * \brief Construct a bitset of which only the least significant bits are set.
 *
 * \a low_bits must never have more bits set than the size of the bitset n.
 */
template<unsigned int n>
  inline
  bitset<n>::bitset(bitset_digit_t low_bits)
  {
#if ECC_DEBUG
    assert( digits > 1 || (low_bits & ~valid_bits) == 0 );
#endif
    std::memset(this->vector, 0, sizeof(this->vector));
    this->vector[0] = low_bits;
  }

/**
 * \brief Set all bits to 1.
 */
template<unsigned int n>
  void
  bitset<n>::setall(void)
  {
    this->vector[digits - 1] = valid_bits;
    if (digits > 1)
    {
      int d = digits - 2;
      do { this->vector[d] = ~static_cast<bitset_digit_t>(0); } while(--d >= 0);
    }
  }

/**
 * \brief Assignment operator.
 *
 * The least significant bits (at position 0) of \a expr and this bitset are lined up.
 * If \a expr is wider than this bitset then excess bits are lost;
 * if narrower then missing bits are set to zero or one depending on whether the
 * expression contains inversion bits.
 *
 * For example, when x is "0000000010001111" (16 bits), y is "10000000" (8 bits)
 * and z is "11110110" (8 bits), then
 *
 * \code
 * x ^= ~y;
 * \endcode
 *
 * results in x being "1111111111110000".
 * And
 *
 * \code
 * x |= ~y ^ ~z;
 * \endcode
 *
 * results in x being "0000000001110110".
 */
template<unsigned int n>
  template<typename Expression>
    inline
    bitset<n>& bitset<n>::operator=(Expression const& expr)
    {
      this->assign(expr, Operator::bitsetAssign());
      return *this;
    }

// Special case, need to define this or else we'd get a default assignment operator.
template<unsigned int n>
  inline bitset<n>&
  bitset<n>::operator=(bitset const& expr)
  {
    this->assign(expr, Operator::bitsetAssign());
    return *this;
  }

/**
 * \brief Copy constructor.
 *
 * The least significant bit (at position 0) of \a bits and the constructed bitset are lined up.
 * If \a bits is wider than the constructed bitset then excess bits are lost;
 * if narrower then missing bits are set to \a inverted.  That is, set to zero when \a bits
 * is not inverted and set to one when \a bits represents an inverted bitset.
 *
 * For example, when x is "0000000010001111" (16 bits), y is "10000000" (8 bits), then
 *
 * \code
 * x ^= ~y;
 * \endcode
 *
 * results in x being "1111111111110000".
 */
template<unsigned int n>
  template<unsigned int m, bool inverted>
    inline bitset<n>::bitset(bitset_invertible<m, inverted> const& bits) : bitset_invertible<n, false>(bits)
    {
    }

/**
 * \brief Construct a bitset from an expression.
 *
 * \see bitset::operator=
 */
template<unsigned int n>
  template<unsigned int m, bool i1, bool i2, typename OP>
    inline bitset<n>::bitset(Operator::bitsetExpression<m, i1, i2, OP> const& expr) : bitset_invertible<n, false>(expr)
    {
    }

/**
 * \brief Equivalence operator.
 *
 * This operator can only compare bitsets and inverted bitsets with eachother.
 *
 * For example:
 *
 * \code
 * if (x == ~y) ...
 * \endcode
 */
template<unsigned int n1, bool inverted1, unsigned int n2, bool inverted2>
  bool operator==(bitset_invertible<n1, inverted1> const& bits1, bitset_invertible<n2, inverted2> const& bits2)
  {
    unsigned int d;
    if (bitset_invertible<n1, inverted1>::digits > bitset_invertible<n2, inverted2>::digits)
    {
      d = bitset_base<n1>::digits - 1;
      do
      {
	if (bits1.vector[d] != (inverted1 == inverted2 ? 0 : ~static_cast<bitset_digit_t>(0)))
	  return false;
      }
      while (--d != bitset_base<n2>::digits - 1);
    }
    if (bitset_base<n2>::digits > bitset_base<n1>::digits)
    {
      d = bitset_base<n2>::digits - 1;
      do
      {
	if (bits2.vector[d] != (inverted1 == inverted2 ? 0 : ~static_cast<bitset_digit_t>(0)))
	  return false;
      }
      while (--d != bitset_base<n1>::digits - 1);
    }
    if (bitset_base<n1>::digits > 1 && bitset_base<n2>::digits > 1)
    {
      if (bitset_base<n1>::digits == bitset_base<n2>::digits)
	d = bitset_base<n1>::digits - 1;
      do
      {
	if (inverted1 == inverted2)
	{
	  if (bits1.vector[d] != bits2.vector[d])
	    return false;
	}
	else
	{
	  if (bits1.vector[d] != ~(bits2.vector[d]))
	    return false;
	}
      }
      while(--d != 0);
    }
    if (inverted1 != inverted2)
      return (bits1.vector[0] == ~(bits2.vector[0]));
    return (bits1.vector[0] == bits2.vector[0]);
  }

/**
 * \brief Unequivalence operator.
 *
 * \returns <code>!(bits1 == bits2)</code>
 */
template<unsigned int n1, bool inverted1, unsigned int n2, bool inverted2>
  inline bool
  operator!=(bitset_invertible<n1, inverted1> const& bits1, bitset_invertible<n2, inverted2> const& bits2)
  {
    return !(bits1 == bits2);
  }

/** \brief Assignment operator with bitwise AND.
 *
 * \see bitset::operator=
 */
template<unsigned int n>
  template<typename Expression>
    inline bitset<n>&
    bitset<n>::operator&=(Expression const& expr)
    {
      this->assign(expr, Operator::bitsetANDAssign());
      return *this;
    }

/** \brief Assignment operator with bitwise OR.
 *
 * \see bitset::operator=
 */
template<unsigned int n>
  template<typename Expression>
    inline bitset<n>&
    bitset<n>::operator|=(Expression const& expr)
    {
      this->assign(expr, Operator::bitsetORAssign());
      return *this;
    }

/** \brief Assignment operator with bitwise XOR.
 *
 * \see bitset::operator=
 */
template<unsigned int n>
  template<typename Expression>
    inline bitset<n>&
    bitset<n>::operator^=(Expression const& expr)
    {
      this->assign(expr, Operator::bitsetXORAssign());
      return *this;
    }

/** \brief Assignment operator with bitwise AND for expressions.
 *
 * \see bitset::operator=
 */
template<unsigned int m, bool inverted1, bool inverted2>
  Operator::bitsetExpression<m, inverted1, inverted2, Operator::bitsetAND>
  operator&(bitset_invertible<m, inverted1> const& arg1, bitset_invertible<m, inverted2> const& arg2)
    {
      return Operator::bitsetExpression<m, inverted1, inverted2, Operator::bitsetAND>(arg1, arg2);
    }

/** \brief Assignment operator with bitwise OR for expressions.
 *
 * \see bitset::operator=
 */
template<unsigned int m, bool inverted1, bool inverted2>
  Operator::bitsetExpression<m, inverted1, inverted2, Operator::bitsetOR>
  operator|(bitset_invertible<m, inverted1> const& arg1, bitset_invertible<m, inverted2> const& arg2)
    {
      return Operator::bitsetExpression<m, inverted1, inverted2, Operator::bitsetOR>(arg1, arg2);
    }

/** \brief Assignment operator with bitwise XOR for expressions.
 *
 * \see bitset::operator=
 */
template<unsigned int m, bool inverted1, bool inverted2>
  Operator::bitsetExpression<m, inverted1, inverted2, Operator::bitsetXOR>
  operator^(bitset_invertible<m, inverted1> const& arg1, bitset_invertible<m, inverted2> const& arg2)
    {
      return Operator::bitsetExpression<m, inverted1, inverted2, Operator::bitsetXOR>(arg1, arg2);
    }

/**
 * \brief Operation used with libecc::bitset::shift_op
 */
struct assign {
  static int const id = 1;
  static bool const __clear = true;
  static inline void op(bitset_digit_t& digit1, bitset_digit_t digit2) { digit1 = digit2; }
  static inline void mixed_op(bitset_digit_t& digit1, bitset_digit_t digit2) { digit1 = digit2; }
};

/**
 * \brief Operation used with libecc::bitset::shift_op
 */
struct exor {
  static int const id = 2;
  static bool const __clear = false;
  static inline void op(bitset_digit_t& digit1, bitset_digit_t digit2) { digit1 ^= digit2; }
  static inline void mixed_op(bitset_digit_t& digit1, bitset_digit_t digit2) { digit1 ^= digit2; }
};

#ifndef HIDE_FROM_DOXYGEN
struct rotate_phase1 {
  static int const id = 3;
  static bool const __clear = false;
  static inline void op(bitset_digit_t& digit1, bitset_digit_t digit2) { digit1 = digit2; }
  static inline void mixed_op(bitset_digit_t& digit1, bitset_digit_t digit2) { digit1 = digit2; }
};

struct rotate_phase2 {
public:
  static int const id = 4;
  static bool const __clear = false;
  static inline void op(bitset_digit_t& digit1, bitset_digit_t digit2) { digit1 = digit2; }
  static inline void mixed_op(bitset_digit_t& digit1, bitset_digit_t digit2) { digit1 |= digit2; }
};
#endif

/** \brief Perform a shift operation followed by a bit operation.
 *
 * Shift this object \a shift bits in \a DIRECTION and then
 * perform operation \a OPERATION on \a result with it.
 *
 * \param shift the number of bits to shift this object.
 * \param DIRECTION can be libecc::left or libecc::right.
 * \param OPERATION can be either libecc::assign or libecc::exor.
 * \param result The bitset that is used to XOR with, or to assign to.
 */
template<unsigned int n>
  template<unsigned int shift, class DIRECTION, class OPERATION>
    void
    bitset<n>::shift_op(bitset& result) const
    {
      LibEccDout(dc::bitsetshift, "Entering shift_op<" << shift << ", " << type_info_of<DIRECTION>().demangled_name() <<
	  ", " << type_info_of<OPERATION>().demangled_name() << '>');
      LibEccDout(dc::bitsetshift|flush_cf, "Input : " <<
	  cwprint_using(*static_cast<bitset_invertible<n, false> const*>(this), &bitset<n>::base2_print_on));
      if (shift == 1)
      {
	// Specialization for shift == 1.
	// digit_shift = 0
	// bit_shift = 1
	// zeroed_digits = 0 (likely and if not - then assumed).
	// Here we scan in the opposite direction of when shift > 1.
	static unsigned int const initial = DIRECTION::__left ? 0 : digits - 1;
	static unsigned int const count = digits - ((DIRECTION::__left && has_excess_bits) ? 1 : 0);
#ifdef __i386__
	if (count)
	{
	  bitset_digit_t const volatile* ptr1;
	  bitset_digit_t volatile* ptr2;
	  int c;
	  if (DIRECTION::__left)
	  {
	    if (OPERATION::id == libecc::exor::id)
	    {
	      if (has_excess_bits)
		LIBECC_ASMSHIFTLEFT(
		    LIBECC_LEFT_PRESERVE_CF("xorl -4(%%edi), %%edx"),
		    LIBECC_LEFT_PRESERVE_CF("xorl (%%edi), %%edx"),
		    LIBECC_ASMSHIFTLEFT_FINISH("xorl (%%edi),%%edx\n\t"));
	      else
		LIBECC_ASMSHIFTLEFT(
		    LIBECC_LEFT_PRESERVE_CF("xorl -4(%%edi), %%edx"),
		    LIBECC_LEFT_PRESERVE_CF("xorl (%%edi), %%edx"), "");
	    }
	    else if (OPERATION::id == libecc::rotate_phase2::id)
	    {
	      if (has_excess_bits)
		LIBECC_ASMSHIFTLEFT(
		    LIBECC_LEFT_PRESERVE_CF("orl -4(%%edi), %%edx"), "", LIBECC_ASMSHIFTLEFT_FINISH(""));
	      else
		LIBECC_ASMSHIFTLEFT(
		    LIBECC_LEFT_PRESERVE_CF("orl -4(%%edi), %%edx"), "", "");
	    }
	    else
	    {
	      if (has_excess_bits)
		LIBECC_ASMSHIFTLEFT("", "", LIBECC_ASMSHIFTLEFT_FINISH(""));
	      else
		LIBECC_ASMSHIFTLEFT("", "", "");
	    }
	  }
	  else if (count > 1)
	  {
	    if (OPERATION::id == libecc::exor::id)
	      LIBECC_ASMSHIFTRIGHT1(
		  LIBECC_RIGHT_PRESERVE_CF("xorl " LIBECC_OP1_DESTINATION "," LIBECC_WORKREG),
		  LIBECC_RIGHT_PRESERVE_CF("xorl " LIBECC_OP2_DESTINATION "," LIBECC_WORKREG),
		  LIBECC_PRESERVE_CF_CLOBBER);
	    else if (OPERATION::id == libecc::rotate_phase2::id)
	      LIBECC_ASMSHIFTRIGHT1(
		  LIBECC_RIGHT_PRESERVE_CF("orl " LIBECC_OP1_DESTINATION "," LIBECC_WORKREG),
		  "",
		  LIBECC_PRESERVE_CF_CLOBBER);
	    else
	      LIBECC_ASMSHIFTRIGHT1(
		  "",
		  "",
		  LIBECC_CLOBBER);
	  }
	  else
	  {
	    if (OPERATION::id == libecc::exor::id)
	      LIBECC_ASMSHIFTRIGHT0("xorl 4(%%edi), %%edx\n\t");
	    else if (OPERATION::id == libecc::rotate_phase2::id)
	      LIBECC_ASMSHIFTRIGHT0("orl 4(%%edi), %%edx\n\t");
	    else
	      LIBECC_ASMSHIFTRIGHT0("");
	  }
	}
	else if (DIRECTION::__left && has_excess_bits)
	  OPERATION::op(result.vector[0], ((vector[0] << 1) & valid_bits));
#else
	static unsigned int complement_shift = bitset_digit_bits - 1;
	static bitset_digit_t const mask1 = DIRECTION::__left ? (1 << complement_shift) : 1;
	static bitset_digit_t const mask2 = DIRECTION::__right ? (1 << complement_shift) : 1;
	bitset_digit_t const* ptr1 = &this->vector[initial];
	bitset_digit_t* ptr2 = &result.vector[initial];
	bool carry;
	if (count)
	{
	  bitset_digit_t digit = *ptr1;
	  carry = (digit & mask1);
	  if (DIRECTION::__left)
	    digit <<= 1;
	  else
	    digit >>= 1;
	  OPERATION::mixed_op(*ptr2, digit);
	  for (int c = count - 1; c; --c)
	  {
	    ptr1 -= DIRECTION::direction;
	    ptr2 -= DIRECTION::direction;
	    digit = *ptr1;
	    if (carry)
	    {
	      carry = (digit & mask1);
	      if (DIRECTION::__left)
		digit <<= 1;
	      else
		digit >>= 1;
	      digit |= mask2;
	    }
	    else
	    {
	      carry = (digit & mask1);
	      if (DIRECTION::__left)
		digit <<= 1;
	      else
		digit >>= 1;
	    }
	    OPERATION::op(*ptr2, digit);
	  }
	}
	if (DIRECTION::__left && has_excess_bits)
	{
	  bitset_digit_t digit;
	  if (count)
	    digit = ptr1[-DIRECTION::direction];
	  else
	    digit = *ptr1;
	  digit <<= 1;
	  if (count && carry)
	    digit |= mask2;
	  if (count)
	    OPERATION::op(ptr2[-DIRECTION::direction], (digit & valid_bits));
	  else
	    OPERATION::op(*ptr2, (digit & valid_bits));
	}
#endif
      }
      else
      {
	static unsigned int const digit_shift = shift / bitset_digit_bits;
	static unsigned int const bit_shift = shift % bitset_digit_bits;

	static unsigned int const zeroed_digits =
	  DIRECTION::__left ? ((shift < n) ? digit_shift : digits)
			    : ((bitset_digit_bits - number_of_valid_bits + shift) / bitset_digit_bits);

	if (zeroed_digits < digits)
	{
	  static unsigned int const complement_shift = (bit_shift == 0) ? 0 : bitset_digit_bits - bit_shift;
	  static unsigned int const initial_to = DIRECTION::__right ? 0 : digits - 1;
	  static unsigned int const initial_from = initial_to + DIRECTION::direction * digit_shift;
	  static unsigned int const final_from = DIRECTION::__left ? 0 : digits - 1;

	  register bitset_digit_t digit = this->vector[initial_from];
	  if (initial_from != final_from)
	  {
	    register bitset_digit_t next_digit;
	    unsigned int to = initial_to;
	    unsigned int from = initial_from + DIRECTION::direction;
	    if (from != final_from)
	      do
	      {
		next_digit = this->vector[from];
		if (bit_shift != 0)
		{
		  if (DIRECTION::direction == -1)
		    digit <<= bit_shift;
		  else
		    digit >>= bit_shift;
		  digit |= DIRECTION::reverse_shift_copy(next_digit, complement_shift);
		}
		OPERATION::op(result.vector[to], digit);
		digit = next_digit;
		to += DIRECTION::direction;
		from += DIRECTION::direction;
	      }
	      while (from != final_from);
	    if (DIRECTION::__left || bit_shift < number_of_valid_bits || bit_shift != 0)
	      next_digit = this->vector[final_from];
	    if (bit_shift != 0)
	    {
	      if (DIRECTION::direction == -1)
		digit <<= bit_shift;
	      else
		digit >>= bit_shift;
	      digit |= DIRECTION::reverse_shift_copy(next_digit, complement_shift);
	    }
	    if (DIRECTION::__left || bit_shift < number_of_valid_bits)
	    {
	      OPERATION::op(result.vector[final_from - DIRECTION::direction * (digit_shift + 1)], digit);
	      digit = DIRECTION::shift_copy(next_digit, bit_shift);
	    }
	  }
	  else
	  {
	    if (DIRECTION::direction == -1)
	      digit <<= bit_shift;
	    else
	      digit >>= bit_shift;
	  }
	  static bool const have_mixed_digit = shift != 0 && (DIRECTION::__left ? shift : (n - shift)) % bitset_digit_bits != 0;
	  static unsigned int const last_digit = (DIRECTION::__left ? shift : (n - 1 - shift)) / bitset_digit_bits;
	  if (have_mixed_digit)
	    OPERATION::mixed_op(result.vector[last_digit], digit);
	  else
	    OPERATION::op(result.vector[last_digit], digit);
	  if (DIRECTION::__left && has_excess_bits)
	    result.vector[digits - 1] &= valid_bits;
	}
	if (OPERATION::__clear && zeroed_digits > 0)
	{
	  static unsigned int const final_to = DIRECTION::__left ? 0 : digits - 1;
	  static unsigned int const initial_to = final_to - DIRECTION::direction * (zeroed_digits - 1);
	  unsigned int to = initial_to;
	  if (to != final_to)
	    do
	    {
	      result.vector[to] = 0;
	      to += DIRECTION::direction;
	    }
	    while(to != final_to);
	  result.vector[to] = 0;
	}
      }
      LibEccDout(dc::bitsetshift|flush_cf, "Output: " <<
	  cwprint_using(*static_cast<bitset_invertible<n, false> const*>(&result), &bitset<n>::base2_print_on));
    }

/**
 * \brief Shift bitset \a shift bits to the left.&nbsp; Use the faster \ref bitset::shift_op for constant distance shifts.
 */
template<unsigned int n>
  bitset<n>&
  bitset<n>::operator<<=(unsigned int shift)
  {
    unsigned int const digit_shift = shift >> bitset_digit_bits_log2;
    unsigned int const bit_shift = shift & (bitset_digit_bits - 1);
    int digit1 = digits - digit_shift - 2;
    bitset_digit_t d0 = (digit1 >= -1) ? vector[digit1 + 1] : 0;
    for (int digit_to = digits - 1; digit_to >= 0; --digit_to)
    {
      bitset_digit_t d1 = (digit1 >= 0) ? vector[digit1] : 0;
      if (bit_shift == 0)
	vector[digit_to] = d0;
      else
	vector[digit_to] = (d0 << bit_shift) | (d1 >> (bitset_digit_bits - bit_shift));
      d0 = d1;
      --digit1;
    }
    if (has_excess_bits)
      this->vector[digits - 1] &= valid_bits;		// Reset possibly set excess bits.
    return *this;
  }

/**
 * \brief Shift bitset \a shift bits to the right.&nbsp; Use the faster \ref bitset::shift_op for constant distance shifts.
 */
template<unsigned int n>
  bitset<n>&
  bitset<n>::operator>>=(unsigned int shift)
  {
    unsigned int const digit_shift = shift >> bitset_digit_bits_log2;
    unsigned int const bit_shift = shift & (bitset_digit_bits - 1);
    int digit1 = digit_shift + 1;
    bitset_digit_t d0 = (digit1 <= digits) ? vector[digit1 - 1] : 0;
    for (int digit_to = 0; digit_to < digits; ++digit_to)
    {
      bitset_digit_t d1 = (digit1 < digits) ? vector[digit1] : 0;
      if (bit_shift == 0)
	vector[digit_to] = d0;
      else
	vector[digit_to] = (d0 >> bit_shift) | (d1 << (bitset_digit_bits - bit_shift));
      d0 = d1;
      ++digit1;
    }
    return *this;
  }

/**
 * \brief Construct a bitset of \a n bits and initialize it with the string \a input, a hexadecimal value in ASCII representation.
 *
 * \a Input is a string of alphanumeric characters ([0-9A-F]) not case sensitive.
 * The input string is read right to left. Any non-space character that is not a hexadecimal digit will terminate reading.
 *
 * Spaces will be ignored, allowing one to write for example
 * "<CODE>C20 93B0D12C F78A9001 F6E2841F 0A918FCD</CODE>" instead of
 * the less readable "<CODE>C2093B0D12CF78A9001F6E2841F0A918FCD</CODE>".
 * It is common practise to let each "word" represent an internal digit (32 bits in this example).
 *
 * If the value passed is larger than fits in the bitset then the most significant bits (the left most characters)
 * are ignored as if the value \em did fit and then a bit-wise AND was performed with a bitset of \a n 1's.
 *
 * \param input A hexadecimal value in ASCII representation.
 */
template<unsigned int n>
  bitset<n>::bitset(std::string const& input)
  {
    reset();						// Reset internal digits to zero.
    unsigned int d = 0;					// Current index of internal digit.
    unsigned int u = 0;					// Current bit-shift of input digit relative to internal digit.

    for (std::string::const_reverse_iterator iter = input.rbegin(); iter != input.rend(); ++iter)	// Read right to left.
    {
      bitset_digit_t c = toupper(*iter);		// Get next hexadecimal input character.
      if (c == ' ')					// Skip spaces.
	continue;
      if (c < '0')					// Terminate reading when not a hexadecimal character.
	break;
      if (c <= '9')
	c -= '0';					// Set c to the value that the input character represents.
      else
      {
	if (c > 'F')
	  break;
	if (c >= 'A')
	  c -= ('A' - 10);
	else
	  break;
      }
      this->vector[d] |= c << u;			// Set internal bits.
      if ((u += 4) == bitset_digit_bits)		// Update bit/digit 'pointers'.
      {
	u = 0;
	if (++d == digits)				// Terminate reading when bitset is full.
	  break;
      }
    }
    if (has_excess_bits)
      this->vector[digits - 1] &= valid_bits;		// Reset possibly set excess bits.
  }

/**
 * \brief Read bitset from istream.
 */
template<unsigned int n>
  std::istream&
  operator>>(std::istream& is, bitset<n>& bitsetx)
  {
    std::string tmp;
    is >> tmp;
    bitsetx.bitset(tmp);
    return is;
  }

/**
 * \brief Write bitset to ostream.
 */
template<unsigned int n>
  std::ostream&
  operator<<(std::ostream& os, bitset<n> const& bits)
  {
    // Hexadecimal representation
    os.fill('0');
    os << std::hex;
    for (int d = bitset<n>::digits - 1; d >= 0; --d)
    {
      os.width((d == bitset<n>::digits - 1 && bitset<n>::has_excess_bits) ?
          (((n % bitset_digit_bits) - 1) / 4 + 1) :
	  (bitset_digit_bits / 4));
      os << bits.digit(d);
      if (d > 0)
	os << ' ';
    }
    os << std::dec;
    return os;
  }

/**
 * \brief Reset all bits to 0.
 */
template<unsigned int n>
  bitset<n>&
  bitset<n>::reset(void)
  {
    std::memset(this->vector, 0, sizeof(this->vector));
    return *this;
  }

/**
 * \brief Test bit at position \a pos.
 */
template<unsigned int n>
  inline bool
  bitset<n>::test(size_t pos) const
  {
#ifdef __i386__
    int result;
    __asm__ __volatile__ (
	"btl %2, %1\n\t"
	"sbbl %0, %0"
	: "=r" (result)
	: "m" ((*(bitset_digit_t volatile*)this->vector)), "Ir" (pos));
    return result;
#else
    unsigned int d = pos / bitset_digit_bits;
    bitset_digit_t mask = static_cast<bitset_digit_t>(1) << (pos % bitset_digit_bits);
    return (this->vector[d] & mask);
#endif
  }

/**
 * \brief Test a bit at a fixed position.
 */
template<unsigned int n>
  template<unsigned int pos>
    inline bool
    bitset<n>::test(void) const
    {
      static unsigned int const d = pos / bitset_digit_bits;
      static bitset_digit_t const mask = static_cast<bitset_digit_t>(1) << (pos % bitset_digit_bits);
      return (this->vector[d] & mask);
    }

/**
 * \brief Test a bit at position \a index.
 */
template<unsigned int n>
  inline bool
  bitset<n>::test(bitset_index const& index) const
  {
#ifdef __i386__
    return this->test(index.get_index());
#else
    return (this->vector[index.get_digit()] & index.get_mask());
#endif
  }

/**
 * \brief Set bit at position \a pos.
 */
template<unsigned int n>
  inline void
  bitset<n>::set(size_t pos)
  {
#ifdef __i386__
    __asm__ __volatile__ (
        "btsl %1, %0"
        : "=m" ((*(bitset_digit_t volatile*)this->vector))
        : "Ir" (pos));

#else
    unsigned int d = pos / bitset_digit_bits;
    bitset_digit_t mask = static_cast<bitset_digit_t>(1) << (pos % bitset_digit_bits);
    this->vector[d] |= mask;
#endif
  }

/**
 * \brief Set bit at position \a index.
 */
template<unsigned int n>
  inline void
  bitset<n>::set(bitset_index const& index)
  {
#ifdef __i386__
    this->set(index.get_index());
#else
    this->vector[index.get_digit()] |= index.get_mask();
#endif
  }

/**
 * \brief Set a bit at a fixed position.
 */
template<unsigned int n>
  template<unsigned int pos>
    inline void
    bitset<n>::set(void)
    {
      static unsigned int const d = pos / bitset_digit_bits;
      static bitset_digit_t const mask = static_cast<bitset_digit_t>(1) << (pos % bitset_digit_bits);
      this->vector[d] |= mask;
    }

/**
 * \brief Clear bit at position \a pos.
 */
template<unsigned int n>
  inline void
  bitset<n>::clear(size_t pos)
  {
#ifdef __i386__
    __asm__ __volatile__ (
	"btrl %1, %0"
	: "=m" ((*(bitset_digit_t volatile*)this->vector))
	: "Ir" (pos)
    );
#else
    unsigned int d = pos / bitset_digit_bits;
    bitset_digit_t mask = static_cast<bitset_digit_t>(1) << (pos % bitset_digit_bits);
    this->vector[d] &= ~mask;
#endif
  }

/**
 * \brief Clear bit at position \a index.
 */
template<unsigned int n>
  inline void
  bitset<n>::clear(bitset_index const& index)
  {
#ifdef __i386__
    this->clear(index.get_index());
#else
    this->vector[index.get_digit()] &= ~(index.get_mask());
#endif
  }

/**
 * \brief Clear a bit at a fixed position.
 */
template<unsigned int n>
  template<unsigned int pos>
    inline void
    bitset<n>::clear(void)
    {
      static unsigned int const d = pos / bitset_digit_bits;
      static bitset_digit_t const mask = ~(static_cast<bitset_digit_t>(1) << (pos % bitset_digit_bits));
      this->vector[d] &= mask;
    }

/**
 * \brief Toggle bit at position \a pos.
 */
template<unsigned int n>
  inline void
  bitset<n>::flip(size_t pos)
  {
#ifdef __i386__
    __asm__ __volatile__ (
        "btcl %1, %0"
        : "=m" ((*(bitset_digit_t volatile*)this->vector))
        : "Ir" (pos)
    );
#else
    unsigned int d = pos / bitset_digit_bits;
    bitset_digit_t mask = static_cast<bitset_digit_t>(1) << (pos % bitset_digit_bits);
    this->vector[d] ^= mask;
#endif
  }

/**
 * \brief Toggle bit at position \a index.
 */
template<unsigned int n>
  inline void
  bitset<n>::flip(bitset_index const& index)
  {
#ifdef __i386__
    this->flip(index.get_index());
#else
    this->vector[index.get_digit()] ^= index.get_mask();
#endif
  }

/**
 * \brief Toggle a bit at a fixed position.
 */
template<unsigned int n>
  template<unsigned int pos>
    inline void
    bitset<n>::flip(void)
    {
      static unsigned int const d = pos / bitset_digit_bits;
      static bitset_digit_t const mask = static_cast<bitset_digit_t>(1) << (pos % bitset_digit_bits);
      this->vector[d] ^= mask;
    }

/**
 * \brief Return true if any bit is set.
 */
template<unsigned int n>
  bool
  bitset<n>::any(void) const
  {
    unsigned int to = digits - 1;
    if (digits > 1)
      do
      {
	if (this->vector[to] != 0)
	  return true;
	--to;
      }
      while(to != 0);
    return (this->vector[0] != 0);
  }

static bool const oddnumberofbits[] = {
  false, true, true, false, true, false, false, true, true, false, false, true, false, true, true, false,
  true, false, false, true, false, true, true, false, false, true, true, false, true, false, false, true,
  true, false, false, true, false, true, true, false, false, true, true, false, true, false, false, true,
  false, true, true, false, true, false, false, true, true, false, false, true, false, true, true, false,
  true, false, false, true, false, true, true, false, false, true, true, false, true, false, false, true,
  false, true, true, false, true, false, false, true, true, false, false, true, false, true, true, false,
  false, true, true, false, true, false, false, true, true, false, false, true, false, true, true, false,
  true, false, false, true, false, true, true, false, false, true, true, false, true, false, false, true,
  true, false, false, true, false, true, true, false, false, true, true, false, true, false, false, true,
  false, true, true, false, true, false, false, true, true, false, false, true, false, true, true, false,
  false, true, true, false, true, false, false, true, true, false, false, true, false, true, true, false,
  true, false, false, true, false, true, true, false, false, true, true, false, true, false, false, true,
  false, true, true, false, true, false, false, true, true, false, false, true, false, true, true, false,
  true, false, false, true, false, true, true, false, false, true, true, false, true, false, false, true,
  true, false, false, true, false, true, true, false, false, true, true, false, true, false, false, true,
  false, true, true, false, true, false, false, true, true, false, false, true, false, true, true, false };

/**
 * \brief Returns true when the bitset contains an odd number of bits.
 */
template<unsigned int n>
  bool
  bitset<n>::odd(void) const
  {
    unsigned int from = digits - 1;
    bitset_digit_t sum = this->vector[0];
    if (digits > 1)
      do
      {
	sum ^= this->vector[from];
	--from;
      }
      while(from != 0);
    bitset_digit_t ssum = sum;
    if (sizeof(bitset_digit_t) >= 2)
    {
      ssum >>= (bitset_digit_bits / 2);
      sum ^= ssum;
    }
    if (sizeof(bitset_digit_t) >= 4)
    {
      ssum = sum;
      ssum >>= (bitset_digit_bits / 4);
      sum ^= ssum;
    }
    if (sizeof(bitset_digit_t) >= 8)
    {
      ssum = sum;
      ssum >>= (bitset_digit_bits / 8);
      sum ^= ssum;
    }
    if (sizeof(bitset_digit_t) == 16)
    {
      ssum = sum;
      ssum >>= (bitset_digit_bits / 16);
      sum ^= ssum;
    }
    return oddnumberofbits[sum & 0xff];
  }

/**
 * \brief Shift direction used with libecc::bitset::shift_op
 */
struct left {
public:
  typedef struct right inverse;
  static int const direction = -1;
  static bool const __left = true;
  static bool const __right = false;
  static inline bitset_digit_t shift_copy(bitset_digit_t digit, unsigned int shift) { return digit << shift; }
  static inline bitset_digit_t reverse_shift_copy(bitset_digit_t digit, unsigned int shift) { return digit >> shift; }
};

/**
 * \brief Shift direction used with libecc::bitset::shift_op
 */
struct right {
public:
  typedef struct left inverse;
  static int const direction = 1;
  static bool const __left = false;
  static bool const __right = true;
  static inline bitset_digit_t shift_copy(bitset_digit_t digit, unsigned int shift) { return digit >> shift; }
  static inline bitset_digit_t reverse_shift_copy(bitset_digit_t digit, unsigned int shift) { return digit << shift; }
};

/** \brief Rotate bitset.
 *
 * Rotate bitset \a shift bits in \a DIRECTION.
 *
 * \param shift The number of bits to rotate this bitset.
 * \param DIRECTION The direction into which this bitset must be rotated, can be libecc::left or libecc::right.
 * \param result The result of the rotation.
 */
template<unsigned int n>
  template<unsigned int shift, class DIRECTION>
    void
    bitset<n>::rotate(bitset<n>& result) const
    {
      LibEccDout(dc::bitsetshift|flush_cf, "Entering bitset<" << n << ">::rotate<" << shift << ", " << type_info_of<DIRECTION>().demangled_name() << ">");
      LibEccDout(dc::bitsetshift|flush_cf, "Rotate input : " <<
	  cwprint_using(*static_cast<bitset_invertible<n, false> const*>(this), &bitset<n>::base2_print_on));
      shift_op<shift % n, DIRECTION, rotate_phase1>(result);
      LibEccDout(dc::bitsetshift|flush_cf, "After phase1 : " <<
	  cwprint_using(*static_cast<bitset_invertible<n, false> const*>(&result), &bitset<n>::base2_print_on));
      shift_op<n - (shift % n), typename DIRECTION::inverse, rotate_phase2>(result);
      LibEccDout(dc::bitsetshift|flush_cf, "After phase2 : " <<
	  cwprint_using(*static_cast<bitset_invertible<n, false> const*>(&result), &bitset<n>::base2_print_on));
      LibEccDout(dc::bitsetshift|flush_cf, "Leaving bitset<" << n << ">::rotate<" << shift << ", " << type_info_of<DIRECTION>().demangled_name() << ">");
    }

} // namespace libecc

#endif // LIBECC_BITS_H
