/*
 * libevaluator - library of procedures for evaluating mathematical
 * functions.
 * 
 * Copyright (C) 2002 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef EVALUATOR_H
#define EVALUATOR_H 1

#ifdef __cplusplus
extern          "C"
{
#endif

	/*
	 * Create evaluator from string representing function.  Function
	 * returns pointer that should be passed as first argument to all
	 * other library functions.  If an error occurs, function will return
	 * null pointer.
	 */
	extern void    *evaluator_create(char *string);

	/* Destroy evaluator specified.  */
	extern void     evaluator_destroy(void *evaluator);

	/*
	 * Evaluate function represented by evaluator given.  Variable names
	 * and respective values are represented by function third and fourth
	 * argument. Number of variables i.e. length of these two arrays is
	 * given by second argument.  Function returns evaluated function
	 * value.  In case that function contains variables with names not
	 * given through third function argument, value of this variable is
	 * undeterminated.
	 */
	extern double   evaluator_evaluate(void *evaluator, int count, char **names,
					                   double *values);

	/*
	 * Write textual representation of evaluator (i.e. corresponding
	 * function) to standard output.
	 */
	extern void     evaluator_write(void *evaluator);

	/*
	 * Create evaluator for first derivative of function represented by
	 * evaluator given as first argument using derivative variable given
	 * as second argument.
	 */
	extern void    *evaluator_derivate(void *evaluator, char *name);

	/*
	 * Helper functions to simplify evaluation when variable names are
	 * "x", "x" and "y" or "x" and "y" and "z" respectively.
	 */
	extern double   evaluator_evaluate_x(void *evaluator, double x);
	extern double   evaluator_evaluate_x_y(void *evaluator, double x, double y);
	extern double   evaluator_evaluate_x_y_z(void *evaluator, double x, double y,
						                 double z);

	/*
	 * Helper functions to simplify derivation when variable names are
	 * "x" or "y" or "z" respectively.
	 */
	extern void    *evaluator_derivate_x(void *evaluator);
	extern void    *evaluator_derivate_y(void *evaluator);
	extern void    *evaluator_derivate_z(void *evaluator);

#ifdef __cplusplus
}
#endif

#endif
