/* hashmap - a chaining hash map
 * Copyright (c) 2002 Michael B. Allen <mballen@erols.com>
 *
 * The MIT License
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdlib.h>
#include <errno.h>
#include "linkedlist.h"
#include "hashmap.h"

/* This guard dissolves the msgno macros allowing the
 * module to be compiled stand alone.
 */
#ifndef MSGNO
  #define MSG(fmt, args...)
  #define MNO(msgno)
  #define MNF(msgno, fmt, args...)
  #define PMSG(fmt, args...)
  #define PMNO(msgno)
  #define PMNF(msgno, fmt, args...)
  #define AMSG(fmt, args...)
  #define AMNO(msgno)
  #define AMNF(msgno, fmt, args...)
#else
#include "msgno.h"
#endif

struct entry {
	unsigned int hash;
	void *key;
	void *data;
};
struct hashmap {
	unsigned int (*hash_fn)(const void *);
	void (*free_key_fn)(void *);
	void (*free_data_fn)(void *);
	unsigned int size;
	unsigned int iter;
	int iter_has_list;
	unsigned int table_size;
	struct linkedlist **table;
};

unsigned int
hash_string(const void *s)
{
	unsigned int h = 0;
	const char *sc = (char *)s;
	while (*sc != '\0') {
		h = 31 * h + *sc++;
	}
	return h;
}
unsigned int
hash_ptr(const void *ptr)
{
	return (unsigned int)ptr;
}

struct hashmap *
hashmap_new(const unsigned int size,
				unsigned int (*hash_fn)(const void *),
				void (*free_key_fn)(void *),
				void (*free_data_fn)(void *))
{
	struct hashmap *h;

	if ((h = malloc(sizeof *h)) == NULL) {
		PMNO(errno);
		return NULL;
	}
	h->hash_fn = hash_fn != NULL ? hash_fn : hash_ptr;
	h->free_key_fn = free_key_fn;
	h->free_data_fn = free_data_fn;
	h->size = 0;
	h->table_size = size;
	if ((h->table = calloc(size, sizeof *h->table)) == NULL) {
		PMNO(errno);
		free(h);
		return NULL;
	}
	return h;
}
void
hashmap_del(struct hashmap *h)
{
	unsigned int i;
	struct linkedlist *l;
	struct entry *e;

	if (h == NULL) {
		return;
	}
	for (i = 0; i < h->table_size; i++) {
		l = h->table[i];
		if (l) {
			linkedlist_iterate(l);
			while ((e = linkedlist_next(l)) != NULL) {
				if (h->free_key_fn) {
					h->free_key_fn(e->key);
				}
				if (h->free_data_fn) {
					h->free_data_fn(e->data);
				}
			}
			linkedlist_del(l, free);
		}
	}
	free(h->table);
	free(h);
}

int
hashmap_put(struct hashmap *h, void *key, void *data)
{
	struct entry *n, *e;
	struct linkedlist *l;
	unsigned int i;

	if (h == NULL || key == NULL || data == NULL) {
		errno = EINVAL;
		PMNF(errno, ": h=%p,key=%p,data=%p", h, key, data);
		return 0;
	}
	n = malloc(sizeof *n);
	if (n == NULL) {
		PMNO(errno);
		return 0;
	}
	n->hash = h->hash_fn(key);
	n->key = key;
	n->data = data;

	l = h->table[n->hash % h->table_size];
	if (l == NULL) {
		if ((l = linkedlist_new(h->table_size)) == NULL) {
			AMSG("max_size=%u", h->table_size);
			free(n);
			return 0;
		}
		h->table[n->hash % h->table_size] = l;
	} else {
		linkedlist_iterate(l);
		for (i = 0; (e = linkedlist_next(l)) != NULL; i++) {
			if (e->hash == n->hash) {
				linkedlist_remove(l, i);
				if (h->free_key_fn) {
					h->free_key_fn(e->key);
				}
				if (h->free_data_fn) {
					h->free_data_fn(e->data);
				}
				free(e);
				break;
			}
		}
	}

	if (linkedlist_insert(l, 0, n) == 0) {
		AMSG("key=%p,data=%p", key, data);
		free(n);
		return 0;
	}
	h->size++;
	return 1;
}
int
hashmap_is_empty(struct hashmap *h)
{
	return h == NULL || h->size == 0;
}
unsigned int
hashmap_size(struct hashmap *h)
{
	return h == NULL ? 0 : h->size;
}
void *
hashmap_get(const struct hashmap *h, const void *key)
{
	unsigned int hash;
	struct linkedlist *l;
	struct entry *e;

	if (h == NULL || key == NULL) {
		return NULL;
	}

	hash = h->hash_fn(key);
	l = h->table[hash % h->table_size];
	if (l != NULL) {
		linkedlist_iterate(l);
		while((e = linkedlist_next(l)) != NULL) {
			if (e->hash == hash) {
				return e->data;
			}
		}
	}
	return NULL;
}
void
hashmap_iterate(struct hashmap *h)
{
	if (h) {
		h->iter = 0;
		h->iter_has_list = 0;
	}
}
void *
hashmap_next(struct hashmap *h)
{
	struct linkedlist *l;
	struct entry *e;

	if (h == NULL) {
		return NULL;
	}

	while (h->iter < h->table_size) {
		l = h->table[h->iter];
		if (l != NULL) {
			if (h->iter_has_list == 0) {
				linkedlist_iterate(l);
				h->iter_has_list = 1;
			}
			if ((e = linkedlist_next(l)) != NULL) {
				return e->key;
			}
			h->iter_has_list = 0;
		}
		h->iter++;
	}
	return NULL;
}
void *
hashmap_remove(struct hashmap *h, const void *key)
{
	unsigned int hash, i;
	struct linkedlist *l;
	struct entry *e;
	void *result;

	if (h == NULL || key == NULL) {
		errno = EINVAL;
		PMNO(errno);
		return NULL;
	}

	hash = h->hash_fn(key);
	l = h->table[hash % h->table_size];
	if (l != NULL) {
		linkedlist_iterate(l);
		for (i = 0; (e = linkedlist_next(l)) != NULL; i++) {
			if (e->hash == hash) {
				if ((e = linkedlist_remove(l, i)) == NULL) {
					AMSG("");
					return NULL;
				}
				if (h->free_key_fn) {
					h->free_key_fn(e->key);
				}
				result = e->data;
				free(e);
				h->size--;
				return result;
			}
		}
	}
	return NULL;
}
