/* profile - a memory management profiler
 * Copyright (c) 2001 Michael B. Allen <mballen@erols.com>
 *
 * The MIT License
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */ 

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "hashmap.h"

#define MAX_OUTSTANDING 50

#ifndef strdup
char *strdup(const char *s);
#endif

static struct hashmap *tbl = NULL;
static size_t profile_alloc_size = 0;
static size_t profile_free_size = 0;
static unsigned int profile_alloc_calls = 0;
static unsigned int profile_free_calls = 0;
static unsigned int table_miss = 0;

struct entry {
	size_t size;
	char *file;
	unsigned int line;
};

void
profile_del(void)
{
	hashmap_del(tbl);
}

void
profile_add(const void *ptr, size_t size, const char *file, unsigned int line)
{
	struct entry *e;

	if (ptr == NULL || size == 0) {
		return;
	}

	if (tbl == NULL) {
		tbl = hashmap_new(HASHMAP_MEDIUM, NULL, NULL, NULL);
		if(tbl == NULL) {
			return;
		}
	}
	if ((e = malloc(sizeof *e)) == NULL) {
		return;
	}
	e->size = size;
	if ((e->file = strdup(file)) == NULL) {
		free(e);
		return;
	}
	e->line = line;
	if (hashmap_put(tbl, (void *)ptr, e) == 0) {
		free(e->file);
		free(e);
		return;
	}
	profile_alloc_calls++;
	profile_alloc_size += size;
}
void *
profile_malloc(size_t size, const char *file, unsigned int line)
{
	void *result;
	struct entry *e;

	if (tbl == NULL) {
		tbl = hashmap_new(HASHMAP_MEDIUM, NULL, NULL, NULL);
		if (tbl == NULL) {
			return NULL;
		}
	}
	if ((result = malloc(size)) == NULL) {
		return NULL;
	}
	if ((e = malloc(sizeof *e)) == NULL) {
		free(result);
		return NULL;
	}
	e->size = size;
	if ((e->file = strdup(file)) == NULL) {
		free(e);
		free(result);
		return NULL;
	}
	e->line = line;
	if (hashmap_put(tbl, result, e) == 0) {
		free(e->file);
		free(e);
		free(result);
		return NULL;
	}
	profile_alloc_calls++;
	profile_alloc_size += size;

	return result;
}
void *
profile_calloc(size_t nmemb, size_t size, const char *file, unsigned int line)
{
	void *result;

	size *= nmemb;
	result = profile_malloc(size, file, line);
	if (result) {
		memset(result, 0, size);
	}
	return result;
}
void
profile_free(void *ptr)
{
	struct entry *e;

	if (tbl && ptr) {
		e = hashmap_remove(tbl, ptr);
		if (e) {
			profile_free_calls++;
			profile_free_size += e->size;
			free(e->file);
			free(e);
		} else {
			table_miss++;
		}
	}
	free(ptr);
}
void *
profile_realloc(void *ptr, size_t size, const char *file, unsigned int line)
{
	void *result;

	if ((result = profile_malloc(size, file, line)) == NULL) {
		return NULL;
	}
	if (ptr) {
		memcpy(result, ptr, size);
		profile_free(ptr);
	}

	return result;
}
void
profile_report(FILE *out)
{
	int call_total;
	int size_total;
	void *key;
	struct entry *e;
	int i;

	call_total = profile_alloc_calls - profile_free_calls;
	size_total = profile_alloc_size - profile_free_size;

	fprintf(out, "\n  -- MEMORY REPORT --\n       calls      size\n");
	fprintf(out, "alloc%7u%10u\n", profile_alloc_calls, profile_alloc_size);
	fprintf(out, " free%7u%10u\n", profile_free_calls, profile_free_size);
	fprintf(out, "total%7d%10d\n", call_total, size_total);
	fprintf(out, "%u outstanding allocations\n", hashmap_size(tbl));
	fprintf(out, "%u free calls not in table\n", table_miss);

	hashmap_iterate(tbl);
	for (i = 0; ((key = hashmap_next(tbl)) != NULL); i++) {
		e = hashmap_get(tbl, key);
		if (e) {
			if (i < MAX_OUTSTANDING) {
				fprintf(out, "%s:%u: %u bytes\n", e->file, e->line, e->size);
			} else {
				fflush(out);
				fprintf(out, "more than %u outstanding allocations\n", MAX_OUTSTANDING);
				return;
			}
		}
	}
}

