/* domnode - simple XML DOM-like interface
 * Copyright (c) 2002 Michael B. Allen <mballen@erols.com>
 *
 * The MIT License
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <langinfo.h>
#include <expat.h>
#ifdef USE_ENCDEC
  #include <encdec.h>
#endif
#include "stack.h"
#include "domnode.h"

/* This guard dissolves the msgno macros allowing the
 * module to be compiled stand alone.
 */
#ifndef MSGNO
  #define MSG(fmt, args...)
  #define MNO(msgno)
  #define MNF(msgno, fmt, args...)
  #define PMSG(fmt, args...)
  #define PMNO(msgno)
  #define PMNF(msgno, fmt, args...)
  #define AMSG(fmt, args...)
  #define AMNO(msgno)
  #define AMNF(msgno, fmt, args...)
#else
  #include "msgno.h"
#endif

#ifndef PROFILE_H
  #define PROFILE_ADD(ptr, size)
#endif

#define NL "\n"

#ifndef strdup
char *strdup(const char *s);
#endif
#ifndef strnlen
size_t strnlen(const char *s, size_t maxlen);
#endif

struct user_data {
	char *buf;
	size_t siz;
	struct stack *stk;
	int err;
};

static struct domnode *
_domnode_new(const char *name, const char *value, int is_elem)
{
	struct domnode *this;

	if ((this = malloc(sizeof *this)) == NULL) {
		PMNO(errno);
		return NULL;
	}

	this->name = name == NULL ? NULL : strdup(name);
	this->value = value == NULL ? NULL : strdup(value);
	this->children = NULL;
	this->attrs = NULL;

	if (this->name) {
		PROFILE_ADD(this->name, strlen(this->name) + 1);
	}
	if (this->value) {
		PROFILE_ADD(this->value, strlen(this->value) + 1);
	}

	if (is_elem) {
		if ((this->children = linkedlist_new(0)) == NULL ||
					(this->attrs = linkedlist_new(0)) == NULL) {
			AMSG("");
			domnode_del(this);
			return NULL;
		}
	}

	return this;
}

struct domnode *
domnode_new(const char *name, const char *value)
{
	return _domnode_new(name, value, 1);
}
void
domnode_del(void *this)
{
	struct domnode *dn;

	if (this == NULL) {
		return;
	}
	dn = (struct domnode *)this;
	free((char *)dn->name);
	free((char *)dn->value);
	linkedlist_del(dn->children, domnode_del);
	linkedlist_del(dn->attrs, domnode_del);
	free(dn);
}

int
fputds(const char *s, FILE *stream)
{
	return fputs(s, stream);
}

size_t
utf8tods(const char *src, size_t sn, struct user_data *ud)
{
	size_t n;
#ifdef USE_ENCDEC
	char *s;

	s = (char *)src;
	if ((n = dec_mbsncpy(&s, sn, NULL, -1, -1, "UTF-8")) == (size_t)-1) {
		ud->err = errno;
		PMNO(errno);
		return -1;
	}
#else
	n = strnlen(src, sn) + 1;
#endif
	if (n > ud->siz) {
		ud->siz = n > (2 * ud->siz) ? n : (2 * ud->siz);
		if ((ud->buf = realloc(ud->buf, ud->siz)) == NULL) {
			ud->err = errno;
			PMNO(errno);
			return -1;
		}
	}
#ifdef USE_ENCDEC
	if ((n = dec_mbsncpy(&s, sn, ud->buf, ud->siz, -1, "UTF-8")) == (size_t)-1) {
		ud->err = errno;
		PMNO(errno);
		return -1;
	}
#else
	strncpy(ud->buf, src, n);
	ud->buf[n - 1] = '\0';
#endif
	return n;
}

static void
start_fn(void *userData, const XML_Char *name, const XML_Char **atts)
{
	struct user_data *ud = userData;
	struct domnode *parent, *child, *attr;
	char *a;
	int i;

	if (ud->err) {
		return;
	}
	if (ud == NULL || name == NULL || atts == NULL) {
		ud->err = errno = EINVAL;
		PMNO(errno);
		return;
	}

	parent = stack_peek(ud->stk);
	if (parent == NULL) {
		ud->err = errno = EIO;
		PMNO(errno);
		return;
	}
	if (utf8tods(name, -1, ud) == (size_t)-1) {
		AMSG("");
		return;
	}
	if ((child = _domnode_new(ud->buf, NULL, 1)) == NULL) {
		ud->err = errno;
		AMNO(EIO);
		return;
	}
	for (i = 0; atts[i]; i += 2) {
		if (utf8tods(atts[i], -1, ud) == (size_t)-1) {
			AMSG("");
			return;
		}
		if ((a = strdup(ud->buf)) == NULL) {
			ud->err = errno;
			PMNO(errno);
			return;
		}
		PROFILE_ADD(a, strlen(a) + 1);
		if (utf8tods(atts[i + 1], -1, ud) == (size_t)-1) {
			AMSG("");
			free(a);
			return;
		}
		if ((attr = _domnode_new(a, ud->buf, 0)) == NULL) {
			ud->err = errno;
			AMNO(EIO);
			free(a);
			return;
		}
		free(a);
		if (linkedlist_add(child->attrs, attr) == 0) {
			ud->err = errno;
			AMNO(EIO);
			return;
		}
	}
	if (linkedlist_add(parent->children, child) == 0) {
		ud->err = errno;
		domnode_del(child);
		AMNO(EIO);
		return;
	}
	if (stack_push(ud->stk, child) == 0) {
		ud->err = errno;
		linkedlist_remove_last(parent->children);
		domnode_del(child);
		AMNO(EIO);
		return;
	}
}

static void
end_fn(void *userData, const XML_Char *name)
{
	stack_pop(((struct user_data *)userData)->stk);
}
static void
_data_fn(void *userData, const char *name, const XML_Char *s, int len)
{
	struct user_data *ud = userData;
	XML_Char *str;
	struct domnode *parent, *tex;

	if (ud->err) {
		return;
	}
	if (ud == NULL || s == NULL || len == 0) {
		ud->err = errno = EINVAL;
		PMNO(errno);
		return;
	}

	str = (XML_Char *)s + len;
	while (s < str && isspace(*s)) {
		s++;
		len--;
	}
	if (s == str) {
		return;
	}
	str = (XML_Char *)s + len - 1;
	while (str > s && isspace(*str)) {
		str--;
		len--;
	}

	parent = stack_peek(ud->stk);
	if (parent == NULL) {
		ud->err = errno = EIO;
		PMNO(errno);
		return;
	}
	if (utf8tods(s, len, ud) == (size_t)-1) {
		AMSG("");
		return;
	}
	if ((tex = _domnode_new(name, ud->buf, 0)) == NULL) {
		ud->err = errno;
		AMNO(EIO);
		return;
	}
	if (linkedlist_add(parent->children, tex) == 0) {
		ud->err = errno;
		domnode_del(tex);
		AMNO(EIO);
	}
}
static void
chardata_fn(void *userData, const XML_Char *s, int len)
{
	_data_fn(userData, "#text", s, len);
}
static void
comment_fn(void *userData, const XML_Char *s)
{
	_data_fn(userData, "#comment", s, strlen(s));
}
size_t
domnode_read(struct domnode *this, FILE *stream)
{
	XML_Parser *p;
	struct user_data ud;
	size_t n;
	void *buf;
	int ret, done;
	struct domnode *dummy_node, *root;

	if (this == NULL || stream == NULL) {
		errno = EINVAL;
		PMNF(errno, ": this=%p,stream=%p", this, stream);
		return -1;
	}

	if ((p = XML_ParserCreate(NULL)) == NULL) {
		errno = EIO;
		PMNO(errno);
		return -1;
	}

	if ((dummy_node = domnode_new(NULL, NULL)) == NULL) {
		AMNO(EIO);
		XML_ParserFree(p);
		return -1;
	}

	ud.buf = NULL;
	ud.siz = 0;
	ud.stk = stack_new(500);
	ud.err = 0;

	if (ud.stk == NULL || stack_push(ud.stk, dummy_node) == 0) {
		AMNO(EIO);
		stack_del(ud.stk, NULL);
		domnode_del(dummy_node);
		XML_ParserFree(p);
		return -1;
	}

	XML_SetElementHandler(p, start_fn, end_fn);
	XML_SetCharacterDataHandler(p, chardata_fn);
	XML_SetCommentHandler(p, comment_fn);
	XML_SetUserData(p, &ud);

	ret = 0;
	for ( ;; ) {
		if ((buf = XML_GetBuffer(p, BUFSIZ)) == NULL) {
			errno = EIO;
			PMNF(errno, ": buf=NULL");
			ret = -1;
			break;
		}
		if ((n = fread(buf, 1, BUFSIZ, stream)) == 0 && ferror(stream)) {
			errno = EIO;
			PMNO(errno);
			ret = -1;
			break;
		}
		ret += n;
		if (XML_ParseBuffer(p, n, (done = feof(stream))) == 0 || ud.err) {
			if (ud.err == 0) {
				errno = EIO;
				PMNF(errno, ": %s: line %u",
							XML_ErrorString(XML_GetErrorCode(p)),
							XML_GetCurrentLineNumber(p));
			} else {
				AMNO(EIO);
			}
			ret = -1;
			break;
		}
		if (done) {
			break;
		}
	}

	free(ud.buf);
	stack_del(ud.stk, NULL);
	XML_ParserFree(p);

	root = linkedlist_remove(dummy_node->children, 0);
	domnode_del(dummy_node);

	if (root) {
		free((char *)this->name);
		free((char *)this->value);
		linkedlist_del(this->children, domnode_del);
		linkedlist_del(this->attrs, domnode_del);

		this->name = root->name;
		this->value = NULL;
		this->children = root->children;
		this->attrs = root->attrs;

		root->name = NULL;
		root->value = NULL;
		root->children = NULL;
		root->attrs = NULL;
		free(root);
	}

	return ret;
}
int
domnode_load(struct domnode *this, const char *filename)
{
	FILE *fd;
	int ret;

	if (this == NULL || filename == NULL) {
		errno = EINVAL;
		PMNF(errno, ": this=%p,filename=%s", this, filename);
		return 0;
	}

	fd = fopen(filename, "r");
	if (fd == NULL) {
		PMNF(errno, ": %s", filename);
		return 0;
	}

	ret = domnode_read(this, fd) != (size_t)-1;

	fclose(fd);

	return ret;
}

static size_t
_domnode_write(struct domnode *this, FILE *stream, int indent)
{
	struct domnode *node;
	size_t ret = 0;
	int i;

    if (this == NULL || stream == NULL) {
		errno = EINVAL;
		PMNF(errno, ": this=%p,stream=%p", this, stream);
		return -1;
    }

	if (errno) {
		AMNO(EIO);
		return -1;
	}

	for (i = 0; i < indent; i++) {
		fprintf(stream, "    ");
	}

	if (strcmp(this->name, "#comment") == 0) {
		fputs("<!--", stream);
		fputds(this->value, stream);
		fputs("-->", stream);
	} else if (strcmp(this->name, "#text") == 0) {
		fputds(this->value, stream);
	} else {
		fputc('<', stream);
		fputds(this->name, stream);
		linkedlist_iterate(this->attrs);
		while ((node = linkedlist_next(this->attrs)) != NULL) {
			fputc(' ', stream);
			fputds(node->name, stream);
			fputs("=\"", stream);
			fputds(node->value, stream);
			fputc('"', stream);
		}
		if (linkedlist_is_empty(this->children) == 0) {
			fputc('>', stream);
			linkedlist_iterate(this->children);
			while ((node = linkedlist_next(this->children)) != NULL) {
				if (_domnode_write(node, stream, indent + 1) == (size_t)-1) {
					return -1;
				}
			}
			for (i = 0; i < indent; i++) {
				fprintf(stream, "    ");
			}
			fputs("</", stream);
			fputds(this->name, stream);
			fputc('>', stream);
		} else {
			fputs("/>", stream);
		}
	}

	return ret;
}
size_t
domnode_write(struct domnode *this, FILE *stream)
{
	size_t n;

	fputs("<?xml version=\"1.0\" encoding=\"", stream);
	fputs(nl_langinfo(CODESET), stream);
	fputs("\"?>"NL NL, stream);
	if ((n = _domnode_write(this, stream, 0)) == (size_t)-1) {
		AMSG("");
		return -1;
	}
	fputs(NL, stream);
	return n;
}
int
domnode_store(struct domnode *this, const char *filename)
{
	FILE *fd;
	int ret;

	if (this == NULL || filename == NULL) {
		errno = EINVAL;
		PMNF(errno, ": this=%p,filename=%s", this, filename);
		return 0;
	}

	fd = fopen(filename, "w");
	if (fd) {
		ret = domnode_write(this, fd);
		fclose(fd);
		return ret > 0;
	}
	PMNF(errno, ": %s", filename);

	return 0;
}

struct domnode *
domnode_attrs_get(struct linkedlist *attrs, const char *name)
{
	struct domnode *node;

	if (attrs && name && *name) {
		linkedlist_iterate(attrs);
		while ((node = linkedlist_next(attrs))) {
			if (strcoll(node->name, name) == 0) {
				return node;
			}
		}
	}

	return NULL;
}
struct domnode *
domnode_search(struct domnode *this, const char *name)
{
	struct domnode *node;

	linkedlist_iterate(this->children);
	while ((node = linkedlist_next(this->children))) {
		if (strcoll(node->name, name) == 0) {
			return node;
		}
	}
	linkedlist_iterate(this->attrs);
	while ((node = linkedlist_next(this->attrs))) {
		if (strcoll(node->name, name) == 0) {
			return node;
		}
	}
	linkedlist_iterate(this->children);
	while ((node = linkedlist_next(this->children))) {
		if ((node = domnode_search(node, name)) != NULL) {
			return node;
		}
	}

	return NULL;
}
struct domnode *
domnode_attrs_remove(struct linkedlist *attrs, const char *name)
{
	struct domnode *node;
	unsigned int idx;

	if (attrs && name && *name) {
		linkedlist_iterate(attrs);
		for (idx = 0; (node = linkedlist_next(attrs)); idx++) {
			if (strcoll(node->name, name) == 0) {
				return linkedlist_remove(attrs, idx);
			}
		}
	}

	return NULL;
}
int
domnode_attrs_put(struct linkedlist *attrs, struct domnode *attr)
{
	struct domnode *node;
	unsigned int idx;

	if (attrs == NULL || attr == NULL || attr->value == NULL) {
		errno = EINVAL;
		PMNF(errno, ": attrs=%p,attr=%p", attrs, attr);
		return 0;
	}

	linkedlist_iterate(attrs);
	for (idx = 0; (node = linkedlist_next(attrs)); idx++) {
		if (strcoll(node->name, attr->name) == 0) {
			domnode_del(linkedlist_remove(attrs, idx));
			return linkedlist_insert(attrs, idx, attr);
		}
	}
	return linkedlist_add(attrs, attr);
}

