/* mbs - locale dependent multi-byte string functions
 * Copyright (c) 2002 Michael B. Allen <mballen@erols.com>
 *
 * The MIT License
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include <wchar.h>
#include <limits.h>
#include <string.h>
#include <errno.h>
#include "mbs.h"

/* This guard dissolves the msgno macros allowing the
 * module to be compiled stand alone.
 */
#ifndef MSGNO
  #define MSG(fmt, args...)
  #define MNO(msgno)
  #define MNF(msgno, fmt, args...)
  #define PMSG(fmt, args...)
  #define PMNO(msgno)
  #define PMNF(msgno, fmt, args...)
  #define AMSG(fmt, args...)
  #define AMNO(msgno)
  #define AMNF(msgno, fmt, args...)
#else
#include "msgno.h"
#endif

int wcwidth(wchar_t ucs);

int
mbslen(const char *src)
{
	return mbsnlen(src, -1, -1);
}
int
mbsnlen(const char *src, size_t sn, int cn)
{
	wchar_t ucs;
	int count, w;
	size_t n;
	mbstate_t ps;

	ucs = 1;
	count = 0;
	memset(&ps, 0, sizeof(ps));

	if (sn > INT_MAX) {
		sn = INT_MAX;
	}
	if (cn < 0) {
		cn = INT_MAX;
	}

	while (ucs && (n = mbrtowc(&ucs, src, sn, &ps)) != (size_t)-2) {
		if (n == (size_t)-1) {
			PMNO(errno);
			return -1;
		}
		if ((w = wcwidth(ucs)) == -1) {
			w = 1;
		}
		if (w > cn) {
			break;
		}
		cn -= w;
		sn -= n;
		src += n;
		count += w;
	}

	return count;
}

size_t
mbssize(const char *src)
{
	return mbsnsize(src, -1, -1);
}
size_t
mbsnsize(const char *src, size_t sn, int cn)
{
	wchar_t ucs;
	int w;
	size_t tot, n;
	mbstate_t ps;

	tot = n = 0;
	ucs = 1;
	memset(&ps, 0, sizeof(ps));

	if (sn > INT_MAX) {
		sn = INT_MAX;
	}
	if (cn < 0) {
		cn = INT_MAX;
	}

	while (ucs && (n = mbrtowc(&ucs, src, sn, &ps)) != (size_t)-2 && n) {
		if (n == (size_t)-1) {
			PMNO(errno);
			return -1;
		}
		if ((w = wcwidth(ucs)) == -1) {
			w = 1;
		}
		if (w > cn) {
			break;
		}
		cn -= w;
		sn -= n;
		src += n;
		tot += n;
	}

	return tot;
}

char *
mbsdup(const char *src)
{
	return mbsndup(src, -1, -1);
}
char *
mbsndup(const char *src, size_t sn, int cn)
{
	size_t n;
	char *dst;

	if (src == NULL) {
		errno = EINVAL;
		PMNO(errno);
		return NULL;
	}
	if ((n = mbsnsize(src, sn, cn)) == (size_t)-1) {
		AMSG("");
		return NULL;
	}
	if ((dst = malloc(n + 1)) == NULL) {
		PMNO(errno);
		return NULL;
	}
	memcpy(dst, src, n);
	dst[n] = '\0';

	return dst;
}

char *
mbsoff(char *src, int off)
{
	return mbsnoff(src, off, -1);
}
char *
mbsnoff(char *src, int off, size_t sn)
{
	wchar_t ucs;
	int w;
	size_t n;
	mbstate_t ps;

	memset(&ps, 0, sizeof(ps));

	if (sn > INT_MAX) {
		sn = INT_MAX;
	}
	if (off < 0) {
		off = INT_MAX;
	}

	while (sn > 0 && (n = mbrtowc(&ucs, src, sn, &ps)) != (size_t)-2) {
		if (n == (size_t)-1) {
			PMNO(errno);
			return NULL;
		}
		if (n == 0 || (w = wcwidth(ucs)) == -1) {
			w = 1;
		}
		if (w > off) {
			break;
		}
		off -= w;
		sn -= n;
		src += n;
	}

	return src;
}

