/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#ifndef _SMSG_H_
#define _SMSG_H_

#ifdef __VXWORKS_OS__
#include <string.h>
#else

#if defined (HAVE_CONFIG_H)
#include <config.h>
#  if defined (HAVE_STRING_H)
#    include <string.h>
#  else
#    include <strings.h>
#  endif /* HAVE_STRING_H */
#else
#  include <string.h>
#endif /* !HAVE_CONFIG_H */

#endif

/* for types */
#include <osip/smsgtypes.h>


/* parse the message. */
/* return -1 on error */
int   msg_parse(sip_t *dest , char *message, err_t *err);
/* allocate the sip_t structure.               */
/* return -1 on error */
int   msg_init(sip_t **sip, err_t *err);
/* deallocate the sip_t structure.              */
void  msg_free(sip_t *sip);
/* Add a header in a SIP message. (for "unknown" headers */
/* that do not appear in the sip_t structure             */
/* INPUT : char *hname | pointer on a header name.       */
/* INPUT : char *hvalue | pointer on a header value.     */
/* OUTPUT: sip_t *sip | structure to save results.       */
/* return -1 on error. */
int   msg_setheader(sip_t *sip, char *hname,
		    char *hvalue, err_t *err);
/* Get a header in a SIP message.                     */
/* INPUT : int pos | position of number in message.   */
/* OUTPUT: sip_t *sip | structure to look for header .*/
/* return null on error. */
int
msg_getheader(sip_t *sip, int pos, header_t **dest);
/* Get a specific header in a SIP message.              */
/* INPUT : int pos | position where we start the search */
/* OUTPUT: sip_t *sip | structure to look for header.   */
/* return the position where the the header is found    */
/* and -1 on error. */
int
msg_getfindheader(char *hname, sip_t *sip, int pos, header_t **dest);

int   msg_setcall_id(sip_t *sip, char *hvalue, err_t *err);
call_id_t*     msg_getcall_id(sip_t *sip);
int   msg_setcontact(sip_t *sip, char *hvalue, err_t *err);
int   msg_getcontact(sip_t *sip, int pos, contact_t **contact);
int   msg_setcseq   (sip_t  *sip, char *hvalue, err_t *err);
cseq_t* msg_getcseq   (sip_t  *sip);
int   msg_setfrom(sip_t *sip, char *hvalue, err_t *err);
from_t* msg_getfrom(sip_t *sip);
int   msg_setrecord_route(sip_t *sip, char *hvalue, err_t *err);
int   msg_getrecord_route(sip_t *sip, int pos, record_route_t **record_route);
int   msg_setroute(sip_t *sip, char *hvalue, err_t *err);
int   msg_getroute(sip_t *sip, int pos, route_t **route);
int   msg_setto(sip_t *sip, char *hvalue, err_t *err);
to_t* msg_getto(sip_t *sip);
int   msg_setvia(sip_t *sip, char *hvalue, err_t *err);
int   msg_getvia(sip_t *sip,int pos, via_t **dest);
int   msg_setcontent_length(sip_t *sip, char *hvalue, err_t *);
content_length_t *msg_getcontent_length(sip_t *sip);
int   msg_setcontent_type(sip_t *sip, char *hvalue,err_t *err);
content_type_t * msg_getcontent_type(sip_t *sip);
int   msg_setmime_version(sip_t *sip, char *hvalue, err_t *err);
mime_version_t * msg_getmime_version(sip_t *sip);
#define msg_setcontent_encoding(C, S, E) msg_setcontent_length(C, S, E) 
#define msg_getcontent_encoding(C)       msg_getcontent_length(C)

/* fill the body of message.                              */
/* INPUT : char *buf | pointer on begenning of body. */
/* OUTPUT: sip_t *sip | structures to save results.        */
/* return -1 on error. */
int   msg_setbody(sip_t *sip, char *buf, err_t *err);
int   msg_setbody_mime(sip_t *sip,  char *buf, err_t *err);
/* returns the pos of body header.                   */
/* INPUT : int pos | pos of body header.             */
/* INPUT : sip_t *sip | sip message.                 */
/* OUTPUT: body_t **body | structure to save results. */
/* returns -1 on error. */
int   msg_getbody(sip_t *sip, int pos, body_t **dest);



/* return the message as a string.           */
/* INPUT : sip_t *sip | message to be done.  */
/* return null on error. */
int     msg_2char(sip_t *sip, char **dest, err_t *err);
#ifdef ENABLE_TRACE
void    msg_logrequest(sip_t *sip,char *fmt);
void    msg_logresponse(sip_t *sip,char *fmt);
#else 
#define msg_logrequest(P,Q) ;
#define msg_logresponse(P,Q) ;
#endif

/* a few MACROS to ease readability of code */
#define MSG_IS_RESPONSE(resp)     (resp->strtline->statuscode!=NULL)
#define MSG_IS_REQUEST(req)       (req->strtline->statuscode==NULL)

#define MSG_IS_INVITE(msg)  (0==strncmp(msg->strtline->sipmethod,"INVITE",6))
#define MSG_IS_ACK(msg)     (0==strncmp(msg->strtline->sipmethod,"ACK",6))
#define MSG_IS_BYE(msg)     (0==strncmp(msg->strtline->sipmethod,"BYE",6))
#define MSG_IS_REGISTER(msg) (0==strncmp(msg->strtline->sipmethod,"REGISTER",6))
#define MSG_IS_CANCEL(msg)  (0==strncmp(msg->strtline->sipmethod,"CANCEL",6))
#define MSG_IS_OPTIONS(msg) (0==strncmp(msg->strtline->sipmethod,"OPTIONS",6))
#define MSG_IS_INFO(msg)    (0==strncmp(msg->strtline->sipmethod,"INFO",6))
#define MSG_IS_PRACK(msg)   (0==strncmp(msg->strtline->sipmethod,"PRACK",6))

#define MSG_IS_STATUS_1XX(msg)    (0==strncmp(msg->strtline->statuscode,"1",1))
#define MSG_IS_STATUS_2XX(msg)    (0==strncmp(msg->strtline->statuscode,"2",1))
#define MSG_IS_STATUS_3XX(msg)    (0==strncmp(msg->strtline->statuscode,"3",1))
#define MSG_IS_STATUS_4XX(msg)    (0==strncmp(msg->strtline->statuscode,"4",1))
#define MSG_IS_STATUS_5XX(msg)    (0==strncmp(msg->strtline->statuscode,"5",1))
#define MSG_IS_STATUS_6XX(msg)    (0==strncmp(msg->strtline->statuscode,"6",1))
#define MSG_TEST_CODE(resp, code) (resp->strtline->statuscode!=NULL \
				   && code==(int)satoi(resp->strtline->statuscode))
#define MSG_IS_RESPONSEFOR(resp,requestname) \
                                  (0==strcmp(resp->cseq->method,requestname))

/* Those methods are internal only */
int find_next_crlf(char *start_of_header,
		   char **end_of_header, err_t *err);
int find_next_crlfcrlf(char *start_of_part,
		       char **end_of_part, err_t *err);
int find_next_occurence(char *str, char *buf,
			char **index_of_str, err_t *err);


/* You can replace this modules if you intend  */
/* to use a private search algorythm.          */
/* This should be improved to a dynamic        */
/* configuration of the parser.                */
#define NUMBER_OF_HEADERS 11
/* MUST be called once before the parser can be used */
int   parser_configure();
int   parser_callmethod(int i, sip_t *dest, char *hvalue, err_t *err);
int   parser_isknownheader(char *hname);

/* return the reason.                    */
/* INPOUT : int replycode | given code.   */
/* return NULL on error. */
char *parser_getreason(int replycode);

int     body_init(body_t **body, err_t *err);
int     body_parse(body_t *body, char *buf, err_t *err);
int     body_parse_mime(body_t *body, char *start_of_body, err_t *err);
/* return the body as a string.          */
/* INPUT : body_t *body | body.  */
/* return null on error. */
int     body_2char(body_t *body, char **dest, err_t *err);
/* deallocate a body structure.  */
/* INPUT : body_t *body | body.  */
void    body_free(body_t *body);



int   generic_param_init    (generic_param_t **gen_param, err_t *err);
void  generic_param_free    (generic_param_t *gen_param);
void  generic_param_freelist(list_t *params);
int   generic_parse_params  (list_t *gen_params, char *params, err_t *err);
int   generic_param_set     (generic_param_t *gen_param, char *pname,
			     char *pvalue, err_t *err);
int   generic_param_add     (list_t *gen_params, generic_param_t *gen_param,
			     err_t *err);
char *generic_param_getname (generic_param_t *fparam);
char *generic_param_getvalue(generic_param_t *fparam);
generic_param_t *generic_param_findname(char *name, list_t *from_params);



int     header_init    (header_t **header, err_t *err);
void    header_free    (header_t *header);
int     header_2char   (header_t *header, char **dest, err_t *err);
char   *header_getname (header_t *header);
char   *header_getvalue(header_t *header);


int     call_id_init(call_id_t **callid, err_t *err);
void    call_id_free(call_id_t *callid);
int     call_id_parse(call_id_t *callid, char *hvalue, err_t *err);
int     call_id_2char(call_id_t *callid, char **dest, err_t *err);
char*   call_id_getnumber(call_id_t *callid);
char*   call_id_gethost(call_id_t *callid);
void    call_id_setnumber(call_id_t *callid, char *number);
void    call_id_sethost(call_id_t *callid, char *host);
int     call_id_clone(call_id_t *callid, call_id_t **dest, err_t *err);

int     contact_init(contact_t **contact, err_t *err);
void    contact_free(contact_t *contact);
int     contact_parse(contact_t *contact, char *hvalue, err_t *err);
int     contact_2char(contact_t *contact, char **dest, err_t *err);
#define contact_getdisplayname(C) from_getdisplayname((from_t*)C)
#define contact_geturl(C)         from_geturl((from_t*)C)
#define contact_getparam(C)       from_getparam((from_t*)C)
#define contact_findparam(N, C)   generic_param_findname(N, C->from_params)

int     cseq_init(cseq_t **cseq, err_t *err);
void    cseq_free     (cseq_t *cseq );
int     cseq_parse    (cseq_t *cseq, char *hvalue, err_t *err);
int     cseq_2char    (cseq_t *header, char **dest, err_t *err);
char*   cseq_getnumber(cseq_t *cseq);
char*   cseq_getmethod(cseq_t *cseq);
void    cseq_setnumber(cseq_t *cseq, char *number);
void    cseq_setmethod(cseq_t *cseq, char *method);


int     from_init(from_t **from, err_t *err);
void    from_free(from_t *from);
int     from_parse(from_t *from, char *string, err_t *err);
int     from_2char(from_t *from, char **dest, err_t *err);
char   *from_getdisplayname(from_t *from);
url_t  *from_geturl(from_t *from);
int     from_getparam(from_t *from, int pos, generic_param_t **fparam);
#define from_gettag(F)       generic_param_findname("tag", F->from_params)
#define from_findparam(N, F) generic_param_findname(N, F->from_params)
#define from_add_generic_param(F,FP, E) generic_param_add(F->from_params, FP, E)

/* Compare the username, host and tag part of the two froms */
int     from_compare(from_t *from1, from_t *from2);


int     record_route_init(record_route_t **rr, err_t *err);
void    record_route_free(record_route_t *rr);
int     record_route_parse(record_route_t *rr, char *hvalue, err_t *err);
int     record_route_2char(record_route_t *rr, char **dest, err_t *err);
#define record_route_geturl(R)        from_geturl((from_t*)F)
#define record_route_getparam(R,I,FP) from_getparam((from_t*)F, I, FP)
#define record_route_findparam(R, F)  generic_param_findname(N, R->from_params)

int     route_init(route_t **rr, err_t *err);
void    route_free(route_t *rr);
int     route_parse(route_t *rr, char *hvalue, err_t *err);
int     route_2char(route_t *rr, char **dest, err_t *err);
#define route_geturl(R)        from_geturl((from_t*)F)
#define route_getparam(R,I,FP) from_getparam((from_t*)F, I, FP)
#define route_findparam(R, F)  generic_param_findname(N, R->from_params)

int     to_init(to_t **to, err_t *err);
void    to_free(to_t *to);
int     to_parse(to_t *to, char *hvalue, err_t *err);
int     to_2char(to_t *to, char **dest, err_t *err);
#define to_getdisplayname(F) from_getdisplayname((from_t*)F)
#define to_geturl(F)         from_geturl((from_t*)F)
#define to_getparam(F,I,FP)  from_getparam((from_t*)F, I, FP)
#define to_gettag(P)         generic_param_findname("tag", P->from_params)
#define to_findparam(N, F)   generic_param_findname(N, F->from_params)
#define to_add_generic_param(F, FP, E) generic_param_add(F->from_params, FP, E)

/* Compare the username, host and tag part of the two froms */
#define to_compare(T1, T2) from_compare((from_t *)F1, (from_t *)F2)


int     via_init(via_t **via,err_t *err);
void    via_free(via_t *via);
int     via_parse (via_t *via, char *hvalue, err_t *err);
int     via_2char(via_t *head, char **dest, err_t *err);
void    via_setversion(via_t *via, char *version);
char   *via_getversion(via_t *via);
void    via_setprotocol(via_t *via, char *protocol);
char   *via_getprotocol(via_t *via);
void    via_sethost(via_t *via, char *host);
char   *via_gethost(via_t *via);
void    via_setport(via_t *via, char *port);
char   *via_getport(via_t *via);
void    via_setcomment(via_t *via, char *comment);
char   *via_getcomment(via_t *via);

#define via_getparam(V,I,GP)  generic_getparam(V, I, GP)
#define via_findparam(N, F)   generic_param_findname(N, F->via_params)
#define via_add_generic_param(F, FP, E) generic_param_add(F->via_params, FP, E)


int     content_length_init(content_length_t **cl, err_t *err);
void    content_length_free(content_length_t *content_length);
int     content_length_parse(content_length_t *contentlength,
			       char *hvalue, err_t *err);
int     content_length_2char(content_length_t *content_length
			       , char **dest, err_t *err);



int     content_type_init(content_type_t **content_type, err_t *err);
void    content_type_free(content_type_t *content_type);
int     content_type_parse(content_type_t *content_type,
			   char *hvalue, err_t *err);
int     content_type_2char(content_type_t *content_type,
			     char **dest, err_t *err);
/* add a content_type parameter in the sip_t structure                */
/* INPUT: content_type_t *content_type  | content_type to be updated. */
/* OUTPUT: generic_param_t *content_type_param | param to store.      */
/* OUTPUT: err_t *err | structure to store error.                     */
/* return -1 on error */
#define content_type_add_generic_param(F, FP, E) generic_param_add(F->content_type_params, FP, E)


#define mime_version_init(C, E)      content_length_init(C, E)
#define mime_version_parse(C, S, E)  content_length_parse(C, S, E)
#define mime_version_2char(C, D, E)  content_length_2char(C, D, E)
#define mime_version_free(C)         content_length_free(C)

#define content_encoding_init(C, E)      content_length_init(C, E)
#define content_encoding_parse(C, S, E)  content_length_parse(C, S, E)
#define content_encoding_2char(C, D, E)  content_length_2char(C, D, E)
#define content_encoding_free(C)         content_length_free(C)


#define msg_setaccept(P,R,E)          msg_setheader((sip_t *)P,(char *)"accept",R,E)
#define msg_getaccept(P,Q,R)          msg_getnheader(( sip_t *)P,"accept",Q,(header_t *)R)
#define msg_setaccept_encoding(P,R,E) msg_setheader((sip_t *)P,(char *)"accept-encoding",R,E)
#define msg_getaccept_encoding(P,Q,R) msg_getnheader(( sip_t *)P, "accept-encoding",Q,(header_t *)R)
#define msg_setaccept_language(P,R,E) msg_setheader((sip_t *)P, (char *)"accept-language",R,E)
#define msg_getaccept_language(P,Q,R) msg_getnheader(( sip_t *)P,"accept-language",Q,(header_t *)R)
#define msg_setcall_info(P,R,E)       msg_setheader((sip_t *)P,(char *)"call-info",R,E)
#define msg_getcall_info(P,Q,R)       msg_getnheader(( sip_t *)P,"call-info",Q,(header_t *)R)
#define msg_setdate(P,R,E)            msg_setheader((sip_t *)P,(char *)"date",R,E)
#define msg_getdate(P,Q,R)            msg_getnheader(( sip_t *)P,"date",Q,(header_t *)R)
#define msg_setencryption(P,R,E)      msg_setheader((sip_t *)P,(char *)"encryption",R,E)
#define msg_getencryption(P,Q,R)      msg_getnheader(( sip_t *)P,"encryption",Q,(header_t *)R)
/* #define msg_setmime_version(P,R,E)    msg_setheader((sip_t *)P,(char *)"mime-version",R,E)
   #define msg_getmime_version(P,Q,R)    msg_getnheader(( sip_t *)P,"mime-version",Q,(header_t *)R) */
#define msg_setorganization(P,R,E)    msg_setheader((sip_t *)P,(char *)"organization",R,E)
#define msg_getorganization(P,Q,R)    msg_getnheader(( sip_t *)P,"organization",Q,(header_t *)R)
#define msg_setrequire(P,R,E)         msg_setheader((sip_t *)P,(char *)"require",R,E)
#define msg_getrequire(P,Q,R)         msg_getnheader(( sip_t *)P,"require",Q,(header_t *)R)
#define msg_setsupported(P,R,E)       msg_setheader((sip_t *)P,(char *)"supported",R,E)
#define msg_getsupported(P,Q,R)       msg_getnheader(( sip_t *)P,"supported",Q,(header_t *)R)
#define msg_settimestamp(P,R,E)       msg_setheader((sip_t *)P,(char *)"timestamp",R,E)
#define msg_gettimestamp(P,Q,R)       msg_getnheader(( sip_t *)P,"timestamp",Q,(header_t *)R)
#define msg_setuser_agent(P,R,E)      msg_setheader((sip_t *)P,(char *)"user-agent",R,E)
#define msg_getuser_agent(P,Q,R)      msg_getnheader(( sip_t *)P,"user-agent",Q,(header_t *)R)
#define msg_setallow(P,R,E)           msg_setheader((sip_t *)P,(char *)"allow",R,E)
#define msg_getallow(P,Q,R)           msg_getnheader(( sip_t *)P,"allow",Q,(header_t *)R)
#define msg_setcontent_disposition(P,R,E) msg_setheader((sip_t *)P,(char *)"content-disposition",R,E)
#define msg_getcontent_disposition(P,Q,R) msg_getnheader(( sip_t *)P,"content-disposition",Q,(header_t *)R)
/* #define msg_setcontent_encoding(P,R,E) msg_setheader((sip_t *)P,(char *)"content-encoding",R,E)
   #define msg_getcontent_encoding(P,Q,R) msg_getnheader(( sip_t *)P,"content-encoding",Q,(header_t *)R) */
#define msg_setcontent_language(P,R,E) msg_setheader((sip_t *)P,(char *)"content-language",R,E)
#define msg_getcontent_language(P,Q,R) msg_getnheader(( sip_t *)P,"content-language",Q,(header_t *)R)
#define msg_setexpires(P,R,E)         msg_setheader((sip_t *)P,(char *)"expires",R,E)
#define msg_getexpires(P,Q,R)         msg_getnheader(( sip_t *)P,"expires",Q,(header_t *)R)
#define msg_setalert_info(P,R,E)      msg_setheader((sip_t *)P,(char *)"alert-info",R,E)
#define msg_getalert_info(P,Q,R)      msg_getnheader(( sip_t *)P,"alert-info",Q,(header_t *)R)
#define msg_setauthorization(P,R,E)   msg_setheader((sip_t *)P,(char *)"authorization",R,E)
#define msg_getauthorization(P,Q,R)   msg_getnheader(( sip_t *)P,"authorization",Q,(header_t *)R)
#define msg_setin_reply_to(P,R,E)     msg_setheader((sip_t *)P,(char *)"in-reply-to",R,E)
#define msg_getin_reply_to(P,Q,R)     msg_getnheader(( sip_t *)P,"in-reply-to",Q,(header_t *)R)
#define msg_setmax_forward(P,R,E)     msg_setheader((sip_t *)P,(char *)"max-forward",R,E)
#define msg_getmax_forward(P,Q,R)     msg_getnheader(( sip_t *)P,"max-forward",Q,(header_t *)R)
#define msg_setpriority(P,R,E)        msg_setheader((sip_t *)P,(char *)"priority",R,E)
#define msg_getpriority(P,Q,R)        msg_getnheader(( sip_t *)P,"priority",Q,(header_t *)R)
#define msg_setproxy_authorization(P,R,E) msg_setheader((sip_t *)P,(char *)"proxy-authorization",R,E)
#define msg_getproxy_authorization(P,Q,R) msg_getnheader(( sip_t *)P,"proxy-authorization",Q,(header_t *)R)
#define msg_setproxy_require(P,R,E)   msg_setheader((sip_t *)P,(char *)"proxy-require",R,E)
#define msg_getproxy_require(P,Q,R)   msg_getnheader(( sip_t *)P,"proxy-require",Q,(header_t *)R)
#define msg_setresponse_key(P,R,E)    msg_setheader((sip_t *)P,(char *)"response-key",R,E)
#define msg_getresponse_key(P,Q,R)    msg_getnheader(( sip_t *)P,"response-key",Q,(header_t *)R)
#define msg_setsubject(P,R,E)         msg_setheader((sip_t *)P,(char *)"subject",R,E)
#define msg_getsubject(P,Q,R)         msg_getnheader(( sip_t *)P,"subject",Q,(header_t *)R)
#define msg_seterror_info(P,R,E)      msg_setheader((sip_t *)P,(char *)"error-info",R,E)
#define msg_geterror_info(P,Q,R)      msg_getnheader(( sip_t *)P,"error-info",Q,(header_t *)R)
#define msg_setproxy_authenticate(P,R,E) msg_setheader((sip_t *)P,(char *)"proxy-authenticate",R,E)
#define msg_getproxy_authenticate(P,Q,R) msg_getnheader(( sip_t *)P,"proxy-authenticate",Q,(header_t *)R)
#define msg_setretry_after(P,R,E)     msg_setheader((sip_t *)P,(char *)"retry-after",R,E)
#define msg_getretry_after(P,Q,R)     msg_getnheader(( sip_t *)P,"retry-after",Q,(header_t *)R)
#define msg_setserver(P,R,E)          msg_setheader((sip_t *)P,(char *)"server",R,E)
#define msg_getserver(P,Q,R)          msg_getnheader(( sip_t *)P,"server",Q,(header_t *)R)
#define msg_setunsupported(P,R,E)     msg_setheader((sip_t *)P,(char *)"unsupported",R,E)
#define msg_getunsupported(P,Q,R)     msg_getnheader(( sip_t *)P,"unsupported",Q,(header_t *)R)
#define msg_setwarning(P,R,E)         msg_setheader((sip_t *)P,(char *)"warning",R,E)
#define msg_getwarning(P,Q,R)         msg_getnheader(( sip_t *)P,"warning",Q,(header_t *)R)
#define msg_setwww_authenticate(P,R,E) msg_setheader((sip_t *)P,(char *)"www-authenticate",R,E)
#define msg_getwww_authenticate(P,Q,R) msg_getnheader(( sip_t *)P,"www-authenticate",Q,(header_t *)R)


#endif


