/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef _URLS_H_
#define _URLS_H_

/* Definition of the SIP-URL
   SIP-URL         = "sip:" [ userinfo "@" ] hostport
   url-parameters [ headers ]

*/

typedef enum {
  SIP,
  TEL
}urischeme_t;


/*
  url-parameters  = *( ";" url-parameter )
  url-parameter   = transport-param | user-param | method-param
                  | ttl-param | maddr-param | other-param
  transport-param = "transport=" ( "udp" | "tcp" )
  ttl-param       = "ttl=" ttl
  ttl             = 1*3DIGIT       ; 0 to 255
  maddr-param     = "maddr=" host
  user-param      = "user=" ( "phone" | "ip" )
  method-param    = "method=" Method
  tag-param       = "tag=" UUID
  UUID            = 1*( hex | "-" )
  other-param     = ( token | ( token "=" ( token | quoted-string )))
  token           =  1*(alphanum | "-" | "." | "!" | "%" | "*"
                     | "_" | "+" | "`" | "'" | "~" )
*/
typedef struct _url_param_t {
  char *pname;
  char *pvalue;
} url_param_t;

/*
  headers         = "?" header *( "&" header )
  header          = hname "=" hvalue
  hname           = 1*uric
  hvalue          = *uric
  uric            = reserved | unreserved | escaped
  reserved        = ";" | "/" | "?" | ":" | "@" | "&" | "=" | "+" |
                    "$" | ","
*/
typedef struct _url_header_t {
  char *hname;
  char *hvalue;
} url_header_t;

typedef struct _sipurl_t {
  char *scheme;

  /*
    userinfo        = [ user | telephone-subscriber ] [ ":" password ]
    user            = *( unreserved | escaped
    | "&" | "=" | "+" | "$" | "," | ";" | "?" | "/" )
  */
  char *username;

  /*
    password        = *( unreserved | escaped  | "&" | "=" | "+" | "$" | "," )
  */
  char *password;

  /*
    hostport        = host [ ":" port ]
    host            = hostname | IPv4address | IPv6reference
    hostname        = *( domainlabel "." ) toplabel [ "." ]
    domainlabel     = alphanum | alphanum *( alphanum | "-" ) alphanum
    toplabel        = alpha | alpha *( alphanum | "-" ) alphanum
    IPv4address     = 1*3DIGIT "." 1*3DIGIT "." 1*3DIGIT "." 1*3DIGIT
    IPv6reference   = "[" IPv6address "]"
    IPv6address     = hexpart [ ":" IPv4address ]
    hexpart         = hexseq | hexseq "::" [ hexseq ] | "::" [ hexseq ]
    hexseq          = hex4 *( ":" hex4)
    hex4            = 1*4HEX
    port            = 1*DIGIT
  */
  char *host;
  char *port;

  list_t *url_params;

  list_t *url_headers;
} url_t ;

int     url_init(url_t **url,err_t *err);
void    url_free(url_t *url);
int     url_parse(url_t *url, char *buf,err_t *err);
int     url_parse_headers(url_t *url, char *headers, err_t *err);
int     url_parse_params (url_t *url, char *params , err_t *err);
int     url_2char(url_t *url,char **dest,err_t *err);

int     url_add_header(url_t *url, char *hname, char *hvalue, err_t *err);
int     url_find_param(list_t *params, char *pname, url_param_t **url_param);
int     url_add_param(url_t *url, char *pname, char *pvalue, err_t *err);

#define url_set_transport_udp(U,E)   url_add_param(U, "transport", "udp", E)
#define url_set_transport_tcp(U,E)   url_add_param(U, "transport", "tcp", E)
#define url_set_transport_sctp(U,E)  url_add_param(U, "transport", "sctp", E)
#define url_set_transport_tls(U,E)   url_add_param(U, "transport", "tls", E)
#define url_set_transport(U,P,E)     url_add_param(U, "transport", P, E)

#define url_set_user_phone(U, E)     url_add_param(U, "user", "phone", E)
#define url_set_user_ip(U, E)        url_add_param(U, "user", "ip", E)
#define url_set_user(U, P, E)        url_add_param(U, "user", P, E)

#define url_set_method_invite(U, E)  url_add_param(U, "method", "INVITE", E)
#define url_set_method_ack(U, E)     url_add_param(U, "method", "ACK", E)
#define url_set_method_options(U, E) url_add_param(U, "method", "OPTIONS", E)
#define url_set_method_bye(U, E)     url_add_param(U, "method", "BYE", E)
#define url_set_method_cancel(U, E)  url_add_param(U, "method", "CANCEL", E)
#define url_set_method_register(U, E) url_add_param(U,"method", "REGISTER", E)
#define url_set_method(U, P, E)      url_add_param(U, "method", P, E)

#define url_set_ttl(U, P, E)         url_add_param(U, "ttl", P, E)

#define url_set_maddr(U, P, E)         url_add_param(U, "maddr", P, E)


int   url_param_init(url_param_t **url_param, err_t *err);
void  url_param_free(url_param_t *url_param);
void  url_param_freelist(list_t *params);
int   url_param_set (url_param_t *url_param, char *pname,
		     char *pvalue, err_t *err);
int   url_param_add(list_t *url_params, url_param_t *url_param, err_t *err);


int   url_header_init(url_header_t **url_header, err_t *err);
void  url_header_free(url_header_t *url_header);
int   url_header_set (url_header_t *url_header, char *pname,
		      char *pvalue, err_t *err);
int   url_header_add(list_t *url_headers, url_header_t *url_header, err_t *err);


#endif /*  _URLS_H_ */
