/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <osip/sip.h>
#include <osip/smsg.h>


static parser_config_t pconfig[NUMBER_OF_HEADERS];


/* This method must be called before using the parser */
int parser_configure() {

  pconfig[0].hname=CALL_ID;
  pconfig[0].setheader=&msg_setcall_id;
  pconfig[1].hname=CONTACT;
  pconfig[1].setheader=&msg_setcontact;
  pconfig[2].hname=CONTENT_LENGTH;
  pconfig[2].setheader=&msg_setcontent_length;
  pconfig[3].hname=CONTENT_TYPE;
  pconfig[3].setheader=&msg_setcontent_type;
  pconfig[4].hname=CSEQ;
  pconfig[4].setheader=&msg_setcseq;
  pconfig[5].hname=FROM;
  pconfig[5].setheader=&msg_setfrom;
  pconfig[6].hname=MIME_VERSION;
  pconfig[6].setheader=&msg_setmime_version;
  pconfig[7].hname=RECORD_ROUTE;
  pconfig[7].setheader=&msg_setrecord_route;
  pconfig[8].hname=ROUTE;
  pconfig[8].setheader=&msg_setroute;
  pconfig[9].hname=TO;
  pconfig[9].setheader=&msg_setto;
  pconfig[10].hname=VIA;
  pconfig[10].setheader=&msg_setvia;

  return 0;
}

/* This method calls the method that is able to parse the header */
int
parser_callmethod(int i,sip_t *dest,char *hvalue,err_t *err)
{
  return pconfig[i].setheader(dest,hvalue,err);
}

/* search the header hname through pconfig[] tab. 
   A quicker algorithm should be used.
   It returns the index of the header in the parser_config_t tab.
*/
int
parser_isknownheader(char *hname) {
  size_t length ;
  int iinf   = 0;
  int isup   = NUMBER_OF_HEADERS;
  int i      = NUMBER_OF_HEADERS/2;
  
  length = strlen(hname);

    while (1)
    {
      if (i<0||i>NUMBER_OF_HEADERS-1)
	return -1;

      if ((length==strlen(pconfig[i].hname))
	  &&strncmp((const char *) hname,
		    (const char *) pconfig[i].hname,
		    length)==0)
	return i;
      
      if (iinf==isup)
	return -1; /* not found */
      if (iinf==isup-1)
	{
	  if ((i<NUMBER_OF_HEADERS-1)
	      &&(length==strlen(pconfig[i+1].hname))
	      &&strncmp((const char *) hname,
			(const char *) pconfig[i+1].hname,
			length)==0)
	      return i+1;
	  else
	    return -1;
	  if ((i>0)&&(length==strlen(pconfig[i-1].hname))
	      &&strncmp((const char *) hname,
			(const char *) pconfig[i-1].hname,
			length)==0)
	      return i-1;
	  else
	    return -1;
	}
      if (0<strncmp((const char *) hname,
		    (const char *) pconfig[i].hname , length)) {
	/* if this is true, search further */
	iinf = i;	      
	if (i == i + (isup-i)/2) i++;
	else  i = i + (isup-i)/2;
      } else {
	isup = i;
	if (i == i - (i-iinf)/2) i--;
	else i = i - (i-iinf)/2;
      }
    } /* end of (while (1)) */
  return -1;
} 



/* this method returns the string associated to the response code.
   Add your own here...!
 */
char *
parser_getreason(int replycode)
{
  int i;
  i = replycode/100;
  if (i==1)
    {/* 1xx  */
      if (replycode==100)
	return sgetcopy("Trying");
      if (replycode==180)
	return sgetcopy("Ringing");
      if (replycode==181)
	return sgetcopy("Call Is Being Forwarded");
      if (replycode==182)
	return sgetcopy("Queued");
      if (replycode==183)
	return sgetcopy("Session Progress");
    }
  if (i==2)
    {/* 2xx */
	return sgetcopy("OK");
    }
  if (i==3)
    {/* 3xx */
      if (replycode==300)
	return sgetcopy("Multiple Choices");
      if (replycode==301)
	return sgetcopy("Moved Permanently");
      if (replycode==302)
	return sgetcopy("Moved Temporarily");
      if (replycode==305)
	return sgetcopy("Use Proxy");
      if (replycode==380)
	return sgetcopy("Alternative Service");
    }
  if (i==4)
    {/* 4xx */
      if (replycode==400)
	return sgetcopy("Bad Request");
      if (replycode==401)
	return sgetcopy("Unauthorized");
      if (replycode==402)
	return sgetcopy("Payment Required");
      if (replycode==403)
	return sgetcopy("Forbidden");
      if (replycode==404)
	return sgetcopy("Not Found");
      if (replycode==405)
	return sgetcopy("Method Not Allowed");
      if (replycode==406)
	return sgetcopy("Not Acceptable");
      if (replycode==407)
	return sgetcopy("Proxy Authentication Required");
      if (replycode==408)
	return sgetcopy("Request Timeout");
      if (replycode==409)
	return sgetcopy("Conflict");
      if (replycode==410)
	return sgetcopy("Gone");
      if (replycode==411)
	return sgetcopy("Length Required");
      if (replycode==413)
	return sgetcopy("Request Entity Too Large");
      if (replycode==414)
	return sgetcopy("Request-URI Too Large");
      if (replycode==415)
	return sgetcopy("Unsupported Media Type");
      if (replycode==420)
	return sgetcopy("Bad Extension");
      if (replycode==480)
	return sgetcopy("Temporarily not available");
      if (replycode==481)
	return sgetcopy("Call Leg/Transaction Does Not Exist");
      if (replycode==482)
	return sgetcopy("Loop Detected");
      if (replycode==483)
	return sgetcopy("Too Many Hops");
      if (replycode==484)
	return sgetcopy("Address Incomplete");
      if (replycode==485)
	return sgetcopy("Ambiguous");
      if (replycode==486)
	return sgetcopy("Busy Here");
      if (replycode==487)
	return sgetcopy("Request Cancelled");
      if (replycode==488)
	return sgetcopy("Not Acceptable Here");
    }
  if (i==5)
    {/* 5xx */
      if (replycode==500)
	return sgetcopy("Internal Server Error");
      if (replycode==501)
	return sgetcopy("Not Implemented");
      if (replycode==502)
	return sgetcopy("Bad Gateway");
      if (replycode==503)
	return sgetcopy("Service Unavailable");
      if (replycode==504)
	return sgetcopy("Gateway Time-out");
      if (replycode==505)
	return sgetcopy("SIP Version not supported");
    }
  if (i==6)
    {/* 6xx */
      if (replycode==600)
	return sgetcopy("Busy Everywhere");
      if (replycode==603)
	return sgetcopy("Decline");
      if (replycode==604)
	return sgetcopy("Does not exist anywhere");
      if (replycode==606)
	return sgetcopy("Not Acceptable");
    }

  return NULL;
}
