/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <osip/sip.h>
#include <osip/trlayer.h>

void *timers_thread(osip_t *config);

int
timersd_start(osip_t *config)
{
  /* timers for uac and uas transactions. */
  config->uas_timerff   = (fifo_t *)smalloc(sizeof(fifo_t)); 
  config->uac_timerff   = (fifo_t *)smalloc(sizeof(fifo_t)); 
  fifo_init(config->uas_timerff);
  fifo_init(config->uac_timerff);

  config->uas_timerthread = sthread_create(20000,NULL,(void *(*)(void *))timers_thread,(void *)config);

  /* verify that this test also works on VxWorks */
  if (config->uas_timerthread==NULL)
    return -1;
  return 0;
}

int
trn_timers_4uas_needretransmission(transaction_t *transaction)
{
  int counter = transaction->retransmissioncounter;
  transaction->retransmissioncounter++; 
  if (counter==1)   /* 500.000s    */
    return 1;
  if (counter==3)  /* 1.000.000s  */
     return 1;
  if (counter==7)  /* 2.000.000s  */
     return 1;
  if (counter==15) /* 4.000.000s  */
     return 1;
  if (counter==23) /* 8.000.000s  */
     return 1;
  if (counter==31) /* 12.000.000s */
     return 1;
  if (counter==39) /* 16.000.000s */
     return 1;
  if (counter==47) /* 20.000.000s */
     return 1;
  if (counter==55) /* 24.000.000s */
     return 1;
  if (counter==63) /* 28.000.000s */
     return 1;
  if (counter==71) /* 32.000.000s */
     return 1;

  return 0;
}

int
trn_timers_4uac_needretransmission(transaction_t *transaction)
{
  int counter = transaction->retransmissioncounter;
  transaction->retransmissioncounter++; 
  if (counter==1)   /* 500.000s    */
    return 1;
  if (counter==3)  /* 1.000.000s  */
     return 1;
  if (counter==7)  /* 2.000.000s  */
     return 1;
  if (counter==15) /* 4.000.000s  */
     return 1;
  if (counter==23) /* 8.000.000s  */
     return 1;
  if (counter==39) /* 16.000.000s */
     return 1;
  if (counter==71) /* 32.000.000s */
     return 1;
  return 0;
}

int
timers_execute(list_t *transactions)
{
  transaction_t *transaction;
  long cur_time;
  int pos = 0;

  cur_time = time(NULL);
  while (!list_eol(transactions,pos))
    {
      int i;
      transaction = list_get(transactions,pos);
      i = cur_time-transaction->birth_time;
      if (transaction->state!=COMPLETED)
	{
	  if (i>NO_RESPONSE_TIMEOUT)
	    {
	      trace(TRACE_LEVEL0,NULL,"<trn_timers.c> (UAX) killing lost transaction %i %s\n",transaction->transactionid,transaction->callid->number);
	      
	      list_remove(transactions,pos);
	      trn_free(transaction);
	      sfree(transaction);
	    }
	  else
	    {
	      /* retransmission is only for INVITE request    */
	      /* (for other requests, ANSWERED is never used) */
	      if (transaction->state==ANSWERED)
		{
		  if (1==trn_timers_4uas_needretransmission(transaction))
		    {
		      sipevent_t *sipevent;
		      sipevent = evt_new_fromtimer(TIMEOUT,transaction->transactionid);
		      fifo_add(transaction->transactionff,sipevent);
		    }
		}
	      if (transaction->state==CALLING)  /* retransmission case */
		{
		  if (1==trn_timers_4uac_needretransmission(transaction))
		    {
		      sipevent_t *sipevent;
		      sipevent = evt_new_fromtimer(TIMEOUT,transaction->transactionid);
		      fifo_add(transaction->transactionff,sipevent);
		    }
		}
	    }
	}
      else 
	{ /*  (transaction->state==COMPLETED) */
	  i = cur_time-transaction->completed_time;
	  if (i>END_TRANSACTION_TIMEOUT)
	    {
	      trace(TRACE_LEVEL2,NULL,"<trn_timers.c> (UAX) Time to kill transaction %i %s\n",transaction->transactionid,transaction->callid->number);
	      
	      list_remove(transactions,pos);
	      trn_free(transaction);
	      sfree(transaction);
	    }
	}
      pos++;
    }
  return 0;
}


void *
timers_thread(osip_t *config)
{
  transaction_t *transaction;
  
  config->uas_transactions  = (list_t *)smalloc(sizeof(list_t));
  list_init(config->uas_transactions);

  config->uac_transactions  = (list_t *)smalloc(sizeof(list_t));
  list_init(config->uac_transactions);

  while (1) {
    susleep(500000);

    timers_execute(config->uas_transactions);
    timers_execute(config->uac_transactions);

    transaction = (transaction_t *)fifo_tryget(config->uas_timerff);
    while (transaction!=NULL)
      {
	list_add(config->uas_transactions,transaction,-1);
	transaction = (transaction_t *)fifo_tryget(config->uas_timerff);
      }

    transaction = (transaction_t *)fifo_tryget(config->uac_timerff);
    while (transaction!=NULL)
      {
	list_add(config->uac_transactions,transaction,-1);
	transaction = (transaction_t *)fifo_tryget(config->uac_timerff);
      }
  }
}
