/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#ifndef _PORT_H_
#define _PORT_H_

#include <stdio.h>

#define VERBOSE(P) P

#ifdef __VXWORKS_OS__
#include <string.h>
#include <time.h>
#else 
#if defined (HAVE_CONFIG_H)
#include <config.h>
#endif
#endif

#ifdef __cplusplus
#  define BEGIN_C_DECLS      extern "C" {
#  define END_C_DECLS        }
#else
#  define BEGIN_C_DECLS
#  define END_C_DECLS
#endif

#if __STDC__
#  ifndef NOPROTOS
#    define PARAMS(args)   args
#  endif
#endif

#ifdef TRACE_MEM_LEAKS
#define SMALLOC(size)      smalloc(size)
#define SFREE(ptr)         sfree(ptr)
#else
#define SMALLOC(size)      malloc(size)
#define SFREE(ptr)         do {      \
                           if (ptr)  {  free(ptr); ptr = 0; }  \
                           } while (0)
#endif

#ifndef PARAMS
#  define PARAMS(args)     ()
#endif

#if __STDC__
typedef void *void_ptr;
#else /* !__STDC__ */
typedef char *void_ptr;
#endif
#ifdef __VXWORKS_OS__
#include <stdarg.h>
#  define VA_START(a, f)  va_start(a, f)
#else

#if STDC_HEADERS
#    include <stdlib.h>
#    include <string.h>
#elif HAVE_STRINGS_H
#  include <strings.h>
#endif /* !STDC_HEADERS */

#if HAVE_STDARG_H
#  include <stdarg.h>
#  define VA_START(a, f)  va_start(a, f)
#else
#  if HAVE_VARARGS_H
#    include <varargs.h>
#    define VA_START(a, f) va_start(a)
#  endif
#endif

#endif

#ifndef VA_START
#  error no variadic api
#endif


#include <errno.h>
#include <semaphore.h>

#ifdef __VXWORKS_OS__
#include <semLib.h>
typedef struct semaphore smutex_t;
#else

#ifdef __sun__
#undef getdate
#include <synch.h>
#endif

#include <pthread.h>
typedef pthread_mutex_t smutex_t;
typedef sem_t           ssem_t;
#endif

#ifdef __VXWORKS_OS__
#include <taskLib.h>
typedef int pthread_t;
typedef sem_t  ssem_t;
#endif

/**************************/
/* SIP library error code */
/**************************/
typedef int err_t;

/* success */
#define SIP_OK              ((err_t)0)

#define SIP_SYNTAX_ERROR    ((err_t)-1)
#define SIP_NETWORK_ERROR   ((err_t)-2)
#define SIP_ECONNREFUSED    ((err_t)-3)
#define SIP_RESSOURCE_ERROR ((err_t)-4)
#define SIP_GLOBAL_ERROR    ((err_t)-5)


BEGIN_C_DECLS

/**************************/
/* LIST implementation    */
/**************************/

typedef struct _node_t {
  void *next;  /* next node_t */
  void *element;
} node_t;

typedef struct _list_t {

  int  nb_elt;
  node_t *node;

  /* would be cool to add ref to lastnode and curnode... */

} list_t;

int   list_init  (list_t *li);
void  listofchar_free(list_t *li);
int   list_size  (list_t *li);
int   list_eol   (list_t *li,int i);
int   list_add   (list_t *li,void *el,int pos);
void *list_get   (list_t *li,int pos);
int   list_remove(list_t *li,int pos);


/**************************/
/* FIFO implementation    */
/**************************/

#define MAX_LEN 1000
typedef enum {ok , plein , vide} fifo_etat;
typedef struct  {

  smutex_t *qislocked;
  ssem_t  *qisempty;
  list_t *queue;
  int     nb_elt;
  fifo_etat etat;
  
} fifo_t;

void *fifo_tryget(fifo_t *ff);
void  fifo_init(fifo_t *ff);
int   fifo_add(fifo_t *ff , void *el);
void *fifo_get(fifo_t *ff);
void  freefifo_t(fifo_t *ff);

/**************************/
/* FSM  implementation    */
/**************************/

typedef enum _state_t {

  INITIAL,

  /* this are the states for UAC/UAC, REDIRECT SERVER, */
  /* Gateway , and Conference Server */
  CALLING,     /* for OUTGOING */
  PROCEEDING,
  ANSWERED,    /* for INCOMING */


  /* these are the states for PROXY-STATEFULL */
  REQUEST_FORWARDED,
  RESPONSE_FORWARDED,
  ACK_FORWARDED,

  /* these are NO states for STATE-LESS-PROXY  */
  /* But info SHOULD be kept to avoid complete */
  /* operation to be done twice on same message*/

  COMPLETED,

  /* Possible reason of ERRORS */
  NETWORK_ERROR,
  SYNTAX_ERROR,
  GLOBAL_ERROR

} state_t;

typedef enum _type_t {
  TIMEOUT,

  /* FOR INCOMING TRANSACTION */
  RCV_REQINVITE,
  RCV_REQACK,
  RCV_REQUEST,
  RCV_STATUS_1XX,
  RCV_STATUS_23456XX,

  /* FOR OUTGOING TRANSACTION */
  SND_REQINVITE,
  SND_REQACK,
  SND_REQUEST,
  
  SND_STATUS_1XX,
  SND_STATUS_23456XX,

  KILL_TRANSACTION,
  UNKNOWN_EVT
} type_t;

typedef struct _statemachine_t {
  list_t *transitions;
} statemachine_t;

typedef struct _transition_t {
  state_t   state;
  type_t    type;
  void       (*method)(void *,void *);
} transition_t;


/**************************/
/* MALLOC redirections    */
/**************************/

void      *smalloc(size_t size);
void       sfree  (void *ptr);

/**************************/
/* SEMAPHORE support      */
/**************************/

smutex_t     *smutex_init();
void         smutex_destroy(smutex_t *mut);
int          smutex_lock(smutex_t *mut);
int          smutex_unlock(smutex_t *mut);

ssem_t      *ssem_init(unsigned int value);
int          ssem_destroy(ssem_t *sem);
int          ssem_post(ssem_t *sem);
int          ssem_wait(ssem_t *sem);
int          ssem_trywait(ssem_t *sem);

/**************************/
/* THREAD support         */
/**************************/



#ifdef __VXWORKS_OS__
/* launch a new thread.                                          */
/* INPUT :  int stacksize | zie of the stack (for VxWorks only). */
/* OUTPUT:  pthread_t *thread | thread ID. (on unix plateforms)  */
/* INPUT :  void *(*func)(void *) | method to start.             */
/* INPUT :  void *arg | arg is the paramter for func.            */
/* returns SIP_NULL on error. */
int sthread_create(int stacksize,
		   pthread_t *thread,  /* MUST BE NULL ON VxWorks */
		   void *(*func)(void *),
		   void *arg);
int sthread_join(int thread);
int sthread_setpriority(int thread, int priority);
#else
pthread_t *sthread_create(int stacksize,
			  pthread_t *thread,
			  void *(*func)(void *),
			  void *arg);
int sthread_join(pthread_t *thread);
/* WARNING UNUSED METHOD */
int sthread_setpriority(pthread_t *thread, int priority);
#endif
int sthread_exit();


/**************************/
/* RANDOM number support  */
/**************************/

unsigned int new_random_number();
void         init_random_number();

/**************************/
/* TIMER support          */
/**************************/

#define SP   " \0"

void       susleep(int useconds);

/**************************/
/* STRING support         */
/**************************/

int        satoi(char *number);
char      *sstrncpy(char *dest,const char *src,int length);
char      *sgetcopy(char *ch);
int        stolowercase(char *word);
int        sclrspace(char *word);
/* find the next unescaped quote and  */
/* return its index.                  */
/* return NULL on error.              */
char      *quote_find(char *qstring);
int        sclrlws(char *word);

/**************************/
/* LOG&DEBUG support      */
/**************************/

/* define log possible value */
typedef int boolean;
#define LOG_TRUE  1
#define LOG_FALSE 0
/* levels */
typedef enum _trace_level {
  TRACE_LEVEL0      = 0,
  TRACE_LEVEL1      = 1,
  TRACE_LEVEL2      = 2,
  TRACE_LEVEL3      = 3,
  TRACE_LEVEL4      = 4,
  TRACE_LEVEL5      = 5,
  END_TRACE_LEVEL   = 6
} trace_level_t;

#ifndef ENABLE_TRACE
#define TRACE_INITIALIZE(level, file) do { \
fprintf(stderr, \
"Error: TRACE_INITIALIZE() is disabled. (use configure --enable-trace)\n"); \
                                      } while (0)
#define TRACE_ENABLE_LEVEL(level)      do { } while (0)
#define TRACE_DISABLE_LEVEL(level)     do { } while (0)
#define IS_TRACE_LEVEL_ACTIVATE(level) do { } while (0)

#else

#define TRACE_INITIALIZE(level, file)  trace_initialize(level, file)
#define TRACE_ENABLE_LEVEL(level)      trace_enable_level ( level )
#define TRACE_DISABLE_LEVEL(level)     trace_disable_level ( level )
#define IS_TRACE_LEVEL_ACTIVATE(level) is_trace_level_activate( level )

extern void    trace_initialize        ( trace_level_t level, FILE *file );
extern void    trace_enable_level      ( trace_level_t level );
extern void    trace_disable_level     ( trace_level_t level );
extern boolean is_trace_level_activate ( trace_level_t level );
#endif

/* log facility. */
/* if f is NULL, current default log file is used. */
/* INPUT: level | level of the trace               */
/* INPUT: f | use f instead of default log file    */
/* INPUT: chfr | format string for next args       */
int
#if defined(HAVE_STDARG_H) || defined(__VXWORKS_OS__)
trace(trace_level_t level,FILE *f,char *chfr, ...);
#else
/* ERROR? I never test this */
trace(level,f,chfr,va_list);
#endif

END_C_DECLS

#endif  /* _PORT_H_ */
