/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#ifndef _TRLAYER_H_
#define _TRLAYER_H_

#ifdef __linux
#include <time.h>
#endif

#include <osip/port.h>

#include <osip/const.h>
#include <osip/smsg.h>


#define NO_RESPONSE_TIMEOUT 60
#define END_TRANSACTION_TIMEOUT 10
#define SIP_MESSAGE_MAX_LENGTH 20000

typedef struct _osip_t {

  url_t *proxy;

  /* this is the (udp) port where sipd is waiting */
  int port;

  list_t *transactions;

  /* When application request a timer for a transaction   */
  /* a pointer to this transaction is added in those fifo */
  fifo_t *uas_timerff;
  fifo_t *uac_timerff;
  /* The timers method MAY use those list to store the       */
  /* actual list of transactions that needs retransmissions. */
  list_t *uas_transactions;
  list_t *uac_transactions;

#ifdef __VXWORKS_OS__
  int uas_timerthread;
  int uac_timerthread;
#else
  pthread_t *uas_timerthread;
  pthread_t *uac_timerthread;
#endif

} osip_t;

typedef struct _transaction_t {
  
  /* could be NULL when you don't */
  /* this transaction for this    */
  /* transaction. */
  /* COMMENT: this is just a copy of the  */
  /* osip->proxy parameter for the moment */
  url_t *proxy;

  from_t    *from;
  to_t      *to;
  call_id_t *callid;
  cseq_t    *cseq;

  int transactionid;
#ifdef __VXWORKS_OS__
  int threadid;
#else
  pthread_t *threadid;
#endif
  state_t state;
  statemachine_t *statemachine;

  /* int status; */
  sip_t *lastrequest;
  sip_t *lastresponse;

  time_t birth_time;
  time_t completed_time;
  int retransmissioncounter;

  fifo_t *transactionff;

  /* each transaction is part of a osip_t instance */
  /* (=each upper layer can use different config)       */
  osip_t *config;
} transaction_t;

/*
  This is for later improvements on the
  transport managements side
  typedef struct {
  list_t *siplisteners;
  fifo_t *evtmngrff;
  } evtmngr_t;
  
  typedef struct {
  evtmngr_t *evtmngr;
  int port;
  } siplistener_t;
  
  typedef struct {
  evtmngr_t *evtmngr;
  fifo_t *dispff;
  } dispatcher_t;
*/

typedef struct _sipevent_t {
  type_t type;
  int transactionid;
  sip_t *sip;
} sipevent_t;



/* INPUT :  osip_t *config | config of stack.                */
void evt_sendevent       (osip_t *config,sipevent_t *sipevent);

/* allocates an event from retransmitter.             */
/* USED ONLY BY THE STACK.                            */
/* INPUT : int transactionid | id of the transaction. */
/* INPUT : type_t type | type of event.               */
/* returns null on error. */
sipevent_t    *evt_new_fromtimer(type_t type,int transactionid);
/* allocates an event from user.                      */
/* USED ONLY BY THE USER.                             */
/* INPUT : sip_t *sip | sip message for transaction.  */
/* returns null on error. */
sipevent_t    *evt_new_fromuser     (sip_t *sip);
/* allocates an event from transport.                 */
/* USED ONLY BY THE TRANSPORT LAYER.                  */
/* INPUT : sip_t *sip | sip message for transaction.  */
/* returns null on error. */
sipevent_t    *evt_new_fromip(sip_t *sip);

/* This is for internal use only.                      */
type_t 	       evt_settype_fromuser        (sip_t *sip);
type_t 	       evt_settype_fromip   (sip_t *sip);

  /* FOR INCOMING TRANSACTION */
#define EVT_IS_RCV_INVITE(event)       (event->type==RCV_REQINVITE)
#define EVT_IS_RCV_ACK(event)          (event->type==RCV_REQACK)
#define EVT_IS_RCV_REQUEST(event)      (event->type==RCV_REQUEST)
#define EVT_IS_RCV_STATUS_1XX(event)   (event->type==RCV_STATUS_1XX)
#define EVT_IS_RCV_STATUS_23456XX(event)   (event->type==RCV_STATUS_23456XX)
/* #define EVT_IS_RCV_STATUS_2XX(event)   (event->type==RCV_STATUS_2XX)
   #define EVT_IS_RCV_STATUS_ERROR(event) (event->type==RCV_STATUS_ERROR)
*/

/* FOR OUTGOING TRANSACTION */
#define EVT_IS_SND_INVITE(event)       (event->type==SND_REQINVITE)
#define EVT_IS_SND_ACK(event)          (event->type==SND_REQACK)
#define EVT_IS_SND_REQUEST(event)      (event->type==SND_REQUEST)
#define EVT_IS_SND_STATUS_1XX(event)   (event->type==SND_STATUS_1XX)
#define EVT_IS_SND_STATUS_23456XX(event)   (event->type==SND_STATUS_23456XX)
/* #define EVT_IS_SND_STATUS_2XX(event)   (event->type==SND_STATUS_23456XX)
   #define EVT_IS_SND_STATUS_ERROR(event) (event->type==SND_STATUS_ERROR)
*/
#define EVT_IS_INCOMINGMSG(event)      (event->type>=RCV_REQINVITE \
                	               &&event->type<=RCV_STATUS_23456XX)
#define EVT_IS_INCOMINGREQ(event)      (EVT_IS_RCV_INVITE(event) \
                                       ||EVT_IS_RCV_ACK(event) \
                                       ||EVT_IS_RCV_REQUEST(event))
#define EVT_IS_INCOMINGRESP(event)     (EVT_IS_RCV_STATUS_1XX(event) \
                                       ||EVT_IS_RCV_STATUS_23456XX(event))
#define EVT_IS_OUTGOINGMSG(event)      (event->type>=SND_REQINVITE \
                	               &&event->type<=SND_STATUS_23456XX)
#define EVT_IS_OUTGOINGREQ(event)      (EVT_IS_SND_INVITE(event) \
                                       ||EVT_IS_SND_ACK(event) \
                                       ||EVT_IS_SND_REQUEST(event))
#define EVT_IS_OUTGOINGRESP(event)     (EVT_IS_SND_STATUS_1XX(event) \
                                       ||EVT_IS_SND_STATUS_23456XX(event))

#define EVT_IS_MSG(event)              (event->type>=RCV_REQINVITE \
                	               &&event->type<=SND_STATUS_23456XX)
#define EVT_IS_KILL_TRANSACTION(event) (event->type==KILL_TRANSACTION)
#define EVT_IS_UNKNOWN_EVT(event)      (event->type==UNKNOWN_EVT)
#define EVT_IS_TIMEOUT(event)          (event->type==TIMEOUT)


/* INPUT :  osip_t *config | config of stack.                */
int            sipd_start(osip_t *config);


/* send a request on UDP.                       */
/* INPUT : sip_t *request | request to be sent. */
/* returns null on error. */
int            udp_send_request(sip_t *request, url_t *proxy, err_t *err);
/* send a response on UDP.                       */
/* INPUT : sip_t *response | response to be sent.*/
/* returns null on error. */
int            udp_send_response(sip_t *response, err_t *err);


#endif
