/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdlib.h>
#include <stdio.h>

#include <osip/port.h>
#include <osip/sip.h>
#include <osip/ua.h>
#include <osip/fsm.h>

static statemachine_t *uac_4req;


statemachine_t *
fsm_getfsm_uac4req()
{
  return uac_4req;
}

int 
fsm_load_uac4req() {

  transition_t *transition;

  uac_4req       = (statemachine_t *) smalloc(sizeof(statemachine_t));
  uac_4req->transitions = (list_t *)  smalloc(sizeof(list_t));
  list_init(uac_4req->transitions);

  transition         = (transition_t *) smalloc(sizeof(transition_t));
  transition->state  = INITIAL;
  transition->type   = SND_REQUEST;
  transition->method = (void(*)(void *,void *))&uac_r_sendrequest;
  list_add(uac_4req->transitions,transition,-1);

  transition         = (transition_t *) smalloc(sizeof(transition_t));
  transition->state  = CALLING;
  transition->type  = TIMEOUT;
  transition->method = (void(*)(void *,void *))&uac_r_retransmitrequest;
  list_add(uac_4req->transitions,transition,-1);

  transition         = (transition_t *) smalloc(sizeof(transition_t));
  transition->state  = CALLING;
  transition->type  = RCV_STATUS_1XX;
  transition->method = (void(*)(void *,void *))&uac_rcv1XX;
  list_add(uac_4req->transitions,transition,-1);

  transition         = (transition_t *) smalloc(sizeof(transition_t));
  transition->state  = CALLING;
  transition->type  = RCV_STATUS_23456XX;
  transition->method = (void(*)(void *,void *))&uac_rcv23456XX;
  list_add(uac_4req->transitions,transition,-1);

  transition         = (transition_t *) smalloc(sizeof(transition_t));
  transition->state  = PROCEEDING;
  transition->type  = RCV_STATUS_1XX;
  transition->method = (void(*)(void *,void *))&uac_rcv1XX;
  list_add(uac_4req->transitions,transition,-1);

  transition         = (transition_t *) smalloc(sizeof(transition_t));
  transition->state  = PROCEEDING;
  transition->type  = RCV_STATUS_23456XX;
  transition->method = (void(*)(void *,void *))&uac_rcv23456XX;
  list_add(uac_4req->transitions,transition,-1);

  return 0;
}



/* Called when STATE=INITIAL and EVENT=SND_REQINVITE */
void
uac_r_sendrequest(sipevent_t *sipevent,transaction_t *transaction)
{
  ua_sendrequest(sipevent,transaction);

  /* always inform the timer of new transactions */
  transaction->retransmissioncounter = 1;
  fifo_add(transaction->config->uac_timerff,transaction);
  return ;    
}

/* Called when STATE=CALLING and EVENT=TIMEOUT */
void
uac_r_retransmitrequest(sipevent_t *sipevent,transaction_t *transaction)
{
  ua_retransmitrequest(sipevent,transaction);
}

/* Called when STATE=CALLING and EVENT=RCV_STATUS_1XX */
void
uac_rcv1XX(sipevent_t *sipevent,transaction_t *transaction)
{
  ua_rcvresponse(sipevent,transaction);

  transaction->state = PROCEEDING;
  /* invoke the right callback! */
  OnEvent_New_Incoming1xxResponse(sipevent,transaction->transactionid);

  return ; /* OK */
}

/* Called when STATE=CALLING||PROCEEDING and EVENT=RCV_STATUS_3456XX */
void
uac_rcv23456XX(sipevent_t *sipevent,transaction_t *transaction)
{
  ua_rcvresponse(sipevent,transaction);
  
  /* invoke the right callback! */
  if (MSG_IS_STATUS_2XX(sipevent->sip))
    OnEvent_New_Incoming2xxResponse(sipevent,transaction->transactionid);
  else if (MSG_IS_STATUS_3XX(sipevent->sip))
    OnEvent_New_Incoming3xxResponse(sipevent,transaction->transactionid);
  else if (MSG_IS_STATUS_4XX(sipevent->sip))
    OnEvent_New_Incoming4xxResponse(sipevent,transaction->transactionid);
  else if (MSG_IS_STATUS_5XX(sipevent->sip))
    OnEvent_New_Incoming5xxResponse(sipevent,transaction->transactionid);
  else if (MSG_IS_STATUS_6XX(sipevent->sip))
    OnEvent_New_Incoming6xxResponse(sipevent,transaction->transactionid);
  
  transaction->completed_time = time(NULL);
  transaction->state = ANSWERED;
  return ; /* OK */
}

void
ua_sendrequest(sipevent_t *sipevent,transaction_t *transaction)
{
  int i;
  err_t err;
  i = udp_send_request(sipevent->sip,
		       transaction->proxy, &err);
  if (i==-1)
    { 
      transaction->state=NETWORK_ERROR;
      /* invoke the right callback! */
      /* err must be tested.... NOT USABLE
	 if (code==SIP_ECONNREFUSED)
	 OnEvent_connection_refused(transaction->transactionid);
	 else */
      OnEvent_network_error(transaction->transactionid);
      
      msg_free(sipevent->sip);
      sfree(sipevent->sip);
      return ;
    }
  
  transaction->lastrequest = sipevent->sip;
  transaction->state = CALLING;

  /* invoke the right callback! */
  if (MSG_IS_INVITE(sipevent->sip))
    OnEvent_New_OutgoingInvite(sipevent, transaction->transactionid);
  else if (MSG_IS_BYE(sipevent->sip))
    OnEvent_New_OutgoingBye(sipevent, transaction->transactionid);
  else if (MSG_IS_REGISTER(sipevent->sip))
    OnEvent_New_OutgoingRegister(sipevent, transaction->transactionid);
  else if (MSG_IS_CANCEL(sipevent->sip))
    OnEvent_New_OutgoingCancel(sipevent, transaction->transactionid);
  else if (MSG_IS_INFO(sipevent->sip))
    OnEvent_New_OutgoingInfo(sipevent, transaction->transactionid);
  else if (MSG_IS_OPTIONS(sipevent->sip))
    OnEvent_New_OutgoingOptions(sipevent, transaction->transactionid);
  else if (MSG_IS_PRACK(sipevent->sip))
    OnEvent_New_OutgoingPrack(sipevent, transaction->transactionid);
  else
    OnEvent_New_OutgoingUnknownRequest(sipevent, transaction->transactionid);
  return;
}

void
ua_retransmitrequest(sipevent_t *sipevent,transaction_t *transaction)
{
  int i;
  err_t err;
  i =   udp_send_request(transaction->lastrequest,
			 transaction->proxy, &err);
  if (i==-1)
    {
      transaction->state=NETWORK_ERROR;
      /*
	if (code==SIP_ECONNREFUSED)
	OnEvent_connection_refused(transaction->transactionid);
	else
	{
	trace(TRACE_LEVEL1,NULL,"<fsm_uac4req.c> retransmission of request failed.\n"); */
      OnEvent_network_error(transaction->transactionid);
	/*} */
      return ;
    }

  /* invoke the right callback! */
  OnEvent_sndreq_retransmission(transaction->transactionid);
  return; /* OK */
}

void
ua_rcvresponse(sipevent_t *sipevent,transaction_t *transaction)
{
  if (transaction->lastresponse!=NULL)
    {
    msg_free(transaction->lastresponse);
    sfree(transaction->lastresponse);
    } /* else this is the first response received */

  transaction->lastresponse = sipevent->sip;
  msg_logresponse(transaction->lastresponse,"RCV %s %s (%s) f:%s t:%s cseq:%s callid:%s\n");

  return ; /* OK */
}



