/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdlib.h>
#include <stdio.h>

#include <osip/port.h>
#include <osip/sip.h>
#include <osip/ua.h>
#include <osip/fsm.h>

#ifdef __VXWORKS_OS__
#include <time.h>
#endif

static statemachine_t *uas_4req;


statemachine_t *
fsm_getfsm_uas4req()
{
  return uas_4req;
}

int
fsm_load_uas4req() {

  transition_t *transition;

  uas_4req       = (statemachine_t *) smalloc(sizeof(statemachine_t));
  uas_4req->transitions = (list_t *)  smalloc(sizeof(list_t));
  list_init(uas_4req->transitions);

  transition         = (transition_t *) smalloc(sizeof(transition_t));
  transition->state  = INITIAL;
  transition->type  = RCV_REQUEST;
  transition->method = (void(*)(void *,void *))&uas_rcvrequest;
  list_add(uas_4req->transitions,transition,-1);

  transition         = (transition_t *) smalloc(sizeof(transition_t));
  transition->state  = PROCEEDING;
  transition->type  = RCV_REQUEST;
  transition->method = (void(*)(void *,void *))&uas_r_rcvretransmitrequest;
  list_add(uas_4req->transitions,transition,-1);

  transition         = (transition_t *) smalloc(sizeof(transition_t));
  transition->state  = PROCEEDING;
  transition->type  = SND_STATUS_1XX;
  transition->method = (void(*)(void *,void *))&uas_send1XX;
  list_add(uas_4req->transitions,transition,-1);

  transition         = (transition_t *) smalloc(sizeof(transition_t));
  transition->state  = PROCEEDING;
  transition->type  = SND_STATUS_23456XX;
  transition->method = (void(*)(void *,void *))&uas_r_send23456XX;
  list_add(uas_4req->transitions,transition,-1);

  transition         = (transition_t *) smalloc(sizeof(transition_t));
  transition->state  = COMPLETED;
  transition->type  = RCV_REQUEST;
  transition->method = (void(*)(void *,void *))&uas_r_rcvretransmitrequest;
  list_add(uas_4req->transitions,transition,-1);

  return 0;
}


void
uas_rcvrequest(sipevent_t *sipevent,transaction_t *transaction)
{
  msg_logrequest(sipevent->sip,"RCV %s f:%s t:%s cseq:%s callid:%s\n");

  transaction->lastrequest = sipevent->sip;
  transaction->state = PROCEEDING;

  transaction->retransmissioncounter = 1;
  fifo_add(transaction->config->uas_timerff,transaction);

  /* invoke the right callback! */
  if (MSG_IS_INVITE(sipevent->sip))
    OnEvent_New_IncomingInvite(sipevent,transaction->transactionid);
  else if (MSG_IS_BYE(sipevent->sip))
    OnEvent_New_IncomingBye(sipevent,transaction->transactionid);
  else if (MSG_IS_REGISTER(sipevent->sip))
    OnEvent_New_IncomingRegister(sipevent,transaction->transactionid);
  else if (MSG_IS_CANCEL(sipevent->sip))
    OnEvent_New_IncomingCancel(sipevent,transaction->transactionid);
  else if (MSG_IS_OPTIONS(sipevent->sip))
    OnEvent_New_IncomingOptions(sipevent,transaction->transactionid);
  else if (MSG_IS_INFO(sipevent->sip))
    OnEvent_New_IncomingInfo(sipevent,transaction->transactionid);
  else if (MSG_IS_PRACK(sipevent->sip))
    OnEvent_New_IncomingPrack(sipevent,transaction->transactionid);
  else
    OnEvent_New_IncomingUnknownRequest(sipevent,transaction->transactionid);
}

void
uas_r_rcvretransmitrequest(sipevent_t *sipevent,transaction_t *transaction)
{
  ua_retransmitresponse(sipevent,transaction);
}

void
uas_r_send23456XX(sipevent_t *sipevent,transaction_t *transaction)
{
  if (uas_send23456XX(sipevent,transaction)==-1)
    return ;

  transaction->completed_time = time(NULL);
  transaction->state = COMPLETED;
}


/* COMMON METHODS FOR incoming INVITE and OTHERREQ */

void
ua_retransmitresponse(sipevent_t *sipevent,transaction_t *transaction)
{
  /* invoke the right callback! */
  OnEvent_rcvreq_retransmission(transaction->transactionid);
  msg_logrequest(sipevent->sip,"RCV (retransmission) %s f:%s t:%s cseq:%s callid:%s\n");
  
  if (transaction->lastresponse!=NULL)
    {
      err_t err;
      int i;
      i = udp_send_response(transaction->lastresponse, &err);
      if (i==-1)
	{
	  trace(TRACE_LEVEL1,NULL,"<fsm_uas4req.c> Error Could not retransmit response.\n");
	  OnEvent_network_error(transaction->transactionid);
	}
      else
	/* invoke the right callback! */
	OnEvent_sndresp_retransmission(transaction->transactionid);
    }
  else
    /* if no response is given by user, retransmission has no effect */
    {
      trace(TRACE_LEVEL4,NULL,"<fsm_uas4req.c> Retransmission has no effect. (Response is not ready?)\n");
    }

  msg_free(sipevent->sip);
  sfree(sipevent->sip);
}

void
uas_send1XX(sipevent_t *sipevent,transaction_t *transaction)
{
  err_t err;
  int i;
  i = udp_send_response(sipevent->sip, &err);
  if (i==-1)
    {
    trace(TRACE_LEVEL1,NULL,"<fsm_uas4req.c> Error Could not send new response.\n");
    OnEvent_network_error(transaction->transactionid);
    msg_free(sipevent->sip);
    sfree(sipevent->sip);
    return ;
    }
  
  /* invoke the right callback! */
  OnEvent_New_Outgoing1xxResponse(sipevent,transaction->transactionid);
  
  if (transaction->lastresponse!=NULL)
    {
    msg_free(transaction->lastresponse);
    sfree(transaction->lastresponse);
    }
  transaction->lastresponse = sipevent->sip;
}

int
uas_send23456XX(sipevent_t *sipevent,transaction_t *transaction)
{
  err_t err;
  int i;
  i = udp_send_response(sipevent->sip, &err);
  if (i==-1)
    {
    trace(TRACE_LEVEL1,NULL,"<fsm_uas4req.c> response not sent.\n");
    OnEvent_network_error(transaction->transactionid);
    msg_free(sipevent->sip);
    sfree(sipevent->sip);
    return -1;
    }
  
  if (transaction->lastresponse!=NULL)
    {
    msg_free(transaction->lastresponse);
    sfree(transaction->lastresponse);
    }
  transaction->lastresponse = sipevent->sip;
  transaction->state = ANSWERED;
  
  /* invoke the right callback! */
  if (MSG_IS_STATUS_2XX(sipevent->sip))
    OnEvent_New_Outgoing2xxResponse(sipevent,transaction->transactionid);
  else if (MSG_IS_STATUS_3XX(sipevent->sip))
    OnEvent_New_Outgoing3xxResponse(sipevent,transaction->transactionid);
  else if (MSG_IS_STATUS_4XX(sipevent->sip))
    OnEvent_New_Outgoing4xxResponse(sipevent,transaction->transactionid);
  else if (MSG_IS_STATUS_5XX(sipevent->sip))
    OnEvent_New_Outgoing5xxResponse(sipevent,transaction->transactionid);
  else if (MSG_IS_STATUS_6XX(sipevent->sip))
    OnEvent_New_Outgoing6xxResponse(sipevent,transaction->transactionid);
  else
    trace(TRACE_LEVEL1,NULL,"<fsm_uas4req.c> Unknown response code?\n");
  return 0;
}
