/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdlib.h>
#include <stdio.h>

#include <osip/port.h>
#include <osip/smsg.h>


int
content_type_init(content_type_t **content_type, err_t *err)
{
  *content_type = (content_type_t *)smalloc(sizeof(content_type_t));
  (*content_type)->type    = NULL;
  (*content_type)->subtype = NULL;

  (*content_type)->content_type_params = (list_t *) smalloc(sizeof(list_t));
  list_init((*content_type)->content_type_params);
  
  return 1;
}

/* adds the content_type header to message.              */
/* INPUT : char *hvalue | value of header.    */
/* OUTPUT: sip_t *sip | structure to save results.  */
/* returns -1 on error. */
int
msg_setcontent_type(sip_t *sip, char *hvalue,err_t *err)
{
  int i;
  i = content_type_init(&(sip->content_type), err);
  if (i==-1)
    return -1;
  return content_type_parse(sip->content_type, hvalue, err);
}


/* returns the content_type header.            */
/* INPUT : sip_t *sip | sip message.   */
/* returns null on error. */
content_type_t *
msg_getcontent_type(sip_t *sip) {
  return sip->content_type;
}

/* parses the string as a content_type header.                   */
/* INPUT : const char *string | pointer to a content_type string.*/
/* OUTPUT: content_type_t *content_type | structure to save results.     */
/* returns -1 on error. */
int
content_type_parse(content_type_t *content_type, char *hvalue, err_t *err)
{
  char *subtype;
  char *content_type_params;
  
  /* How to parse:

     we'll place the pointers:
     subtype              =>  beginning of subtype
     content_type_params  =>  beginning of params

     examples:

     application/multipart ; boundary=
                ^          ^
  */

  subtype = strchr(hvalue,'/');
  content_type_params = strchr(hvalue,';');

  if (subtype == NULL)
    return -1; /* do we really mind such an error */

  if (content_type_params!=NULL)
    {
      if (generic_parse_params(content_type->content_type_params,
			       content_type_params, err)==-1)
	return -1;
    }
  else
    content_type_params = subtype + strlen (subtype);

  if (subtype-hvalue+1<2) return -1;
  content_type->type = (char *)smalloc(subtype-hvalue+1);
  sstrncpy(content_type->type, hvalue, subtype-hvalue);
  sclrspace(content_type->type);

  if (content_type_params-subtype<2) return -1;
  content_type->subtype = (char *)smalloc(content_type_params-subtype);
  sstrncpy(content_type->subtype, subtype+1, content_type_params-subtype-1);
  sclrspace(content_type->subtype);

  return 0;
}



/* returns the content_type header as a string.  */
/* INPUT : content_type_t *content_type | content_type header.   */
/* returns null on error. */
int
content_type_2char(content_type_t *content_type, char **dest, err_t *err)
{
  char *buf ;

  *dest = NULL;
  if ((content_type==NULL)||(content_type->type==NULL)
      ||(content_type->subtype==NULL))
    return -1;

  buf =  (char *)smalloc(200);
  *dest = buf;

  sprintf(buf,"%s/%s ", content_type->type, content_type->subtype);

  buf = buf + strlen(buf);
  {
    int pos = 0;
    generic_param_t *u_param;
    while (!list_eol(content_type->content_type_params,pos))
      {
	u_param = (generic_param_t *)list_get(content_type->content_type_params,pos);
	sprintf(buf,";%s=%s",u_param->pname,u_param->pvalue);
	buf = buf + strlen(buf);
	pos++;
      }
  }
  return 0;
}


/* deallocates a content_type_t structure.  */
/* INPUT : content_type_t *content_type | content_type. */
void
content_type_free(content_type_t *content_type)
{
  sfree(content_type->type);
  sfree(content_type->subtype);

  generic_param_freelist(content_type->content_type_params);
  sfree(content_type->content_type_params);

  content_type->type    = NULL;
  content_type->subtype = NULL;
  content_type->content_type_params = NULL;
}

