/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <time.h>
#include <stdlib.h>
#include <signal.h>

#include <osip/sip.h>
#include <osip/fsm.h>

static smutex_t *fastmutex;

int
trn_global_init()
{
  fastmutex = smutex_init();  
  return 0;
}

/* BUG: 21/08/2001
   add a call to smutex_unlock() before evry call to return -1
*/
int
trn_init(osip_t *config,
	 transaction_t *dest,
	 to_t *to,
	 from_t *from,
	 call_id_t *callid,
	 cseq_t *cseq,
	 err_t *err)
{
  static int current_transactionid = 0;

  char *tmp;
  int i;

  current_transactionid++;
  trace(TRACE_LEVEL3,NULL,"<trn_manager.c> Creating trnsaction with id = %i\n",current_transactionid);

  dest->config = config;
  dest->transactionid = current_transactionid;

  /* could be improved by selecting individually wich */
  /* transaction should go through a proxy and wich should not */
  dest->proxy = config->proxy; /* set to NULL if not used */

  i = from_2char(from, &tmp, err);
  if (i==-1) return -1;
  i = from_init(&(dest->from), err);
  if (i==-1) return -1;
  i = from_parse(dest->from, tmp, err);
  sfree(tmp);
  if (i==-1) return -1;

  i = to_2char(to, &tmp, err);
  if (i==-1) return -1;
  i = to_init(&(dest->to), err);
  if (i==-1) return -1;
  i = to_parse(dest->to, tmp, err);
  sfree(tmp);
  if (i==-1) return -1;

  i = call_id_2char(callid, &tmp, err);
  if (i==-1) return -1;
  i = call_id_init(&(dest->callid), err);
  if (i==-1) return -1;
  i = call_id_parse(dest->callid, tmp, err);
  sfree(tmp);
  if (i==-1) return -1;

  i = cseq_2char(cseq, &tmp, err);
  if (i==-1) return -1;
  i = cseq_init(&(dest->cseq), err);
  if (i==-1) return -1;
  i = cseq_parse(dest->cseq, tmp, err);
  sfree(tmp);
  if (i==-1) return -1;

  dest->state         = INITIAL;
  dest->statemachine  = NULL;
  dest->lastrequest   = NULL;
  dest->lastresponse  = NULL;
  dest->retransmissioncounter=0;

  dest->transactionff = (fifo_t *)smalloc(sizeof(fifo_t));
  fifo_init(dest->transactionff);

  dest->birth_time = time(NULL);
  dest->completed_time = 0;
  dest->threadid = sthread_create(20000,NULL,(void *(*)(void *))trn_thread,(void *)dest);

  /* try to give a highest priority to this task     */
  /* this is only supported by some OS and currently */
  /* only used on VxWorks. */
  sthread_setpriority(dest->threadid, 1);

  smutex_lock(fastmutex);
  list_add(config->transactions,dest,-1);
  smutex_unlock(fastmutex);
  return 0;
}

void
trn_free(transaction_t *transaction)
{
  if (transaction==NULL)
    return ;

  /* first: delete transaction from active ones */
  trn_removeid(transaction->config->transactions,transaction->transactionid);
#ifndef __VXWORKS_OS__
  if (transaction->threadid!=NULL)
#else
  if (transaction->threadid!=ERROR)
#endif
    {
      int i;
      sipevent_t *sipevent;
      sipevent = evt_new_fromtimer(KILL_TRANSACTION,transaction->transactionid);
      fifo_add(transaction->transactionff,sipevent);

      /* arg is a pointer to a pthread on linux, solaris... */
      /* and an int on VxWorks */
      i = sthread_join(transaction->threadid); 
#ifndef __VXWORKS_OS__
      sfree(transaction->threadid);
#endif
    }
  
  /* do not delete value for proxy....  */
  /* it is shared by all transactions   */
  /* THIS SHOULD BE CHANGED? */
  transaction->proxy = NULL;

  from_free(transaction->from);
  sfree(transaction->from);
  to_free(transaction->to);
  sfree(transaction->to);
  call_id_free(transaction->callid);
  sfree(transaction->callid);
  cseq_free(transaction->cseq);
  sfree(transaction->cseq);
  
  if (transaction->lastrequest != NULL)
    {
      sip_t *tmp;
      tmp = transaction->lastrequest;
      transaction->lastrequest=NULL;
      msg_free(tmp);
      sfree(tmp);
    }
  if (transaction->lastresponse != NULL)
    {
      msg_free(transaction->lastresponse);
      sfree(transaction->lastresponse);
     }
  if (transaction->transactionff!= NULL)
    {
      freefifo_t(transaction->transactionff);
      sfree(transaction->transactionff);
    }

}

void *
trn_thread(transaction_t *transaction)
{
  sipevent_t *se;
#ifdef __VXWORKS_OS__
  taskSafe(); /* prevent from deletion */
#endif

  while (1) {
    se = (sipevent_t *)fifo_get(transaction->transactionff);

    /* to kill the process, simply send this type of event. */
    if (EVT_IS_KILL_TRANSACTION(se))
      { /* This event cannot be used by the timer. */
	sfree(se);
#ifdef __VXWORKS_OS__
	taskUnsafe(); /* thread is safe for deletion */
#endif
	return NULL;
      }
    trace(TRACE_LEVEL3,NULL,"<trn_manager.c> sipevent se->transactionid: %i\n",se->transactionid);
    trace(TRACE_LEVEL3,NULL,"<trn_manager.c> sipevent se->type: %i\n",se->type);
    trace(TRACE_LEVEL3,NULL,"<trn_manager.c> sipevent se->sip: %x\n",se->sip);

  /* statemachine is found automaticly... */
  if (transaction->statemachine==NULL)
    {
      if (EVT_IS_SND_INVITE(se))
	transaction->statemachine = fsm_getfsm_uac4inv();
      else
	transaction->statemachine = fsm_getfsm_uac4req();
    }

    if (-1==fsm_callmethod(se->type,transaction->state,
		     transaction->statemachine,
		     se,transaction ))
      {
	/* message is useless. */
	if(EVT_IS_MSG(se))
	  {
	    if (se->sip!=NULL)
	      {
	      msg_free(se->sip);
	      sfree(se->sip);
	      }
	  }
      }
    sfree(se); /* this is the ONLY place for freeing event!! */
    trace(TRACE_LEVEL3,NULL,"<trn_manager.c> method called: %x\n",transaction->transactionff);

  }
  sthread_exit(); /* in fact, we will never go there.... */

}

transaction_t *
trn_findfromtransport(osip_t *config, list_t *transactions, sipevent_t *se)
{
  transaction_t *transaction;
  int            pos=0;


  smutex_lock(fastmutex);

  while (!list_eol(transactions,pos))
    {
      transaction = (transaction_t *)list_get(transactions,pos);
      /* PSM 30/07/2001 */
      /* if (transaction->lastrequest!=NULL)
	 { */
      if (0==call_id_match(transaction->callid,
			   se->sip->call_id)
	  &&0==callleg_match(transaction->to,
			     transaction->from,
			     se->sip->to,se->sip->from)
	  &&0==cseq_match(transaction->cseq
			  ,se->sip->cseq))
	{
	if (0==strncmp(transaction->cseq->method,
		       se->sip->cseq->method,
		       strlen(transaction->cseq->method))
	    ||
	    ((0==strncmp(transaction->cseq->method,"INVITE",6)||
	      0==strncmp(transaction->cseq->method,"ACK",3))
	     &&0==strncmp(se->sip->cseq->method,"ACK",3)))
	  {
	  smutex_unlock(fastmutex);
	  return transaction;
	  }
	}
      /* PSM 30/07/2001 } */
      pos++;
    }
   /*not found */
  smutex_unlock(fastmutex);

  if (EVT_IS_RCV_STATUS_1XX(se)
      ||EVT_IS_RCV_STATUS_23456XX(se)
      ||EVT_IS_RCV_ACK(se))
    return NULL;

  transaction = (transaction_t *) smalloc(sizeof(transaction_t));
  {
    int i;
    err_t err;
    i = trn_init(config,
		 transaction,
		 se->sip->to,
		 se->sip->from,
		 se->sip->call_id,
		 se->sip->cseq,
		 &err);
    if (i==-1)
      {
	sfree(transaction);
	return NULL;
      }
  }
  /* Only first requests are not part of a transaction */
  transaction->lastrequest = se->sip;

  if (MSG_IS_RESPONSEFOR(se->sip,"INVITE"))
    transaction->statemachine = fsm_getfsm_uas4inv();
  else 
    transaction->statemachine = fsm_getfsm_uas4req();

  se->transactionid = transaction->transactionid;

  return transaction;
}

/* return NULL when transaction does not exist */
transaction_t *
trn_findfromuser(list_t *transactions,sipevent_t *se)
{
  transaction_t *transaction;
  /* transaction MUST already exist in the list of transaction */

  transaction = trn_findid(transactions, se->transactionid);
  se->transactionid = transaction->transactionid;

  return transaction;
}

transaction_t *
trn_findid(list_t *transactions, int transactionid)
{
  transaction_t *transaction;
  int            pos=0;
    
  smutex_lock(fastmutex);
  
  while (!list_eol(transactions,pos))
    {
    transaction = (transaction_t *)list_get(transactions,pos);
    
    if (transaction->transactionid==transactionid)
      {
      smutex_unlock(fastmutex);
      return transaction;
      }
    pos++;
    }
  /*not found */ 
  smutex_unlock(fastmutex);
  return NULL;
}

int
trn_removeid(list_t *transactions, int transactionid)
{
  transaction_t *transaction;
  int            pos=0;
  
  smutex_lock(fastmutex);
  
  while (!list_eol(transactions,pos))
    {
    transaction = (transaction_t *)list_get(transactions,pos);
    
    if (transaction->transactionid==transactionid)
      {
      list_remove(transactions,pos);
      smutex_unlock(fastmutex);
      return 0;
      }
    pos++;
    }
  /*not found */ 
  trace(TRACE_LEVEL1,NULL,"<trn_manager.c> Error while Deleting transaction ...\n");

  smutex_unlock(fastmutex);
  return -1;
}


int
call_id_match(call_id_t *callid1,call_id_t *callid2){

  if (callid1==NULL||callid2==NULL) return -1;
  if (callid1->number==NULL||callid2->number==NULL) return -1;

  if (0!=strcmp(callid1->number,callid2->number))
    return -1;

  if ((callid1->host==NULL)&&(callid2->host==NULL))
     return 0;
  if ((callid1->host==NULL)&&(callid2->host!=NULL))
     return -1;
  if ((callid1->host!=NULL)&&(callid2->host==NULL))
     return -1;
  if (0!=strcmp(callid1->host,callid2->host))
    return -1;

  return 0;
}

int
callleg_match(to_t *to1,from_t *from1,to_t *to2,from_t *from2)
{
  if (to1==NULL||to2==NULL) return -1;
  if (from1==NULL||from2==NULL) return -1;

  if (0==from_compare((from_t *)to1,(from_t *)to2)
      &&0==from_compare(from1,from2))
    return 0;
  return -1;
}

int
cseq_match(cseq_t *cseq1,cseq_t *cseq2)
{
  if (cseq1==NULL||cseq2==NULL) return -1;
  if (cseq1->number==NULL||cseq2->number==NULL
      ||cseq1->method==NULL||cseq2->method==NULL) return -1;

  if (0==strcmp(cseq1->number,cseq2->number))
    {
      if (strcmp(cseq1->method,"INVITE")||
	  strcmp(cseq1->method,"ACK"))
	return 0;
    }
  return -1;
}
