/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <stdio.h>
#include <stdlib.h>

#include <osip/smsg.h>
#include <osip/port.h>

/* enable logging of memory accesses */
#ifdef MEMORY_LEAKS
static int freesipcptr = 0;
#endif


static void
msg_startline_free(startline_t* strtline)
{
  sfree(strtline->sipmethod);
  sfree(strtline->sipversion);
  if (strtline->rquri!=NULL)
    {
      url_free(strtline->rquri);
      sfree(strtline->rquri);
    }
  sfree(strtline->statuscode);
  sfree(strtline->reasonphrase);
}


int
msg_init(sip_t **sip) {
#ifdef MEMORY_LEAKS
  static int comptr = 0;
  comptr++;
  freesipcptr++;
  trace(__FILE__,__LINE__,TRACE_LEVEL0,stdout,"<msg_write.c> msg_init() = %i malloc-free = %i\n",comptr, freesipcptr);
  fflush(stdout); 
#endif
  *sip = (sip_t *) smalloc(sizeof(sip_t));
  (*sip)->from          = NULL;
  (*sip)->to            = NULL;
  (*sip)->call_id       = NULL;
  (*sip)->cseq          = NULL;
  (*sip)->contentlength = NULL;
  (*sip)->mime_version  = NULL;
  (*sip)->content_type  = NULL;

  (*sip)->bodies = (list_t *)smalloc(sizeof(list_t));
  list_init((*sip)->bodies);

  (*sip)->headers = (list_t *)smalloc(sizeof(list_t));
  list_init((*sip)->headers);
  (*sip)->vias = (list_t *)smalloc(sizeof(list_t));
  list_init((*sip)->vias);
  (*sip)->contacts = (list_t *)smalloc(sizeof(list_t));
  list_init((*sip)->contacts);
  (*sip)->record_routes = (list_t *)smalloc(sizeof(list_t));
  list_init((*sip)->record_routes);
  (*sip)->routes = (list_t *)smalloc(sizeof(list_t));
  list_init((*sip)->routes);

  return 0; /* ok */
}

void
msg_setreasonphrase(sip_t *sip, char *reason)
{
  sip->strtline->reasonphrase = reason;
}
void
msg_setstatuscode(sip_t *sip, char *statuscode)
{
  sip->strtline->statuscode = statuscode;
}
void
msg_setmethod(sip_t *sip, char *sipmethod)
{
  sip->strtline->sipmethod = sipmethod;
}
void
msg_setversion(sip_t *sip, char *version)
{
  sip->strtline->sipversion = version;
}
void
msg_seturi(sip_t *sip, url_t *url)
{
  sip->strtline->rquri = url;
}

void
msg_free(sip_t *sip)
{
  int pos = 0;
#ifdef MEMORY_LEAKS
  static int comptr = 0;
  comptr--;
  freesipcptr--;
  trace(__FILE__,__LINE__,TRACE_LEVEL0,stdout,"<msg_write.c> msg_free() = %i malloc-free = %i\n",comptr, freesipcptr);
#endif
  msg_startline_free(sip->strtline);
  sfree(sip->strtline);

  /* mandatory header */
  if (sip->from!=NULL)
    {
      from_free(sip->from);
      sfree(sip->from);
    }
  if (sip->to!=NULL)
    {
      to_free(sip->to);
      sfree(sip->to);
    }
  if (sip->call_id!=NULL)
    {
      call_id_free(sip->call_id);
      sfree(sip->call_id);
    }
  if (sip->cseq!=NULL)
    {
      cseq_free (sip->cseq);
      sfree(sip->cseq);
    }
  if (sip->contentlength!=NULL)
    {
      content_length_free (sip->contentlength);
      sfree(sip->contentlength);
    }
  if (sip->content_type!=NULL)
    {
      content_type_free (sip->content_type);
      sfree(sip->content_type);
    }
  if (sip->mime_version!=NULL)
    {
      mime_version_free (sip->mime_version);
      sfree(sip->mime_version);
    }

  {
    via_t *via;
    while (!list_eol(sip->vias,pos))
      {
	via = (via_t *)list_get(sip->vias,pos);
	list_remove(sip->vias,pos);
	via_free(via);
	sfree(via);
      }
    sfree(sip->vias);
  }
  {
    contact_t *contact;
    while (!list_eol(sip->contacts,pos))
      {
	contact = (contact_t *)list_get(sip->contacts,pos);
	list_remove(sip->contacts,pos);
	contact_free(contact);
	sfree(contact);
      }
    sfree(sip->contacts);
  }
  {
    record_route_t *record_route;
    while (!list_eol(sip->record_routes,pos))
      {
	record_route = (record_route_t *)list_get(sip->record_routes,pos);
	list_remove(sip->record_routes,pos);
	record_route_free(record_route);
	sfree(record_route);
      }
    sfree(sip->record_routes);
  }
  {
    route_t *route;
    while (!list_eol(sip->routes,pos))
      {
	route = (route_t *)list_get(sip->routes,pos);
	list_remove(sip->routes,pos);
	route_free(route);
	sfree(route);
      }
    sfree(sip->routes);
  }
  {
  header_t *header;
  while (!list_eol(sip->headers,pos))
    {
     header = (header_t *)list_get(sip->headers,pos);
     list_remove(sip->headers,pos);
     header_free(header);
     sfree(header);
    }
  sfree(sip->headers);
  }

  {
  body_t *body;
  while (!list_eol(sip->bodies,pos))
    {
     body = (body_t *)list_get(sip->bodies,pos);
     list_remove(sip->bodies,pos);
     body_free(body);
     sfree(body);
    }
  sfree(sip->bodies);
  }
}

