/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <osip/const.h>
#include <osip/sdp.h>
#include <osip/port.h>

#include <stdio.h>
#include <stdlib.h>


/* internal methods */
char *sdp_appendfield        (char *buf, char *field_value    , char *field_name);
char *sdp_appendfields       (char *buf, list_t *field_values , char *field_name);
char *sdp_appendmediadfields (char *buf, list_t *mediads);
char *sdp_appendtimedfields  (char *buf, list_t *timeds);



void
mediad_init(mediad_t *mediad)
{
  mediad->m  = NULL;
  mediad->i  = NULL;

  mediad->cs = (list_t *) smalloc(sizeof(list_t));
  list_init(mediad->cs);

  mediad->bs  = (list_t *) smalloc(sizeof(list_t));
  list_init(mediad->bs);

  mediad->k  = NULL;

  mediad->as = (list_t *) smalloc(sizeof(list_t));
  list_init(mediad->as);
  
}

void
mediad_free(mediad_t *mediad)
{
   FREE_M_FIELD ( mediad );
   FREE_I_FIELD ( mediad );
   FREE_C_FIELDS( mediad );
   FREE_B_FIELDS( mediad );
   FREE_K_FIELD ( mediad );
   FREE_A_FIELDS( mediad );
}

void
timed_init(timed_t *timed)
{
  timed->t  = NULL;
  timed->rs = (list_t *) smalloc(sizeof(list_t));
  list_init(timed->rs);
  
}

void
timed_free(timed_t *timed)
{
   FREE_T_FIELD(timed);
   FREE_R_FIELDS(timed);
}

char *
sdp_initv(char *v)
{
  return sgetcopy(v);
}

char *
sdp_inito(char *username,char *sessid
	  ,char *version,char *nettype
	  ,char *addr_type,char *address)
{

  char *o;
  o = (char *) smalloc(strlen(username)+1
			    +strlen(sessid)+1
			    +strlen(version)+1
			    +strlen(nettype)+1
			    +strlen(addr_type)+1
			    +strlen(address)+1
			    );
  sprintf(o,"%s %s %s %s %s %s",
	  username,sessid,version,
	  nettype,addr_type,address);
  return o;
}


char *
sdp_inits(char *name)
{
  return sgetcopy(name);
}

char *
sdp_initt(char *beg,char *end)
{
  char *t;
  t = (char *) smalloc(strlen(beg)+1+
			    strlen(end)+1);
   sprintf(t,"%s %s",
	   beg,end);
   return t;
}

char *
sdp_initc(char *nettype,char *addr_type,char *addressforconnection)
{
  char *c;
  c = (char *) smalloc(strlen(nettype)+1+
			    strlen(addr_type)+1+
			    strlen(addressforconnection)+1);
   sprintf(c,"%s %s %s",
	   nettype,addr_type,addressforconnection);
   return c;
}

char *
sdp_initm(char *media,char *port,char *transport,char *payloads)
{
  char *m;
  m = (char *) smalloc(strlen(media)+1+
			    strlen(port)+1+
			    strlen(transport)+1+
			    strlen(payloads)+1);
   sprintf(m,"%s %s %s %s",
	   media,port,transport,payloads);
   return m;

}

char *
sdp_inita_asflag(char *flag)
{
  return sgetcopy(flag);
}

char *
sdp_inita_withvalue(char *attr,char *attr_value)
{
  char *a;
  a = (char *) smalloc(strlen(attr)+1+
			    strlen(attr_value)+1);
  sprintf(a,"%s %s",
	  attr,attr_value);
  return a;
}

/* to be changed to sdp_init(sdp_t **dest) */
void
sdp_init(sdp_t *sdp)
{
  sdp->audio_payload = NULL;
  sdp->video_payload = NULL;
  sdp->applications = NULL;

  sdp->v = NULL;
  sdp->o = NULL;
  sdp->s = NULL;
  sdp->i = NULL;
  sdp->u = NULL;

  sdp->es = (list_t *) smalloc(sizeof(list_t));
  list_init(sdp->es);
  sdp->ps = (list_t *) smalloc(sizeof(list_t));
  list_init(sdp->ps);

  sdp->c = NULL;

  sdp->bs = (list_t *) smalloc(sizeof(list_t));
  list_init(sdp->bs);

  sdp->timeds  = (list_t *) smalloc(sizeof(list_t));
  list_init(sdp->timeds);

  sdp->z = NULL;
  sdp->k = NULL;

  sdp->as = (list_t *) smalloc(sizeof(list_t));
  list_init(sdp->as);


  sdp->mediads = (list_t *) smalloc(sizeof(list_t));
  list_init(sdp->mediads);
}

void
sdp_setproperties(sdp_t *sdp            ,list_t *audio_payload,
		  list_t *video_payload ,list_t *applications)
{
  sdp->audio_payload = audio_payload;
  sdp->video_payload = video_payload;
  sdp->applications = applications;
}


int
sdp_parse(sdp_t *sdp,const char *buf)
{

  /* In SDP, headers must be in the right order */
  /* This is a simple example
     v=0
     o=user1 53655765 2353687637 IN IP4 128.3.4.5
     s=Mbone Audio
     i=Discussion of Mbone Engineering Issues
     e=mbone@somewhere.com
     c=IN IP4 224.2.0.1/127
     t=0 0
     m=audio 3456 RTP/AVP 0
     a=rtpmap:0 PCMU/8000
     */

    int pos = 0;
    char *name;
    char *value;

    sdp_init(sdp);
    /* mandatory */
    pos   = sdp_gotonextline   (buf,0);
    name  = sdp_readfieldname  (buf,pos);
    value = sdp_readfieldvalue (buf,pos);
    if (IS_V_FIELD(name))
      {
      sdp->v = value;
      }
    else
      {
      if (name!=NULL)
	sfree(name);
      if (value!=NULL)
	sfree(value);
      DEBUG(fprintf(stdout,"<sdp_rfc2327.c> Error in V while parsing!\n"));
      return SDP_SYNTAXERROR;
      }

    /* mandatory */
    sfree(name);
    pos   = sdp_gotonextline   (buf,pos);
    name  = sdp_readfieldname  (buf,pos);
    value = sdp_readfieldvalue (buf,pos);
    if (IS_O_FIELD(name))
      sdp->o = value;
    else
      {
      if (name!=NULL)
	sfree(name);
      if (value!=NULL)
	sfree(value);
      DEBUG(fprintf(stdout,"<sdp_rfc2327.c> Error in O while parsing!\n"));
      return SDP_SYNTAXERROR;
      }
   
    /* mandatory */
    sfree(name);
    pos   = sdp_gotonextline   (buf,pos);
    name  = sdp_readfieldname  (buf,pos);
    value = sdp_readfieldvalue (buf,pos);
    if (IS_S_FIELD(name))
      {
      sdp->s = value;
      }
    else
      {
      if (name!=NULL)
	sfree(name);
      if (value!=NULL)
	sfree(value);
      DEBUG(fprintf(stdout,"<sdp_rfc2327.c> Error in S while parsing!\n"));
      return SDP_SYNTAXERROR;
      }
  
    sfree(name);
    pos   = sdp_gotonextline   (buf,pos);
    name  = sdp_readfieldname  (buf,pos);
    value = sdp_readfieldvalue (buf,pos);
    if (IS_I_FIELD(name))
      {
      sdp->i = value;
      sfree(name);
      pos   = sdp_gotonextline   (buf,pos);
      name  = sdp_readfieldname  (buf,pos);
      value = sdp_readfieldvalue (buf,pos);
      }

    if (IS_U_FIELD(name))
      {
      sdp->u = value;
      sfree(name);
      pos   = sdp_gotonextline   (buf,pos);
      name  = sdp_readfieldname  (buf,pos);
      value = sdp_readfieldvalue (buf,pos);
      }

    while (IS_E_FIELD(name))
      {
      list_add(sdp->es , value, -1 );
      sfree(name);
      pos   = sdp_gotonextline   (buf,pos);
      name  = sdp_readfieldname  (buf,pos);
      value = sdp_readfieldvalue (buf,pos);
      }

    while (IS_P_FIELD(name))
      {
      list_add(sdp->ps , value, -1 );
      sfree(name);
      pos   = sdp_gotonextline   (buf,pos);
      name  = sdp_readfieldname  (buf,pos);
      value = sdp_readfieldvalue (buf,pos);
      }

    if (IS_C_FIELD(name))
      {
      sdp->c = value;
      sfree(name);
      pos   = sdp_gotonextline   (buf,pos);
      name  = sdp_readfieldname  (buf,pos);
      value = sdp_readfieldvalue (buf,pos);
      }

    if (IS_B_FIELD(name))
      {
      list_add(sdp->bs , value , -1 );
      sfree(name);
      pos   = sdp_gotonextline   (buf,pos);
      name  = sdp_readfieldname  (buf,pos);
      value = sdp_readfieldvalue (buf,pos);
      }
    
    while (IS_T_FIELD(name))
      {
      timed_t *timed;
      timed = (timed_t *) smalloc(sizeof(timed_t));
      timed_init(timed);
      timed->t = value;
      sfree(name);
      pos   = sdp_gotonextline   (buf,pos);
      name  = sdp_readfieldname  (buf,pos);
      value = sdp_readfieldvalue (buf,pos);
      
      while (IS_R_FIELD(name))
	{
	list_add(timed->rs, value , -1 );
	sfree(name);
	pos   = sdp_gotonextline   (buf,pos);
	name  = sdp_readfieldname  (buf,pos);
	value = sdp_readfieldvalue (buf,pos);
	}
      list_add(sdp->timeds , timed , -1 );
      }

    if (IS_Z_FIELD(name))
      {
      sdp->z = value;
      sfree(name);
      pos    = sdp_gotonextline   (buf,pos);
      name   = sdp_readfieldname  (buf,pos);
      value  = sdp_readfieldvalue (buf,pos);
      }
    
    if (IS_K_FIELD(name))
      {
      sdp->k = value;
      sfree(name);
      pos    = sdp_gotonextline   (buf,pos);
      name   = sdp_readfieldname  (buf,pos);
      value  = sdp_readfieldvalue (buf,pos);
      }
    
    while (IS_A_FIELD(name))
      {
      list_add(sdp->as, value , -1 );
      sfree(name);
      pos    = sdp_gotonextline   (buf,pos);
      name   = sdp_readfieldname  (buf,pos);
      value  = sdp_readfieldvalue (buf,pos);
      }
    
    while (IS_M_FIELD(name))
      {
      mediad_t *mediad;
      mediad = (mediad_t *) smalloc(sizeof(mediad_t));
      mediad_init(mediad);
      mediad->m = value;
      sfree(name);
      pos   = sdp_gotonextline   (buf,pos);
      name  = sdp_readfieldname  (buf,pos);
      value = sdp_readfieldvalue (buf,pos);
      
      if (IS_I_FIELD(name))
	{
	mediad->i = value;
	sfree(name);
	pos   = sdp_gotonextline   (buf,pos);
	name  = sdp_readfieldname  (buf,pos);
	value = sdp_readfieldvalue (buf,pos);
	}
      while (IS_C_FIELD(name))
	{
	list_add(mediad->cs , value , -1 );
	sfree(name);
	pos   = sdp_gotonextline   (buf,pos);
	name  = sdp_readfieldname  (buf,pos);
	value = sdp_readfieldvalue (buf,pos);
	}
      while (IS_B_FIELD(name))
	{
	list_add(mediad->bs , value , -1 );
	sfree(name);
	pos   = sdp_gotonextline   (buf,pos);
	name  = sdp_readfieldname  (buf,pos);
	  value = sdp_readfieldvalue (buf,pos);
	}
      if (IS_K_FIELD(name))
	{
	mediad->k = value;
	sfree(name);
	pos   = sdp_gotonextline   (buf,pos);
	name  = sdp_readfieldname  (buf,pos);
	value = sdp_readfieldvalue (buf,pos);
	}
      while (IS_A_FIELD(name))
	{
	list_add(mediad->as , value , -1);
	sfree(name);
	pos   = sdp_gotonextline   (buf,pos);
	name  = sdp_readfieldname  (buf,pos);
	value = sdp_readfieldvalue (buf,pos);
	}
      
      list_add(sdp->mediads,mediad,-1);
      }

    if (name!=NULL)
      {
      TRACE(trace(__FILE__,__LINE__,TRACE_LEVEL3,stdout,"<sdp_rfc2327.c> VALUE END ON NON EMPTY FOR SDP name %s !\n",name));
      sfree(name);
      }
    if (value!=NULL)
      {
      TRACE(trace(__FILE__,__LINE__,TRACE_LEVEL3,stdout,"<sdp_rfc2327.c> VALUE END ON NON EMPTY FOR SDP value %s !\n",value));

      sfree(value);
      }
    
    return 0;
}

/* return the position of the next feild
   return pos if header not found
  */
int
sdp_gotonextline(const char *buf,int pos)
{
  char *ptr;

  ptr = strchr (buf+pos+1, 61) ; /* find next "=" -> 61 */
  if (ptr==NULL)
    {
    return -1; /* error */
    }
/* fprintf(stdout,"<sdp_rfc2327.c> Next header for pos:%i is at pos %i:...\n",pos, ptr-buf);*/

  return ptr-buf; /* so we are right on the = sign */
}

/* return the name of the next field
   return NULL : header not found
*/
char *
sdp_readfieldname(const char *buf,int pos)
{
  char *field_name;

  if (pos==-1)
    return NULL;
  field_name = (char *)smalloc(2);
  sstrncpy(field_name,buf+pos-1,1);

  return field_name;
}

/* return the value of the next field
   return NULL : header not found
*/
char *
sdp_readfieldvalue(const char *buf,int pos)
{
  char *field_value;
  char *end;
  
  if (pos==-1)
    return NULL;
  end = strchr (buf+pos+1, 13) ; /* find "\r" ->13  */


  field_value = (char *)smalloc(end-(buf+pos+1)+1);
  sstrncpy(field_value,buf+pos+1,end-(buf+pos+1));
  return field_value;
}

char *
sdp_appendfield(char *buf,char *field_value, char *field_name)
{
  if (field_value==NULL) 
    return buf;
  sstrncpy(buf,field_name,strlen(field_name)); 
  buf = buf + strlen(buf);
  sstrncpy(buf,field_value,strlen(field_value)); 
  buf = buf + strlen(buf); 
  sstrncpy(buf,CRLF,2);
  buf = buf + 2;
  return buf;
}

char *
sdp_appendfields(char *buf,list_t *field_values, char *field_name)
{
  int pos = 0;
  if (field_values==NULL) 
    return buf;
  
  while (!list_eol(field_values,pos))
    {
    char *field_value;
    field_value = (char *) list_get(field_values , pos);

    sstrncpy(buf,field_name,strlen(field_name)); 
    buf = buf + strlen(buf);
    sstrncpy(buf,field_value,strlen(field_value)); 
    buf = buf + strlen(buf); 
    sstrncpy(buf,CRLF,2);
    buf = buf + 2;

    pos ++;
    }
  return buf;
}

char *
sdp_appendtimedfields(char *buf, list_t *timeds)
{
  int pos = 0;
  if (timeds==NULL) 
    return buf;
  
  while (!list_eol(timeds,pos))
    {
    timed_t *timed;
    timed = (timed_t *) list_get( timeds , pos);

    buf = sdp_appendfield      (buf,  timed->t    ,"t=");
    buf = sdp_appendfields     (buf,  timed->rs   ,"r=");

    pos ++;
    }

  return buf;
}

char *
sdp_appendmediadfields(char *buf, list_t *mediads)
{
  int pos = 0;
  if (mediads==NULL) 
    return buf;
  
  while (!list_eol(mediads,pos))
    {
    mediad_t *mediad;
    mediad = (mediad_t *) list_get( mediads , pos);

    buf = sdp_appendfield      (buf,  mediad->m   ,"m=");
    buf = sdp_appendfield      (buf,  mediad->i   ,"i=");
    buf = sdp_appendfields     (buf,  mediad->cs  ,"c=");
    buf = sdp_appendfields     (buf,  mediad->bs  ,"b=");
    buf = sdp_appendfield      (buf,  mediad->k   ,"k=");
    buf = sdp_appendfields     (buf,  mediad->as  ,"a=");

    pos ++;
    }
  
  return buf;
}

char *
sdp_2char(sdp_t *sdp)
{

  char *tmp;
  char *string;

  if (sdp->v == NULL)
    return NULL;
  if (sdp->o == NULL)
    return NULL;
  if (sdp->s == NULL)
    return NULL;


  tmp = (char *)smalloc(66001*sizeof(char));
  string = tmp;
  sprintf(tmp,"v=%s\r\no=%s\r\ns=%s\r\n",
	  sdp->v , sdp->o , sdp->s );

  tmp = tmp + strlen(tmp);

  tmp = sdp_appendfield       (tmp,  sdp->i   ,"i=");
  tmp = sdp_appendfield       (tmp,  sdp->u   ,"u=");

  tmp = sdp_appendfields      (tmp,  sdp->es  ,"e=");
  tmp = sdp_appendfields      (tmp,  sdp->ps  ,"p=");

  tmp = sdp_appendfield       (tmp,  sdp->c   ,"c=");

  tmp = sdp_appendfields      (tmp,  sdp->bs  ,"b=");

  tmp = sdp_appendtimedfields (tmp,  sdp->timeds);

  tmp = sdp_appendfield       (tmp,  sdp->z   ,"z=");
  tmp = sdp_appendfield       (tmp,  sdp->k   ,"k=");

  tmp = sdp_appendfields      (tmp,  sdp->as  ,"a=");

  tmp = sdp_appendmediadfields(tmp,sdp->mediads);
 
  /* SIPit day1
     sprintf(tmp,"\r\n"); */
  return string;
}

void
sdp_free(sdp_t *sdp)
{
  int pos;

  FREE_V_FIELD( sdp );
  FREE_O_FIELD( sdp );
  FREE_S_FIELD( sdp );
  FREE_I_FIELD( sdp );
  FREE_U_FIELD( sdp );
  FREE_E_FIELDS( sdp );
  FREE_P_FIELDS( sdp );
  FREE_C_FIELD( sdp );
  FREE_B_FIELDS( sdp );
  {
  timed_t *timed;
  pos = 0;
  while (!list_eol(sdp->timeds,pos))
    {
     timed = (timed_t *)list_get(sdp->timeds,pos);
     list_remove(sdp->timeds,pos);
     FREE_TIMED_FIELD(timed);
     sfree(timed);
    }
  sfree(sdp->timeds);
  }

  FREE_Z_FIELD(sdp);
  FREE_K_FIELD(sdp);
  FREE_A_FIELDS(sdp);
  {
  mediad_t *mediad;
  pos = 0;
  while (!list_eol(sdp->mediads,pos))
    {
     mediad = (mediad_t *)list_get(sdp->mediads,pos);
     list_remove(sdp->mediads,pos);
     FREE_MEDIAD_FIELD(mediad);
     sfree(mediad);
    }
  sfree(sdp->mediads);
  }
}

int
mediad_replyto(mediad_t *remote_mediad,mediad_t *dest)
{
  mediad_init(dest);

  if (0==strncmp(remote_mediad->m,"audio",5))
    {
      dest->m = sdp_initm("audio","23456","RTP/AVP","0 1");
    }
  if (0==strncmp(remote_mediad->m,"video",5))
    {
      dest->m = sdp_initm("video","32556","RTP/AVP","32");
    }
  if (0==strncmp(remote_mediad->m,"application",11))
    {
      dest->m = sdp_initm("application","32556","udp","wb");
      list_add(dest->as , sdp_inita_withvalue("orient:","portrait") , -1);
    }
  return 0;
}

int
sdp_replyto(sdp_t *sdp_remote,sdp_t *dest,
	    char *username,char *networktype,
	    char *addr_type,char *localip)
{
  static int sessid = 0;
  sessid++;


  sdp_init(dest);

  dest->v = sdp_initv("0");
  {
    char *sess;
    sess = (char *)smalloc(10);
    sprintf(sess,"%i",sessid);
    dest->o = sdp_inito(username,
			sess,
			sess,
			networktype,
			addr_type,
			localip);
    sfree(sess);
  }

  dest->s = sdp_inits(sdp_remote->s);
  {
    timed_t *timed;
    timed = (timed_t *)smalloc(sizeof(timed_t));
    timed_init(timed);
    timed->t = sdp_initt("0","0");
    list_add(dest->timeds,timed,-1);
  }
  dest->c = sdp_initc("IN","IP4",localip);

  {
    mediad_t *mediad;
    mediad_t *remote_mediad;
    int pos = 0;
    while (!list_eol(sdp_remote->mediads,pos))
      {
	int i;
	remote_mediad = (mediad_t *)list_get(sdp_remote->mediads,pos);

	mediad = (mediad_t *)smalloc(sizeof(mediad_t));
	i = mediad_replyto(remote_mediad,mediad);
	if (0>i)
	  {
	    return i;
	  }
	list_add(dest->mediads,mediad,-1);
	pos++;
      }
  }

  
  return 0; /* no media */
}
