/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#ifndef _OSIP_H_
#define _OSIP_H_

#include <time.h>

#ifdef __sun
#include <sys/types.h>
#endif

#include <osip/const.h>
#include <osip/smsg.h>
#include <osip/fifo.h>


#define NO_RESPONSE_TIMEOUT 60
#define END_TRANSACTION_TIMEOUT 10
#define SIP_MESSAGE_MAX_LENGTH 20000


typedef enum _state_t {
  /* states for UAC/UAS behavior */
  INITIAL,
  CALLING,
  PROCEEDING,
  ANSWERED,
  COMPLETED,

  /* these are the states for PROXY-STATEFULL */
  REQUEST_FORWARDED,
  RESPONSE_FORWARDED,
  ACK_FORWARDED,

  /* Possible reasons of ERRORS */
  NETWORK_ERROR,
  SYNTAX_ERROR,
  GLOBAL_ERROR
} state_t;

typedef enum _type_t {
  TIMEOUT,

  /* FOR INCOMING TRANSACTION */
  RCV_REQINVITE,
  RCV_REQACK,
  RCV_REQUEST,
  RCV_STATUS_1XX,
  RCV_STATUS_23456XX,

  /* FOR OUTGOING TRANSACTION */
  SND_REQINVITE,
  SND_REQACK,
  SND_REQUEST,
  SND_STATUS_1XX,
  SND_STATUS_23456XX,

  KILL_TRANSACTION,
  UNKNOWN_EVT
} type_t;

typedef struct _statemachine_t {
  list_t *transitions;
} statemachine_t;

typedef struct _transaction_t {
  
  void *your_instance;       /* add whatever you want here.       */
  int transactionid;         /* simple id used to identify the tr.*/
  fifo_t *transactionff;     /* events must be added in this fifo */

  from_t    *from;           /* CALL-LEG definition */
  to_t      *to;
  call_id_t *callid;
  cseq_t    *cseq;
  sip_t *lastrequest;        /* last request received or sent     */
  sip_t *lastresponse;       /* last response received or sent    */

  state_t state;             /* state of transaction              */
  statemachine_t *statemachine; /* state-machine of transaction   */

  time_t birth_time;         /* birth_date of transaction         */
  time_t completed_time;     /* end   date of transaction         */
  int retransmissioncounter; /* facilities for internal timer     */

  url_t *proxy;              /* url used to send requests        */    
  void *config;            /* transaction is managed by config */
} transaction_t;

typedef struct _sipevent_t {
  type_t type;
  int transactionid;
  sip_t *sip;
} sipevent_t;

typedef struct _osip_t {

  list_t *transactions;      /* this is the list of transactions */

  /* when proxy is not set, the req-URI is used */
  url_t *proxy;              /* url used to send requests */

  /* callbacks for state machines */
  void (*cb_rcvinvite)(sipevent_t*,transaction_t*);
  void (*cb_rcvack)(sipevent_t*,transaction_t*);
  void (*cb_rcvbye)(sipevent_t*,transaction_t*);
  void (*cb_rcvcancel)(sipevent_t*,transaction_t*);
  void (*cb_rcvinfo)(sipevent_t*,transaction_t*);
  void (*cb_rcvoptions)(sipevent_t*,transaction_t*);
  void (*cb_rcvregister)(sipevent_t*,transaction_t*);
  void (*cb_rcvprack)(sipevent_t*,transaction_t*);
  void (*cb_rcvunkrequest)(sipevent_t*,transaction_t*);

  void (*cb_snd1xx)(sipevent_t*,transaction_t*);
  void (*cb_snd2xx)(sipevent_t*,transaction_t*);
  void (*cb_snd3xx)(sipevent_t*,transaction_t*);
  void (*cb_snd4xx)(sipevent_t*,transaction_t*);
  void (*cb_snd5xx)(sipevent_t*,transaction_t*);
  void (*cb_snd6xx)(sipevent_t*,transaction_t*);

  void (*cb_sndinvite)(sipevent_t*,transaction_t*);
  void (*cb_sndack)(sipevent_t*,transaction_t*);
  void (*cb_sndbye)(sipevent_t*,transaction_t*);
  void (*cb_sndcancel)(sipevent_t*,transaction_t*);
  void (*cb_sndinfo)(sipevent_t*,transaction_t*);
  void (*cb_sndoptions)(sipevent_t*,transaction_t*);
  void (*cb_sndregister)(sipevent_t*,transaction_t*);
  void (*cb_sndprack)(sipevent_t*,transaction_t*);
  void (*cb_sndunkrequest)(sipevent_t*,transaction_t*);

  void (*cb_rcv1xx)(sipevent_t*,transaction_t*);
  void (*cb_rcv2xx)(sipevent_t*,transaction_t*);
  void (*cb_rcv3xx)(sipevent_t*,transaction_t*);
  void (*cb_rcv4xx)(sipevent_t*,transaction_t*);
  void (*cb_rcv5xx)(sipevent_t*,transaction_t*);
  void (*cb_rcv6xx)(sipevent_t*,transaction_t*);

  void (*cb_rcvresp_retransmission)(transaction_t*);
  void (*cb_sndreq_retransmission)(transaction_t*);
  void (*cb_sndresp_retransmission)(transaction_t*);
  void (*cb_rcvreq_retransmission)(transaction_t*);

  void (*cb_killtransaction)(transaction_t*);
  void (*cb_endoftransaction)(transaction_t*);

  void (*cb_connection_refused)(transaction_t*);
  void (*cb_network_error)(transaction_t*);


  /* When application request a timer for a transaction   */
  /* a pointer to this transaction is added in those fifo */
  fifo_t *uas_timerff;
  fifo_t *uac_timerff;
  /* The timers method MAY use those list to store the       */
  /* actual list of transactions that needs retransmissions. */
  list_t *uas_transactions;
  list_t *uac_transactions;

} osip_t;


int  osip_global_init();
int  osip_init(osip_t **osip);
int  osip_init_timers(osip_t *osip);
void osip_init_proxy(osip_t *osip, url_t *prox);
int  osip_execute(osip_t *config);

void osip_setcb_rcvinvite  (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcvack     (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcvbye     (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcvcancel  (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcvinfo    (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcvoptions (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcvregister(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcvprack   (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcvunkrequest(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));

void osip_setcb_sndinvite  (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_sndack     (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_sndbye     (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_sndcancel  (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_sndinfo    (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_sndoptions (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_sndregister(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_sndprack   (osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_sndunkrequest(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));

void osip_setcb_rcv1xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcv2xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcv3xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcv4xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcv5xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_rcv6xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));

void osip_setcb_snd1xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_snd2xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_snd3xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_snd4xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_snd5xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));
void osip_setcb_snd6xx(osip_t *cf,void (*cb)(sipevent_t*,transaction_t*));

void osip_setcb_rcvresp_retransmission(osip_t *cf,void (*cb)(transaction_t*));
void osip_setcb_sndreq_retransmission (osip_t *cf,void (*cb)(transaction_t*));
void osip_setcb_sndresp_retransmission(osip_t *cf,void (*cb)(transaction_t*));
void osip_setcb_rcvreq_retransmission (osip_t *cf,void (*cb)(transaction_t*));

void osip_setcb_killtransaction   (osip_t *cf,void (*cb)(transaction_t*));
void osip_setcb_endoftransaction  (osip_t *cf,void (*cb)(transaction_t*));
void osip_setcb_connection_refused(osip_t *cf,void (*cb)(transaction_t*));
void osip_setcb_network_error     (osip_t *cf,void (*cb)(transaction_t*));

transaction_t *osip_distribute_event(osip_t *config,sipevent_t* sipevent);
sipevent_t    *osip_parse(char *buf);
sipevent_t    *osip_new_event(type_t type,int transactionid);
sipevent_t    *osip_new_outgoing_sipmessage(sip_t *sip);
sipevent_t    *osip_new_incoming_sipmessage(sip_t *sip);
int            osip_execute(osip_t *config);

transaction_t *osip_find_asincomingmessage(osip_t *config,
					   list_t *transactions,
					   sipevent_t *se);
transaction_t *osip_find_asoutgoingmessage(list_t *transactions,
					   sipevent_t *se);
transaction_t *osip_find_byid(list_t *transactions, int transactionid);
int            osip_remove_byid(list_t *transactions, int transactionid);

int           transaction_init(osip_t *config, transaction_t *dest,
			       to_t *to, from_t *from, call_id_t *callid,
			       cseq_t *cseq);
void          transaction_free(transaction_t *transaction);
int           transaction_execute(transaction_t *transaction, sipevent_t *se);

  /* FOR INCOMING TRANSACTION */
#define EVT_IS_RCV_INVITE(event)       (event->type==RCV_REQINVITE)
#define EVT_IS_RCV_ACK(event)          (event->type==RCV_REQACK)
#define EVT_IS_RCV_REQUEST(event)      (event->type==RCV_REQUEST)
#define EVT_IS_RCV_STATUS_1XX(event)   (event->type==RCV_STATUS_1XX)
#define EVT_IS_RCV_STATUS_23456XX(event)   (event->type==RCV_STATUS_23456XX)


/* FOR OUTGOING TRANSACTION */
#define EVT_IS_SND_INVITE(event)       (event->type==SND_REQINVITE)
#define EVT_IS_SND_ACK(event)          (event->type==SND_REQACK)
#define EVT_IS_SND_REQUEST(event)      (event->type==SND_REQUEST)
#define EVT_IS_SND_STATUS_1XX(event)   (event->type==SND_STATUS_1XX)
#define EVT_IS_SND_STATUS_23456XX(event)   (event->type==SND_STATUS_23456XX)

#define EVT_IS_INCOMINGMSG(event)      (event->type>=RCV_REQINVITE \
                	               &&event->type<=RCV_STATUS_23456XX)
#define EVT_IS_INCOMINGREQ(event)      (EVT_IS_RCV_INVITE(event) \
                                       ||EVT_IS_RCV_ACK(event) \
                                       ||EVT_IS_RCV_REQUEST(event))
#define EVT_IS_INCOMINGRESP(event)     (EVT_IS_RCV_STATUS_1XX(event) \
                                       ||EVT_IS_RCV_STATUS_23456XX(event))
#define EVT_IS_OUTGOINGMSG(event)      (event->type>=SND_REQINVITE \
                	               &&event->type<=SND_STATUS_23456XX)
#define EVT_IS_OUTGOINGREQ(event)      (EVT_IS_SND_INVITE(event) \
                                       ||EVT_IS_SND_ACK(event) \
                                       ||EVT_IS_SND_REQUEST(event))
#define EVT_IS_OUTGOINGRESP(event)     (EVT_IS_SND_STATUS_1XX(event) \
                                       ||EVT_IS_SND_STATUS_23456XX(event))

#define EVT_IS_MSG(event)              (event->type>=RCV_REQINVITE \
                	               &&event->type<=SND_STATUS_23456XX)
#define EVT_IS_KILL_TRANSACTION(event) (event->type==KILL_TRANSACTION)
#define EVT_IS_UNKNOWN_EVT(event)      (event->type==UNKNOWN_EVT)
#define EVT_IS_TIMEOUT(event)          (event->type==TIMEOUT)


#endif
