/*
  The oSIP library implements the Session Initiation Protocol (SIP -rfc2543-)
  Copyright (C) 2001  Aymeric MOIZARD jack@atosc.org
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>

#include <osip/smsg.h>
#include <osip/port.h>
#include "msg.h"



static int
startline_parsereq(startline_t *dest, char *buf
		   , char **headers)
{
  char *p1;
  char *p2;
  char *requesturi;
  int i;

  dest->sipmethod    = NULL;
  dest->statuscode   = NULL;
  dest->reasonphrase = NULL;
  
  /* The first token is the method name: */
  p2 = strchr(buf,' ');  
  if (p2==NULL) return -1;
  dest->sipmethod = (char *) smalloc(p2-buf+1);
  sstrncpy(dest->sipmethod,buf,p2-buf);

  /* The second token is a sip-url or a uri: */
  p1 = strchr(p2+2, ' '); /* no space allowed inside sip-url */
  if (p1-p2<2) return -1;
  requesturi = (char *)smalloc(p1-p2);
  sstrncpy(requesturi,p2+1,(p1-p2-1));
  sclrspace(requesturi);

  url_init(&(dest->rquri));
  i = url_parse(dest->rquri, requesturi);
  sfree(requesturi);
  if (i==-1) return -1;

  /* find the the version and the beginning of headers */
  *headers = p1;
  while (strncmp((const char *)(*headers) , (const char *) CRLF, 2))
      (*headers)++;

  if ((*headers)-p1<2) return -1;
  dest->sipversion = (char *) smalloc((*headers)-p1);
  sstrncpy(dest->sipversion,p1+1,((*headers)-p1-1));

  (*headers) = (*headers) + 2;
  return 0;
}

static int
startline_parseresp(startline_t *dest, char *buf,
		    char **headers)
{
  char *statuscode;
  char *reasonphrase;

  dest->rquri     = NULL;
  dest->sipmethod = NULL;

  *headers = buf;

  statuscode = strchr(buf,32);  /* search for fisrt SPACE */
  if (statuscode==NULL) return -1;
  dest->sipversion = (char *) smalloc(statuscode-(*headers)+1);
  sstrncpy(dest->sipversion, *headers, statuscode-(*headers));

  reasonphrase = strchr(statuscode+1,32);
  dest->statuscode = (char *) smalloc(reasonphrase-statuscode);
  sstrncpy(dest->statuscode, statuscode+1, reasonphrase-statuscode-1);
  (*headers) = reasonphrase;
  while (strncmp((const char *) (*headers), (const char *) CRLF, 2)) 
      (*headers)++;
  dest->reasonphrase = (char *) smalloc((*headers)-reasonphrase);
  sstrncpy(dest->reasonphrase, reasonphrase+1, (*headers)-reasonphrase-1);
  (*headers) = (*headers) + 2;
  return 0;
}

int
msg_startline_init(startline_t **stl)
{
  (*stl) = (startline_t *)smalloc(sizeof(startline_t));
  (*stl)->sipmethod    = NULL;
  (*stl)->sipversion   = NULL;
  (*stl)->rquri        = NULL;
  (*stl)->statuscode   = NULL;
  (*stl)->reasonphrase = NULL;
  return 0;
}

/* return size of request URI */
static int
msg_startline_parse(startline_t *dest, char *buf,
		    char **headers) {

  if (0==strncmp((const char *) buf, (const char *) "SIP/", 4))
    return startline_parseresp(dest, buf, headers);
  else
    return startline_parsereq(dest, buf, headers);
}

int
find_next_occurence(char *str, char *buf,
		    char **index_of_str)
{
  char *tmp;
  int len;

  *index_of_str = NULL;
  if (buf == NULL) return -1;

  tmp = buf;
  len = strlen(str);
  while (strlen(tmp)>=len)
    {
      if (strncmp(tmp,str,len)==0)
	{ /* we found the occurrence */
	  *index_of_str = tmp;
	  return 0;
	}
      tmp++;
    }
  return -1;
}

int
find_next_crlf(char *start_of_header,
	       char **end_of_header)
{
  char *tmp;
  char *cr;
  char *lf;

  while (1) {
    cr = strchr(start_of_header, 13); /* find CR */
    lf = strchr(start_of_header, 10);    /* find LF */
    
    *end_of_header = NULL;
    
    if ((cr==NULL)&&(lf==NULL))
      {
	DEBUG(fprintf(stdout,"No crlf found at the end of header..\n"));
	return -1;
      }
    if ((cr!=NULL)&&(lf!=NULL))
      {
	if (cr==lf-1)  tmp = lf; /* case 1: CRLF is the separator */
	else {
	  if (lf<cr)   tmp = lf; /* case 2: LF is the separator */
	  else         tmp = cr; /* case 3: LF is the separator */
	}
      }
    else
      {
	if (cr==NULL)
	  tmp = lf;              /* case 2: LF is the separator */
	else
	  tmp = cr;              /* case 3: LF is the separator */
      }
    
    /* VERIFY if TMP is the end of header or LWS.            */
    /* LWS are extra SP, HT, CR and LF contained in headers. */
    if (strncmp(tmp+1,(const char *)SP,1)==0
	||strncmp(tmp+1,(const char *)"\t",1)==0)
      { /* YES, this is LWS extra characters    */
	/* restart search of the end of headers */
	if ((strncmp(tmp-1,"\n",1)==0)
	    ||(strncmp(tmp-1,"\r",1)==0))
	  {
	    strncpy(tmp-1," ",1);
	  }
	strncpy(tmp," ",1);
	start_of_header = tmp + 1;
      }
    else
      {
	*end_of_header = tmp+1; /* beggining of next header */
	return 0;
      }
  }
}

int
find_next_crlfcrlf(char *start_of_part,
		   char **end_of_part)
{
  char *start_of_line;
  char *end_of_line;
  int i;

  start_of_line = start_of_part;

  while (1) {
      i = find_next_crlf(start_of_line,
			 &end_of_line);
      if (i==-1) { /* error case??? no end of body found */
	DEBUG(fprintf(stdout,"<msg_parser.c> find_next_crlfcrlf failed..\n"));
	return -1;
      }
      if (strncmp(end_of_line, CRLF, 2)==0)
	{
	  *end_of_part = end_of_line;
	  return 0;
	}
      start_of_line = end_of_line;
  }
}

int
msg_set_header(sip_t *sip, char *hname, char *hvalue)
{
  int my_index;
  stolowercase(hname);

  /* some headers are analysed completely      */
  /* this method is used for selective parsing */
  my_index = parser_isknownheader(hname);
  if (my_index>=0) /* ok */
    {
      int ret;
      ret =  parser_callmethod(my_index, sip, hvalue);
      if (ret==-1)
	{
	  TRACE(trace(__FILE__,__LINE__,TRACE_LEVEL1,stdout,"errorinheaderformat \"%s\" %s\n",hname,hvalue));
	  return -1;
	}
      return 0;
    }
  /* unknownheader */
  if (msg_setheader(sip, hname, hvalue)==-1)
    {
      DEBUG(fprintf(stdout,"<msg_parser.c> msg_setheader failed...\n"));
      return -1;
    }

  return 0;
}

int
msg_handle_multiple_values(sip_t *sip, char *hname,
			   char *hvalue)
{
  int i;
  char *ptr;     /* current location of the search */
  char *comma;  /* This is the separator we are elooking for */
  char *beg;     /* beg of a header */
  char *end;     /* end of a header */
  char *quote1; /* first quote of a pair of quotes   */
  char *quote2; /* second quuote of a pair of quotes */


  beg = hvalue;
  end = NULL;
  ptr = hvalue;
  if (hvalue==NULL)
    {
      i = msg_set_header(sip, hname, hvalue);
      if (i==-1) return -1;
      return 0;
    }

    comma = strchr(ptr,',');
  

  stolowercase(hname);

  if (comma==NULL||strncmp(hname,"organization",12)==0
      ||strncmp(hname,"subject",7)==0)
    /* there is no multiple header! likely      */
    /* to happen most of the time...            */
    /* or hname is a TEXT-UTF8-TRIM and may     */
    /* contain a comma. this is not a separator */
    /* THIS DOES NOT WORK FOR UNKNOWN HEADER!!!!*/
    {
      i = msg_set_header(sip, hname, hvalue);
      if (i==-1) return -1;
      return 0;
    }

  while (comma!=NULL)
    {
      quote1  = quote_find(ptr);
      if (quote1!=NULL)
	{
	  quote2  = quote_find(quote1+1);
	  if (quote2==NULL) return -1; /* quotes comes by pair */
	  ptr = quote2+1;
	}

      if ((quote1==NULL)||(quote1>comma))
	{
	  end = comma;
	  comma = strchr(comma+1,',');
	  ptr = comma+1;
	}
      else if ((quote1<comma)&&(quote2<comma))
	{ /* quotes are located before the comma, */
	  /* continue the search for next quotes  */
	  ptr = quote2+1;
	}
      else if ((quote1<comma)&&(comma<quote2))
	{ /* if comma is inside the quotes... */
	  /* continue with the next comma.    */
	  ptr = quote2 + 1;
	  comma = strchr(ptr,',');
	  if (comma==NULL)
	    /* this header last at the end of the line! */
	    { /* this one does not need an allocation... */
	      if (strlen(beg)<2) return 0; /* empty header */
	      sclrspace(beg);
	      i = msg_set_header(sip, hname, beg);
	      if (i==-1) return -1;
	      return 0;
	    }
	}

      if (end!=NULL)
	{
	  char *avalue;
	  if (end-beg+1<2) return -1;
	  avalue = (char *)smalloc(end-beg+1);
	  sstrncpy(avalue, beg, end-beg);
	  sclrspace(avalue);
	  /* really store the header in the sip structure */
	  i = msg_set_header(sip, hname, avalue);
	  sfree(avalue);
	  if (i==-1) return -1;
	  beg = end + 1;
	  end = NULL;
	  if (comma==NULL)
	    /* this header last at the end of the line! */
	    { /* this one does not need an allocation... */
	      if (strlen(beg)<2) return 0; /* empty header */
	      sclrspace(beg);
	      i = msg_set_header(sip, hname, beg);
	      if (i==-1) return -1;
	      return 0;
	    }

	}
    }
  return -1; /* if comma is NULL, we should have already return 0 */
}

/* set all headers */
int
msg_headers_parse(sip_t *sip, char *start_of_header,
		  char **body)
{
  char *colon_index; /* index of ':' */
  char *hname;
  char *hvalue;
  char *end_of_header;
  int  i;

  while (1)
    {
      i = find_next_crlf(start_of_header,
			 &end_of_header);
      if (i==-1)
	{
	  DEBUG(fprintf(stdout,"<msg_parser.c> End of header Not found\n"));       /* end of header not found... */
	  return -1;     /* this is an error case!     */
	}
      /* find the headere name */
      colon_index = strchr(start_of_header, ':');
      if (colon_index==NULL)
	{
	  DEBUG(fprintf(stdout,"<msg_parser.c> End of header Not found\n"));
	  return -1; /* this is also an error case */
	}
      if (colon_index-start_of_header+1<2) return -1;
      hname = (char *)smalloc(colon_index-start_of_header+1);
      sstrncpy(hname, start_of_header, colon_index-start_of_header);
      sclrspace(hname);


      if ((end_of_header-2)-colon_index<2)
	hvalue = NULL; /* some headers (subject) can be empty */
      else
	{
	hvalue = (char *)smalloc((end_of_header-2)-colon_index);
	sstrncpy(hvalue, colon_index+1,
		 (end_of_header-2)-colon_index-1);
	sclrspace(hvalue);
	/* this string must not contain LWS */
	}
      
      /* hvalue MAY contains multiple value. In this case, they   */
      /* are separated by commas. But, a comma may be part of a   */
      /* quoted-string ("here, and there" is an example where the */
      /* comma is not a separator!) */
      i = msg_handle_multiple_values(sip, hname, hvalue);

      sfree(hname);
      sfree(hvalue);
      if (i==-1)
	{
	  DEBUG(fprintf(stdout,"<msg_parser.c> End of header Not found\n"));
	  return -1;
	}

      /* the list of headers MUST always end with  */
      /* CRLFCRLF (also CRCR and LFLF are allowed) */
      if (strncmp(end_of_header, CRLF, 2)==0
	  ||strncmp(end_of_header, "\n", 1)==0
	  ||strncmp(end_of_header, "\r", 1)==0)
	{
	  *body = end_of_header;
	  return 0;      /* end of header found        */
	}
      /* continue on the next header */
      start_of_header = end_of_header;
    }

  DEBUG(fprintf(stdout,"<msg_parser.c> Not possible....\n"));
  return -1;
}


/* internal method to parse the body */
int
msg_body_parse(sip_t *sip, char *start_of_buf,
	       char **next_body)
{
  char *start_of_body;
  char *end_of_body;
  char *tmp;
  int i;

  char *sep_boundary;
  generic_param_t *ct_param;

  /* if MIME-Version: does not exist we just have */
  /* to deal with one body and no header... */
  if (sip->mime_version==NULL)
    {   /* Mime-Version header does NOT exist */
      if (sip->contentlength==NULL)
	{ /* if content_length does not exist, set it to 0 */
	  char *tmp = sgetcopy("0");
	  i = msg_setcontent_length(sip, tmp);
	  sfree(tmp);
	  if (i==-1) return -1;
	}
      if (sip->content_type==NULL)
	return 0; /* no body is attached */
      else
	{
	    int body_len;
	  /* get rid of the first CRLF */	  
	  if (strncmp( start_of_buf, CRLF, 2)==0)
	    start_of_body = start_of_buf + 2;
	  else
	    {
	      if ((strncmp(start_of_buf, "\n", 1)==0)
		  ||(strncmp(start_of_buf, "\r", 1)==0))
		start_of_body = start_of_buf + 1;
	      else return -1; /* message does not end with CRLFCRLF, CRCR or LFLF */
	    }

	  body_len = satoi(sip->contentlength->value);
	  
	  if (body_len>strlen(start_of_body)) /* we do not receive the */
	    return -1;          /* complete message      */
	  /* end_of_body = start_of_body + strlen(start_of_body);*/
	  end_of_body = start_of_body + body_len;
	  tmp = smalloc(end_of_body-start_of_body+1);
	  sstrncpy(tmp, start_of_body, end_of_body-start_of_body);
	  
	  i = msg_setbody(sip, tmp);
	  sfree(tmp);
	  if (i==-1) return -1;
	  return 0;
	}
    }

  /* find the boundary */
 i = generic_param_getbyname(sip->content_type->content_type_params,
			    "boundary",
			    &ct_param);
 if (i!=0) return -1;

 if (ct_param==NULL) return -1;
 if (ct_param->gvalue==NULL) return -1; /* No boundary but multiple headers??? */
  
  sep_boundary = (char *)smalloc(strlen(ct_param->gvalue)+3);
  sprintf(sep_boundary,"--%s",ct_param->gvalue);

  *next_body = NULL;
  start_of_body = start_of_buf;
  while (1)
    {
      i = find_next_occurence(sep_boundary,start_of_body,
			      &start_of_body);
      if (i==-1) {
	sfree(sep_boundary);
	return -1;
      }
      i = find_next_occurence(sep_boundary,start_of_body+strlen(sep_boundary),
			      &end_of_body);
      if (i==-1) {
	sfree(sep_boundary);
	return -1;
      }

      /* this is the real beginning of body */
      start_of_body = start_of_body+strlen(sep_boundary)+2;

      tmp = smalloc(end_of_body-start_of_body+1);
      sstrncpy(tmp, start_of_body, end_of_body-start_of_body);
      
      i = msg_setbody_mime(sip, tmp);
      sfree(tmp);
      if (i==-1) {
	sfree(sep_boundary);
	return -1;
      }

      if (strncmp(end_of_body+strlen(sep_boundary), "--", 2)==0)
	{ /* end of all bodies */
	  *next_body = end_of_body;
	  sfree(sep_boundary);
	  return 0;
	}
      /* continue on the next body */
      start_of_body = end_of_body;
    }
  sfree(sep_boundary);
  return -1;
}


/* sip_t *sip is filled while analysing buf */
int
msg_parse(sip_t *sip, char *buf) {
  int i;
  char *next_header_index;

  /* parse request or status line */
  msg_startline_init(&(sip->strtline));
  i = msg_startline_parse(sip->strtline, buf,
			  &next_header_index);
  if (i==-1)
    {
      DEBUG(fprintf(stdout,"error in startline\n"));
      return -1;
    }
  buf = next_header_index;

  /* parse headers */
  i = msg_headers_parse(sip, buf, &next_header_index);
  if (i==-1) 
    {
      DEBUG(fprintf(stdout,"error in headers\n"));
      return -1;
    }
  buf = next_header_index;


  /* this is mantory in the oSIP stack */
  if (sip->contentlength == NULL)
	msg_setcontent_length(sip, "0");

  /* this is a *very* simple test... */
  if (strlen(buf)<5) return 0; /* no body found */

    
  i = msg_body_parse(sip, buf, &next_header_index);
  if (i==-1) {
    DEBUG(fprintf(stdout,"msg_body_parse return -1\n"));
    return -1;
  }
  buf = next_header_index;

  return 0;
}


char *
msg_getreason(int replycode)
{
  int i;
  i = replycode/100;
  if (i==1)
    {/* 1xx  */
      if (replycode==100)
	return sgetcopy("Trying");
      if (replycode==180)
	return sgetcopy("Ringing");
      if (replycode==181)
	return sgetcopy("Call Is Being Forwarded");
      if (replycode==182)
	return sgetcopy("Queued");
      if (replycode==183)
	return sgetcopy("Session Progress");
    }
  if (i==2)
    {/* 2xx */
	return sgetcopy("OK");
    }
  if (i==3)
    {/* 3xx */
      if (replycode==300)
	return sgetcopy("Multiple Choices");
      if (replycode==301)
	return sgetcopy("Moved Permanently");
      if (replycode==302)
	return sgetcopy("Moved Temporarily");
      if (replycode==305)
	return sgetcopy("Use Proxy");
      if (replycode==380)
	return sgetcopy("Alternative Service");
    }
  if (i==4)
    {/* 4xx */
      if (replycode==400)
	return sgetcopy("Bad Request");
      if (replycode==401)
	return sgetcopy("Unauthorized");
      if (replycode==402)
	return sgetcopy("Payment Required");
      if (replycode==403)
	return sgetcopy("Forbidden");
      if (replycode==404)
	return sgetcopy("Not Found");
      if (replycode==405)
	return sgetcopy("Method Not Allowed");
      if (replycode==406)
	return sgetcopy("Not Acceptable");
      if (replycode==407)
	return sgetcopy("Proxy Authentication Required");
      if (replycode==408)
	return sgetcopy("Request Timeout");
      if (replycode==409)
	return sgetcopy("Conflict");
      if (replycode==410)
	return sgetcopy("Gone");
      if (replycode==411)
	return sgetcopy("Length Required");
      if (replycode==413)
	return sgetcopy("Request Entity Too Large");
      if (replycode==414)
	return sgetcopy("Request-URI Too Large");
      if (replycode==415)
	return sgetcopy("Unsupported Media Type");
      if (replycode==420)
	return sgetcopy("Bad Extension");
      if (replycode==480)
	return sgetcopy("Temporarily not available");
      if (replycode==481)
	return sgetcopy("Call Leg/Transaction Does Not Exist");
      if (replycode==482)
	return sgetcopy("Loop Detected");
      if (replycode==483)
	return sgetcopy("Too Many Hops");
      if (replycode==484)
	return sgetcopy("Address Incomplete");
      if (replycode==485)
	return sgetcopy("Ambiguous");
      if (replycode==486)
	return sgetcopy("Busy Here");
      if (replycode==487)
	return sgetcopy("Request Cancelled");
      if (replycode==488)
	return sgetcopy("Not Acceptable Here");
    }
  if (i==5)
    {/* 5xx */
      if (replycode==500)
	return sgetcopy("Internal Server Error");
      if (replycode==501)
	return sgetcopy("Not Implemented");
      if (replycode==502)
	return sgetcopy("Bad Gateway");
      if (replycode==503)
	return sgetcopy("Service Unavailable");
      if (replycode==504)
	return sgetcopy("Gateway Time-out");
      if (replycode==505)
	return sgetcopy("SIP Version not supported");
    }
  if (i==6)
    {/* 6xx */
      if (replycode==600)
	return sgetcopy("Busy Everywhere");
      if (replycode==603)
	return sgetcopy("Decline");
      if (replycode==604)
	return sgetcopy("Does not exist anywhere");
      if (replycode==606)
	return sgetcopy("Not Acceptable");
    }

  return NULL;
}

