/*
** Copyright (C) 2002,2003 Erik de Castro Lopo <erikd@mega-nerd.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <samplerate.h>

#include "util.h"
#include "calc_snr.h"
#include "config.h"

#define	BUFFER_LEN		(1<<18)
#define	BLOCK_LEN		(234)

#define	ARRAY_LEN(x)	((int) (sizeof (x) / sizeof ((x) [0])))
#define	MIN(x,y)		((x) < (y) ? (x) : (y))

#define	NEW_CODE		0

static void multi_channel_test (int converter, int channel_count) ;

#if NEW_CODE	
static void simple_multi_channel_test (int converter, int channel_count, double target_snr) ;
static void process_multi_channel_test (int converter, int channel_count, double target_snr) ;
#endif

int
main (void)
{
	/* Force output of the Electric Fence banner message. */
	force_efence_banner () ;

	puts ("\n    Zero Order Hold interpolator :") ;
	multi_channel_test (SRC_ZERO_ORDER_HOLD, 1) ;
	multi_channel_test (SRC_ZERO_ORDER_HOLD, 2) ;
	multi_channel_test (SRC_ZERO_ORDER_HOLD, 3) ;
	multi_channel_test (SRC_ZERO_ORDER_HOLD, 7) ;

	puts ("\n    Linear interpolator :") ;
	multi_channel_test (SRC_LINEAR, 1) ;
	multi_channel_test (SRC_LINEAR, 2) ;
	multi_channel_test (SRC_LINEAR, 3) ;
	multi_channel_test (SRC_LINEAR, 7) ;

	puts ("\n    Sinc interpolator :") ;
	multi_channel_test (SRC_SINC_FASTEST, 1) ;
	multi_channel_test (SRC_SINC_FASTEST, 2) ;
	multi_channel_test (SRC_SINC_FASTEST, 3) ;
	multi_channel_test (SRC_SINC_FASTEST, 7) ;

#if NEW_CODE	
	simple_multi_channel_test  (SRC_SINC_FASTEST, 1, 120.0) ;
	simple_multi_channel_test  (SRC_SINC_FASTEST, 2, 120.0) ;
	process_multi_channel_test (SRC_SINC_FASTEST, 3, 120.0) ;
	process_multi_channel_test (SRC_SINC_FASTEST, 4, 120.0) ;
	process_multi_channel_test (SRC_SINC_FASTEST, 7, 120.0) ;

	puts ("\n    Extra tests :") ;
	simple_multi_channel_test (SRC_LINEAR, 1, 118.0) ;
	simple_multi_channel_test (SRC_LINEAR, 2, 120.0) ;
	simple_multi_channel_test (SRC_LINEAR, 3, 120.0) ;
	simple_multi_channel_test (SRC_LINEAR, 4, 120.0) ;
	simple_multi_channel_test (SRC_LINEAR, 7, 120.0) ;
#endif

	puts ("") ;

	return 0 ;
} /* main */

/*==============================================================================
*/

#define	CHANNEL_FUNC(x)		(2.0 / ((x) + 1) - 1.0)

static void
multi_channel_test (int converter, int channel_count)
{	static float input [BUFFER_LEN], output [BUFFER_LEN] ;

	SRC_DATA	src_data ;

	int ch, k, error, ignore ;

	printf ("\tmulti_channel_test     (%d channel%c) ............. ", channel_count, channel_count > 1 ? 's' : ' ') ;
	fflush (stdout) ;

	/* Choose a converstion ratio < 1.0. */
	src_data.src_ratio = 0.95 ;

	src_data.data_in = input ;
	src_data.input_frames = BUFFER_LEN / channel_count ;

	for (k = 0 ; k < src_data.input_frames ; k++)
		for (ch = 0 ; ch < channel_count ; ch++)
			src_data.data_in [channel_count * k + ch] = CHANNEL_FUNC (ch) ;

	src_data.data_out = output ;
	src_data.output_frames = BUFFER_LEN / channel_count ;

	if ((error = src_simple (&src_data, converter, channel_count)))
	{	printf ("\n\nLine %d : %s\n\n", __LINE__, src_strerror (error)) ;
		exit (1) ;
		} ;

	if (fabs (src_data.output_frames_gen - src_data.src_ratio * src_data.input_frames) > 2)
	{	printf ("\n\nLine %d : bad output data length %ld should be %d.\n", __LINE__,
					src_data.output_frames_gen, (int) floor (src_data.src_ratio * src_data.input_frames)) ;
		printf ("\tsrc_ratio  : %.4f\n", src_data.src_ratio) ;
		printf ("\tinput_len  : %ld\n", src_data.input_frames) ;
		printf ("\toutput_len : %ld\n\n", src_data.output_frames_gen) ;
		exit (1) ;
		} ;

	ignore = src_data.output_frames_gen / 8 ;

	for (k = ignore ; k < src_data.output_frames_gen - ignore ; k++)
		for (ch = 0 ; ch < channel_count ; ch++)
			if (fabs (src_data.data_out [channel_count * k + ch] - CHANNEL_FUNC (ch)) > 1.5e-5)
			{	printf ("\n\nLine %d    %5d : %f -> %f   (%f difference)\n", __LINE__, k - ignore,
						CHANNEL_FUNC (ch), src_data.data_out [channel_count * k + ch],
						fabs (src_data.data_out [channel_count * k + ch] - CHANNEL_FUNC (ch))) ;
				save_oct_data ("multi_channel_test.dat", src_data.data_in, src_data.input_frames,
							src_data.data_out, src_data.output_frames_gen) ;
				exit (1) ;
				} ;

	puts ("ok") ;

	return ;
} /* multi_channel_test */

/*==============================================================================
*/

#if NEW_CODE	
static void
simple_multi_channel_test (int converter, int channel_count, double target_snr)
{	static float buffer1 [BUFFER_LEN], buffer2 [BUFFER_LEN] ;

	SRC_DATA	src_data ;

	double	freq, snr ;
	int		ch, error, frames ;

	printf ("\tsimple_multi_channel_test  (%d channel%c) ......... ", channel_count, channel_count > 1 ? 's' : ' ') ;
	fflush (stdout) ;

	frames = MIN (ARRAY_LEN (buffer1) / channel_count, 1 << 15) ;

	/* Calculate channel_count seprate sine waves. */
	for (ch = 0 ; ch < channel_count ; ch++)
	{	freq = 440.0 / 44100.0 + (13.456789 / 44100.0 * ch) ;
		gen_windowed_sines (buffer2 + ch * frames, frames, &freq, 1) ;
		} ;

	/* Interleave the data in preparation for SRC. */
	interleave_data (buffer2, buffer1, frames, channel_count) ;

	/* Choose a converstion ratio < 1.0. */
	src_data.src_ratio = 0.95 ;

	src_data.data_in = buffer1 ;
	src_data.input_frames = frames ;

	src_data.data_out = buffer2 ;
	src_data.output_frames = frames ;

	if ((error = src_simple (&src_data, converter, channel_count)))
	{	printf ("\n\nLine %d : %s\n\n", __LINE__, src_strerror (error)) ;
		exit (1) ;
		} ;

	if (fabs (src_data.output_frames_gen - src_data.src_ratio * src_data.input_frames) > 2)
	{	printf ("\n\nLine %d : bad output data length %ld should be %d.\n", __LINE__,
					src_data.output_frames_gen, (int) floor (src_data.src_ratio * src_data.input_frames)) ;
		printf ("\tsrc_ratio  : %.4f\n", src_data.src_ratio) ;
		printf ("\tinput_len  : %ld\n", src_data.input_frames) ;
		printf ("\toutput_len : %ld\n\n", src_data.output_frames_gen) ;
		exit (1) ;
		} ;

	frames = src_data.output_frames_gen ;

	/* De-interleave data so SNR can be calculated for each channel. */
	deinterleave_data (buffer2, buffer1, frames, channel_count) ;

	for (ch = 0 ; ch < channel_count ; ch++)
	{	snr = calculate_snr (buffer1 + ch * frames, frames) ;
		if (snr < target_snr)
		{	printf ("\n\nLine %d: channel %d snr %f should be %f\n", __LINE__, ch, snr, target_snr) ;
			save_oct_data ("output.dat", buffer1 + ch * frames, frames, NULL, 0) ;
			exit (1) ;
			} ;
		} ;

	puts ("ok") ;

	return ;
} /* simple_multi_channel_test */

/*==============================================================================
*/

static void
process_multi_channel_test (int converter, int channel_count, double target_snr)
{	static float buffer1 [BUFFER_LEN], buffer2 [BUFFER_LEN] ;

	SRC_STATE	*src_state ;
	SRC_DATA	src_data ;

	double	freq, snr ;
	int		ch, error, frames, current_in, current_out ;

	printf ("\tprocess_multi_channel_test (%d channel%c) ......... ", channel_count, channel_count > 1 ? 's' : ' ') ;
	fflush (stdout) ;

	frames = MIN (ARRAY_LEN (buffer1) / channel_count, 1 << 15) ;

	/* Calculate channel_count seprate sine waves. */
	for (ch = 0 ; ch < channel_count ; ch++)
	{	freq = 567.89 / 44100.0 + (1234.56789 / 44100.0 * ch) ;
		gen_windowed_sines (buffer2 + ch * frames, frames, &freq, 1) ;
		} ;

	/* Interleave the data in preparation for SRC. */
	interleave_data (buffer2, buffer1, frames, channel_count) ;

	/* Perform sample rate conversion. */
	if ((src_state = src_new (converter, channel_count, &error)) == NULL)
	{	printf ("\n\nLine %d : src_new() failed : %s\n\n", __LINE__, src_strerror (error)) ;
		exit (1) ;
		} ;

	src_data.end_of_input = 0 ; /* Set this later. */

	/* Choose a converstion ratio < 1.0. */
	src_data.src_ratio = 1.0 ;

	src_data.data_in = buffer1 ;
	src_data.data_out = buffer2 ;

	current_in = current_out = 0 ;

	while (1)
	{	src_data.input_frames	= MIN (BLOCK_LEN, frames - current_in) ;
		src_data.output_frames	= MIN (BLOCK_LEN, frames - current_out) ;

		if ((error = src_process (src_state, &src_data)))
		{	printf ("\n\nLine %d : %s\n\n", __LINE__, src_strerror (error)) ;
			exit (1) ;
			} ;

		if (src_data.end_of_input && src_data.output_frames_gen == 0)
			break ;

		current_in	+= src_data.input_frames_used ;
		current_out += src_data.output_frames_gen ;

		src_data.data_in	+= src_data.input_frames_used * channel_count ;
		src_data.data_out	+= src_data.output_frames_gen * channel_count ;

		src_data.end_of_input = (current_in >= frames) ? 1 : 0 ;
		} ;

	src_state = src_delete (src_state) ;

	if (fabs (current_out - src_data.src_ratio * current_in) > 2)
	{	printf ("\n\nLine %d : bad output data length %d should be %d.\n", __LINE__,
					current_out, (int) floor (src_data.src_ratio * current_in)) ;
		printf ("\tsrc_ratio  : %.4f\n", src_data.src_ratio) ;
		printf ("\tinput_len  : %d\n", frames) ;
		printf ("\toutput_len : %d\n\n", current_out) ;
		exit (1) ;
		} ;

	frames = current_out ;

	/* De-interleave data so SNR can be calculated for each channel. */
	deinterleave_data (buffer2, buffer1, frames, channel_count) ;

	for (ch = 0 ; ch < channel_count ; ch++)
	{	snr = calculate_snr (buffer1 + ch * frames, frames) ;
		if (snr < target_snr)
		{	printf ("\n\nLine %d: channel %d snr %f should be %f\n\n", __LINE__, ch, snr, target_snr) ;
			save_oct_data ("output.dat", buffer1 + ch * frames, frames, NULL, 0) ;
			exit (1) ;
			} ;
		} ;

	puts ("ok") ;

	return ;
} /* process_multi_channel_test */

#endif
