///////////////////////////////////////////////////////////////////////////////
// Project    : libswf v0.1
// Module     : nc_swf_encode.cpp
// Author     : Eduardo Aguiar - 2000.08.27
// Description: main module
///////////////////////////////////////////////////////////////////////////////
#include <errno.h>
#include <string.h>
#include "nc_swf_encode.hpp"

nc_swf_encode::nc_swf_encode(void){
   i_errno = 0;
   i_errtext = "";
   fileswf = NULL;
   i_pos_framecount = 0;
   i_framecount = 0;
   i_encodebitmask = 0;
   i_encodebitchar = 0;
   i_tagid = -1;
}
nc_swf_encode::~nc_swf_encode(void){
   if (fileswf != NULL) close();
}



///////////////////////////////////////////////////////////////////////////////
// Method     : create
// Description: Creates a valid .swf file
// Parameters : filename, version (1-4), width, height (TWIPS), framerate
// Returns    : none
// 1. Both width and height must be given em TWIPS. Each pixel is equal to 20
//    twips
// 2. Framerate is integer
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::create(const typestring& filename, int version, int width,
                           int height, int framerate){

   ////////////////////////////////////////////////////////////////////////////
   // Check if fswf is already opened
   ////////////////////////////////////////////////////////////////////////////
   if (fileswf != NULL){
      i_errno = -1;
      i_errtext = "File already opened";
      return;
   }

   ////////////////////////////////////////////////////////////////////////////
   // Format filename
   ////////////////////////////////////////////////////////////////////////////
   typestring l_filename;
   if (strlower(strright(filename,4)) == ".swf")
      l_filename = filename;
   else
      l_filename = filename + ".swf";

   ////////////////////////////////////////////////////////////////////////////
   // Try to create .swf
   ////////////////////////////////////////////////////////////////////////////
   fileswf = fopen(!l_filename,"wb");
   if (fileswf == NULL){
      i_errno = errno;
      i_errtext = strerror(errno);
      return;
   }

   ////////////////////////////////////////////////////////////////////////////
   // Writes file header
   ////////////////////////////////////////////////////////////////////////////
   encodeUI8('F');                 // signature FWS
   encodeUI8('W');
   encodeUI8('S');
   encodeUI8(byte(version));       // version
   encodeUI32(0);                  // filesize (0 until close)
   encodeRECT(0,0,width,height);
   encodeUI8(0);                   // framerate
   encodeUI8(byte(framerate));

   ////////////////////////////////////////////////////////////////////////////
   // Saves framecount position and set it on 0 by now
   ////////////////////////////////////////////////////////////////////////////
   fgetpos(fileswf,&i_pos_framecount);
   encodeUI16(0);                  

   ////////////////////////////////////////////////////////////////////////////
   // Returns ok
   ////////////////////////////////////////////////////////////////////////////
   i_errno = 0;
   i_errtext = "";
}

///////////////////////////////////////////////////////////////////////////////
// Method     : close
// Description: Closes .swf file recording filesize and framecount
// Parameters : none
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::close(void){

   ////////////////////////////////////////////////////////////////////////////
   // Verifies if it is a valid fileswf
   ////////////////////////////////////////////////////////////////////////////
   if (fileswf == NULL){
      i_errno = -1;
      i_errtext = "File not opened";
      return;
   }

   ////////////////////////////////////////////////////////////////////////////
   // Writes filesize
   ////////////////////////////////////////////////////////////////////////////
   fpos_t pos;
   fgetpos(fileswf,&pos);
   fseek(fileswf,4,SEEK_SET);
   encodeUI32(dword(pos));

   ////////////////////////////////////////////////////////////////////////////
   // Writes framecount
   ////////////////////////////////////////////////////////////////////////////
   fseek(fileswf,i_pos_framecount,SEEK_SET);
   encodeUI16(word(i_framecount));
   fclose(fileswf);
   fileswf = NULL;

   i_errno = 0;
   i_errtext = "";
}
   



//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
//                               PRIVATES
//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

///////////////////////////////////////////////////////////////////////////////
// Method     : encodeUI8
// Description: encodes a byte
// Parameters : byte (8 bits)
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodeUI8(byte b){

   ////////////////////////////////////////////////////////////////////////////
   // Verifies if we are writing a TAG
   ////////////////////////////////////////////////////////////////////////////
   if (i_tagid != -1){
      tag.push(b);
      i_errno = 0;
      i_errtext = "";
      return;
   }

   ////////////////////////////////////////////////////////////////////////////
   // If not, tries to write directly to file
   ////////////////////////////////////////////////////////////////////////////
   if (fputc(b,fileswf) != b){
      i_errno = errno;
      i_errtext = strerror(errno);
   }
   else {
      i_errno = 0;
      i_errtext = "";
   }
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodeUI16
// Description: encodes a word
// Parameters : word (16 bits)
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodeUI16(word value){
   encodeUI8(byte(value & 0xff));
   if (i_errno) return;
   encodeUI8(byte((value>>8) & 0xff));
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodeUI32
// Description: encodes a dword
// Parameters : dword (32 bits)
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodeUI32(dword value){
   encodeUI8(byte(value & 0xff));
   if (i_errno) return;
   encodeUI8(byte((value>>8) & 0xff));
   if (i_errno) return;
   encodeUI8(byte((value>>16) & 0xff));
   if (i_errno) return;
   encodeUI8(byte((value>>24) & 0xff));
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodeRGB
// Description: encodes a color in RGB format
// Parameters : R,G,B components
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodeRGB(byte r,byte g,byte b){
   encodeUI8(r);
   if (i_errno) return;
   encodeUI8(g);
   if (i_errno) return;
   encodeUI8(b);
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodeRGBA
// Description: encodes a color in RGBA format
// Parameters : R,G,B,A components
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodeRGBA(byte r,byte g,byte b,byte a){
   encodeUI8(r);
   if (i_errno) return;
   encodeUI8(g);
   if (i_errno) return;
   encodeUI8(b);
   if (i_errno) return;
   encodeUI8(a);
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodeblock
// Description: encodes a block
// Parameters : byte *, size
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodeblock(byte *buffer, int size){

   ////////////////////////////////////////////////////////////////////////////
   // Verifies if we are writing a TAG
   ////////////////////////////////////////////////////////////////////////////
   if (i_tagid != -1){
      while(size){
         tag.push(*buffer);
         size--;
      }
      i_errno = 0;
      i_errtext = "";
   }

   ////////////////////////////////////////////////////////////////////////////
   // If not, tries to write to file
   ////////////////////////////////////////////////////////////////////////////
   else {
      fwrite(buffer,1,size,fileswf);
      if (ferror(fileswf)){
         i_errno = ferror(fileswf);
         i_errtext = strerror(i_errno);
         clearerr(fileswf);
      }
      else {
         i_errno = 0;
         i_errtext = "";
      }
   }
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodeRECT
// Description: encodes a RECT structure
// Parameters : ix,iy,fx,fy
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodeRECT(int ix,int iy,int fx,int fy){
   int bits = getbitsofvalue(ix);
   if (bits < getbitsofvalue(iy)) bits = getbitsofvalue(iy);
   if (bits < getbitsofvalue(fx)) bits = getbitsofvalue(fx);
   if (bits < getbitsofvalue(fy)) bits = getbitsofvalue(fy);
   bits = bits + 1;
   encodebitstart();
   encodeUB(5,bits);
   encodeSB(bits,ix);
   encodeSB(bits,fx);
   encodeSB(bits,iy);
   encodeSB(bits,fy);
   encodebitend();
}

///////////////////////////////////////////////////////////////////////////////
// Method     : getbitsofvalue
// Description: Returns mininum amount of bits needed to encode a value
// Parameters : value
// Returns    : amount of bits
///////////////////////////////////////////////////////////////////////////////
int nc_swf_encode::getbitsofvalue(int value){
   dword uval = abs(value);
   dword mask = (1<<31);
   byte  bits = 32;
   while(((uval & mask) == 0)&&(mask > 0)){
      bits = bits - 1;
      mask = (mask>>1);
   }
   return bits;
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodebitstart
// Description: Starts the encoding bit block
// Parameters : none
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodebitstart(void){
   i_encodebitmask = 128;
   i_encodebitchar = 0;
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodebitend
// Description: Ends the encoding bit block
// Parameters : none
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodebitend(void){
   if (i_encodebitmask != 128){
      encodeUI8(i_encodebitchar);
      i_encodebitmask = 128;
      i_encodebitchar = 0;
   }
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodebit
// Description: Encodes a bit in a bit block
// Parameters : bit (0/1)
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodebit(byte bit){
   if (bit != 0) i_encodebitchar = (i_encodebitchar | i_encodebitmask);
   i_encodebitmask = (i_encodebitmask>>1);

   if (i_encodebitmask == 0){
      encodeUI8(i_encodebitchar);
      i_encodebitmask = 128;
      i_encodebitchar = 0;
   }
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodeUB
// Description: Encodes a dword using amount of bits given
// Parameters : bits, value
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodeUB(int bits, dword value){
   dword mask = (1 << (bits-1));
   while(mask > 0){
      encodebit((value & mask) != 0);
      mask = (mask>>1);
   }
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodeSB
// Description: Encodes an integer using amount of bits given
// Parameters : bits, value
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodeSB(int bits, int value){
   dword mask  = (1 << (bits-2));

   if (value < 0){
      encodebit(1);
      value = 0 - value;
   }
   else{
      encodebit(0);
   }

   while(mask > 0){
      encodebit((value & mask) != 0);
      mask = (mask>>1);
   }
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodetagstart
// Description: Starts to encode a .swf TAG
// Parameters : tagid
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodetagstart(dword tagid){

   ////////////////////////////////////////////////////////////////////////////
   // Checks if there is a tag already opened
   ////////////////////////////////////////////////////////////////////////////
   if (i_tagid != -1){
      i_errno = -1;
      i_errtext = "Already encoding a tag";
   }
   else {
      i_tagid = tagid;
      i_errno = 0;
      i_errtext = "";
   }
}

///////////////////////////////////////////////////////////////////////////////
// Method     : encodetagend
// Description: Terminate to encode a .swf TAG
// Parameters : none
// Returns    : none
///////////////////////////////////////////////////////////////////////////////
void nc_swf_encode::encodetagend(void){

   ////////////////////////////////////////////////////////////////////////////
   // Checks if there is not a tag already opened
   ////////////////////////////////////////////////////////////////////////////
   if (i_tagid == -1){
      i_errno = -1;
      i_errtext = "Was not encoding a tag";
      return;
   }

   ////////////////////////////////////////////////////////////////////////////
   // Writes tag header
   ////////////////////////////////////////////////////////////////////////////
   int tagid   = i_tagid;
   int tagsize = tag.size();
   i_tagid = -1;

   if (tagsize < 0x3f){
      encodeUI16((tagid<<6) | tagsize);
   }
   else {
      encodeUI16((tagid<<6) | 0x3f);
      encodeUI32(tagsize);
   }

   ////////////////////////////////////////////////////////////////////////////
   // Copies a tag to buffer, writes to file and exit
   ////////////////////////////////////////////////////////////////////////////
   byte buffer[tagsize];
   int i;
   for(i = 0; i < tagsize;i++){
      buffer[i] = tag.front();
      tag.pop();
   }

   encodeblock(buffer,tagsize);
}
