/* xml++.cc
 * libxml++ and this file are copyright (C) 2000 by Ari Johnson, and
 * are covered by the GNU Lesser General Public License, which should be
 * included with libxml++ as the file COPYING.
 */

#include "xml++.h"

static XMLNode *readnode(xmlNodePtr);
static void writenode(xmlDocPtr, XMLNode *, xmlNodePtr, int = 0);

XMLTree::XMLTree(const XMLTree *from) {
  _filename = from->filename();
  _root = new XMLNode(from->root());
  _compression = from->compression();
  _initialized = true;
}

XMLTree::~XMLTree() {
  if(_initialized && _root)
    delete _root;
}

const std::string & XMLTree::set_encoding(const std::string & e){
	if(xmlFindCharEncodingHandler(e.c_str()))
		_encoding = e;
	return _encoding;
}

int XMLTree::set_compression(int c) {
  if(c > 9)
    c = 9;

  if(c < 0)
    c = 0;

  _compression = c;

  return _compression;
}

bool XMLTree::read(void) {
  xmlDocPtr doc;

  if(_root) {
    delete _root;
    _root = NULL;
  }

  xmlKeepBlanksDefault(0);
  xmlLineNumbersDefault(1);

  doc = xmlParseFile(_filename.c_str());
  if(!doc) {
    _initialized = false;
    return false;
  }

  _root = readnode(xmlDocGetRootElement(doc));
  xmlFreeDoc(doc);
  _initialized = true;

  return true;
}

bool XMLTree::read_buffer(const std::string &buffer) {
  xmlDocPtr doc;

  _filename = "";

  if(_root) {
    delete _root;
    _root = NULL;
  }

  xmlLineNumbersDefault(1);

  doc = xmlParseMemory((char *) buffer.c_str(), buffer.length());
  if(!doc) {
    _initialized = false;
    return false;
  }

  _root = readnode(xmlDocGetRootElement(doc));
  xmlFreeDoc(doc);
  _initialized = true;

  return true;
}

bool XMLTree::write(void) const {
  xmlDocPtr doc;
  XMLNodeList children;
  int result;

  xmlKeepBlanksDefault(0);
  doc = xmlNewDoc((xmlChar *) "1.0");
  xmlSetDocCompressMode(doc, _compression);
  writenode(doc, _root, doc->children, 1);
  if(_encoding != std::string()) {
	  result = xmlSaveFormatFileEnc(_filename.c_str(), doc, _encoding.c_str(), 1);
  } else {
	  result = xmlSaveFormatFile(_filename.c_str(), doc, 1);
  }
  xmlFreeDoc(doc);

  if(result == -1)
    return false;

  return true;
}

const std::string & XMLTree::write_buffer(void) const {
  static std::string retval;
  char *ptr;
  int len;
  xmlDocPtr doc;
  XMLNodeList children;

  xmlKeepBlanksDefault(0);
  doc = xmlNewDoc((xmlChar *) "1.0");
  xmlSetDocCompressMode(doc, _compression);
  writenode(doc, _root, doc->children, 1);
  xmlDocDumpMemory(doc, (xmlChar **) &ptr, &len);
  xmlFreeDoc(doc);

  retval = ptr;

  free(ptr);

  return retval;
};

XMLNode::XMLNode(const std::string &n)
 : _name(n), _is_content(false), _content(std::string()), _line(0) {
  if(_name.empty())
    _initialized = false;
  else
    _initialized = true;
}

XMLNode::XMLNode(const std::string &n, int line)
 : _name(n), _is_content(false), _content(std::string()), _line(line) {
  if(_name.empty())
    _initialized = false;
  else
    _initialized = true;
}

XMLNode::XMLNode(const std::string &n, const std::string &c)
 : _name(std::string()), _is_content(true), _content(c), _line(0) {
  _initialized = true;
}

XMLNode::XMLNode(const XMLNode *from) {
  XMLPropertyList props;
  XMLPropertyIterator curprop;
  XMLNodeList nodes;
  XMLNodeIterator curnode;

  _name = from->name();
  set_content(from->content());

  props = from->properties();
  for(curprop = props.begin(); curprop != props.end(); curprop++)
    add_property((*curprop)->name(), (*curprop)->value());

  nodes = from->children();
  for(curnode = nodes.begin(); curnode != nodes.end(); curnode++)
    add_child(new XMLNode(*curnode));
}

XMLNode::~XMLNode() {
  XMLNodeIterator curchild;
  XMLPropertyIterator curprop;

  for(curchild = _children.begin(); curchild != _children.end(); curchild++)
    delete *curchild;

  for(curprop = _proplist.begin(); curprop != _proplist.end(); curprop++)
    delete *curprop;
}

const std::string & XMLNode::set_content(const std::string &c) {
  if(c.empty())
    _is_content = false;
  else
    _is_content = true;

  _content = c;

  return _content;
}

const XMLNodeList & XMLNode::children(const std::string &n) const {
  static XMLNodeList retval;
  XMLNodeConstIterator cur;

  if(n.length() == 0)
    return _children;

  retval.erase(retval.begin(), retval.end());

  for(cur = _children.begin(); cur != _children.end(); cur++)
    if((*cur)->name() == n)
      retval.insert(retval.end(), *cur);

  return retval;
}

XMLNode *XMLNode::add_child(const std::string &n) {
  XMLNode *tmp;

  tmp = new XMLNode(n);

  return add_child(tmp);
}

XMLNode *XMLNode::add_child(XMLNode *n) {
  if(!n)
    return NULL;

  _children.insert(_children.end(), n);
  return n;
}

XMLNode *XMLNode::add_content(const std::string &c) {
  XMLNode *tmp;

  tmp = new XMLNode(std::string(), c);

  return add_child(tmp);
}

void XMLNode::remove_child(XMLNode *n) {
  if(n)
    _children.remove(n);
}

XMLProperty *XMLNode::property(const std::string &n) {
  if(_prophash.find(n) == _prophash.end())
    return NULL;

  return _prophash[n];
}

XMLProperty *XMLNode::add_property(const std::string &n, const std::string &v) {
  XMLProperty *tmp;

  if(_prophash.find(n) != _prophash.end())
    return NULL;

  tmp = new XMLProperty(n, v);

  if(!tmp)
    return NULL;

  _prophash[tmp->name()] = tmp;
  _proplist.insert(_proplist.end(), tmp);

  return tmp;
}

void XMLNode::remove_property(const std::string &n) {
  if(_prophash.find(n) != _prophash.end()) {
    _proplist.remove(_prophash[n]);
    _prophash.erase(n);
  }
}

static XMLNode *readnode(xmlNodePtr node) {
  std::string name, content;
  xmlNodePtr child;
  XMLNode *tmp;
  xmlAttrPtr attr;

  if(node->name)
    name = (char *) node->name;
  else
    name = std::string();

  tmp = new XMLNode(name, XML_GET_LINE(node));

  for(attr = node->properties; attr; attr = attr->next) {
    name = (char *) attr->name;
    content = "";
    if(attr->children)
      content = (char *) attr->children->content;
    tmp->add_property(name, content);
  }

  if(XML_GET_CONTENT(node))
    tmp->set_content((char *) XML_GET_CONTENT(node));
  else
    tmp->set_content(std::string());

  for(child = node->children; child; child = child->next)
      tmp->add_child(readnode(child));

  return tmp;
}

static void writenode(xmlDocPtr doc, XMLNode *n, xmlNodePtr p, int root) {
  XMLPropertyList props;
  XMLPropertyIterator curprop;
  XMLNodeList children;
  XMLNodeIterator curchild;
  xmlNodePtr node;

  if(root)
    node = doc->children = xmlNewDocNode(doc, NULL,
				(xmlChar *) n->name().c_str(),
				(xmlChar *) n->content().c_str());
  else
    node = xmlNewChild(p, NULL, (xmlChar *) n->name().c_str(),
			(xmlChar *) n->content().c_str());

  if(n->is_content()) {
    node->type = XML_TEXT_NODE;
    xmlNodeSetContentLen(node, (const xmlChar *) n->content().c_str(),
				n->content().length());
  }

  props = n->properties();
  for(curprop = props.begin(); curprop != props.end(); curprop++)
    xmlSetProp(node, (xmlChar *) (*curprop)->name().c_str(),
		(xmlChar *) (*curprop)->value().c_str());

  children = n->children();
  for(curchild = children.begin(); curchild != children.end(); curchild++)
    writenode(doc, *curchild, node);
}

